(function(factory) {
	if (!jQuery) {
		throw 'cvWizard depends on jQuery and seems to be missing...';
	} else {
		factory(jQuery);
	}
}(function($) {
	"use strict";

	$.fn.cvWizard = function(config) {
		return this.each(function() {
			var $this = $(this);
			$this.data('cvWizard', new Wizard(this, config));
		});
	};
}));

function Wizard(element, _config) {
	this.$element = $(element);
	this.config = _config;
	this.$wrapper = null;
	this.$formWrapper = null;
	//each step contains step no, html to be loaded, validation function and an initialization function
	this.steps = {};

	this.theme = this.config.theme;

	this.userInput = {};
	this.aggregatedInput = {};
	this.nextButton = null;
	this.prevButton = null;
	this.submitButton = null;
	this.currentStep = 1;
	this.submitCallback = null;
	this.init();
}

Wizard.prototype.Themes = {
	simple : 0,
	collapse : 1,
	slide : 2
};

Wizard.prototype.Constants = {
	WRAPPER : "cvWizard-wrapper",
	FORM_WRAPPER : "cvWizard-form-wrapper",
	STEP_WRAPPER : "cvWizard-step-wrapper",
	WIZARD_STEP : "cvWizard-step",
	BTN_NEXT : "btn-next",
	BTN_PREV : "btn-prev",
	BTN_SUBMIT : "btn-submit",
	BTN_CANCEL : "btn-cancel",
	ANIMATION_DURATION : 500
};

Wizard.prototype.templates = {
    getNextButtonTemplate: function() {
      return $("<button tabindex = '0' class='btn btn-primary " + Wizard.prototype.Constants.BTN_NEXT +"'>" + localMsg.Next + "</button>");
	},
    getPrevButtonTemplate: function() {
      return $("<button tabindex = '0' class='btn btn-default " + Wizard.prototype.Constants.BTN_PREV +"'>" + localMsg.Previous + "</button>");
	},
		getSubmitButtonTemplate: function() {
			return $("<button tabindex= '0' class='btn btn-primary " + Wizard.prototype.Constants.BTN_SUBMIT +"'>" + localMsg.Submit + "</button>");
	},
		getCancelButtonTemplate: function() {
			return $("<button tabindex= '0' class='btn btn-default " + Wizard.prototype.Constants.BTN_CANCEL + "'>" + localMsg.Cancel + "</button>");
	},
	getCollapseTab : function() {
		return $("<div class='step-content-header'><label>collapsed</label></div>");
	}
};

//alternateForm - To be shown when a step is collapsed e.t.c
Wizard.prototype.step = function(config) {
	var id = config.id;
	var form = config.form;
	var alternateForm = config.alternateForm;
	var validator = config.validator;
	var initializer = config.initializer;
	var fetcher = config.fetcher;

	var stepData = {};

	if (form != undefined)
		stepData.form = form;
	if (alternateForm != undefined)
		stepData.alternateForm = alternateForm;
	if (typeof validator == "function")
		stepData.validator = validator;
	if (typeof initializer == "function")
		stepData.initializer = initializer;
	if (typeof fetcher == "function")
		stepData.fetcher = fetcher;

	this.steps[id] = stepData;
};

Wizard.prototype.init = function() {
	this.$wrapper = $("<div class='" + this.Constants.WRAPPER + "' margin-top:25px;></div>");
	this.$formWrapper = $("<div class='" + this.Constants.FORM_WRAPPER + "'></div>");
	this.$wrapper.append(this.$formWrapper);
	this.createOuterWrapper();
	this.attachListeners();
	this.$element.append(this.$wrapper);
};

Wizard.prototype.createOuterWrapper = function() {
	this.nextButton = this.templates.getNextButtonTemplate();
	this.prevButton = this.templates.getPrevButtonTemplate();
	this.submitButton = this.templates.getSubmitButtonTemplate();
	this.cancelButton = this.templates.getCancelButtonTemplate();
	switch (this.theme) {
	case this.Themes.simple:
	case this.Themes.slide:
		this.$wrapper.append(this.prevButton);
		this.$wrapper.append(this.nextButton);
		this.$wrapper.append(this.submitButton);
		break;
	case this.Themes.collapse:
		break;
	}
};

Wizard.prototype.attachListeners = function() {
	var self = this;

	this.$formWrapper.off("click", ".btn-next").on("click",
			".btn-next",
			function() {
				var validator = self.steps[self.currentStep].validator;
				var fetcher = self.steps[self.currentStep].fetcher;
				var $stepHolder = self.$formWrapper.find("." + self.Constants.WIZARD_STEP + "[data-step='" +
						self.currentStep + "']");
				if (validator) {
					var userInput = fetcher(self.currentStep, $stepHolder);
					validator(self.currentStep, $stepHolder, self.aggregatedInput, function(valid) {
						if (valid === true) {
							self.userInput[self.currentStep] = userInput;
							self.calculateAggregatedInput();
							self.currentStep++;
							self.draw(self.currentStep, false);
						} else
							self.handleFailedValidationTransition(self.currentStep);
					});
				}
			});

	this.$formWrapper.off("click", ".btn-prev").on("click", ".btn-prev", function() {
		self.currentStep--;
		self.draw(self.currentStep, true);
	});

	this.$formWrapper.off("click", ".btn-cancel").on("click", ".btn-cancel", function() {
		self.cancelCallback();
	});

	this.$formWrapper.off("click", ".step-content-header").on("click", ".step-content-header", function() {
		var parent = $(this).parent();
		var stepId = parent.attr("data-step");
		var validator = self.steps[self.currentStep].validator;
		var prev = true;
		self.currentStep = parseInt(stepId);
		self.draw(self.currentStep, prev);

	});

	this.$formWrapper
			.off("keydown", "input")
			.on("keydown",
					"input",
					function(e) {
						if (e.keyCode === 13 &&
								($(this).data("proceedToNextStep") === undefined || $(this).data("proceedToNextStep") === true)) {
							if (!self.nextButton.is(":hidden")) {
								self.nextButton.trigger("click");
							}
							e.preventDefault();
							e.stopPropagation();
						}
					});

	this.$formWrapper.off("click", ".btn-submit").on("click",
			".btn-submit",
			function() {
				var validator = self.steps[self.currentStep].validator;
				var fetcher = self.steps[self.currentStep].fetcher;
				var $stepHolder = self.$formWrapper.find("." + self.Constants.WIZARD_STEP + "[data-step='" +
						self.currentStep + "']");
				if (validator) {
					var userInput = fetcher(self.currentStep, $stepHolder);
					validator(self.currentStep, self.$wrapper, self.aggregatedInput, function(valid) {
						if (valid === true) {
							self.userInput[self.currentStep] = userInput;
							self.calculateAggregatedInput();
							self.submitCallback(self.aggregatedInput);
						} else
							self.handleFailedValidationTransition(self.currentStep);
					});
				}
			});
};

Wizard.prototype.addInitializer = function(stepId, initializer) {
	if (stepId === undefined)
		return;

	this.steps[stepId].initializer = initializer;
};

Wizard.prototype.draw = function(stepId, prev) {
	this.handleWrapperTransition(stepId, prev);
	this.handlerFormWrapperTransition(stepId, prev);
	this.updateButtons(stepId);

	/*
	 * Once the form is on the DOM, populate it or do whatever initializing you want with it
	 */
	if (typeof this.steps[stepId].initializer == "function")
		this.steps[stepId].initializer(this.currentStep, this.$formWrapper);
};

Wizard.prototype.updateButtons = function(stepId) {
	this.submitButton.hide();
	if ($.isEmptyObject(this.steps[stepId + 1])) {
		this.nextButton.hide();
		this.submitButton.show();
	} else
		this.nextButton.show();

	if ($.isEmptyObject(this.steps[stepId - 1]))
		this.prevButton.attr("disabled", true);
	else
		this.prevButton.attr("disabled", false);
};

Wizard.prototype.setSubmit = function(callback) {
	this.submitCallback = callback;
};

/*
 * Animate and change the outer UI when going from one step to another Prev is a bool. If true, it means we
 * are going backwards - user pressed the 'previous' button
 */
Wizard.prototype.handleWrapperTransition = function(stepId, prev) {

};

/*
 * Decides what to do with the elements already in the form when going from on step to another. Depending on
 * the theme selected, either removes the previous step, collapses the previous step, changes the style of the
 * previous step e.t.c*
 * 
 * prev is a boolean
 */

Wizard.prototype.handlerFormWrapperTransition = function(stepId, prev) {
	var self = this;
	var alreadyExpanded = this.$formWrapper.find(".step-content").not(":hidden");
	var expandedStepId = alreadyExpanded.parent().attr("data-step");

	//The form of the the step that is being drawn - if it is already in the DOM
	var currentFormInDom = this.$formWrapper.find("." + this.Constants.WIZARD_STEP + "[data-step='" + stepId + "']");

	/*
	 * Why clone instead of using directly? If using directly, suppose we change the color of the number
	 * circle to green. The user then goes back from step C to A, and we remove all the subsequent steps (B
	 * and C) from the dom. Then we append disabled headers using addCollapsedStepHeaders(). But header for
	 * step B is already green even though we are only at step A. This will be because we had turned its
	 * header green while the user was at step B and we did not change it back when we removed the header from
	 * DOM (the changes are still there in the this.step[id].form variable). Possible solutions are - a) clone
	 * the form before changing it b) Manually undo all changes made. a) is easier.
	 */
	var currentForm = this.steps[stepId].form.clone();

	switch (this.theme) {
	case this.Themes.simple:
		this.$formWrapper.html(this.steps[stepId].form);
		break;
	case this.Themes.slide:
		this.steps[stepId].form.hide();
		this.$formWrapper.find("[data-step]").remove();
		this.$formWrapper.append(this.steps[stepId].form);
		this.steps[stepId].form.show("slide", {
			direction : "left"
		}, this.Constants.ANIMATION_DURATION);
		break;
	case this.Themes.collapse:
		this.$formWrapper.find(".button-holder").empty();
		// this.removeCollapsedStepHeaders();
		/*
		 * This get executed only when we press 'next'. Hence we should collapse the holder of the previous
		 * step
		 */
		/*
		 * stepId = 1 and prev = false for the initial draw call from dsController.js. Prev will be undefined
		 * and step will be 1. However, we cannot 'come from step 0'. So handling the special case here by
		 * adding an if check
		 */
		if (!prev) {

			if (stepId > 1) {
				this.collapseForm(alreadyExpanded.parent(), expandedStepId);
			}

			if (currentFormInDom.length > 0) {
				currentFormInDom.show();
				this.expandForm(currentFormInDom);
			} else {
				//Hide the yet unrendered form
				currentForm.hide();
				this.setStepStatusClass(currentForm.find(".number-circle"), "active");

				//Add our hidden form to the DOM
				this.$formWrapper.append(currentForm);
				currentForm.find(".step-content").prepend(this.steps[stepId].alternateForm.clone());
				//Show the hidden form
				currentForm.show();
				this.focusInput(currentForm);
			}
		} else {
			//Close whatever step is open now and expand the current step
			if (currentFormInDom.length !== 0) {
				this.collapseForm(alreadyExpanded.parent(), expandedStepId);
				this.expandForm(currentFormInDom);
			}
		}
		var buttonHolder = this.$formWrapper.find("." + this.Constants.WIZARD_STEP + "[data-step='" + stepId +
				"'] .button-holder");
		buttonHolder.append(this.prevButton);
		buttonHolder.append(this.nextButton);
		buttonHolder.append(this.submitButton);
		buttonHolder.append(this.cancelButton);
		this.$formWrapper.find("." + this.Constants.WIZARD_STEP + "[data-step='" + stepId + "'] .step-content")
				.append(buttonHolder);
		this.addCollapsedStepHeaders(stepId);
		break;
	}
};

Wizard.prototype.collapseForm = function($holder, stepId) {
	var collapseTab = this.steps[stepId].alternateForm.clone();
	this.setStepStatusClass(collapseTab, "done");
	this.setStepStatusClass($holder.find(".number-circle"), "done");
	$holder.find(".step-content").hide();
	$holder.find(".step-content").after(collapseTab);
};

Wizard.prototype.expandForm = function($holder) {
	this.setStepStatusClass($holder.find(".number-circle"), "active");
	$holder.children(".step-content-header").remove();
	$holder.find(".step-content").show();
	this.setStepStatusClass($holder.find(".step-content .step-content-header"), "active");
	this.focusInput($holder);
};

/*
 * Given a form, focus on one of the input fields inside. This allows the user to fill up the form without
 * using a mouse
 */
Wizard.prototype.focusInput = function($holder) {
	$holder.find(":not(select,.no-autofocus).form-control:first").focus();
};

/*
 * When using collapsed style, this function adds just the headers of the coming steps below the html of the
 * current step
 */
Wizard.prototype.addCollapsedStepHeaders = function(stepId) {
	var form = null;
	for (var i = stepId + 1; i <= Object.keys(this.steps).length; i++) {
		form = this.$formWrapper.find("." + this.Constants.WIZARD_STEP + "[data-step='" + i + "']");
		if (form.length == 0) {
			form = this.steps[i].form.clone();
			this.$formWrapper.append(form);
			form.find(".step-content").prepend(this.steps[i].alternateForm).hide().after(this.steps[i].alternateForm
					.clone());
			this.setStepStatusClass(form.find(".step-content-header"), "disabled");
			this.setStepStatusClass(form.find(".step-content .step-conent-header"), "none");
			this.setStepStatusClass(form.find(".number-circle"), "disabled");
		} else if (form.is(":hidden")) {
			form.show();
			continue;
		} else if (form.find(".step-content-header").length > 0) {
			this.setStepStatusClass(form.find(".number-circle"), "disabled");
			this.setStepStatusClass(form.find(".step-content-header"), "disabled");
			continue;
		}
	}
};

Wizard.prototype.removeCollapsedStepHeaders = function() {
	this.$formWrapper.find(".step-content-header.disabled").parent().hide();
};

Wizard.prototype.handleFailedValidationTransition = function(stepId) {
	var currentForm = this.$formWrapper.find("." + this.Constants.WIZARD_STEP + "[data-step='" + stepId + "']");
	switch (this.theme) {
	case this.Themes.collapse:
		this.setStepStatusClass(currentForm.find(".number-circle"), "failed");
		this.setStepStatusClass(currentForm.find(".step-content .step-content-header"), "failed");
		break;
	}
};

Wizard.prototype.setCancel = function(callback) {
	this.cancelCallback = callback;
};
//Adds a css class to an step in the dom to indicate the status of that step
Wizard.prototype.setStepStatusClass = function($holder, statusClass) {
	switch (statusClass) {
	case 'active':
		$holder.removeClass("failed done disabled").addClass("active");
		break;
	case 'done':
		$holder.removeClass("active failed disabled").addClass("done");
		break;
	case 'failed':
		$holder.removeClass("active done disabled").addClass("failed");
		break;
	case 'disabled':
		$holder.removeClass("active done failed").addClass("disabled");
		break;
	case 'none':
		$holder.removeClass("active done failed disabled");
		break;
	}
};

/* Rebuilds the aggregatedInput up to the current step */
Wizard.prototype.calculateAggregatedInput = function() {
	/*
	 * If we do not destroy aggregatedInput and rebuild from scratch, $.extend will cause trouble. For
	 * example, suppose in step 2 an array 'A' has values [1, 2, 3] entered by the user. If user now goes back
	 * to step 1, then again to step 2 and enter 'A' as [3, 4], it won't be reflected in aggregatedInput since
	 * array 'A' already exists. Hence aggregatedInput needs to be cleared after every step.
	 */
	this.aggregatedInput = {};

	/*
	 * Adding a reference to wizard in aggregatedInput will allow validator, initializer functions access to
	 * the wizard object, should they need it
	 */
	this.aggregatedInput.wizard = this;
	for (var i = 1; i <= this.currentStep; i++)
		$.extend(true, this.aggregatedInput, this.userInput[i]);

};

/*
 * For easily referring to the next/submit button of a step without having to parse the DOM
 */
Wizard.prototype.getNextButton = function() {
	if (!this.nextButton.is(":hidden"))
		return this.nextButton;
	else if (!this.prevButton.is(":hidden"))
		return this.prevButton;
};
