/**
 * @author Ryan Cebulko
 */

/**
 * An object representing a variable or parameter required by one or more APIs
 * @typedef {Object} API_VARIABLE
 *
 * @property {String} label A string containing the label for the variable/parameter input
 * @property {String} description A string containing a description of the variable/parameter to appear in a tooltip
 * @property {String} value A string containing the default value for the variable/parameter
 */

/**
 * An object representing sample XML code to be submitted
 * @typedef {Object} SAMPLE_OBJECT
 *
 * @property {String} label Button text for the sample
 * @property {String} input Sample XML
 */

/**
 * An object representing a specific API method
 * @typedef {Object} API_METHOD
 *
 * @property {String} snippet A string containing a label for the tree view
 * @property {String} description A string containing a description of the effect of invoking the given method on the given API
 * @property {Array<SAMPLE_OBJECT>} sample Sample XML code
 * @property {Array<String>} params Necessary parameters; ideally each a key to an {@link API_VARIABLE}
 */

/**
 * An object representing a path on the web service
 * @typedef {Object} API_OBJECT
 *
 * @property {Obect} __self__
 * @property {String} __self__.key A string containing the key for the {@link API_OBJECT}
 * @property {...METHOD_OBJECT} __self__.get An object representing the GET API
 * @property {...METHOD_OBJECT} __self__.post An object representing the POST API
 * @property {...METHOD_OBJECT} __self__.put An object representing the PUT API
 * @property {...METHOD_OBJECT} __self__.del An object representing the DELETE API
 * @property {...API_OBJECT} *
 */

 /**
  * Primary object; controls operations of entire page
  *
  * @type {sandboxObj}
  * @globalcurrApi.
  */
 var sandbox;

/**
 * Escapes any characters that could cause a problem in HTML attributes
 *
 * @param  {String} s the string to be escaped
 * @param  {Bool} preserveCR whether or not to preserve whitespace
 * @return {String} a string which is safe to include in an HTML attribute
 */
function quoteattr(s, preserveCR) {
  return ('' + s)
    .replace(/&/g,'&amp;')
    .replace(/"/g,'&quot;')
    .replace(/</g,'&lt;')
    .replace(/>/g,'&gt;')
    .replace(/\r\n|[\r\n]/g, preserveCR ? '&#13;' : '\n');
}

/**
 * Initializes the page view, binds initial event handlers, and creates the sandbox object
 * @function $documentReady
 */
$(document).ready(function() {
  sandbox = new sandboxObj();

  // Set the api tree to the correct size whenever the window is resized
  $("#apiListDiv").css("height",
    parseInt($("#apiTreeContainer").css("height"), 10) -
    parseInt($("#searchbar").css("height"), 10) -
    parseInt($("#apiTreeContainer > .divHeader").css("height"), 10)
  ).css("margin-top", parseInt($("#searchbar").css("height"), 10) + 7);
  $(window).resize(function() {
    $("#apiListDiv").css("height",
      parseInt($("#apiTreeContainer").css("height"), 10) -
      parseInt($("#searchbar").css("height"), 10) -
      parseInt($("#apiTreeContainer > .divHeader").css("height"), 10)
    ).css("margin-top", parseInt($("#searchbar").css("height"), 10) + 7);
  });

  // Enhance appearance with JQuery UI and JQuery Uniform plugins
  $("select, textarea, input, button").uniform();

  // Initialize jstree plugin for tree view
  $("#apiListDiv").bind("loaded.jstree", function() {
    $(".jstree-open > a, .jstree-closed > a").click(function(e) {
      e.stopPropagation();
      var node = $(e.currentTarget).parent();
      $("#apiListDiv").jstree("open_node", node);
      node.children("ul").children("li:first-child").children("a").trigger("click");
    });

    $(".jstree-leaf").click(function(e) {
      e.stopPropagation();

      $(".selectedApi").removeClass("selectedApi");
      $(e.currentTarget).addClass("selectedApi");
    });

    $("#searchbar").on("keyup", function() {
      $("#apiListDiv").jstree("search", $(this).val()).jstree("close_node", $("li[rel=file]").parent().parent()).scrollTop(0);

      if ($(this).val() === "") {
        $("#APIs li > a").parent().show();
      } else {
        $("#APIs li:not([rel=file]) > a").each(function () {
          if ($(this).hasClass("jstree-search")) {
            $(this).parent().show();
          } else  {
            $(this).parent().hide();
          }
        });
      }

      $("#APIs").siblings("a").removeClass("jstree-search");
    });

    // When the page is fully loaded and set up, trigger clicks to select the Login API
    $("#APIs-Login").parent().children("a").trigger("click");
    $(".loginParam").trigger("keyup");
  }).jstree({
      plugins: ["themes", "html_data", "types", "search"],
      core: {
        initially_open: ["APIs"],
        animation: 200
      },
      types: {
        file: {}
      },
      search: {
        case_insensitive: true
      }
  });

  // Adjust headers appropriately
  $("input[type=radio], input[type=checkbox]").change(function() {
    sandbox.setHeaders();
  });

  // Inkvoke button functionality
  $("#invokeButton").click(function(e) {
    e.stopPropagation();
    sandbox.apiCall();
  });

  // React to clicks on the hide/show buttons
  $(".hideShowLink > a").click(function() {
    if ($("#requestDiv, #responseDiv").is(":not(:visible)")) {
      sandbox.split();
    } else {
      sandbox.minimize($(this).parent().parent());
    }
  });

  // Update headers if options change

  // Clear leading and trailing whitespace from request and response bodies
  $("textarea").change(function() {
    $(this).val($.trim($(this).val()));
  });

  // Event handler to allow tabbing in text areas
  $(document).on('keydown', 'textarea', function(e) {
    var keyCode = e.keyCode || e.which;

    if (keyCode == 9) {
      e.preventDefault();
      var start = $(this).get(0).selectionStart;
      var end = $(this).get(0).selectionEnd;

      // Set textarea value to: text before caret + tab + text after caret
      $(this).val($(this).val().substring(0, start) + "\t" + $(this).val().substring(end));

      // Put caret at right position again
      $(this).get(0).selectionStart = $(this).get(0).selectionEnd = start + 1;
    }
  });
});

/**
 * Initializes API tree and draws the tree view
 * @class sandboxObj Handles all major functionality of the sandbox
 *
 * @property {String} pass Password to use when making authorization requests
 * @property {String} user Username to use when making authorization requests
 * @property {String} token Authorization token from the server
 * @property {API_OBJECT} apis The main API tree (either {@link staticTree} or {@link dynamicTree})
 * @property {Object} currApi Information regarding the currently selected API
 * @property {API_OBJECT} currApi.api The corresponding {@link API_OBJECT}
 * @property {String} currApi.method Either "get" or "post" or "put" or "del"
 * @property {String} currApi.url The request URI with all parameters incuded
 */
function sandboxObj() {
  this.giveParents();
  this.createTier();
}

/**
 * Invoke an API *
 * @return {sandboxObj} Return <i>this</i> to allow chaining
 */
sandboxObj.prototype.apiCall = function() {
  // Alias for "this" within anonymous function scope
  var self = this;

  // If a token is needed, exit function and request one
  if (typeof this.token === "undefined" && this.currApi.api != this.apis.Login) {
    $("#responseContent").val("No authorization token provided.");
    return;
  }

  // Clear the response textarea
  $("#responseContent").val("Waiting for response...");

  // Content of request body
  var content = $("#requestContent").val();

  // Format of request and response bodies
  var type;

  // Index of the beginning of the actual request body
  var start;

  // Filter out headers that may be left in the textarea from the previous request
  if ($("#xmlRequestRadio").prop("checked")) {
	  type = "xml";
    start = content.indexOf("<");
  } else {
	  type = "json";
    start = content.indexOf("\n{");
    if (start === -1) {
    	// Fall back to xml
    	start = content.indexOf("<");
    }
    else {
		start++;
    }
  }
  if (start == -1) start = content.length;
  content = content.substring(start);


  $.ajax({
    type: "POST",
    url: "make_request.jsp",
    data: {
      "token": self.token,
      "uri": self.currApi.uri,
      "method": self.currApi.method,
      "content": content,
      "type": type
    },
    success: function(data) {
      data = data.trim().replace(/\\/, "\\\\");

      if (data.substr(0, 4) == "ERR_") {
    	if (data.substr(4,7) == "401") {
    		$("#responseContent").val("Invalid authorization token provided.");
    	} else {
    		$("#responseContent").val("There was a problem with the webservice.");
    	}

        return;
      }

		// Use new token if a login API was called
		var token = null;
		if (data.indexOf("DM2ContentIndexing_CheckCredentialResp") !== -1) {
			// XML case
			token = $($.parseXML(data)).find("DM2ContentIndexing_CheckCredentialResp").attr("token");
		} else {
			// JSON case
			try {
				token = JSON.parse(data).token;
			} catch (e) {
			}
		}
		if (typeof token !== "undefined" && token !== null) {
			self.token = token;
		}

      // Replace pairs of spaces before XML tags with tabs and fill textareas with server response
      $("#responseContent").val(data.replace(/  (?=(  )*<)/g, "\t"));
      if (self.currApi.method == "post" || self.currApi.method == "put") {
        $("#requestContent").val(content.replace(/  (?=(  )*<)/g, "\t"));
      } else { $("#requestContent").val(""); }

      self.setHeaders();
    }
  });

  return this;
};

/**
 * React to changes in parameter/variable values
 *
 * @return {sandboxObj} Return <i>this</i> to allow chaining
 */
sandboxObj.prototype.bindParamInputs = function() {
  var self = this;

  // Update URI as parameter input values are changed and prepare tooltips
  $("#apiParamsSpan input:not(.loginParam)").keyup(function() {
    if (typeof $(this).attr("placeholder") === "string") {
      var key = $(this).attr("placeholder");
      if (!$(this).hasClass("param")) {
        key = key.substr(1, key.length - 2);
      }
      self.vars[key].value = $.trim($(this).val());
    }

    self.currApi.uri = self.genReqUri();
    $("#apiUriInput").val(self.currApi.uri);

    self.setHeaders();
  }).change(function() {
    // Trim leading and trailing whitespace from text fields
    $(this).val($.trim($(this).val()));

    if (typeof $(this).attr("placeholder") === "string") {
      var key = $(this).attr("placeholder");
      if (!$(this).hasClass("param")) {
        key = key.substr(1, key.length - 2);
      }
      self.vars[key].value = $.trim($(this).val());
    }

    self.currApi.uri = self.genReqUri();
    $("#apiUriInput").val(self.currApi.uri);

    self.setHeaders();
  });

  /*$("#apiParamsSpan input").qtip({
    style: "qtip-light",
    position: {
      my: "right center",
      at: "left center"
    },
    show: {
      solo: true,
      event: "focus"
    },
    hide: {
      event: "unfocus"
    }
  });*/

  $(".loginParam").keyup(function() {
    $("#requestContent").val(
      "<DM2ContentIndexing_CheckCredentialReq mode=\"Webconsole\" username=\"" +
      $("#apiParamsSpan input.loginParam[type=text]").val() +
      "\" password=\"" +
      base64_encode($("#apiParamsSpan input.loginParam[type=password]").val()) +
      "\" />"
    );

    self.setHeaders();
  });

  return this;
};

/**
 * Place appropriate details, parameter inputs, and URI into the page
 *
 * @return {sandboxObj} Return <i>this</i> to allow chaining
 */
sandboxObj.prototype.genReqDetails = function() {
  api = this.currApi.api;
  method = this.currApi.method;
  var self = api["__self__"][method];

  $("#apiDescriptionSpan").html(self.description);

  var $span = $("#apiParamsSpan");
  $span.html("");
  var newHtml = "";

  for (var child = api; child !== this.apis; child = child["__parent__"]) {
    if (child["__self__"].key.charAt(0) == "{") {
      var strippedKey = child["__self__"].key.substr(1, child["__self__"].key.length - 2);
      if (typeof this.vars[strippedKey] === "undefined") {
        this.vars[strippedKey] = {
          label: strippedKey,
          value: "",
          description: strippedKey
        };
      }

      newHtml += "<tr><td><b>" +
        this.vars[strippedKey].label +
        ":</b></td><td><input type='text' size='15' class='var' placeholder='" +
        child["__self__"].key +
        "' value='" +
        quoteattr(this.vars[strippedKey].value) +
        "' title='" +
        quoteattr(this.vars[strippedKey].label) +
        "'></td><tr>";
    }
  }

  if (typeof self.params !== "undefined") {
    for (var i = 0; i < self.params.length; ++i) {
      if (typeof this.vars[self.params[i]] === "undefined") {
        this.vars[self.params[i]] = {
          label: self.params[i],
          value: "",
          description: self.params[i]
        };
      }

      newHtml += "<tr><td><b>" +
        this.vars[self.params[i]].label +
        ":</b></td><td><input type='text' size='15' class='param' placeholder='" +
        self.params[i] +
        "' value='" +
        quoteattr(this.vars[self.params[i]].value) +
        "' title='" +
        quoteattr(this.vars[self.params[i]].description) +
        "'></td><tr>";
    }
  }

  if (this.currApi.api == this.apis.Login) {
    newHtml = "<tr><td><b>Username:</b></td><td><input type='text' size='15' class='loginParam' placeholder='" +
      "Username' title='Username'></td><tr><tr><td><b>Password:</b></td><td><input type='password' size='15' class='loginParam' placeholder='" +
      "Password' title='Password'></td><tr>";
  }

  if (newHtml === "") {
    newHtml = "<center>No variables or parameters required</center>";
  } else {
    newHtml = "<center><table id='varTable'>" + newHtml + "</table></center>";
  }

  $span.append(newHtml);
  $("input", $span).uniform();
  this.bindParamInputs();
  $("#apiUriInput").val(this.currApi.uri);
};

/**
 * Give each API object a reference to its parent API object
 *
 * @param  {API_OBJECT} api Any branch API on the API tree
 * @return {sandboxObj} Return <i>this</i> to allow chaining
 */
sandboxObj.prototype.giveParents = function(api) {
  if (typeof api === "undefined") {
    api = this.apis;
  }

  for (var key in api) {
    if (key == "__self__" || key == "__parent__") continue;

    api[key]["__parent__"] = api;
    this.giveParents(api[key]);
  }
};

/**
 * Generate tiers of APIs recursively
 *
 * @param  {String} [currUri=""] The URI at the current branch
 * @param  {API_OBJECT} [api=this.apis] The object with data for the relevant API
 * @param  {String} [parent=$("#apiListDiv > ul")] Selector for the parent element in which the tier will be added
 * @return {sandboxObj} Return <i>this</i> to allow chaining
 */
sandboxObj.prototype.createTier = function(currUri, api, parent) {
  var self = this;

  if (typeof currUri === "undefined" && typeof api === "undefined" && typeof parent === "undefined") {
    currUri = "";
    api = self.apis;
    parent = $("#apiListDiv > ul");
  }

  var currApi = api["__self__"].key;
  if (currApi === "") {
    currApi = "Webservice APIs";
  }

  // Create a dropdown for each group of APIs
  var id = "APIs" + currUri.replace(/[\{\}]/g, "__").replace(/\//g, "-");
  $(parent).append(
    "<li rel='folder'><a href='javascript:void(0)'>" +
    currApi +
    "</a><ul id='" +
    id +
    "' class='apiDropdown'></ul></li>"
  );

  // Create an entity for each individual method
  for (var key in api["__self__"]) if (key == "get" || key == "post" || key == "put" || key == "del") {
    var rel;
    if (typeof api["__self__"][key].sample === "undefined") {
      rel = "file";
    } else {
      rel = "folder";
    }

    var anchor = "anchor-" + base64_encode(currUri + key + "undefined").replace(/=/g, "-");

    $("#" + id).append(
      "<li rel='" +
      rel +
      "'><a href='javascript:void(0)' id='" +
      anchor +
      "'>" +
      api["__self__"][key].snippet +
      "</a><span class='uriSpan'>" +
      currUri +
      "</span><ul id='" +
      id + "-" + key +
      "' class='apiDropdown'></ul></li>"
    );

    $("#" + anchor).on(
      "click",
      (function(uri, method) {
        return function() {
          self.invokeHandler(uri, method);
          $("#apiListDiv").scrollTo($("#" + anchor).parents("#APIs > li"), 250);
        };
      })(currUri, key)
    );

    if (typeof api["__self__"][key].sample !== "undefined") {
      var sample = api["__self__"][key].sample;

      for (var i = 0; i < sample.length; ++i) {
        anchor = "anchor-" + base64_encode(currUri + key + i).replace(/=/g, "-");
        var newHtml =
          "<li rel='file'><a href='javascript:void(0)' id='" +
          anchor +
          "'>" +
          sample[i].label +
          "</a><span class='uriSpan'>" +
          currUri +
          "</span><ul id='" +
          id + "-" + key +
          "' class='apiDropdown'></ul></li>";

        if (sample.length == 1) {
          $("#" + id + "-" + key).parent()[0].outerHTML = newHtml;
        } else {
          $("#" + id + "-" + key).append(newHtml);
        }

        $("#" + anchor).on(
          "click",
          (function(uri, method, index) {
            return function() {
              self.invokeHandler(uri, method, index);
              $("#apiListDiv").scrollTo($("#" + anchor).parents("#APIs > li"), 250);
            };
          })(currUri, key, i)
        );
      }
    }
  }

  for (var prop in api) {
    if (prop == "__self__" || prop == "__parent__") continue;

    // Create a dropdown for each child API
    self.createTier(
      currUri + "/" + prop,
      api[prop],
      parent.find("#" + id)
    );
  }
};

/**
 * Get URI for an API object by traversing the tree upwards
 *
 * @param  {API_OBJECT} api Any branch API on the API tree
 * @return {String} The URI at the requested branch
 */
sandboxObj.prototype.getUri = function(api) {
  if (api === this.apis) {
    return "";
  } else {
    var curr = api["__self__"].key;
    var strippedCurr = curr.substr(1, curr.length - 2);
    if (api["__self__"].key.charAt(0) == "{" && typeof this.vars[strippedCurr] !== "undefined" && this.vars[strippedCurr].value !== "") {
      curr = encodeURI(this.vars[strippedCurr].value);
    }

    return this.getUri(api["__parent__"]) + "/" + curr + "";
  }
};

/**
 * Generate the URI for a given api with variables and parameters
 *
 * @param  {API_OBJECT} api Any branch API on the API tree
 * @param  {String} method "get" or "post" or "put" or "del"
 * @return {String} The request URI with parameter and variable values pulled from the input fields
 */
sandboxObj.prototype.genReqUri = function(api, method) {
  if (typeof api === "undefined") {
    api = this.currApi.api;
  }
  if (typeof method === "undefined") {
    method = this.currApi.method;
  }

  var currUri = this.webserviceUri + this.getUri(api);
  if (typeof api["__self__"][method].params !== "undefined") {
    var vars = {};
    var params = api["__self__"][method].params;
    for (var i = 0; i < params.length; ++i) {
      if (typeof this.vars[params[i]] === "undefined") {
        this.vars[params[i]] = {
          label: params[i],
          value: "",
          description: params[i]
        };
      }

      if (this.vars[params[i]].value !== "") {
        vars[params[i]] = this.vars[params[i]].value;
      }
    }

    params = "?" + $.param(vars);
    if (params !== "?") {
      currUri += params;
    }
  }

  return currUri;
};

/**
 * Fill content boxes with sample values on buttonpress
 *
 * @param  {Number} index Index of {@link SAMPLE_OBJECT} in {@link METHOD_OBJECT}.sample to be used
 * @return {sandboxObj} Return <i>this</i> to allow chaining
 */
sandboxObj.prototype.useSample = function(index) {
  var sample = this.currApi.api["__self__"][this.currApi.method].sample[index];
  $("#requestContent").val(sample.input);
  if (this.currApi.api == this.apis.Login) $(".loginParam").keyup();
};

/**
 * Get API object and pass it to the request detail generation function
 *
 * @param  {String} uri URI associated with API
 * @param  {String} method "get" or "post" or "put" or "del"
 * @param  {Number} [index] Index of {@link SAMPLE_OBJECT} in {@link METHOD_OBJECT}.sample to be used
 * @return {sandboxObj} Return <i>this</i> to allow chaining
 */
sandboxObj.prototype.invokeHandler = function(uri, method, index) {
  var path = uri.split("/");
  var self = this.apis;

  for (var i = 0; i < path.length; i++) {
    if (path[i] === "") continue;
    self = self[path[i]];
  }

  this.currApi = {
    api: self,
    method: method,
    uri: this.genReqUri(self, method),
  };

  this.genReqDetails();
  if (method == "get" || method == "del") {
    $("#requestContent").attr("disabled", true).attr("placeholder", "No request content");
    $.uniform.update("#requestContent");
  } else {
    $("#requestContent").attr("disabled", false).attr("placeholder", "API request body");
    $.uniform.update("#requestContent");
  }
  $("#requestAndResponse textarea").val("");

  if (typeof index !== "undefined") this.useSample(index);

  this.setHeaders();
  $(".loginParam").trigger("keyup");
};

/**
 * Function to show both textareas
 * @return {sandboxObj} Return <i>this</i> to allow chaining
 */
sandboxObj.prototype.split = function() {
  $("textarea").animate({height: "89%"});
  $("#requestDiv").animate({top: "10%", bottom: "45%"}, function() { $(this).fadeIn(200); });
  $("#responseDiv").animate({top: "55%", bottom: "1%"}, function() { $(this).fadeIn(200); });
  $(".hideShowLink > a").html("hide");
};

/**
 * Function to show both textareas
 * @return {sandboxObj} Return <i>this</i> to allow chaining
 */
sandboxObj.prototype.minimize = function(selector) {
  $(selector).fadeOut(200, function() {
    $("textarea").animate({height: "95%"});
    $("#requestDiv:visible, #responseDiv:visible").animate({top: "10%", bottom: "1%"}).show();
  });
  $(".hideShowLink > a").html("restore");
};

/**
 * Insert the request header at the beginning of the textarea
 * @return {sandboxObj} Return <i>this</i> to allow chaining
 */
sandboxObj.prototype.setHeaders = function() {
  var newContents = "";
  var acceptType;
  var contentType = "xml";
  var content = $("#requestContent").val();

  var start;
  if (content.indexOf("<") != -1) {
    start = content.indexOf("<");
  } else if (content.indexOf("\n{") != -1) {
    start = content.indexOf("\n{") + 1;
  } else { start = content.length; }

  if ($("#xmlRequestRadio").prop("checked")) {
	  acceptType = "xml";
  } else { acceptType = "json"; }
  if (start == -1) start = content.length;
  content = content.substring(start);

  if ($("#headersOn").prop("checked")) {

    newContents = {get:"GET",post:"POST",put:"PUT",del:"DELETE"}[this.currApi.method] + " ";
    var uri = this.currApi.uri.split("//")[1].split("/");
    var host = uri.shift();
    var path = uri.join("/");

    newContents += path + " HTTP/1.1\nHost: " + host + "\nAccept: application/" + acceptType + "\n";
    if (this.currApi.api["__self__"].key != "Login" && typeof this.token !== "undefined") newContents += "Cookie2: " + this.token + "\n";
    if (this.currApi.method == "post" || this.currApi.method == "put") newContents += "Content-type: application/" + contentType + "\n";
    newContents += "\n";
  }

  $("#requestContent").val(newContents + content);

  return this;
};