# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
Parses the given Input JSON and updates its values and structure according to the given base JSON.

Finally it then overrides the given input JSON with the newly formatted JSON.

Points to Note:
- Input JSON is a required parameter
- Base JSON is optional, and defaulted to **v2.json**
- For any issues, please write to **Dev-SimpleRestAPI@commvault.com**

Steps
-----

    - Open your APIMatic dashboard, and click on Import
    - Select the Postman Collection to be converted into Swagger JSON
    - After the API is created, click on Options (Triple vertical dots) --> Export API
    - Select OpenAPI / Swagger v2.0 (JSON) as the export format
    - Ensure Include Extensions checkbox is **NOT** checked

    This will convert the Postman Collection into Swagger JSON

    Store all Swagger JSON files under sandbox/apiexplorer/apiJson path

Usage
-----

    Launch cmd.exe from under the sandbox/apiexplorer directory

    - Run the command:

    >>> python parser.py --input-JSON apiJson/{{ generated_json }}


    If cmd is launched from a different path, or you wish to use a different base JSON,
    please provide the value for --base-JSON as well

    >>> python parser.py --input-JSON {{ full_path_of_generated_json }} --base-JSON {{ full_path_of_base_json }}

"""

import argparse
import json
import logging
import os


def main():
    """Main function"""
    parser = argparse.ArgumentParser(
        description="Commvault Swagger JSON generator",
        formatter_class=argparse.RawTextHelpFormatter
    )

    parser.add_argument(
        '--input-JSON',
        '-json',
        dest='input_json',
        help='Input JSON to be formatted with Commvault Swagger specs',
        required=True,
        default=None
    )

    parser.add_argument(
        '--base-JSON',
        '-b-json',
        dest='base_json',
        help='Base JSON written with Commvault Swagger specs, to be used for formatting input JSON',
        required=False,
        default='apiJson/v2.json'
    )

    logging.info("Parsing the command line arguments")

    args = parser.parse_args()

    logging.info("Base JSON: %s", args.base_json)
    logging.info("Input JSON: %s", args.input_json)

    with open(args.base_json) as file_object:
        base_json = json.load(file_object)

    with open(args.input_json) as file_object:
        input_json = json.load(file_object)

    logging.info("Loaded the Input JSON and Base JSON successfully")

    input_json['basePath'] = base_json['basePath']
    input_json['securityDefinitions'] = base_json['securityDefinitions']
    input_json['info']['contact'] = base_json['info']['contact']
    input_json['info']['license'] = base_json['info']['license']

    logging.info("Updated the input JSON properties")

    try:
        del input_json['host']
    except KeyError:
        pass

    security_token_tag = [{
        "token": []
    }]

    if 'https' not in input_json['schemes']:
        input_json['schemes'].insert(0, 'https')

    logging.info("Added HTTPS to schemes")

    for path in input_json['paths']:
        for operation in input_json['paths'][path]:
            input_json['paths'][path][operation]['security'] = security_token_tag

            if 'todo' in input_json['paths'][path][operation].get('description', '').lower():
                del input_json['paths'][path][operation]['description']

            parameters = []

            for parameter in input_json['paths'][path][operation]['parameters']:
                if parameter['name'].lower() not in ['accept', 'authtoken', 'content-type']:
                    parameters.append(parameter)

            input_json['paths'][path][operation]['parameters'] = parameters

    logging.info("Added security to all APIs, and removed additional attributes")

    with open(args.input_json, 'w') as file_object:
        json.dump(input_json, file_object)

    logging.info("Updated the JSON successfully")


if __name__ == "__main__":
    logging.basicConfig(
        format='%(process)-6d %(asctime)-25s %(lineno)-6d %(message)s',
        filename=os.path.join(os.getcwd(), "parser.log"),
        level=logging.DEBUG
    )
    logging.info("Starting parser")
    main()
    logging.info("Parsing finished")
