#!/usr/bin/env python

from flask import Flask
from flask import request
from flask import render_template
from flask import Response, render_template_string
from flask import get_template_attribute
from flask import redirect, url_for
from flask import jsonify
from flask import send_from_directory
from collections import OrderedDict
from operator import itemgetter
from netifaces import interfaces, ifaddresses, AF_INET
import subprocess
import platform
import ConfigParser
import os
import re
import sys
import time
import json
import socket
import psutil
import logging
import binascii
import collections
import subprocess
import xml.etree.ElementTree as ET
import tempfile
import ipaddress
import pprint
from time import gmtime, strftime
import shutil
import inspect
import traceback
# Need below for supporting standby CS creation
import threading
# Need the below for processing multi-node progress monitoring
import glob


sys.path.append('scripts/')
import avahi_browse_and_print as avahi_browse
import cvavahi
import cvnwhelper

app = Flask(__name__)

#Variables
username="temp"
passwd="temp"
setup_process = None
setup_process_standby = None
CVLT_BASE = "/opt/commvault/Base"
json_cloudcfg = None

#Scripts
verify_password_script = "scripts/verify_password.sh"
avahibrowse = "scripts/avahi_browse_and_print.py"

# config files
avahibrowse_config = "config/avahibrowse.cfg"
silentinstall_filepath = "config/silentinstall.cfg"
temp_passfilepath = "config/tempuuid.cfg"
passfilepath = "config/uuid.cfg"
decrypted_file = "config/duuid.cfg"
jsonfilepath = "config/silentinstall.json"
progress_filepath = "config/progress_{0}.txt".format(cvavahi.get_current_node_cvlt_uuid())
scratch_file = "config/scratch_file.txt"
logs_basename  = ""

#Last error
last_error="HyperScale configuration failed"

# Map which contains information about current node and peer nodes
# which are getting configured
hyperscale_node_map = { }
out_file = None
err_file = None

#logging information
log_file_path = "logs/cvzeroconf.log"
logging.basicConfig(filename=log_file_path, format='%(asctime)s : %(levelname)s : %(message)s', datefmt='%m/%d/%Y %I:%M:%S %p', level=logging.DEBUG)

# Method to validate existence of folders. Not adding logs folder as it would anyway result in webserver not starting up
def _validate_folders():
    folders = ['scripts', 'static', 'templates', 'config', 'static/css', 'static/img', 'static/js', 'static/fonts', 'static/screenshots']
    is_validated = True
    for folder in folders:
        if (not os.path.isdir(folder)):
            is_validated = False
            logging.error('Unable to find {0} directory in /var/www'.format(folder))
    if is_validated:
        logging.info ('Completed folder validations successfully')

_validate_folders()


CV_CONSTANTS = {
    "CS_OVA"        : r'/ws/ddb/ova/cvhcics.ova',
    "MA_REG_PATH"   : r'/etc/CommVaultRegistry/Galaxy/Instance001/MediaAgent/.properties',
    "CS_REG_PATH"   : r'/etc/CommVaultRegistry/Galaxy/Instance001/CommServe/.properties'
}

####### Helper methods to populate the configurations are defined here ##########
def get_hyperscale_type(node):
    cv_ma_reg_path = CV_CONSTANTS["MA_REG_PATH"]
    platform_type=cvavahi.get_remote_reg_key(node, cv_ma_reg_path, "sScaleoutPlatformType")
    if (platform_type=="HyperScale"):
        #Get HyperScale Appliance type
        hsappliance_type=cvavahi.get_remote_reg_key(node, cv_ma_reg_path, "sHyperScaleApplianceType")
        return hsappliance_type
    return "ReferenceArchitecture"


def is_lab_install():
    cv_ma_reg_path = CV_CONSTANTS["MA_REG_PATH"]
    install_type = cvavahi.get_remote_reg_key("localhost", cv_ma_reg_path, "sHSLabInstall")
    if (install_type.lower() == "yes" or install_type.lower() == "y"):
        return True
    return False

def is_metallic_install():
    cv_ma_reg_path = CV_CONSTANTS["MA_REG_PATH"]
    install_type = cvavahi.get_remote_reg_key("localhost", cv_ma_reg_path, "sHSMetallicInstall")
    if (install_type.lower() == "yes" or install_type.lower() == "y"):
        return True
    return False

def get_metallic_cs():
    cv_cs_reg_path = CV_CONSTANTS["CS_REG_PATH"]
    metallic_cs = cvavahi.get_remote_reg_key("localhost", cv_cs_reg_path, "sCSHOSTNAME")
    return metallic_cs
####### Helper method definittions end here ##########

HS_CONFIG = {
    "HS_TYPE"               : get_hyperscale_type("localhost"),
    "IS_LAB_INSTALL"        : is_lab_install(),
    "IS_METALLIC_INSTALL"   : is_metallic_install()
}


class ValidationError(Exception):
    # Before installation begins, there are numerous validations.
    # This class will set validation message and code that the Javascript promise can use.
 
    status_code = 400
 
    def __init__(self, message, status_code=400, payload=None, is_cors=False):
        super(ValidationError, self).__init__()
        self.message = message
        if status_code is not None:
            self.status_code = status_code
        self.payload = payload
        self.is_cors = is_cors
 
    def to_dict(self):
        rv = dict(self.payload or ())
        rv['message'] = self.message
        return rv

class cvlt_avahi_node_info:
    def __init__(self, uuid, hostname, ip, status, setup_in_progress, reboot_pending, restart_level, blkid, data_protection_nw, storage_pool_nw, mgmt_nw):
        self.uuid = uuid
        self.host = host
        self.ip = ip
        self.status = status
        self.setup_in_progress = setup_in_progress
        self.reboot_pending = reboot_pending
        self.restart_level = restart_level
        self.blkid = blkid
        self.missing = False
        self.arch_error = False
        self.data_protection_nw = data_protection_nw
        self.storage_pool_nw = storage_pool_nw
        self.mgmt_nw = mgmt_nw


    def __init__(self, cvlt_avahi_node_dict):
        self.uuid = cvlt_avahi_node_dict["uuid"]
        self.host = cvlt_avahi_node_dict["host"]
        self.ip = cvlt_avahi_node_dict["ip"]
        self.status = cvlt_avahi_node_dict["status"]
        self.setup_in_progress = cvlt_avahi_node_dict["setup_in_progress"]
        self.reboot_pending = cvlt_avahi_node_dict["reboot_pending"]
        self.restart_level = cvlt_avahi_node_dict["restart_level"]
        self.blkid = cvlt_avahi_node_dict["blkid"]
        self.data_protection_nw = cvlt_avahi_node_dict["data_protection_nw"]
        self.storage_pool_nw = cvlt_avahi_node_dict["storage_pool_nw"]
        self.mgmt_nw = cvlt_avahi_node_dict["mgmt_nw"]
        self.missing = False
        self.arch_error = False

    def __repr__(self):
        d_avahi_node_info = {
            "ip"            : self.ip,
            'host'          : self.host,
            "uuid"          : self.uuid,
            "blkid"         : self.blkid,
            "status"        : self.status,
            "arch_error"    : self.arch_error
        }
        return str(d_avahi_node_info)

class hyperscale_node_info(cvlt_avahi_node_info):
    hyperscale_blk_size = 3
    
    def __init__(self, node_desc, av_node_info):
        self.node_desc = node_desc
        self.uuid = av_node_info.uuid
        self.host = av_node_info.host
        self.ip = av_node_info.ip
        self.status = av_node_info.status
        self.setup_in_progress = av_node_info.setup_in_progress
        self.reboot_pending = av_node_info.reboot_pending
        self.restart_level = av_node_info.restart_level
        self.blkid = av_node_info.blkid
        
        # Below fields are populated by populate_hw_info() method
        self.data_protection_mac_addr = ""
        self.data_protection_ip_addr = ""
        self.storagepool_ip_addr = ""
        self.mgmt_ip_addr = ""
        self.mgmt_netmask = ""
        self.ipmi_ipaddr = "" 
        self.livefs_ipaddr = "" 
        self.hwdesc = []
        self.data_protection_netmask = ""
        self.storagepool_netmask = ""
        self.tengig_list = []
        self.onegig_list = []
        self.populate_hw_info()

    def __repr__(self):
        d_node_info = {
            'ip'                    : self.ip,
            'host'                  : self.host,
            'uuid'                  : self.uuid,
            'blkid'                 : self.blkid,
            'status'                : self.status,
            'restart_level'         : self.restart_level,
            'setup_in_progress'     : self.setup_in_progress,
            'storagepool_ip_addr'   : self.storagepool_ip_addr,
            'dp_ip'                 : self.data_protection_ip_addr
        }
        return str(d_node_info)

    def populate_avahi_info(self, av_node_info):
        self.host = av_node_info.host
        self.ip = av_node_info.ip
        self.status = av_node_info.status
        self.setup_in_progress = av_node_info.setup_in_progress
        self.reboot_pending = av_node_info.reboot_pending
        self.restart_level = av_node_info.restart_level
    
    def populate_hw_info(self):
        ret = 0
        hwinfo_config = "config/" + self.uuid+"_hwinfo.cfg"
        self.advanced_nw_config = cvnwhelper.query_existing_nw_config_for_node(self)
        cvnwhelper.populate_basicnode_info (self)

        # Get the 10G network information corresponding to this node
        nwinfo = get_nwintfx(self.ip, 10000, hwinfo_config)
        if (len(nwinfo) == 0):
            # This could happen either because arch is down
            # or no network interfaces are found
            ret = -1
            blkdevinfo = ""
        else:    
            blkdevinfo = get_blkdev_info(hwinfo_config)
            self.hwdesc = append_blkdev_desc(self.hwdesc, blkdevinfo) 

        return ret
    
    def mark_setup_in_progress(self):
        cvavahi.update_remote_avahi_config(self.ip, "setup_in_progress", True)
        self.setup_in_progress = True
    
    def clear_setup_in_progress(self):
        cvavahi.update_remote_avahi_config(self.ip, "setup_in_progress", False)
        self.setup_in_progress = False
        
    def get_restart_level(self):
        return self.restart_level
    
    def is_setup_in_progress(self):
        return self.setup_in_progress
        
    def is_configured(self):
        return self.status == "HyperScaleConfigured"

class NonInstallConfiguration(object):
    # Class to drive configurations that are not directly related to the install. 
    # This can be extended in the future to include other features
    def __init__(self):
        self.time_zone = None
        # 'centos' or 'redhat'
        self.os_type = platform.dist()[0]

    def get_all_timezones(self):
        # Returns a list of all available timezones
        p1 = subprocess.Popen(["timedatectl", "list-timezones"], stdout=subprocess.PIPE)
        output = p1.communicate()[0]
        return output.split()

    def get_current_timezone(self):
        p1 = subprocess.Popen(["timedatectl"], stdout=subprocess.PIPE)
        p2 = subprocess.Popen(["grep", "Time zone"], stdin=p1.stdout, stdout=subprocess.PIPE)
        p1.stdout.close()
        output = p2.communicate()[0]
        return output.strip().split()[-3]

    def set_current_timezone(self, _tz):
        if self.time_zone is None:
            self.time_zone = self.get_current_timezone()
        if _tz != self.time_zone:
            # Execute timedatectl set-timezone time_zone
            p1 = subprocess.Popen(["sudo", "timedatectl", "set-timezone", _tz], stdout=subprocess.PIPE)
            output = p1.communicate()
            logging.info("Command to set timezone completed with output: {0}".format(output))


def update_remote_txt_record(remote_host):
    ret=subprocess.call(["arch", "manage", "commvault", "execute", remote_host, "cvupdate_avahi.sh", "/etc/avahi/services/cvlt.service"])
    if (ret):
        logging.error("Failed to update avahi configuration file on remote host " + remote_host)
    return ret

def get_avahi_ip(config_file_path, serialno):
    # Check if management network is configured
    config = ConfigParser.ConfigParser()
    config.read(config_file_path)

    if (config.has_section("network") and config.has_option("network", "config_mgmt")):
        config_mgmt = config.getboolean("network", "config_mgmt")
    else:
        config_mgmt = False
    if (not config_mgmt):
        return get_from_cfgfile(config, "data_protection_ips", serialno)
    else:    
        return get_from_cfgfile(config, "management_ips", serialno)

def browse_all_cvlt_avahi_nodes(node_ids = []):
    done = False
    retries = 0
    max_retries = 5

    while ((not done) and (retries < max_retries)):    
        cvlt_avahi_map = { }
        ordered_cvlt_avahi_map = { }

        cmd = "sudo " + avahibrowse + " "
        cmd += avahibrowse_config
        os.system(cmd)
        Config = ConfigParser.ConfigParser()
        Config.read(avahibrowse_config)
        
        current_blkid = cvavahi.get_current_node_cvlt_blkid()
        all_ips_are_valid = True

        # Sort avahi nodes by uuid
        for section in Config.sections():
            if ' ' in section:
                continue
            cvlt_avahi_node_dict = dict(Config.items(section))
            remote_ipaddr = cvlt_avahi_node_dict["ip"]

            if (current_blkid != cvlt_avahi_node_dict["blkid"]):
                continue

            if (not cvnwhelper.is_redhat() and
                cvnwhelper.is_livefs_ip(remote_ipaddr)):
                remote_ipaddr = cvnwhelper.query_dataprotection_ip(silentinstall_filepath, section) 
                cvlt_avahi_node_dict["ip"] = remote_ipaddr

            ret = cvnwhelper.execute_command(["sudo","ping","-c","1",remote_ipaddr])
            if (ret != 0):
                # Let's clear ARP cache and retry 
                cvnwhelper.clear_arp_cache()
                all_ips_are_valid = False
                break

            #Read the avahi config file from the remote node 
            #We will update the TXT records reported by avahi with those read from 
            #avahi config file on the remote node       
            avahi_cfg_file_dict = cvavahi.parse_remote_node_avahi_config(remote_ipaddr)
            for key in avahi_cfg_file_dict:
                if key in cvlt_avahi_node_dict:
                    cvlt_avahi_node_dict[key]=avahi_cfg_file_dict[key]              
            cvlt_avahi_node_dict["uuid"] = section
            cvlt_avahi_map[section] = cvlt_avahi_node_info(cvlt_avahi_node_dict)
            logging.info("Found node with id {0}".format(section))
        if all_ips_are_valid:
            done = True
            break 
        retries = retries + 1
    
    if (len(node_ids) > 0 and 
        (len(cvlt_avahi_map.keys()) < len(node_ids)) and 
        os.path.exists(silentinstall_filepath)):
        # Check if we can locate nodes using IPs from silentinstall.cfg if it exists
        for node_id in node_ids:
            avahi_ip = get_avahi_ip(silentinstall_filepath, node_id)
            ret = cvnwhelper.execute_command(["sudo","ping","-c","1",avahi_ip])
            if (ret == 0):
                cvlt_avahi_node_dict = cvavahi.parse_remote_node_avahi_config(avahi_ip)
                if ("status" in cvlt_avahi_node_dict):
                    cvlt_avahi_node_dict["uuid"] = node_id
                    cvlt_avahi_node_dict["ip"] = avahi_ip
                    cvlt_avahi_node_dict["host"] = node_id
                    cvlt_avahi_map[node_id] = cvlt_avahi_node_info(cvlt_avahi_node_dict)

    sorted_srnos = sorted(cvlt_avahi_map.keys())
    ordered_cvlt_avahi_map = collections.OrderedDict({})
    for srno in sorted_srnos:
        ordered_cvlt_avahi_map[srno] = cvlt_avahi_map[srno]        
    return ordered_cvlt_avahi_map

def is_valid_ipaddress(address):
    try:
        ipaddress.ip_address(unicode(address))
        return True
    except:
        return False


def get_from_cfgfile(config, section, key, boolean=False):
    if config.has_section(section) and config.has_option(section,key):
        if (boolean):
            return config.getboolean(section, key)
        return config.get(section,key)
    else:
        logging.error("Failed to find "+key+" inside section "+section)
        return ""

# Browse and pick 3 nodes for configuration including the current node 
# Returns True if all 3 nodes are found 
# else returns False with partially built hyperscale node map.
def select_avahi_nodes_for_configuration(node_ids = []):
    global hyperscale_node_map
    global last_error   
    ret = True
    logging.info("Querying to find peer information...")
    ordered_cvlt_avahi_map = browse_all_cvlt_avahi_nodes(node_ids)
    logging.info("Found {0} avahi nodes".format(len(ordered_cvlt_avahi_map.keys())))
    logging.info("Node Details:  {0}".format(dict(ordered_cvlt_avahi_map)))
    if (len(ordered_cvlt_avahi_map.keys()) < hyperscale_node_info.hyperscale_blk_size):
        last_error = "Failed to detect minimum number of HyperScale nodes"
        raise ValidationError ("Failed to detect minimum number of HyperScale nodes. Please ensure that cvavahi is exported from all nodes")

    if (len(ordered_cvlt_avahi_map.keys()) > 3):
        raise ValidationError("Detected more than 3 nodes. Please ensure that you have only 3 unconfigured HyperScale nodes. Refer to Commvault documentation for troubleshooting")

    hyperscale_node_map = { }

    if (len(node_ids) == 0):
        #
        # create a map
        # of 3 nodes (i.e. hyperscale block size)
        # which will be configured.
        #

        # Locate current node 1st
        current_nodeid = cvavahi.get_current_node_cvlt_uuid()
        current_blkid = cvavahi.get_current_node_cvlt_blkid()
        logging.info("Current node has UUID {0} and blkid {1}".format(current_nodeid, current_blkid))

        try:
            if (current_nodeid == ""):
                last_error = "Failed to detect UUID of current node"
                logging.error("select_avahi_nodes_for_configuration: Make sure commvault avahi service is exported from current node")
                return False

            if (current_nodeid in ordered_cvlt_avahi_map.keys()):
                hyperscale_node_map["CurrentNode"] = hyperscale_node_info("CurrentNode", ordered_cvlt_avahi_map[current_nodeid])
            else:
                last_error = "Failed to detect minimum number of HyperScale nodes"  
                logging.error("select_avahi_nodes_for_configuration: Make sure commvault avahi service is exported from current node")
                return False
        except Exception as str_e:
            logging.error("Exception thrown while trying to query peer information: {0}".format(str_e))
            logging.error("Please try clearing out the config directory and restart avahi, archd and httpd services on all nodes")
            raise ValidationError("Unable to discover peer nodes. Please check network and avahi settings.")

        del ordered_cvlt_avahi_map[current_nodeid]
        if (len(ordered_cvlt_avahi_map.keys()) == 0):
            return False

        # Pick up top 2 unconfigured nodes from rest of the node list
        # as peer nodes
        peerindex = 0
        for key in ordered_cvlt_avahi_map.keys():
            # Select hyperscale nodes only for current block id
            if (ordered_cvlt_avahi_map[key].blkid != current_blkid):
                continue

            # Filter out nodes which are already configured
            if (ordered_cvlt_avahi_map[key].status == "HyperScaleConfigured"):
                continue 
            try:
                hyperscale_node_map["PeerNode"+str(peerindex+1)] = hyperscale_node_info("PeerNode"+str(peerindex+1), 
                                                                                        ordered_cvlt_avahi_map[key])
            except Exception as str_e:
                logging.warning("Unable to add node with UUID {0} to hyperscale_node_map".format(key))
            peerindex = peerindex + 1 
            if (peerindex == 2):
                logging.info("Minimum number of nodes for configuration satisfied")
                break
        if (peerindex != 2):
            logging.error("Unable to find 3 unconfigured nodes for configuration.")
            logging.error("Please check network and avahi configurations")
            return False
    else:
        #
        # Collect avahi node info for only given node IDs.
        # This is needed when we are resuming failed setup.
        #
        peerIndex = 0
        current_blkid = cvavahi.get_current_node_cvlt_blkid()
        for node_id in node_ids:
            if (not (node_id in ordered_cvlt_avahi_map.keys())):
                logging.error("select_avahi_nodes_for_configuration: Could not locate avahi service with guid "+ node_id +" to resume the setup")
                ret = False
            elif (ordered_cvlt_avahi_map[node_id].blkid != current_blkid):
                logging.error("select_avahi_nodes_for_configuration: Invalid UUID entry in config file, please make sure all nodes from config file belong to same hyperscale block")
                ret = False
            else:
                if (node_id == cvavahi.get_current_node_cvlt_uuid()):
                    hyperscale_node_map["CurrentNode"] = hyperscale_node_info("CurrentNode", 
                                                                              ordered_cvlt_avahi_map[node_id])
                else:
                    hyperscale_node_map["PeerNode"+str(peerIndex+1)] = hyperscale_node_info("PeerNode"+str(peerIndex+1), 
                                                                                            ordered_cvlt_avahi_map[node_id])
                    peerIndex = peerIndex + 1 
    return ret
        

# Re-populates avahi node information
# Additionally, refresh hardware information for all nodes, if refresh_hw_info = True
# Returns True if refresh is successful for all nodes , returns False otherwise
def refresh_avahi_nodes(refresh_hw_info = True):
    global hyperscale_node_map

    start_time = current_milli_time()
    timeout_in_seconds = 120
    node_ids = []
    for key in hyperscale_node_map.keys():
        node_ids.append(hyperscale_node_map[key].uuid)
    while (1):
        ordered_cvlt_avahi_map = browse_all_cvlt_avahi_nodes(node_ids)
        ret = True
        nodes_found = 0
    
        for key in hyperscale_node_map.keys():
            node = hyperscale_node_map[key]
            if (not (node.uuid in ordered_cvlt_avahi_map)):
                node.missing = True
                logging.error("refresh_avahi_nodes: Could not locate avahi service with guid "+ node.uuid)
                ret = False

                current_time = current_milli_time()
                if ((current_time - start_time) >= (timeout_in_seconds * 1000)):
                    logging.error("Failed to connect to the remote host UUID " + node.uuid)
                    node.missing = True
                    return False
                else:
                    # Re-browse and check if node is available
                    logging.error("Failed to connect to the remote host UUID " + node.uuid + ". Let's rebrowse and see if it is available now")
                    time.sleep(5)
                    break
            else:
                logging.info("@ refresh_avahi_nodes Key: "+key+" Value: "+node.uuid)
                node.missing = False

            nodes_found = nodes_found + 1
            node.populate_avahi_info(ordered_cvlt_avahi_map[node.uuid])
            # Repopulate hardware info, in case of error empty strings will be assigned
            if (refresh_hw_info):
                i_ret = node.populate_hw_info()
                if (i_ret):
                    logging.error("Failed to retrieve hardware info for "+key+" ip="+node.ip+" uuid= "+ node.uuid)
                    node.arch_error = True
                    ret = False
                else:
                    node.arch_error = False

        # If all required nodes are located then exit  
        if (nodes_found == len(hyperscale_node_map.keys())):
            logging.info("Found the required nodes while refreshing avahi nodes")
            logging.info("Status of the nodes while refreshing avahi: {0}".format(pprint.pformat(hyperscale_node_map)))
            ret = True
            break      

    return ret


def gather_mgmt_ipaddr(config_file_path, node_id):
    config = ConfigParser.ConfigParser()
    config.read(config_file_path)

    if (not config.getboolean("network", "config_mgmt")):
        return ""
        
    mgmt_ip_addr = get_from_cfgfile(config,"management_ips", node_id) 
    return mgmt_ip_addr

def gather_network_config_params(config_file_path, node_id):
    config = ConfigParser.ConfigParser()
    config.read(config_file_path)
    
    eth_hashmap = { }
    # Gather data protection IP and storage pool IP for given UUID
    eth_hashmap['dp_ipaddr'] = get_from_cfgfile(config,"data_protection_ips", node_id)
    eth_hashmap['sp_ipaddr'] = get_from_cfgfile(config,"storagepool_ips", node_id)
    eth_hashmap['hname'] = cvnwhelper.gethname_from_ip(get_from_cfgfile(config,"data_protection_ips", node_id))

    # Gather data protection network info from 'network' section
    eth_hashmap['dp_netmask'] = get_from_cfgfile(config,"network", 'dp_netmask')
    eth_hashmap['sp_netmask'] = get_from_cfgfile(config,"network", 'sp_netmask')
    eth_hashmap['default_gw'] = get_from_cfgfile(config,"network", 'default_gw')
    eth_hashmap['nameserver1'] = get_from_cfgfile(config,"network", 'nameserver1')
    if (config.has_option("network", "nameserver2")):
        eth_hashmap['nameserver2'] = get_from_cfgfile(config,"network", 'nameserver2')
    else:
        logging.warning("Unable to find nameserver2 in config file: {0}".format(config_file_path))
        eth_hashmap['nameserver2'] = ""
    if (config.has_option("network", "nameserver3")):
        eth_hashmap['nameserver3'] = get_from_cfgfile(config,"network", 'nameserver3')
    else:
        logging.warning("Unable to find nameserver3 in config file: {0}".format(config_file_path))
        eth_hashmap['nameserver3'] = ""
    return eth_hashmap

def is_advanced_config_set(config_file_path):
    config = ConfigParser.ConfigParser()
    config.read(config_file_path)

    if (config.has_option("network", "jsonpath")):
        return True

    return False

def load_advanced_json(config_file_path):
    config = ConfigParser.ConfigParser()
    config.read(config_file_path)

    if (config.has_option("network", "jsonpath")):
        jsonpath = config.get("network", "jsonpath") 
        with open(jsonpath, "r") as fd:
            advanced_json = json.load(fd)
        return advanced_json 

    return {}


def configure_remote_node_nw(node, config_file_path):
    global jsonfilepath
    avahi_node_ip = node.ip
    if (node.status == "NwConfigured" or node.status == "NwConfigFailed" or node.status == "PrivateNwConfigured"):
        cvavahi.update_remote_avahi_config(avahi_node_ip, "status", "Unattempted")

    logging.info("Configuring network for node {0} using {1}".format(node, config_file_path))
    
    use_advanced_config = is_advanced_config_set(config_file_path)
    # Reset network config status fields for cvlt.service
    cvavahi.update_remote_avahi_config(avahi_node_ip, "data_protection_nw", "Unattempted")
    cvavahi.update_remote_avahi_config(avahi_node_ip, "storage_pool_nw", "Unattempted")
    cvavahi.update_remote_avahi_config(avahi_node_ip, "mgmt_nw", "Unattempted")

    logging.info("Completed updating avahi information for node {0}".format(node))

    # Send arch command to configure network
    if use_advanced_config:
        args = ["arch", "config-nw", avahi_node_ip]
        args.append(full_filepath(jsonfilepath))
    else:
        args = ["arch", "config-nw", avahi_node_ip]
        args.append(config_file_path)

    return cvnwhelper.execute_command(args)


def check_for_ip_conflicts(config_file_path, config_mgmt, nw_config_error_map):
    global hyperscale_node_map
    ret = 0

    for key in hyperscale_node_map:
        hyperscale_node = hyperscale_node_map[key]
        hashmap = gather_network_config_params(config_file_path, hyperscale_node.uuid)

        if (is_ip_already_in_use(hyperscale_node, "data_protection", hashmap["dp_ipaddr"]) == True):
            nw_config_error_map[key]["data_protection"] = get_error_message("ip_conflict")
            ret = -1
        
        if (is_ip_already_in_use(hyperscale_node, "storage_pool", hashmap["sp_ipaddr"]) == True):
            nw_config_error_map[key]["storage_pool"] = get_error_message("ip_conflict")
            ret = -1
            
        if (config_mgmt):
           mgmt_ipaddr = gather_mgmt_ipaddr(config_file_path, hyperscale_node.uuid)
           if (is_ip_already_in_use(hyperscale_node, "management", mgmt_ipaddr) == True):
               nw_config_error_map[key]["management"] = get_error_message("ip_conflict")
               ret = -1 
    return ret


def start_hyperscale_node_nwconfig(hyperscale_node, config_file_path, config_mgmt, nw_config_error, config_only_sp_nw = False):
    hashmap = gather_network_config_params(config_file_path, hyperscale_node.uuid)
    logging.info("Gathered network config parameters: {0}".format(hashmap))
    
    ret = configure_remote_node_nw(hyperscale_node, config_file_path)
    if (ret):
        # For advanced network configuration populate error message
        if ("error" in nw_config_error):
            nw_config_error["error"] = get_error_message("arch_service")
        else:
            nw_config_error["data_protection"] = get_error_message("arch_service")
        return ret
    return ret

def wait_for_hyperscale_node_nwconfig(hyperscale_node, config_file_path, config_mgmt, nw_config_error, config_only_sp_nw = False):
    hashmap = gather_network_config_params(config_file_path, hyperscale_node.uuid)
    b_ret = wait_for_nwconfig(hyperscale_node.uuid, config_mgmt, nw_config_error, config_only_sp_nw, 300)
    if (b_ret == False):
        logging.error("Failed to configure network for avahi node with UUID = "+hyperscale_node.uuid)
        return -1

    if config_only_sp_nw:
        # No need to configure the host yet.
        return 0

    # Execute arch command which will execute setupsds on remote node
    ret = cvnwhelper.execute_command(["arch", "manage", "commvault", "execute", 
                    hyperscale_node.ip,
                    cvnwhelper.setup_exec(), config_file_path])
    if (ret):
        # For advanced network configuration populate error message
        if ("error" in nw_config_error):
            nw_config_error["error"] = get_error_message("setupsds")
        else:
            nw_config_error["data_protection"] = get_error_message("setupsds")

    return ret

def populate_nw_config_error(avahi_node, config_mgmt, nw_config_error):
    if (avahi_node.data_protection_nw != "Configured"):
        nw_config_error["data_protection"] = get_error_message("nw_config")
   
    if (avahi_node.storage_pool_nw != "Configured"):
        nw_config_error["storage_pool"] = get_error_message("nw_config")

    if (config_mgmt and avahi_node.mgmt_nw != "Configured"):
        nw_config_error["management"] = get_error_message("nw_config")

current_milli_time = lambda: int(round(time.time() * 1000))   
def wait_for_nwconfig(remote_uuid, config_mgmt, nw_config_error, config_only_sp_nw, timeout_in_seconds):
    global hyperscale_node_map
    start_time = current_milli_time()
    attempt = 1

    key = ""
    for node_key in hyperscale_node_map.keys():
        if (hyperscale_node_map[node_key].uuid == remote_uuid):
            key = node_key
            break
 
    while (1):
        logging.info("Checking if network for avahi node " + remote_uuid  + " is configured. Attempt #" + str(attempt))
        attempt = attempt + 1
        
        # Let's browse all avahi nodes
        avahi_nodes = browse_all_cvlt_avahi_nodes()
        avahi_node_not_found = False   
        if (len(avahi_nodes.keys()) == 0) or (not (remote_uuid in avahi_nodes.keys())):
            avahi_node_not_found = True
   
        # Refresh avahi IP for node
        if (not avahi_node_not_found):    
            hyperscale_node_map[key].ip = avahi_nodes[remote_uuid].ip

        if ((not avahi_node_not_found) and avahi_nodes[remote_uuid].status == "NwConfigured"):
            logging.info("Avahi status for node " + remote_uuid  + " is now changed to Nwconfigured.")
            return True
        elif ( (not avahi_node_not_found) and avahi_nodes[remote_uuid].status == "NwConfigFailed") :
            # For advanced network configuration populate error message
            if ("error" in nw_config_error):
                nw_config_error["error"] = "Network configuration failed for node " + remote_uuid
            else:
                populate_nw_config_error(avahi_nodes[remote_uuid], config_mgmt, nw_config_error)
            logging.error("Network configuration failed for avahi node id " + remote_uuid)
            return False
        elif (config_only_sp_nw and (not avahi_node_not_found) and 
              avahi_nodes[remote_uuid].status == "PrivateNwConfigured"):
            if (avahi_nodes[remote_uuid].status == "PrivateNwConfigured"):
                logging.info("Avahi status for node " + remote_uuid  + " is now changed to PrivateNwConfigured.")
            else:
                logging.info("Avahi status for node " + remote_uuid  + " is now changed to storage_pool_nw=Configured.")
            return True  
        else:
            current_time = current_milli_time()
            if ((current_time - start_time) >= (timeout_in_seconds * 1000)):
                logging.error("Failed to connect to the remote host UUID" + remote_uuid)
                # For advanced network configuration populate error message
                if ("error" in nw_config_error):
                    nw_config_error["error"] = get_error_message("avahi_service")
                else:
                    nw_config_error["data_protection"] = get_error_message("avahi_service")
                return False
            else:
                # Sleep and try after 5 seconds
                time.sleep(5)
    return False    

def removeregistryentry(inputf,key,output):
    with open(inputf) as f:
        entries = f.read().splitlines()

    wfh = open(output, "w")

    for entry in entries:
        entry = entry.strip()
        lst = entry.split()
        if (len(lst) == 2):
            val = lst[1]
            if (lst[0] != key):
                wfh.write(lst[0]+"  "+ val+"\n")
        else:
            wfh.write(lst[0]+"\n")
    wfh.close()
    f.close()
            
def read_remote_registry(node,registry,key):
    #Copy the registry from remote node to this node
    ret = cvnwhelper.execute_command(["arch", "read-file", node, registry, scratch_file])
    if (ret != 0):
        logging.error ("Failed to read registry file "+registry+" from remote node "+node)
        return ""

    value=cvavahi.getregistryentry(registry,key)
    if (value == ""):
        logging.error ("Failed to read registry file "+registry+" from remote node "+node)
        return ""

    return value.strip()

def update_remote_registry(node,registry,key,value):
    #Copy the registry from remote node to this node
    ret = cvnwhelper.execute_command(["arch", "read-file", node, registry, scratch_file])
    if (ret != 0):
        logging.error ("Failed to read registry file "+registry+" from remote node "+node)
        return -1

    #Create a new registry file after updating the required key
    cvavahi.setregistryentry(scratch_file,key,value)

    #Copy the updated registry file to the remote node
    ret = cvnwhelper.execute_command(["arch", "write-file", node, scratch_file, registry])
    if (ret != 0):
        logging.error ("Failed to copy registry file "+registry+" to remote node "+node)
        return -1

def remove_remote_registry_entry(node,registry,key):
    #Copy the registry from remote node to this node
    ret = cvnwhelper.execute_command(["arch", "read-file", node, registry, scratch_file])
    if (ret != 0):
        logging.error ("Failed to read registry file "+registry+" from remote node "+node)
        return -1

    #Create a new registry file after updating the required key
    removeregistryentry(scratch_file,key,scratch_file+".out")

    #Copy the updated registry file to the remote node
    ret = cvnwhelper.execute_command(["arch", "write-file", node, scratch_file+".out", registry])
    if (ret != 0):
        logging.error ("Failed to copy registry file "+registry+" to remote node "+node)
        return -1


def verify_password():
    global username
    global passwd
    global verify_password_script
    if not os.path.isfile(verify_password_script):
        logging.error('Failed to find files required to verify passwords')
        return False

    cmd = "sudo {0} {1} '{2}'".format(verify_password_script, username, passwd)
    try:
        ret = os.system(cmd)
        if 0 == ret:
            return True
        return False
    except Exception as str_e:
        logging.error("Exception while verifying password: {0}".format(str_e))
        return False

# If user tries to skip to page2 without entering the credentials
def page_validate_password():
    global username
    global passwd
    return verify_password()

def write_to_configfile(json_data, last_node, write_cluster_info = True):
    global silentinstall_filepath
    global hyperscale_node_map
    global temp_passfilepath


    # Check if advanced configuration is being used
    if ("useAdvancedSettings" in json_data and json_data["useAdvancedSettings"] == True):
        ret = cvnwhelper.write_to_configfile(json_data, last_node, silentinstall_filepath, hyperscale_node_map, write_cluster_info)
        return ret


    Config = ConfigParser.ConfigParser ()
    cfgfile = open (silentinstall_filepath, "w+")

    # Store password in separate file
    PassConfig = ConfigParser.ConfigParser ()
    passcfgfile = open (temp_passfilepath, "w+")

    if (json_data["global"]["different_rootpass"] == False):
        use_diff_password = False
        common_root_password = json_data["global"]["root_pwd"]
    else:
        use_diff_password = True


    # Populate IP addresses and passwords for 3 nodes.
    section_list_pernode = ["data_protection_ips", "passwords", "storagepool_ips", "ipmi_ips"]
    allowed_node_identifiers = ["CurrentNode", "PeerNode1", "PeerNode2"]
    for section in section_list_pernode:
        Config.add_section(section)
        PassConfig.add_section(section)
        logging.info ("Looking for section {0}".format(section))
        inner_dict = json_data[section]
        for inner_key in inner_dict.keys():
            logging.info ("@write_to_configfile Section: {0} Key: {1}".format(section, inner_key))
            if not (inner_key in allowed_node_identifiers):
                logging.error("Please provide valid JSON; Invalid key: {0} found in section: {1}".format(inner_key, section))
                sys.exit(1)

            # Enrypt and store the password in a separate file
            if "passwords" == section and (not use_diff_password):
                PassConfig.set (section, hyperscale_node_map[inner_key].uuid, common_root_password)
            elif True == cvnwhelper.needs_encrytion(section, inner_key):
                PassConfig.set (section, hyperscale_node_map[inner_key].uuid, inner_dict[inner_key])
            else:    
                Config.set (section, hyperscale_node_map[inner_key].uuid, inner_dict[inner_key])
     
    # Populate network section 
    Config.add_section("network")
    Config.set("network", "sp_netmask", json_data["network"]["sp_netmask"])
    inner_dict = json_data["network"]
    for inner_key in inner_dict.keys():
        if (inner_key == "nameservers"):
           continue
        Config.set ("network", inner_key, inner_dict[inner_key])
    cnt = 1
    for name_server in inner_dict["nameservers"]:
        Config.set ("network", "nameserver"+str(cnt), name_server)
        cnt = cnt + 1
    if write_cluster_info:
        # Populate cluster section with commserve details 
        Config.add_section("cluster")
        PassConfig.add_section("cluster")
        inner_dict = json_data["cluster"]
        for inner_key in inner_dict.keys():
            if True == cvnwhelper.needs_encrytion("cluster", inner_key):
                PassConfig.set ("cluster", inner_key, inner_dict[inner_key])
            else:
                Config.set ("cluster", inner_key, inner_dict[inner_key])
  
        Config.set ("cluster", "cshname", json_data["cluster"]["cshname"])
        Config.set ("cluster", "csip", cvnwhelper.hostname_to_ipaddr(hyperscale_node_map, json_data["cluster"]["cshname"]))
        if (inner_dict["existing_cs"] == "on"):
            Config.set ("cluster", "username",  json_data["cluster"]["username"])
        else:
            # Existing CS case and standby KVM case are disjoint
            if "standby_kvm_cs" in inner_dict:
                standby_kvm_cs = json_data["cluster"]["standby_kvm_cs"]
                if standby_kvm_cs:
                    Config.set ("cluster", "standby_kvm_csname", json_data["cluster"]["standby_kvm_csname"])
                    Config.set ("cluster", "standby_kvm_csprid", json_data["cluster"]["standby_kvm_csprid"])

        if (True == json_data["network"]["config_mgmt"]):
            Config.set("cluster", "csbackupnetworkip", cvnwhelper.hostname_to_ipaddr(hyperscale_node_map, json_data["cluster"]["commserve_backup_network"]))
            Config.set("cluster", "csbackupnetworkhname", json_data["cluster"]["commserve_backup_network"])
    
    Config.add_section("global")
    Config.set("global", "last_node", last_node)
    # Set the Standby KVM CS boolean here
    # For MA only case and RA, the code in setup.py should be able to ignore this value
    # Set this value only when cluster details are obtained
    if write_cluster_info:
        Config.set("global", "standby_kvm_cs", json_data["cluster"]["standby_kvm_cs"])
    # Add the field creator to silent install config file
    Config.set("global", "creator", hyperscale_node_map["CurrentNode"].uuid)
    Config.set("global", "enable_bonding", json_data["global"]["enable_bonding"])
    if (json_data["global"]["enable_bonding"] == True):
        Config.set("global", "bonding_mode", json_data["global"]["bonding_mode"])

    PassConfig.write (passcfgfile)
    Config.write (cfgfile)
    cfgfile.close()
    passcfgfile.close()
    return True

def get_value(json_data):
    key_value_status = { }
    failed = False  
    # Perform the needed backend function to extract the value for the key
    key = json_data['Key']
    if (key == 'RESTART_LEVEL'):
        #Check whether this is a fresh install or a failed install
        key_value_status['Value'] = "FRESH_INSTALL"
    return key_value_status 

def validate_ptr_records(json_data):
    ptr_status = { }
    failed = False  
    # Check whether each of the IP address has a valid PTR record
    network_types = ["data_protection_ips"]
    allowed_node_identifiers = ["CurrentNode", "PeerNode1", "PeerNode2"]
    for network in network_types:
        ptr_status[network] = {}
        for node in allowed_node_identifiers:
            ret = cvnwhelper.validate_ipaddr_to_hname(json_data[network][node])
            if ("Failed" == ret):
                failed = True
            ptr_status[network][node] = ret

    if (True == failed):
        ptr_status["status"] = "FAILED"
    else:       
        ptr_status["status"] = "OK"
    return ptr_status 


def validate_hardware_configuration(json_data):
    global hyperscale_node_map
    hw_status = { }
    failed = False 
    management_network = json_data["useSeparateManagementNetwork"]

    # All hardware validations are ignored in case of lab installs
    if HS_CONFIG["IS_LAB_INSTALL"]:
        hw_status["status"] = "OK"
        return hw_status
    
    allowed_node_identifiers = ["CurrentNode", "PeerNode1", "PeerNode2"]

    # Precautionary step to make sure the Node map is populated before proceeding further. The index page does the population. 
    # This will handle cases where the end point is invoked directly from the browser
    for node in allowed_node_identifiers:
        if node not in hyperscale_node_map:
            logging.error("Peer details are not populated. Please invoke {0} after avahi nodes are dicovered".format(validate_hardware_configuration.__name__))
            raise ValueError("Peer details are not populated to proceed further")

    # Execute commands to get disk configurations
    for key in allowed_node_identifiers:
        obj = hyperscale_node_map[key]
        ip = hyperscale_node_map[key].ip
        uuid = hyperscale_node_map[key].uuid
        logging.info("Getting disk configurations from node {0} with serial {1}..".format(ip, uuid))

        # cmdfilepath = os.getcwd() + "/config/cvavahi_cmd"
        cmdfilepath = "/tmp/cvavahi_cmd"
        with open(cmdfilepath, "w") as fd:
            fd.write("get_all_drives")

        # Write this file into remote node
        args = ["arch", "write-file", obj.ip, cmdfilepath, "/tmp/hwebtool_command"]
        ret = cvnwhelper.execute_command(args)
        if (ret):
            logging.error("arch write failed on: {0}, serial: {1}".format(ip, uuid))
            continue

        ret = cvnwhelper.execute_command(["arch", "manage", "commvault", "execute", ip, "cvavahi.py", "/tmp/hwebtool_command"])
        if (ret):
            logging.error("Unable to execute avahi command")
        logging.info("Completed executing command on the remote node: {0}".format(ip))

        args = ["arch", "read-file", ip, "/tmp/cvdrivelist.json", "/tmp/cvdrivelist_{0}.json".format(uuid)]
        ret = cvnwhelper.execute_command(args)
        if (ret):
            logging.error("arch read failed on: {0}, serial: {1}".format(ip, uuid))
            continue

    logging.info("Gathered storage configurations from all nodes. Validating storage constraints..")
    d_disks = {}
    for filename in glob.glob('/tmp/cvdrivelist_*.json'):
        serialno = filename.split('.')[0].split('/')[-1].split('_')[-1]
        with open(filename, 'r') as fd:
            d_disks[serialno] = json.load(fd)
    logging.info("Storage information: {0}".format(d_disks))

    l_sas = []
    l_nvme = []
    # Make sure all nodes have same # of SAS and NVME drives
    for serial in d_disks.keys():
        if len(l_sas) == 0 or l_sas[0] != len(d_disks[serial]['sas']):
            l_sas.append(len(d_disks[serial]['sas']))
        if len(l_nvme) == 0 or l_nvme[0] != len(d_disks[serial]['nvme']):
            l_nvme.append(len(d_disks[serial]['nvme']))
    
    if len(l_sas) > 1:
        raise ValidationError('Uneven number of SAS drives across nodes detected. Please check the nodes for device errors.')
    if len(l_nvme) > 1:
        raise ValidationError('Uneven number of NVMe drives across nodes detected. Please check the nodes for device errors')

    logging.info("Successfully validated number of SAS and NVMe drives across nodes...")

    for serial in d_disks.keys():
        for mount_point in d_disks[serial]["hv_mounts"].keys():
            if not d_disks[serial]["hv_mounts"][mount_point]:
                raise ValidationError("The node {0} does not have {1} mounted. Please verify".format(serial, mount_point))

    l_sas = []
    l_nvme = []
    # Make sure the size of drives is same across nodes. Use the current node as reference
    for drives in d_disks[hyperscale_node_map["CurrentNode"].uuid]['sas']:
        l_sas.append(drives['devsize'])
    for drives in d_disks[hyperscale_node_map["CurrentNode"].uuid]['nvme']:
        l_nvme.append(drives['devsize'])

    for node in ["PeerNode1", "PeerNode2"]:
        serial = hyperscale_node_map[node].uuid
        for drives in d_disks[serial]['sas']:
            if drives['devsize'] not in l_sas:
                raise ValidationError('SAS Drive sizes are not the same as this node for {0}. Please verify the storage configuration'.format(node.ip))
        for drives in d_disks[serial]['nvme']:
            if drives['devsize'] not in l_nvme:
                raise ValidationError('NVMe Drive sizes are not the same as this node for {0}. Please verify the storage configuration'.format(node.ip))

    logging.info("Completed validating storage information successfully...")

    #Check whether each node has a valid hardware configuration
    for node in allowed_node_identifiers:
        hyperscale_type = get_hyperscale_type(hyperscale_node_map[node].ip)
        logging.info("Validating {0}...".format(node))
        logging.info("Node: {0} HyperScale type: {1}".format(hyperscale_node_map[node], hyperscale_type))
        uuid = hyperscale_node_map[node].uuid
        hwinfo_config = "config/" + uuid+"_hwinfo.cfg"
        config = ConfigParser.ConfigParser()
        config.read(hwinfo_config)

        num_10G_links = len(hyperscale_node_map[node].tengig_list)
        num_1G_links = len(hyperscale_node_map[node].onegig_list)
        num_flash_drives = 0
        num_sas_drives = 0

        # Check if IPMI link is connected
        if (hyperscale_node_map[node].ipmi_ipaddr == "0.0.0.0"):
            hw_status[node] = "IPMI_MISSING"
            failed = True
            continue
            
        logging.info ("Number of 10G links: "+str(num_10G_links))
        if (num_10G_links < 2):
            hw_status[node] = "NWLINK_MISSING"   
            failed = True
            continue
            
        # Check that 4 10G links are present for bonding in simple n/w config for appliance
        if (get_hyperscale_type("localhost") != "ReferenceArchitecture"):
            if (json_data['useBondedInterface']):
                if (num_10G_links < 4):
                    logging.error('Number of 10G links: {0}'.format(num_10G_links))
                    hw_status[node] = "BOND10GLINKS_MISSING"
                    failed = True
                    continue

        if (True == management_network):
            logging.info ("Number of 1G links: "+str(num_1G_links))
            if (num_1G_links < 1):
                hw_status[node] = "INSUF_1G_LINK"
                failed = True
                continue

        hw_status[node] = "OK"   

    if (True == failed):
        hw_status["status"] = "FAILED"
    else:       
        hw_status["status"] = "OK"

    return hw_status 


def validate_netmask_entries(json_data):
    netmask_status = { }
    failed = False  
    # Check whether each of the Data protection and StoragePool IP address are from different subnets
    dp_netmask = json_data["network"]["dp_netmask"]
    sp_netmask = json_data["network"]["sp_netmask"]


    allowed_node_identifiers = ["CurrentNode", "PeerNode1", "PeerNode2"]
    for node in allowed_node_identifiers:
        subnet_match = False
        dp_ipaddr = json_data["data_protection_ips"][node]
        sp_ipaddr = json_data["storagepool_ips"][node]
        if (True == cvnwhelper.compare_ipv4_networks(dp_ipaddr,dp_netmask,sp_ipaddr,sp_netmask)):
            logging.error ("Data protection and StoragePool networks are from same subnet for: "+node)
            subnet_match = True
            failed = True

        if (True == subnet_match):
            netmask_status[node] = "FAILED"
        else:
            netmask_status[node] = "OK"

    if (True == failed):
        netmask_status["status"] = "FAILED"
    else:
        netmask_status["status"] = "OK"
        logging.info("Subnet validation completed successfully")
    return netmask_status

def validate_dns_entries(json_data):
    global hyperscale_node_map
    
    dns_status = {}
    failed = False
    # Check whether the provided CommServe and Management fqdn resolve to valid IP address
    allowed_node_identifiers = ["Commserve_fqdn", "Commserve_backup_fqdn"]
    for node in allowed_node_identifiers:
        if (node == "Commserve_backup_fqdn" and json_data["useSeparateManagementNetwork"] == False):
            continue
        fqdn = json_data[node]
        try:
            ipaddr = cvnwhelper.hostname_to_ipaddr(hyperscale_node_map, fqdn)
        except Exception as str_e:
            logging.error("Error validating FQDN {0}: {1}".format(fqdn, str_e))
            raise ValidationError("Unable to resolve {0} to a valid IP address".format(fqdn))

        if (0 == len(ipaddr)):
            logging.error (fqdn+" does not resolve to valid IP address")
            failed = True
            dns_status[node] = "FAILED"
        else:
            dns_status[node] = "OK"

    dns_status["status"] = "FAILED" if failed else "OK"
    return dns_status

def get_ipmi_ipaddress(node):
    from subprocess import Popen, PIPE, STDOUT
    cmd = "arch getipmiaddress "+node
    p = Popen(cmd, shell=True, stdin=PIPE, stdout=PIPE, stderr=PIPE, close_fds=True)
    for entry in p.stdout:  
        entry=entry.rstrip()
        return entry
    return ""

def get_nwintfx(node,link_speed,hwinfo_config):
    ret = cvnwhelper.execute_command(["arch", "get-hwinfo", node, hwinfo_config])
    if (ret != 0):
        # Return empty list to indicate failure 
        return []

    #Get the IPMI ipaddress from the remote host and update the config file
    ipmi_ipaddress = get_ipmi_ipaddress(node)
    config = ConfigParser.ConfigParser()
    config.read(hwinfo_config)
    if (not config.has_section("IPMI_INFO")):
        config.add_section("IPMI_INFO")
    config.set("IPMI_INFO","IP_ADDRESS",ipmi_ipaddress)
    cfgfile=open(hwinfo_config, "w+")
    config.write(cfgfile)
    cfgfile.close()

    #Get the following from the hardware information config file
    #<device,mac address,link speedi,netmask>
    nwintfx_list = [ ]
    options = config.options("LINK_SPEED_INFO")     
    for option in options:
        speed = get_from_cfgfile(config,"LINK_SPEED_INFO", option)
        if (int(speed) >= link_speed):
            mac = get_from_cfgfile(config,"MAC_ADDRESS_INFO", option)         
            nwinfo = { }
            nwinfo['device'] = option
            nwinfo['link_speed'] = link_speed
            nwinfo['mac_address'] = mac
            nwintfx_list.append(nwinfo)     
    
    #Sort the dictionary of network interfaces by the device name
    nwintfx_list = sorted(nwintfx_list, key = itemgetter('device'))
    return nwintfx_list

def get_links(cfg_file):
    # Parse the hardware configuration file and extract the active 10gig network interfaces
    config = ConfigParser.ConfigParser()
    config.read(cfg_file)
    links = config.options("LINK_SPEED_INFO")
    return links

def get_mac_addr(cfg_file, device):
    config = ConfigParser.ConfigParser()
    config.read(cfg_file)
    return config.get("MAC_ADDRESS_INFO", device)


def get_link_speed(cfg_file, device):
    config = ConfigParser.ConfigParser()
    config.read(cfg_file)
    return config.get("LINK_SPEED_INFO", device)

def get_ipaddr_info(cfg_file):
    #Parse the hardware configuration file and extract the network interface and ipaddress assigned to the network interface
    config = ConfigParser.ConfigParser()
    config.read(cfg_file)
    options = config.options("IP_ADDRESS_INFO")
    ipaddr_map = { }
    for option in options:
        ipaddr = get_from_cfgfile(config,"IP_ADDRESS_INFO", option)
        if is_valid_ipaddress(ipaddr) == False:
            ipaddr = ""
        ipaddr_map[option] = ipaddr
    return ipaddr_map

def get_netmask_info(cfg_file):
    #Parse the hardware configuration file and extract the netmask of the interface
    config = ConfigParser.ConfigParser()
    config.read(cfg_file)
    options = config.options("NETMASK_INFO")
    netmask_map = { }
    for option in options:
        netmask_map[option] = get_from_cfgfile(config,"NETMASK_INFO", option)
    return netmask_map

def get_data_protection_entry(entrymap,nwinfo,cfg_file):
    #Check whether the first entry is in the entrymap, if not then check whether ovirtmgmt is in the entrymap
    if len(nwinfo):
        device=nwinfo[0]['device']
        if device in entrymap:
            return entrymap[device]
        elif 'hs1300' in entrymap:
            return entrymap['hs1300']
        else:
            return ''
    return ''

def get_storagepool_entry(entrymap,nwinfo,cfg_file):
    if len(nwinfo) >= 2: 
        device=nwinfo[1]['device']
        if device in entrymap:
            return entrymap[device]
        else:
            return ''
    return ''

def get_1gig_ips(cfg_file):
    one_gig_nw_list = [ ]
    config = ConfigParser.ConfigParser()
    config.read(cfg_file)
    options = config.options("LINK_SPEED_INFO")
    for option in options:
        speed = get_from_cfgfile(config,"LINK_SPEED_INFO", option)
        if (int(speed) == 1000):
            mac = get_from_cfgfile(config,"MAC_ADDRESS_INFO", option)         
            nwinfo = { }
            nwinfo['device'] = option
            nwinfo['link_speed'] = 1000
            nwinfo['mac_address'] = mac
            one_gig_nw_list.append(nwinfo)
    
    #Sort the dictionary of network interfaces by the device name
    one_gig_nw_list = sorted(one_gig_nw_list, key = itemgetter('device'))
    return one_gig_nw_list

def get_management_ipaddr(nwinfo,cfg_file):
    ipaddr_map = get_ipaddr_info(cfg_file)
    #Check whether the first mac address is in the ipaddr_map, if not then check whether ovirtmgmt is in the ipaddr_map     
    if len(nwinfo): 
        device=nwinfo[0]['device']
        if device in ipaddr_map:
            return ipaddr_map[device]
        else:
            return ''

def get_blkdev_info(cfg_file):
    #Parse the hardware configuration file and extract the number of block devices of each type
    config = ConfigParser.ConfigParser()
    config.read(cfg_file)
    options = config.options("BLOCK_DEVICE_INFO")
    size_map = { }  
    for option in options:
        blkdev_size = int(get_from_cfgfile(config,"BLOCK_DEVICE_INFO", option))
        if (blkdev_size in size_map):
            size_map[blkdev_size] = size_map[blkdev_size]+1
        else:           
            size_map[blkdev_size] = 1
    od = collections.OrderedDict(sorted(size_map.items()))          
    return od

def get_ipmi_info(cfg_file):
    #Parse the hardware configuration file and extract IPMI information
    config = ConfigParser.ConfigParser()
    config.read(cfg_file)
    ipmi_ipaddr = get_from_cfgfile(config,"IPMI_INFO","IP_ADDRESS")
    return ipmi_ipaddr    

def get_ipaddr_list():
    ip_list = []
    for interface in interfaces():
        address = ifaddresses(interface)
        if AF_INET in address:
            if (address[AF_INET] != None):       
                for link in address[AF_INET]:
                    ip_list.append(link['addr'])
    return ip_list

def get_first_macid(nwinfo):
    return nwinfo[0]['mac_address']

def form_network_desc(nwinfo_item,desc):
    linkspeed=""        
    if (int(nwinfo_item['link_speed']) >= 10000):
        linkspeed="10G"
    elif (int(nwinfo_item['link_speed']) == 1000):
        linkspeed="1G"
    else:    
        linkspeed="No link"
    hwdesc = nwinfo_item['device'] + "  " + nwinfo_item['mac_address'] + "  " + linkspeed + "  " + desc
    return hwdesc    

def form_blkdev_desc(key,value):
    size = int(key)
    disk_type = ""
    # Filter out disks whose size is less than 128GB, as they are not necessary
    if (size < 128):
        return ""
    elif (size >= 128 and size <= 500):
        disk_type = "SSD"
    elif (size > 500 and size <= 2048):
        disk_type = "FLASH"
    elif (size > 2048):
        disk_type = "SAS"
    desc = disk_type + ": " + str(key) + "GB" + " X " + str(value)
    return desc

def append_blkdev_desc(hwdesc_list, blkdevinfo):
    for key,value in blkdevinfo.iteritems():
        blkdesc = form_blkdev_desc(key,value)
        if ("" != blkdesc):
            hwdesc_list.append(blkdesc)


    return hwdesc_list

def parse_nameserver():
    nsentries={ }
    ns_list=[ ]
    search_list=[ ] 
    fhandle=open("/etc/resolv.conf","r")
    for line in fhandle:
        line=line.rstrip()
        lst=line.split()
        if (len(lst)>=2):
            if (lst[0]=="search"):
                for domain in lst[1:]:
                    search_list.append(domain)
            elif (lst[0]=="nameserver"):
                ns_list.append(lst[1])
                
    fhandle.close() 

    nsentries["search"]=search_list
    nsentries["nameservers"]=ns_list
    return nsentries

def get_default_gw():
    cmd="route -n | grep 0.0.0.0 | head -n 1 | awk \'{print $2}\'"
    #Launch the shell command:
    process = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=None, shell=True)
    output = process.communicate()
    vec=output[0].strip().split('\n')
    return vec

def validate_password(password):
    #Check whether the password meets the following criteria
    #Minimum of eight characters
    #One uppercase letter
    #One lowercase letter
    #One number
    #One special character from ~!@#$%^&*_-+=`|\(){}[]:;<>,.?/
    is_upper_case=False 
    is_lower_case=False 
    is_digit=False      
    is_special=False    

    if (len(password)<8):
        return False        

    special_charecter_map=['~','!','@','#','$','%','^','&','*','-','+','=','`','|','(',')','{','}','[',']',':',';','>','.','?','/'] 
    for ch in password:
        if (ch.isupper()):
            is_upper_case=True
        if (ch.islower()):
            is_lower_case=True
        if (ch.isdigit()):
            is_digit=True
        if (ch in special_charecter_map):
            is_special=True

    if (is_upper_case and is_lower_case and is_digit and is_special):
        return True

    return False 

def is_valid_hostname(hostname, iswindowsmachine = False):
    if hostname[-1] == ".":
        # Strip exactly one dot from the right, if present
        hostname = hostname[:-1]

    if len(hostname) > 253:
        return False

    labels = hostname.split(".")
    # TLD must be not all-numeric
    if re.match(r"[0-9]+$", labels[-1]):
        return False

    # Windows by default removes the extra characters>15
    # netbios name
    if (iswindowsmachine and len(labels[0]) > 15):
        return False

    allowed = re.compile(r"(?!-)[a-z0-9\d-]{1,63}(?<!-)$", re.IGNORECASE)
    return all(allowed.match(label) for label in labels)


@app.route ('/timezone', methods=['GET', 'POST'])
def configure_timezone():
    if request.method == 'GET':
        o_config = NonInstallConfiguration()
        o_data = {
            'all_tz': o_config.get_all_timezones(),
            'current_tz': o_config.get_current_timezone()
        }
        resp = jsonify(o_data)
        return resp
    if request.method == 'POST':
        # Check if content type is application/json
        if not request.is_json:
            raise ValidationError('Server expects a JSON post request')
        content = request.get_json()
        if 'time_zone' not in content:
            raise ValidationError('Unable to find time_zone key in client request')
        str_req_tz = str(content['time_zone'])
        logging.info("Client requested time zone to be set to {0}".format(str_req_tz))
        o_config = NonInstallConfiguration()
        o_config.set_current_timezone(str_req_tz)
        return json.dumps({'status':'OK'})


# Caller: index.js
# Called to display summary of setup execution
@app.route ('/summary', methods=['GET', 'POST'])
def summary():
    if (not page_validate_password()):
        logging.error ("Incorrect credentials for user %s for route setup", username)
        return render_template('index.html')
    
    return display_summary()


def display_summary(render = True):
    global progress_filepath

    if (not render):
        return redirect(url_for('summary'))

    # gather warning messages from progress file
    warning_list = []
    for filename in glob.glob(os.path.join('config', 'progress_*.txt')):
        file_path = os.path.join(os.getcwd(), filename)
        file_path = os.path.abspath(file_path)
        l_warn = cvavahi.gather_warnings(file_path)
        warning_list += l_warn

    if (len(warning_list) == 0):
        return render_template('summary.html',
                               status=json.dumps("success"),
                               csname=adminconsole(),
                               warning_list=json.dumps(warning_list))
    else:
        return render_template('summary.html',
                               status=json.dumps("warning"),
                               csname=adminconsole(),
                               warning_list=json.dumps(warning_list))


# Caller: index.js
# Called to display current status of setup
@app.route ('/processList', methods=['GET', 'POST'])
def processList():
    if (not page_validate_password()):
        raise ValueError("Incorrect credentials for user {0} for route setup".format(username))
    return display_progress()



def gather_logs():
    global hyperscale_node_map
    global logs_basename

    if "CurrentNode" in hyperscale_node_map:
        this_ip = hyperscale_node_map["CurrentNode"].ip
    else:
        logging.error("hyperscale_node_map doesn't contain required key.")
        return -1

    curr_time = strftime("%Y-%m-%d", gmtime())
    logs_basename = "cvhyperscale-" + curr_time
    logs_tar_dir = os.getcwd() + "/static/" + logs_basename
    logs_tar_file = os.getcwd() + "/static/" + logs_basename + ".tar.gz"
    refresh_avahi_nodes()

    input_file = os.getcwd() + "/static/input_file"

    for key in hyperscale_node_map:
        node = hyperscale_node_map[key]
        uuid = node.uuid.replace(" ","")
        ip = node.ip

        local_dest_path = logs_tar_dir + "/" + uuid
        cmd = "mkdir -p " + local_dest_path
        ret = os.system(cmd)
        if ret < 0:
            logging.error("Couldn't create log dir " + local_dest_path + " for node : " + uuid)
            continue

        remote_dest_path = os.getcwd() + "/static/" + uuid + ".tar.gz"
        local_dest_path = logs_tar_dir + "/" + uuid + "/" + uuid + ".tar.gz"
        
        # Create a file with the arguments for arch manage execute in it
        f = open(input_file, "w")
        f.write(remote_dest_path+"\n"+local_dest_path+"\n"+this_ip)
        f.flush()
        f.close()

        # Write this file into remote node
        args = ["arch", "write-file", ip, input_file, input_file]
        ret = cvnwhelper.execute_command(args)
        if (ret):
            logging.error("arch write failed on : " + ip + ". Cannot gather logs for node : " + uuid)
            continue

        args = ["arch", "manage", "commvault", "execute", ip, "cvgatherlogs.py", input_file]
        ret = cvnwhelper.execute_command(args)
        if (ret):
            logging.error("Could not gather logs for node : " + uuid)
            continue

        #Fix this perm in remote host otherwise apache won't be able to delete it once done
        args = ["arch", "manage", "commvault", "execute", this_ip, "cvfixperm.py", local_dest_path]
        ret = cvnwhelper.execute_command(args)
        if (ret):
            logging.error("Failed to execute cvfixperm on logs : " + local_dest_path)

    if os.path.exists(input_file):
        os.remove(input_file)

    # Now all the logs have been gathered and put in logs_tar_dir
    # We will tar this dir and display link
    if not os.path.exists(logs_tar_dir):
        logging.error("Failed to create compressed logs file.")
        return -1

    cmd = "sudo tar -zcf " + logs_tar_file + " " + logs_tar_dir
    ret = os.system(cmd)
    if ret < 0:
        if os.path.exists(logs_tar_dir):
            shutil.rmtree(logs_tar_dir)
        logging.error("Failed to create compressed logs file.")
        return ret

    if os.path.exists(logs_tar_dir):
        shutil.rmtree(logs_tar_dir)
    return 0

def display_progress(render = True):
    global hyperscale_node_map
    global progress_filepath
    global setup_process
    global setup_process_standby

    image_number = -1

    if (not render):
        return redirect(url_for('processList'))

    setup_failed = False
    setup_is_active_on = is_setup_in_progress(True)

    setup_status = "Pending"
    completed_milestones = []
    inprogress_milestones = []
    failed_milestones = []
    remaining_milestones = []

    current_node_serial_num = cvavahi.get_current_node_cvlt_uuid()
    d_status = {}
    d_status[current_node_serial_num] = {}

    if setup_is_active_on is None:
        if (hyperscale_configured()):
            # redirect to summary app route
            return display_summary(False)
        node_failed = prev_setup_failed_on()
        if node_failed is not None:
            setup_failed = True
            d_status[current_node_serial_num]["desc"] = "configuring node {0}".format(node_failed.ip)
            if setup_process is not None:
                ret = setup_process.poll()
                if ret is not None:
                    logging.error("Setup failed with exit code: ".format(ret))

    if setup_is_active_on == hyperscale_node_map["CurrentNode"]:
        if setup_process is None:
            logging.error ("No active setup found")
        else:  
            # Poll setup process status
            ret = setup_process.poll()
            if (ret is not None):
                setup_process = None
                if (ret == 0):
                    logging.info ("*** Setup completed successfully ***")
                else:
                    logging.error ("Setup failed with error = ".format(ret))
                    setup_failed = True
            else:
                logging.info ("Setup is still in progress")


    # Progress of the other node in case of standby CS
    for key in hyperscale_node_map:
        obj = hyperscale_node_map[key]

        dest_path = os.path.join(os.getcwd(), "config", "progress_{0}.txt".format(obj.uuid))

        if key != "CurrentNode":
            args = ["arch", "read-file", obj.ip, dest_path, dest_path]
            ret = cvnwhelper.execute_command(args)
            if (ret):
                logging.warning("Unable to copy {0} to this node. It may be absent".format(dest_path))

            args = ["chown", "apache:apache", dest_path]
            ret = cvnwhelper.execute_command(args)
            if (ret):
                logging.warning("Unable to change permissions of {0}".format(dest_path))
    
    for filename in glob.glob(os.path.join('config', 'progress_*.txt')):
        serial_number = filename.split('/')[-1].split('_')[1].split('.')[0]
       
        if serial_number != current_node_serial_num and os.path.getsize(filename) != 0:
            dest_path = os.path.join(os.getcwd(), filename)
            logging.info("Processing file: {0}...".format(filename))

            # Process the standby CS install progress
            progress_config = ConfigParser.ConfigParser()
            progress_config.read(dest_path)
            for milestone in progress_config.sections():
                if int(get_from_cfgfile(progress_config, milestone, "cpp")) == -1:
                    logging.error("HyperScale configuration failed at: {0}".format(get_from_cfgfile(progress_config, milestone, "desc")))
                    setup_failed = True
                    setup_process = None
                    d_status[current_node_serial_num]["desc"] = "Configuring the HyperScale nodes. Please verify the logs for more detail"
                    # Create terminate hookfile to terminate primary CS installation
                    terminate_file = os.path.join('config', 'cvhsterminate')
                    terminate_file = os.path.abspath(terminate_file)
                    with open(terminate_file, 'w') as fd:
                        fd.write('')
                    break
                
    progress_config = ConfigParser.ConfigParser()
    progress_config.read(progress_filepath)
    statusDict = {}
    
    # Process progress on local node
    if not setup_failed:
        for milestone in progress_config.sections():
            statusDict = {}
            statusDict["cpp"] = int(get_from_cfgfile(progress_config, milestone, "cpp"))
            statusDict["desc"] = get_from_cfgfile(progress_config, milestone, "desc")
            statusDict["completed"] = get_from_cfgfile(progress_config, milestone, "done", True)
            statusDict["current_milestone"] = int(get_from_cfgfile(progress_config, milestone, "progress"))

            d_status[current_node_serial_num] = statusDict

            if statusDict["completed"]:
                completed_milestones.append(statusDict)
                if milestone == 'setup_completed':
                    logging.info("Marking the install as complete..")
                    setup_status = "Complete"
            else:
                if (statusDict["cpp"] == -1):
                    setup_failed = True
                    setup_status = "Failed"
                    logging.error("milestone {0} failed, on this node".format(milestone))
                    failed_milestones.append(statusDict)
                    d_status[current_node_serial_num]["desc"] = statusDict["desc"]
                    statusDict["error"] =  get_from_cfgfile(progress_config,milestone, "error")
                    # Copy terminate hookfile to the other nodes
                    terminate_file = os.path.join('config', 'cvhsterminate')
                    terminate_file = os.path.abspath(terminate_file)
                    with open(terminate_file, 'w') as fd:
                        fd.write('')
                    for key in hyperscale_node_map:
                        obj = hyperscale_node_map[key]
                        args = ["arch", "write-file", obj.ip, terminate_file, terminate_file]
                        ret = cvnwhelper.execute_command(args)
                        if (ret):
                            logging.warning("arch write failed on node: {0}".format(ip))
                    break
                elif statusDict["current_milestone"] >= 0 and statusDict["current_milestone"] <= 100:
                    setup_status = "Running"
                    logging.info("The install is still running..")
                    provision_cs = get_from_cfgfile(progress_config, "start_cs_vm", "desc")
                    if (provision_cs == ""):
                        rt_in_minutes = ((100-statusDict["cpp"]) * 15)/100
                    else: 
                        rt_in_minutes = ((100-statusDict["cpp"]) * 45)/100
                    statusDict["rt"] = rt_in_minutes
                    inprogress_milestones.append(statusDict)
                    break
                else:
                    setup_failed = False
                    setup_status = "Pending"

    # if setup_failed and len(failed_milestones) == 0:
    if setup_failed:
        # Gather logs from all the nodes in this block
        ret = gather_logs()
        if ret:
            logging.error("Failed to gather logs.")
        else:
            statusDict["link"] = logs_basename + ".tar.gz"
        setup_status = "Failed"
        if len(failed_milestones) == 0:
            if "link" in statusDict:
                d_status[current_node_serial_num]["link"] = statusDict["link"]
        logging.error("Failed milestones: {0}".format(failed_milestones))


    if setup_status == "Complete":
        # Reload avahi nodes and navigate to admin console if hyperscale is configured
        if (hyperscale_configured(True)):
            # Redirect to summary app route
            return display_summary(False)

    image_number += 1
    logging.info("Setup status: {0}".format(setup_status))

    return render_template('processList.html', 
                           milestone_status = d_status,
                           setup_status = setup_status,
                           driver = current_node_serial_num,
                           image_number = image_number)


def full_filepath(configpath):
    return (os.getcwd() + "/" + configpath)

def push_config(remote_host, config_file_path, last_node):
    config = ConfigParser.ConfigParser()
    config.read(config_file_path)
    config.set("global", "last_node", last_node)
    # create a temporary file
    f = tempfile.NamedTemporaryFile(delete=False)
    src_path = f.name
    config.write(f)
    f.flush()
    f.close()
    return (cvnwhelper.arch_copy_file(remote_host, src_path, full_filepath(config_file_path)))

def add_colon_to_macaddr(mac_addr):
    mac_str = ":".join(mac_addr[i:i+2] for i in range(0, len(mac_addr), 2))
    return mac_str

# Returns None if no avahi service is publishing 
# service state as 'SetupInProgress'
# else returns mac address on node responsible for executing
# setup.
def is_setup_in_progress(refresh = True):
    global hyperscale_node_map

    # Refresh avahi node information
    if (refresh):
        if (refresh_avahi_nodes(False) == False):
            logging.error("Failed to locate all avahi nodes")
            return None

    logging.info("Configuration before node discovery: {0}".format(hyperscale_node_map))

    # Check if any of the nodes have setup_in_progress set to True
    for key in hyperscale_node_map:
        obj = hyperscale_node_map[key]

        logging.info("Checking setup progress state for node {0}".format(key))
        if (obj.status == "SetupInProgress"):

            cmdfilepath = os.getcwd() + "/config/cvavahi_cmd"
            cmdfile = open (cmdfilepath, "w+")
            cmdfile.write("is_setup_alive")
            cmdfile.close()

            # Write this file into remote node
            args = ["arch", "write-file", obj.ip, cmdfilepath, "/tmp/hwebtool_command"]
            ret = cvnwhelper.execute_command(args)
            if (ret):
                logging.error("arch write failed on : " + ip + ". Cannot find out whether setup is active on node : " + uuid)
                continue
 

            ret = cvnwhelper.execute_command(["arch", "manage", "commvault", "execute", hyperscale_node_map[key].ip, "cvavahi.py", "/tmp/hwebtool_command"])
            if (ret == 0): 
                return hyperscale_node_map[key]
            else:
                # Before changing the status to SetupFailed check 
                # whether setupsds completed successfully or if status
                # is already set to SetupFailed.
                if (refresh_avahi_nodes(False) == False):
                    logging.error("Failed to locate all avahi nodes")
                    return None
                
                if (obj.status != "SetupInProgress"):
                    continue

                cmdfile = open (cmdfilepath, "w+")
                cmdfile.write("update_status SetupFailed")
                cmdfile.close()

                # Write this file into remote node
                args = ["arch", "write-file", obj.ip, cmdfilepath, "/tmp/hwebtool_command"]
                ret = cvnwhelper.execute_command(args)
                if (ret):
                    logging.error("arch write failed on : " + ip + ". Failed to change avahi status on node : " + uuid)
                    return obj
 
                ret = cvnwhelper.execute_command(["arch", "manage", "commvault", "execute", hyperscale_node_map[key].ip, "cvavahi.py", "/tmp/hwebtool_command"])
                if (ret):
                    logging.error("cvavahi update_status failed on : " + ip + ". Failed to change avahi status on node : " + uuid)
                    return obj

                obj.status = "SetupFailed"
                return None

    return None

# Checks if any of hyperscale node has status set to configured
def hyperscale_configured(refresh = False):
    # Refresh avahi node information
    if (refresh):
        if (refresh_avahi_nodes(False) == False):
            logging.error("Failed to locate all avahi nodes")
            return False

    # Check if any of the nodes have setup_in_progress set to True
    for key in hyperscale_node_map:
        if (hyperscale_node_map[key].status == "HyperScaleConfigured"):
            return True

    return False

# Loads uuids of avahi nodes previously discovered by the setup.
def get_avahi_nodes_uuids():
    global silentinstall_filepath
    node_uuids = []

    if (os.path.exists(silentinstall_filepath)):
        config = ConfigParser.ConfigParser()
        config.read(silentinstall_filepath)
        return dict(config.items("data_protection_ips")).keys()
    else:
        return node_uuids

# Extracts the network information
def get_network_info():
    # Fill the network information
    nwinfo_map = {} 
    nsentries = parse_nameserver()
    nameservers =  nsentries["nameservers"]
    search_list = nsentries["search"]
    default_gw = get_default_gw()

    # Check whether the config file already exists. In that case we will populate the entries from config file. 
    if (os.path.exists(silentinstall_filepath)):
        config = ConfigParser.ConfigParser()
        config.read(silentinstall_filepath)
        nwinfo_map['nameserver1'] = get_from_cfgfile(config,"network", "nameserver1")
        if (config.has_option("network", "nameserver2")):
            nwinfo_map['nameserver2'] = get_from_cfgfile(config,"network", "nameserver2")
            nwinfo_map['nameserver2'] = ""
        if (config.has_option("network", "nameserver3")):
            nwinfo_map['nameserver3'] = get_from_cfgfile(config,"network", "nameserver3")
            nwinfo_map['nameserver3'] = ""
        nwinfo_map['default_gw'] = get_from_cfgfile(config,"network", "default_gw")
        nwinfo_map['dp_netmask'] = get_from_cfgfile(config,"network", "dp_netmask")
        nwinfo_map['sp_netmask'] = get_from_cfgfile(config,"network", "sp_netmask") 

        if (config.has_section("cluster")):
            nwinfo_map['commserve'] = get_from_cfgfile(config,"cluster", "cshname")
    else:       
        # Initialize all the parameters to valid defaults
        if (len(nameservers) >= 1):
            nwinfo_map['nameserver1'] = nameservers[0]

        if (len(nameservers) >= 2):
            nwinfo_map['nameserver2'] = nameservers[1]

        if (len(nameservers) >= 3):
            nwinfo_map['nameserver3'] = nameservers[2]

        if (len(default_gw) >= 1):
            nwinfo_map['default_gw'] = default_gw[0]

        dp_netmask = hyperscale_node_map["CurrentNode"].data_protection_netmask
        if (len(dp_netmask) > 0):
            nwinfo_map['dp_netmask'] = dp_netmask

        sp_netmask = hyperscale_node_map["CurrentNode"].storagepool_netmask
        if (len(sp_netmask) > 0):
            nwinfo_map['sp_netmask'] = sp_netmask

        if (len(search_list) >= 1):
            nwinfo_map['commserve'] = search_list[0]

    return nwinfo_map            


def create_nwconfig_json_response(status_code, nw_config_error_map):
    resp = { }
    resp["status"] = status_code
    for key in nw_config_error_map:
        resp[key] = nw_config_error_map[key]
    
    logging.error("Error map: {0}".format(pprint.pformat(nw_config_error_map)))
    # Dump network config errors to log
    for key in nw_config_error_map:
        logging.error("Network configuration errors for " + key)
        nw_config_error = nw_config_error_map[key]
        for inner_key in nw_config_error:
            logging.error("nw_config_error["+inner_key+"]="+str(nw_config_error[inner_key]))

    return resp

def init_nw_config_error(nw_config_error):
    nw_config_error["data_protection"] = nw_config_error["management"] = nw_config_error["storage_pool"] = ""


def update_silent_install_config(config_file_path, section, field, value, node_uuid):
    json_file_path = ""
    config = ConfigParser.ConfigParser()
    config.read(config_file_path)
    config.set(section, field, value)
    if config.has_option("network", "jsonpath"):
        json_file_path = config.get("network", "jsonpath")
    f = open(config_file_path, "w+")
    config.write(f)
    f.close()

    # Update config_only_private_ifs fields from advanced config
    if (json_file_path != "" and field == "config_only_sp"):
        # If advanced config file exists then 
        # change config_only_sp key in json as well.
        f = open(json_file_path, "r")
        json_data = json.load(f)
        for node in json_data["advancedNodes"]:
            if node["serialno"] == node_uuid:
                node["config_only_private_ifs"] = value
            elif value == True:
                # For all other nodes in block reset this field to False
                node["config_only_private_ifs"] = False
 
        f.flush()
        f.close()

        outfile = open(json_file_path, 'w+')
        json.dump(json_data, outfile, indent=4)
        outfile.close()
        

# Get current IP address assigned to
# data_protection or storage_pool or management network
def get_current_ip(hyperscale_node, nw_interface_desc):
    if (nw_interface_desc == "data_protection"):
        return hyperscale_node.data_protection_ip_addr
        
    elif (nw_interface_desc == "storage_pool"):
        return hyperscale_node.storagepool_ip_addr
        
    # TBD: Add code to populate 1gig IP address for 
    # management network.
    elif (nw_interface_desc == "management"):
        return hyperscale_node.mgmt_ip_addr
    else:
       logging.error("get_current_ip: Invalid nw_interface_desc provided")
       return "" 


# Check if IP is already in use.
# hyperscale_node : hyperscale node object to which this IP needs to be assigned
# nw_interface_desc : network interface type to which this IP needs to be assigned.
#                     valid values for this argument are "data_protection", "storage_pool"
# ip_to_validate : User provided IP address
def is_ip_already_in_use(hyperscale_node, nw_interface_desc, ip_to_validate):
    ret = cvnwhelper.execute_command(["sudo","ping","-c","1",ip_to_validate])
    if (ret == 0):
        # It's okay to reassign the IP address 
        # if it is used by same network interface 
        # to which we need to assign this IP.
        hyperscale_node.populate_hw_info()
        if (get_current_ip(hyperscale_node, nw_interface_desc) == ip_to_validate):
            return False
        else:
            return True
    return False

def get_error_message(err_desc):
    err_map = {"avahi_service": "Make sure cvlt avahi service is exported",
               "arch_service": "Make sure arch service is up and running",
               "ip_conflict": "This IP is already is in use",
               "setupsds": "Failed to configure the node",
               "write_config": "Failed to push config parameters to this node",
               "nw_config": "Failed to configure network interface"}
    return err_map[err_desc]


def get_json_resp_for_node(json_resp, uuid):
    for node_resp in json_resp["advancedNodes"]:
        if (node_resp["serialno"] == uuid):
            return node_resp
    return {}


def update_hosts_file(cfgfilepath):
    # Update the /etc/hosts on all the nodes with the contents of /tmp/hosts
    for key in hyperscale_node_map:
        ret = cvnwhelper.update_remote_hosts_file(hyperscale_node_map[key].ip, cfgfilepath)
        if (ret):
            logging.error ("Failed to update /etc/hosts on remote host %s", hyperscale_node_map[key].ip)
            return False
    return True

@app.route ('/loadSetupHTML', methods=['GET', 'POST'])
def loadSetupHTML ():
    return render_template('setup.html')


# Caller: index.js
# Called to configure network
@app.route ('/configure_network', methods=['GET', 'POST'])
#  This function can return below json status response:
# 1. OK : Proceed with next input screen which will collect commserve details
# 2. FAILED: network configuration failed
# JSON response will provide details of exactly which network interface's
# network configuration failed.
def configure_network():
    global hyperscale_node_map
    global silentinstall_filepath
    logging.info ("Call to configure network executed...")

    json_data = request.get_json()
    
    if ("useAdvancedSettings" in json_data and json_data["useAdvancedSettings"] == True):
        use_advanced_cfg = True
    else:
        use_advanced_cfg = False

    if use_advanced_cfg:
        config_mgmt = json_data["useSeparateManagementNetwork"]
    else:
        config_mgmt = False

    nw_config_error_map = { }
    resp = {}
    if (use_advanced_cfg):
        # Initialize json response
        advancedNodes = []
        for node in json_data["advancedNodes"]:
            node_validation_resp = {}
            node_validation_resp["error"] = ""
            node_validation_resp["serialno"] = node["serialno"]
            advancedNodes.append(node_validation_resp)
    
        resp["advancedNodes"] = advancedNodes
        resp["status"] = ""
    else:
        for key in hyperscale_node_map:
            nw_config_error = { }
            init_nw_config_error(nw_config_error)
            nw_config_error_map[key] = nw_config_error
    
    # Get latest avahi IPs
    ret = refresh_avahi_nodes()
    if (ret == False):
        for key in hyperscale_node_map:
            if (hyperscale_node_map[key].missing == True):
                if (use_advanced_cfg):
                    node_resp = get_json_resp_for_node(resp, hyperscale_node_map[key].uuid)
                    node_resp["error"] = get_error_message("avahi_service")
                else:
                    nw_config_error_map[key]["data_protection"] = get_error_message("avahi_service")

            if (hyperscale_node_map[key].arch_error == True):
                if (use_advanced_cfg):
                    node_resp = get_json_resp_for_node(resp, hyperscale_node_map[key].uuid)
                    node_resp["error"] = get_error_message("avahi_service")
                else:
                    nw_config_error_map[key]["data_protection"] = get_error_message("arch_service")
        
        if (use_advanced_cfg):
            resp["status"] = "FAILED"
        else:
            resp = create_nwconfig_json_response("FAILED", nw_config_error_map)
        return json.dumps(resp)

    logging.info("Completed generating JSON response for network configurations")

    # Write user provided parameters to config file.
    # This will create config file without cluster section. 
    ret = write_to_configfile(json_data, 
                              False, # Last node = False
                              False) # write cluster info= False
    if (ret == False):
        raise ValidationError("Failed to write configuration parameters")

    logging.info("Completed writing values to configuration file")
   
    full_file_path = full_filepath(silentinstall_filepath)
    #
    # Update local node's config file to indicate that only Storage pool
    # network needs to be configured
    #
    update_silent_install_config(full_file_path, "global", "config_only_sp", True, hyperscale_node_map["CurrentNode"].uuid)

    logging.info("Completed updating configuration file with storage poool network details")


    ret = cvavahi.encrypt(full_filepath(temp_passfilepath), full_filepath(passfilepath), full_filepath(silentinstall_filepath))


    if (ret):
        logging.error("Failed to create encrypted password")
        raise ValidationError("Failed to create encrypted password. Please verify and retry")

    logging.info("Completed encrypting passwords that were provided")

    for key in hyperscale_node_map:
        if (key != "CurrentNode"):
            if (use_advanced_cfg):
                node_resp = get_json_resp_for_node(resp, hyperscale_node_map[key].uuid)
            else:
                node_resp = nw_config_error_map[key]
            ret = cvnwhelper.arch_copy_file(hyperscale_node_map[key].ip, passfilepath, full_filepath(passfilepath))
            if (ret):
                if (use_advanced_cfg):
                    node_resp["error"] = "Failed to copy password config file"
                    resp["status"] = "FAILED"
                else:
                    node_resp["write_config"] = True
                    resp = create_nwconfig_json_response("FAILED", nw_config_error_map)
                return json.dumps(resp)
            logging.info("Completed copying {0} to node {1}".format(passfilepath, hyperscale_node_map[key].ip))

            ret = push_config(hyperscale_node_map[key].ip, silentinstall_filepath, False)
            if (ret):
                node_resp["error"] = "Failed to copy silentinstall.cfg to peers"
                resp["status"] = "FAILED"
                return json.dumps(resp)
            logging.info("Completed copying {0} to node {1}".format(silentinstall_filepath, hyperscale_node_map[key].ip))

            if (use_advanced_cfg):
                ret = cvnwhelper.arch_copy_file(hyperscale_node_map[key].ip, jsonfilepath, full_filepath(jsonfilepath))
                if (ret):
                    if (use_advanced_cfg):
                        node_resp["error"] = "Failed to copy advanced network config parameter json file"
                        resp["status"] = "FAILED"
                    else:
                        node_resp["write_config"] = True
                        resp = create_nwconfig_json_response("FAILED", nw_config_error_map)
                    return json.dumps(resp)
            logging.info("Completed copying {0} to node {1}".format(jsonfilepath, hyperscale_node_map[key].ip))

    logging.info("Completed copying files to all peers")

    # Update /etc/hosts file if required
    ret = update_hosts_file(full_file_path)
    if not ret:
        logging.error("Unable to update /etc/hosts file. Please check /var/log/cvresolvhname.log for more details")
        raise ValidationError("Unable to update hosts files. Please verify name resolution and PTR records in the DNS server")
    
    # Configure storage pool network for current node
    if (use_advanced_cfg):
        node_resp = get_json_resp_for_node(resp, hyperscale_node_map[key].uuid)
    else:
        node_resp = nw_config_error_map["CurrentNode"]

    logging.info("Will start network configuration using file {0} on {1}".format(full_file_path, hyperscale_node_map["CurrentNode"].uuid))

    ret = start_hyperscale_node_nwconfig(hyperscale_node_map["CurrentNode"], 
                                 full_file_path, 
                                 False,
                                 node_resp, 
                                 True)

    if (ret):
        if (use_advanced_cfg):
            resp["status"] = "FAILED"
        else:
            resp = create_nwconfig_json_response("FAILED", nw_config_error_map)
        return json.dumps(resp)


    ret = wait_for_hyperscale_node_nwconfig(hyperscale_node_map["CurrentNode"], 
                                 full_file_path, 
                                 False,
                                 node_resp, 
                                 True)
    if (ret):
        if (use_advanced_cfg):
            resp["status"] = "FAILED"
        else:
            resp = create_nwconfig_json_response("FAILED", nw_config_error_map)
        return json.dumps(resp)
    
    # Reset configure only storage pool flag
    update_silent_install_config(full_file_path, "global", "config_only_sp", False, hyperscale_node_map["CurrentNode"].uuid) 
   
    # Refresh network info for current node
    
 
    # Make sure there are no conflicting IPs
    if (use_advanced_cfg):
        hyperscale_node_map["CurrentNode"].advanced_nw_config = cvnwhelper.query_existing_nw_config_for_node(hyperscale_node_map["CurrentNode"])
        ret = cvnwhelper.check_for_ip_conflicts(json_data, hyperscale_node_map, resp)
    else:
        ret = check_for_ip_conflicts(full_file_path, config_mgmt, nw_config_error_map)
    if (ret):
        raise ValidationError("Some of the IP addresses provided are already in use")

    logging.info("Completed checks for finding conflicting IP addresses")

    # Inputs do look valid, let's go ahead and configure peer nodes
    for key in hyperscale_node_map:
        if (key != "CurrentNode"):
            if (use_advanced_cfg):
                node_resp = get_json_resp_for_node(resp, hyperscale_node_map[key].uuid)
            else:
                node_resp = nw_config_error_map[key]
            logging.info("Configuring push for peer node {0}..".format(hyperscale_node_map[key].ip))
            # Let's configure peer nodes by calling cvremotenwconfig and then setupsds with last_node = False
            ret = push_config(hyperscale_node_map[key].ip, silentinstall_filepath, False)
            if (ret):
                if (use_advanced_cfg):
                    node_resp["error"] = "Failed to push config file"
                    resp["status"] = "FAILED"
                else:
                    node_resp["write_config"] = True
                    resp = create_nwconfig_json_response("FAILED", nw_config_error_map)
                return json.dumps(resp)
            logging.info("Copying file {0} to node {1}..".format(jsonfilepath, hyperscale_node_map[key].ip))

            if (use_advanced_cfg):
                ret = cvnwhelper.arch_copy_file(hyperscale_node_map[key].ip, jsonfilepath, full_filepath(jsonfilepath))
                if (ret):
                    if (use_advanced_cfg):
                        node_resp["error"] = "Failed to push advanced network config parameter json file"
                        resp["status"] = "FAILED"
                    else:
                        node_resp["write_config"] = True
                        resp = create_nwconfig_json_response("FAILED", nw_config_error_map)
                    return json.dumps(resp)

            logging.info("Configuring network for node {0}..".format(hyperscale_node_map[key].ip))
            ret = start_hyperscale_node_nwconfig(hyperscale_node_map[key], 
                                         full_file_path,
                                         config_mgmt,
                                         node_resp)
            if (ret):
                if (use_advanced_cfg):
                    node_resp["error"] = "Failed to send network config RPC"
                    resp["status"] = "FAILED"
                else:
                    resp = create_nwconfig_json_response("FAILED", nw_config_error_map)
                return json.dumps(resp)
            logging.info("Completed network configuration for node {0}".format(hyperscale_node_map[key].ip))

   
    # Wait for network configuration to complete
    for key in hyperscale_node_map:
        if (key != "CurrentNode"):
            if (use_advanced_cfg):
                node_resp = get_json_resp_for_node(resp, hyperscale_node_map[key].uuid)
            else:
                node_resp = nw_config_error_map[key]
            ret = wait_for_hyperscale_node_nwconfig(hyperscale_node_map[key], 
                                         full_file_path,
                                         config_mgmt,
                                         node_resp)
            if (ret):
                if (use_advanced_cfg):
                    node_resp = get_json_resp_for_node(resp, hyperscale_node_map[key].uuid)
                    node_resp["error"] = "Wait for network configuration failed"
                    resp["status"] = "FAILED"
                else:
                    resp = create_nwconfig_json_response("FAILED", nw_config_error_map)
                return json.dumps(resp)
    
    return json.dumps({"status":"OK"})


# Caller: index.js
# Called to validate setup form information
@app.route ('/validate_setup', methods=['GET', 'POST'])
# This function can return below json status response:
# 1. OK : Display progress page
# 2. resume_setup: Resume previous setup by executing setupsds on current node
# 3. redirect_to_peer: Redirect to execute_setup on peer node
# 4. FAILED: An error occured while writing configuration file
def validate_setup():
    global hyperscale_node_map
    global CV_CONSTANTS
    logging.info("Call to validate setup being executed..")


    if (not page_validate_password()):
        raise ValidationError ("Incorrect credentials for user {0} for route setup".format(username))

    json_data = request.get_json()

    if "clusterDetails" in json_data:
        primary_cs = json_data["clusterDetails"][0]["primary_cs"]
        existing_cs = json_data["clusterDetails"][0]["existing_cs"]
        additional_cs = json_data["clusterDetails"][0]["additional_cs"]
        standby_kvm_cs = json_data["clusterDetails"][0]["standby_kvm_cs"]

        if primary_cs:
            if not primary_cs ^ (existing_cs | additional_cs | standby_kvm_cs):
                raise ValidationError("Failed to obtain valid installation type. Additional option detected apart from Primary CS. Please clean up config files and try again")

        if existing_cs:
            if not existing_cs ^ (primary_cs | additional_cs | standby_kvm_cs):
                raise ValidationError("Failed to obtain valid installation type. Additional option detected apart from Existing CS. Please clean up config files and try again")

        if additional_cs:
            if not additional_cs ^ (existing_cs | primary_cs | standby_kvm_cs):
                raise ValidationError("Failed to obtain valid installation type. Additional option detected apart from DR CS. Please clean up config files and try again")

        if standby_kvm_cs:
            if not standby_kvm_cs ^ (existing_cs | additional_cs | primary_cs):
                raise ValidationError("Failed to obtain valid installation type. Additional option detected apart from Primary and Standby CS. Please clean up config files and try again")

        if not existing_cs:
            ova_loc = CV_CONSTANTS["CS_OVA"]
            ret = cvnwhelper.execute_command(["sudo", "ls", ova_loc])
            if (ret != 0):
                raise ValidationError("Failed to find the OVA file. Please ensure that every node has the CommServe OVA for installation")


    ret = refresh_avahi_nodes()
    if (ret == False):
        raise ValidationError("Failed to retrive hardware information for avahi nodes. Please verify the node services and retry")

    # Write user provided parameters to config file.
    try:
        ret = write_to_configfile(json_data, 
                                  False,
                                  True)
        if (ret == False):
            raise ValidationError("Unable to write user provided configuration parameters")
    except Exception as str_e:
        logging.error("Unable to write to config file: {0}".format(str_e))
        raise ValidationError("Unable to prepare configuration files for install. Please check logs for details")

        
    # Explicitely set resume_install as True 
    config = ConfigParser.ConfigParser()
    config.read(silentinstall_filepath)
    config.set("cluster", "resume_install", True)

    # Modify local config file
    cfgfile = open(silentinstall_filepath, "w+")
    config.write(cfgfile)
    cfgfile.close()
 
    # Push config to peer nodes using arch copy command
    # peer node2 will be executing rest of the setup, so
    # set last_node = True
    ret = push_config(hyperscale_node_map["PeerNode1"].ip, silentinstall_filepath, False)
    if (ret):
        raise ValidationError("Unable to push config file to host {0}".format(hyperscale_node_map["PeerNode1"].ip))

    ret = push_config(hyperscale_node_map["PeerNode2"].ip, silentinstall_filepath, True)
    if (ret):
        raise ValidationError("Unable to push config file to host {0}".format(hyperscale_node_map["PeerNode2"].ip))

    ret = cvavahi.encrypt(full_filepath(temp_passfilepath), full_filepath(passfilepath), full_filepath(silentinstall_filepath))
    if (ret):
        raise ValidationError("Unable to encrypt passwords. Please verify the config file parameters and retry")

    ret = cvnwhelper.arch_copy_file(hyperscale_node_map["PeerNode1"].ip, passfilepath, full_filepath(passfilepath))
    if (ret):
        raise ValidationError("Unable to perform arch operation on host {0}".format(hyperscale_node_map["PeerNode1"].ip))

    ret = cvnwhelper.arch_copy_file(hyperscale_node_map["PeerNode2"].ip, passfilepath, full_filepath(passfilepath))
    if (ret):
        raise ValidationError("Unable to perform arch operation on host {0}".format(hyperscale_node_map["PeerNode2"].ip))

    # Update avahi service on peer node1
    # This needs to be done only when network for remote nodes
    # is successfully configured. 
    ret = hyperscale_node_map["PeerNode2"].mark_setup_in_progress()
    if (ret):
        raise ValidationError("Unable to update remote node {0} with setup sttus. Please verify the services and retry".format(hyperscale_node_map["PeerNode2"].ip))

    # Once network is configured, use avahi IP for communication
    mgmt_ip = hyperscale_node_map["PeerNode2"].ip
    logging.info ("Setup will now be redirected to "+cvnwhelper.gethname_from_ip(mgmt_ip)+". Please refer logs from that machine...")
    return json.dumps({'status':'redirect_to_peer','hname':mgmt_ip})


# Caller: index.js
# Called to validate whether each of the node has all the required hardware configuration
@app.route ('/validate_hardware', methods=['POST'])
# This function can return below json status response:
# 1. OK : If each of the node has all the required hardware configuration
# 2. Error descriptor: If any of the hardware components is missing
def validate_hardware():
    logging.info ("Validate Hardware is being invoked")
    json_data = request.get_json()

    json_resp = validate_hardware_configuration(json_data)

    logging.info ("JSON Response: {0}".format(json_resp))
    if (json_resp["status"] == "FAILED"):
        logging.error ("Some of the hardware components are missing")
        raise ValidationError("Some of the hardware checks failed. Please check the backend logs for more details.")
    logging.info ("Detected all the required hardware components successfully")

    return json.dumps(json_resp)


# Caller: index.js
# Called to validate whether each provided IP address has a valid PTR record
@app.route ('/validate_ptr', methods=['POST'])
# This function can return below json status response:
# 1. OK : If each of the Data protection network IP address has a valid PTR entry
# 2. Error descriptor: If any of the IP address does not have a valid PTR entry
def validate_ptr():
    logging.info ("Validate PTR records is being invoked...")
    json_data = request.get_json()
    json_resp = validate_ptr_records(json_data)
    json_resp["net_conf_mode"] = "basic"  
    
    if (json_resp["status"] == "FAILED"):
        logging.error ("Some of the IP address do not have valid PTR records")
    else:       
        logging.info ("All IP address have valid PTR records")

    return json.dumps(json_resp)


# Caller: index.js
# Called to validates advanced network config parameters
@app.route ('/validate_advanced', methods=['POST'])
# This function can return below json status response:
# 1. OK : If network parameters are valid and satisfy below constraints:
#   -- If public IPs (mgmt or CS registration) have valid PTRs
#   -- If all IPs are from different subnet
#   -- Management network IPs are created over 1gig link
#   -- Public IPs (mgmt ot CS registration) are not a tagged VLAN
# 2. Json Response , descriptive error per node mentioning validation error
def validate_advanced():
    logging.info ("Validate advanced config is being invoked...")
    json_data = request.get_json()
    try:
        json_resp = cvnwhelper.validate_advanced_config(json_data)
        json_resp["net_conf_mode"] = "advanced"
    except Exception as str_e:
        raise ValidationError("Error validating network: {0}".format(str_e))
    if (json_resp["status"] == "FAILED"):
        logging.error ("Failed to validate advanced network config parameters")
        raise ValidationError("Error encountered validating network parameters")

    logging.info ("Valid network configuration is provided per node")
    return json.dumps(json_resp)


# Caller: index.js
# Called to validate whether Dataprotection and StoragePool ipaddress for each node are from different subnets
@app.route ('/validate_subnet', methods=['POST'])
# This function can return below json status response:
# 1. OK : If Data protection and StoragePool ipaddress are from different subnets
# 2. Error descriptor: If data protection and storagepool are from same subnet
def validate_subnet():
    logging.info ("Validate Subnet is being invoked")

    # All hardware validations are ignored in case of lab installs
    if HS_CONFIG["IS_LAB_INSTALL"]:
        json_resp = {}
        json_resp["status"] = "OK"
        return json.dumps(json_resp)

    json_data = request.get_json()
    try:
        if ("useAdvancedSettings" in json_data and json_data["useAdvancedSettings"]):
            json_resp = cvnwhelper.validate_gw_nameservers(json_data) 
            json_resp["net_conf_mode"] = "advanced"  
        else: 
            json_resp = validate_netmask_entries(json_data)
            json_resp["net_conf_mode"] = "basic"
    except Exception as str_e:
        logging.error("Exception encountered while validating subnet: {0}".format(str_e))
        raise ValidationError("Unable to validate subnet requirements")
    
    if (json_resp["status"] == "FAILED"):
        raise ValidationError ("Some of the data protection and storagepool IP address are from same subnet")
    logging.info ("All data protection and storagepool IP address are from different subnets")
    return json.dumps(json_resp)


# Caller: index.js
# Called to validate whether each of the provided hostnames maps to a valid IP address.
@app.route ('/validate_dns', methods=['POST'])
# This function can return below json status response:
# 1. OK : If each of the hostnames maps to a valid IP address
# 2. Error descriptor: If any of the hostnames does not map to a valid IP address
def validate_dns():
    logging.info ("Validate DNS is being invoked...")
    json_data = request.get_json()
    json_resp = validate_dns_entries(json_data)
    
    if (json_resp["status"] == "FAILED"):
        raise ValidationError ("Some of the hostnames do not map to valid ipaddress")
    logging.info ("All the hostnames map to valid ipaddress")
    return json.dumps(json_resp)


# Caller: index.js
# Called to validate and execute setup
@app.route ('/execute_setup', methods=['GET', 'POST'])
def execute_setup():
    global hyperscale_node_map
    global setup_process
    global setup_process_standby
    global username
    global passwd
    global silentinstall_filepath
    logging.info ("Call to execute setup being executed..")

    ret = cvnwhelper.validate_reverse_lookup()
    if ret:
        raise ValueError("Unable to find valid PTR records for some IPs. Please update the /etc/hosts file and DNS entries to use static IPs")
    logging.info("Validated PTR records successfully")
 
    ret = cvnwhelper.validate_host_length()
    if ret:
        raise ValueError("Some of FQDNs don't follow length restrictions. Please ensure that the FQDNs are no more than 63 chars")
    logging.info("Validated length restrictions for all hostnames on the DNS servers")

    terminate_file = os.path.join('config', 'cvhsterminate')
    terminate_file = os.path.abspath(terminate_file)
    if os.path.isfile(terminate_file):
        os.remove(terminate_file)

    full_file_path = full_filepath(silentinstall_filepath)

    # Populate global hashmap for mac address, hardware info, IPs
    if (len(hyperscale_node_map.keys()) == 0):
        # Create a map of hyperscale nodes using UUIDs from config file
        ret = select_avahi_nodes_for_configuration(get_avahi_nodes_uuids())
        if (ret == False):
            logging.error("execute_setup: Failed to browse avahi nodes from config file")
            raise ValueError("Failed to browse avahi nodes from config file")
    else:
        # Refresh the hardware info for existing map
        ret = refresh_avahi_nodes()
    
    if (ret == False):
        logging.error("execute_setup: Failed to browse avahi nodes to retrive current node credentials")
        raise ValueError("Failed to browse avahi nodes to retrive current node credentials. Please verify the services and retry")
 
    # Populate current node password from credential config file
    pass_hashmap = cvavahi.decrypt(passfilepath, decrypted_file, silentinstall_filepath, False)
    if (None == pass_hashmap):
        logging.error("Failed to decrypt the config file to retrieve passwords")
        raise ValueError("Failed to decrypt files to retrieve passwords")

    username = "root"
    passwd = pass_hashmap["passwords"][hyperscale_node_map["CurrentNode"].uuid]

    # Populate fields related to network and cluster from silentinstall config file
    config = ConfigParser.ConfigParser()
    config.read(silentinstall_filepath)

    use_advanced_config = is_advanced_config_set(silentinstall_filepath)

    # Configure network for peer node which has not been configured yet.
    peerIndex = 1
    full_file_path = full_filepath(silentinstall_filepath)
    config_mgmt = config.getboolean("network", "config_mgmt")
    standby_kvm_cs = False
    standby_cs_host = None
    creator = None
    if (config.has_option("global", "standby_kvm_cs")):
        standby_kvm_cs = config.getboolean("global", "standby_kvm_cs")
    else:
        logging.warning("Unable to read standby KVM CS requirement. Will proceed with the install without it")
        standby_kvm_cs = False
    
    if config.has_option("global", "creator"):
        creator = config.get("global", "creator")
    else:
        logging.error("Unable to find the creator of config files in {0}".format(full_file_path))
        raise ValueError("Unable to find the creator of config files in {0}".format(full_file_path))

    for peerIndex in range(1, hyperscale_node_info.hyperscale_blk_size):
        peer = "PeerNode"+str(peerIndex)

        # Identify which host should host the standby CS
        if standby_kvm_cs:
            if cvavahi.get_current_node_cvlt_uuid() != hyperscale_node_map[peer].uuid:
                if hyperscale_node_map[peer].uuid != creator:
                    standby_cs_host = hyperscale_node_map[peer].ip
                    logging.info("Setting standby CommServe creation host as: {0}".format(standby_cs_host))

        logging.info("Status of node with IP {0}: {1}".format(hyperscale_node_map[peer].ip, hyperscale_node_map[peer].status))
        if (hyperscale_node_map[peer].status != "NwConfigured"):
            logging.error("Configuring "+ peer + " with avahi ip = " +hyperscale_node_map[peer].ip+" avahi uuid= "+hyperscale_node_map[peer].uuid)  
            nw_config_error = { }
            if (use_advanced_config): 
                nw_config_error["error"] = ""  
            else:  
                init_nw_config_error(nw_config_error)
            ret = start_hyperscale_node_nwconfig(hyperscale_node_map[peer], full_file_path, config_mgmt, nw_config_error)
            if (ret):
                logging.error("execute_setup: start_hyperscale_node_nwconfig failed for "+hyperscale_node_map[peer].ip+" avahi uuid= "+hyperscale_node_map[peer].uuid) 
                return render_template('index.html', error='Unable to start configuring network for {0}'.format(hyperscale_node_map[peer].ip))
            ret = wait_for_hyperscale_node_nwconfig(hyperscale_node_map[peer], full_file_path, config_mgmt, nw_config_error)
            if (ret):
                logging.error("execute_setup: wait_for_hyperscale_node_nwconfig failed for "+hyperscale_node_map[peer].ip+" avahi uuid= "+hyperscale_node_map[peer].uuid) 
                return render_template('index.html', error='Unable to complete network configuration for {0}'.format(hyperscale_node_map[peer].ip))


    # Now that all hyperscale nodes are configured, refresh hardware info, new IPs
    ret = refresh_avahi_nodes()
    if (ret == False):
        logging.error("execute_setup: Failed to refresh avahi node info after configuring remote nodes")
        return render_template('index.html')


    
    # Execute setup on current node
    hostname = socket.gethostname()
    setup_process = subprocess.Popen(["arch", "manage", "commvault", "execute",
                                      hostname, cvnwhelper.setup_exec(), full_file_path])

    # Kick off hyperscale setup for standby in background
    if standby_kvm_cs:
        # Allow the primary installation to begin before standby in case there is a lag
        # Setting delay for 30 seconds
        time.sleep(30)
        # Call the non-creator node to create standby CS
        setup_process_standby = subprocess.Popen(["arch", "manage", "commvault", "execute",
                                                  standby_cs_host, cvnwhelper.setup_exec(), full_file_path])

        if setup_process_standby is None:
            logging.error("Failed to execute setup on node {0}".format(standby_cs_host))
            raise ValueError("Failed to execute setup of Standby CommServe on node {0}".format(standby_cs_host))

    if (setup_process is None):
        logging.error("Failed to execute setup on current node")
        raise ValidationError("Failed to execute setup on current node")
  
    # Redirect to progress list app route, instead of rendering processList page in this contest
    return display_progress(False)


# Returns a hyperscale node in current block
# which has status set to SetupFailed
# returns None otherwise
def prev_setup_failed_on(refresh = True):
    global hyperscale_node_map

    # Refresh avahi node information
    if (refresh):
        if (refresh_avahi_nodes() == False):
            logging.error("Failed to locate all avahi nodes")
            return None

    # Check if any of the nodes have status as SetupFailed
    for key in hyperscale_node_map:
        obj = hyperscale_node_map[key]
        if (obj.status == "SetupFailed" and obj.restart_level != "unknown"):
            return hyperscale_node_map[key]

    return None


# Caller: index.js
# Called if post request validate of index fails
@app.route ('/setup/<action>', methods=['GET', 'POST'])
def setup (action):
    logging.info("Dectecting request for: /setup/{0}".format(action))
    global setup_process
    global hyperscale_node_map
    
    if (not page_validate_password()):
        logging.error ("Incorrect credentials for user %s for route setup", username)
        return render_template('index.html')

    logging.info("Validated credentials successfully")

    if (os.path.exists(silentinstall_filepath) and os.path.getsize(silentinstall_filepath) != 0):
        ret = select_avahi_nodes_for_configuration(get_avahi_nodes_uuids())
    else:
        ret = select_avahi_nodes_for_configuration()

    if (not ret):
        raise ValidationError("Unable to find peer files for install. Please rectify.")
    logging.info("Found the requisite peer information")

    # Check if the installtion needs to be resumed
    needs_restart = "False" 
    if (action == "resume"):
        needs_restart = "True"

    # if setup is still in progress then
    # navigate to progress page
    setup_is_active_on = is_setup_in_progress(False)
    if (setup_is_active_on != None):
        if (setup_is_active_on == hyperscale_node_map['CurrentNode']):
            if (setup_process != None):
                return display_progress()
            # If no active execution of setupsds found
            # and previous setup failed with some error
            # then allow user to resume previous setup
        else:
            return display_progress(False)
    elif (hyperscale_configured()):
        return display_summary(False)  
    
    #Get the network information
    nwinfo_map = get_network_info()
    logging.info("Network map: {0}".format(nwinfo_map))

    if (os.path.exists(silentinstall_filepath)):
        use_advanced_config = is_advanced_config_set(silentinstall_filepath)
    else:
        use_advanced_config = False

    if (action == 'cloud'):
        return render_template('setup.html')

    if not use_advanced_config:
        logging.info("Calling query to get exising network configuration for nodes {0}".format(hyperscale_node_map))
        try:
            advancedJSON = cvnwhelper.query_existing_nw_config(hyperscale_node_map, nwinfo_map)
            advancedJSON["useAdvancedSettings"] = False 
            advancedJSON = cvnwhelper.rearrange_advanced_json(advancedJSON, hyperscale_node_map)
        except Exception as str_e:
            logging.error("Unable to get existing configurations. Avahi block ID may not be set.")
            raise ValidationError("Unable to obtain current configurations")
    else:
        logging.info("Loading silentinstall.json from disk...")
        advancedJSON = load_advanced_json(silentinstall_filepath) 
        advancedJSON = cvnwhelper.rearrange_advanced_json(advancedJSON, hyperscale_node_map)

    # Set advanced network settings as the default for non appliance installations
    advancedJSON["useAdvancedSettings"] = False
    if (get_hyperscale_type("localhost") == "ReferenceArchitecture"):
        advancedJSON["useAdvancedSettings"] = True
        for i in range(0, len(advancedJSON["advancedNodes"])):
            for j in range(0, len(advancedJSON["advancedNodes"][i]["tooltips"])):
                advancedJSON["advancedNodes"][i]["tooltips"][j] = advancedJSON["advancedNodes"][i]["tooltips"][j].replace(" Data Protection", "")
                advancedJSON["advancedNodes"][i]["tooltips"][j] = advancedJSON["advancedNodes"][i]["tooltips"][j].replace(" Storage Pool", "")

    # Setting is_lab_install key so that front end validations can be ignored in case of lab installs
    advancedJSON["is_lab_install"] = HS_CONFIG["IS_LAB_INSTALL"]

    # Setting is_metallic_install key so that some front end validations for existing CS
    advancedJSON["is_metallic_install"] = HS_CONFIG["IS_METALLIC_INSTALL"]
    advancedJSON["status"] = "OK"

    logging.info("Advanced JSON Dumps: {0}".format(pprint.pformat(advancedJSON)))
    if (action == "resume"):
        return render_template('setup.html', needs_restart=needs_restart)

    return json.dumps(advancedJSON)


# Caller: index.js
# This is called if validation of the index post request fails
@app.route ('/error_index', methods=['GET', 'POST'])
def error_index ():
    global last_error
    logging.error (last_error)
    return render_template('index.html', error=last_error)


# Caller: index.js
# This is called after CommServe has been successfully instantiate from OVA
def adminconsole ():
    config = ConfigParser.ConfigParser()
    config.read(silentinstall_filepath)
    cshname = ""
    if is_metallic_install():
        cshname = get_metallic_cs()
    if (get_from_cfgfile(config,"cluster", "additional_cs", boolean=True)):
        cshname = get_from_cfgfile(config,"cluster", "additional_cs_host")
    else:
        cshname = get_from_cfgfile(config,"cluster", "cshname")
    return cshname


# Caller: index.js
# Called to find out restart level
@app.route ('/query_restartlevel', methods=['GET', 'POST'])
def query_restartlevel ():
    global hyperscale_node_map
    logging.info("Call to query restart level being executed...")
    resp = { }
    node = prev_setup_failed_on()
    resp["restart_level"] = node.restart_level
    return json.dumps(resp) 


# Caller: index.js
# Called when user clicks on resume setup button on setup.html
@app.route ('/resume_setup', methods=['GET', 'POST'])
def resume_setup ():
    global hyperscale_node_map
    global silentinstall_filepath
    logging.info("Call to resume setup being executed...")

    if (not page_validate_password()):
        logging.error ("Incorrect credentials for user %s for route setup", username)
        return render_template('index.html')

    # Find out the node where setup failed previously
    node = prev_setup_failed_on()
    config = ConfigParser.ConfigParser()
    config.read(silentinstall_filepath)
    config.set("cluster", "resume_install", "True")         
    config_mgmt = config.getboolean("network","config_mgmt")

    # Modify local config file
    cfgfile = open(silentinstall_filepath, "w+")
    config.write(cfgfile)
    cfgfile.close() 


    if (node == hyperscale_node_map["CurrentNode"]):
        # Modify local config file
        config.set("global", "last_node", "True")
        cfgfile = open(silentinstall_filepath, "w+")
        config.write(cfgfile)
        cfgfile.close()

        return redirect(url_for('execute_setup'))
    else:
        # Modify local config file
        cfgfile = open(silentinstall_filepath, "w+")
        config.write(cfgfile)
        cfgfile.close()

        # Use avahi IP 
        dest = node.ip
        ret = push_config(dest, silentinstall_filepath, True)
        if (ret):
            logging.error ("Failed to push config file to node "+dest)
            return render_template('index.html')

        # Instruct remote node to resume setup  
        execute_setup_url = "http://"+ dest + "/execute_setup"
        return redirect(execute_setup_url, code=302)


# Caller: index.js
# Called when user clicks on start over button on setup.html
@app.route ('/start_over', methods=['GET', 'POST'])
#  This function can return below json status response:
# 1. OK : Cleanup succeeded so reboot all hyperscale nodes in this block
# 2. FAILED: Failed to cleanup
def start_over ():
    global hyperscale_node_map
    global silentinstall_filepath
    resp = {}
    logging.info("Call to execute start over of install executed...")
    
    if (not page_validate_password()):
        logging.error ("Incorrect credentials for user %s for route setup", username)
        return render_template('index.html')
     
    # Find out the node where setup failed previously
    node = prev_setup_failed_on()
    config = ConfigParser.ConfigParser()
    config.read(silentinstall_filepath)
    config.set("cluster", "resume_install", False)
    
    # Modify local config file
    cfgfile = open(silentinstall_filepath, "w+")
    config.write(cfgfile)
    cfgfile.close() 
    full_file_path = os.getcwd() + "/" + silentinstall_filepath
    
    # Use avahi IP which is routable
    dest = node.ip

    if (node != hyperscale_node_map["CurrentNode"]):
        # Push config file to peer node where setup failed
        ret = push_config(dest, silentinstall_filepath, True)
        if (ret):
            logging.error ("Failed to push config file to node "+dest)
            resp["status"] = "FAILED"
            return json.dumps(resp)

    # Execute setupsds with resume_install = False
    # on the node where previous instance of setup failed.
    ret = cvnwhelper.execute_command(["arch", "manage", "commvault", "execute",
                    dest, cvnwhelper.setup_exec(), full_file_path])
    if (ret != 0):
        logging.error ("Failed to execute cleanup: "+dest)
        resp["status"] = "FAILED"
        return json.dumps(resp) 

    resp["status"] = "OK"
    return json.dumps(resp) 


def fix_progress_file_after_failure():
    global hyperscale_node_map
    global progress_filepath

    terminate_file = os.path.join('config', 'cvhsterminate')
    terminate_file = os.path.abspath(terminate_file)
    if os.path.isfile(terminate_file):
        os.remove(terminate_file)

    for filename in glob.glob(os.path.join('config', 'progress_*.txt')):

        args = ["sudo", "chmod", "755", filename]
        ret = cvnwhelper.execute_command(args)
        if (ret):
            logging.warning("Unable to set permissions on file: {0}".format(filename))

        args = ["sudo", "chown", "apache:apache", filename]
        ret = cvnwhelper.execute_command(args)
        if (ret):
            logging.warning("Unable to set ownership on file: {0}".format(filename))

        progress_config = ConfigParser.ConfigParser()
        progress_config.read(filename)
        for milestone in progress_config.sections():
            if (int(get_from_cfgfile(progress_config, milestone, "cpp")) == -1):
                progress_config.set(milestone, "cpp", 0)
            if (int(get_from_cfgfile(progress_config, milestone, "progress")) == -1):
                progress_config.set(milestone, "progress", 0)
        cfgfile = open(filename,'w')
        progress_config.write(cfgfile)
        cfgfile.flush()
        cfgfile.close()

        dest_path = os.path.abspath(filename)

        for key in hyperscale_node_map:
            obj = hyperscale_node_map[key]
            remote_host = obj.ip

            if key == "CurrentNode":
                continue

            # Push progress.txt to other nodes too.
            args = ["arch", "write-file", remote_host, filename, dest_path]
            ret = cvnwhelper.execute_command(args)
            if (ret):
                logging.error("Failed to push progress.txt to " + obj.uuid)
                continue

            try:
                args = ["arch", "manage", "commvault", "execute", remote_host, "cvfixperm.py", dest_path]
                ret = cvnwhelper.execute_command(args)
                if (ret):
                    logging.warning("Failed to fix perm for {0} in {1}".format(dest_path, obj.uuid))
            except Exception as str_e:
                logging.warning("Unable to set permissions for file {0} on node {1}".format(dest_path, obj.uuid))



def reset_hyperscale_configuration():
    global hyperscale_node_map
    global progress_filepath

    if (not os.path.exists(progress_filepath)):
        return
    progress_config = ConfigParser.ConfigParser()
    progress_config.read(progress_filepath)
    for milestone in progress_config.sections():
        progress_config.set(milestone, "cpp", 0)
        progress_config.set(milestone, "progress", 0)
    cfgfile = open(progress_filepath,'w')
    progress_config.write(cfgfile)
    cfgfile.flush()
    cfgfile.close()

    dest_path = os.getcwd() + "/" + progress_filepath
    mediaagent_registry = "/etc/CommVaultRegistry/Galaxy/Instance001/MediaAgent/.properties"
    
    for key in hyperscale_node_map:
        obj = hyperscale_node_map[key]
        remote_host = obj.ip

        #Reset the avahi configuration on the remote node
        cvavahi.update_remote_avahi_config(remote_host, "status", "Unattempted")
        cvavahi.update_remote_avahi_config(remote_host, "setup_in_progress", "False")
        cvavahi.update_remote_avahi_config(remote_host, "restart_level", "unknown")

        #Reset the restart level registry on the remote node
        remove_remote_registry_entry(remote_host,mediaagent_registry,"sHyperScaleRestartLevel")

        if key == "CurrentNode":
            continue

        #Push progress.txt to other nodes
        args = ["arch", "write-file", remote_host, progress_filepath, dest_path]
        ret = cvnwhelper.execute_command(args)
        if (ret):
            logging.error("Failed to push progress.txt to " + obj.uuid)
            continue



# Caller: index.js
# This function will validate if the user provided
# username and password is correct
@app.route ('/validate_index', methods=['GET', 'POST'])
def validate_index ():
    global username
    global passwd
    global hyperscale_node_map
    global last_error
    global silentinstall_filepath
  
    logging.info("Detecting request for validate_index...")
    if "username" in request.form:
        username = request.form['username']
    else:
        logging.error("Unable to obtain username from the HTML request: {0}".format(request.form))
        raise ValidationError("Unable to parse username")

    if "password" in request.form:
        passwd = request.form['password']
    else:
        logging.error("Unable to obtain password from the HTML request")
        raise ValidationError("Unable to parse password")

    if (not verify_password()):
        raise ValidationError("Invalid root user credentials")

    logging.info ("Credentials have been validated successfully")

    if (os.path.exists(silentinstall_filepath) and os.path.getsize(silentinstall_filepath) != 0):
        logging.info("Parsing the existing config file...")
        try:
            ret = select_avahi_nodes_for_configuration(get_avahi_nodes_uuids())
        except Exception as str_e:
            logging.error("Encountered exception: {0}".format(str_e))
            logging.error("Please try cleaning up the {0} file".format(silentinstall_filepath))
            raise ValidationError("Failed to process avahi nodes for this block")
        if (ret == False):
            raise ValidationError("Failed to locate all avahi nodes in current block")

        try:
            # Check if hyperscale is already configured 
            if (is_setup_in_progress(False) != None):
                logging.info ("Hyperscale setup is already in progress, redirecting to processList.html page")
                return json.dumps({'status': 'SetupInProgress'}) 
            elif (hyperscale_configured()):
                logging.info ("Hyperscale is already configured, redirecting to admin console")
                return json.dumps({'status':'HyperScaleConfigured'})
            elif (prev_setup_failed_on(False) != None):
                node = prev_setup_failed_on(False)
                if (node.restart_level != "unknown"):
                    logging.error ("Previous setup failed on node "+node.ip+" Trying to resume setup on that node")
                    fix_progress_file_after_failure()
                    # redirect to setup.html which will provide user an option
                    # to start over or resume install 
                    return json.dumps({'status': 'ShowResumeDialog'})
        except Exception as str_e:
            logging.error('Encountered exception while getting install status: {0}'.format(str_e))
            raise ValidationError('Unable to obtain the status of installation')                

    response_json = {'status':'OK'}
    response_json['nodetype'] = 'HyperScale' if get_hyperscale_type('localhost') != 'ReferenceArchitecture' else 'ReferenceArchitecture'

    logging.info("Executing render template of setup...")
    return json.dumps(response_json);


@app.route('/validate_cscreds', methods=['POST', 'OPTIONS'])
def validate_cs_creds():
    logging.info("Call to validate CS password received...")
    json_data = request.get_json()

    if request.method == 'OPTIONS':
        response = jsonify({"status": "OK"})
        response.headers.add('Access-Control-Allow-Origin', '*')
        response.headers.add('Access-Control-Allow-Headers', 'Content-Type,Authorization')
        return response

    if request.method == 'POST':
        if ('csname' not in json_data):
            raise ValidationError("Unable to find CS name in the data provided", is_cors=True)
        if ('csuser' not in json_data):
            raise ValidationError("Unable to find CS username in the data provided", is_cors=True)
        if ('cspasswd' not in json_data):
            raise ValidationError("Unable to find CS user's password in the data provided", is_cors=True)

        csname = json_data['csname']
        csuser = json_data['csuser']
        cspasswd = json_data['cspasswd']

        logging.info("Validating password entered for CS {0}...".format(csname))

        is_metallic_install = HS_CONFIG["IS_METALLIC_INSTALL"]
        if is_metallic_install:
            cmdfilepath = "/tmp/cvavahi_cmd"
            with open(cmdfilepath, "w") as fd:
                fd.write("validate_metallic_cscreds {0} {1} {2}".format(csname, csuser, cspasswd))

            args = ["arch", "write-file", "localhost", cmdfilepath, "/tmp/hwebtool_command"]
            ret = cvnwhelper.execute_command(args)
            if (ret):
                raise ValidationError("Arch write failed on localhost")
        try:
            output = cvnwhelper.is_valid_cspasswd(csname, csuser, cspasswd, is_metallic_install)
        except Exception as str_e:
            raise ValidationError("Unable to validate CS credentials successfully. Please verify manually and disable CS firewall if present.", is_cors=True)
        if (not output[0]):
            raise ValidationError("Unable to validate credentials for existing CommServe {0}. {1}".format(csname, output[1]), is_cors=True)

        logging.info("Validated CommServe credentials successfully")
        response = jsonify({"status": "OK"})
        response.headers.add('Access-Control-Allow-Origin', '*')
        response.headers.add('Access-Control-Allow-Headers', 'Content-Type,Authorization')
        return response



@app.route('/querycloudparams', methods=['GET'])
def get_cloud_params():
    logging.info('Call to query cloud configuration invoked')
    global json_cloudcfg
    if (json_cloudcfg is None):
        raise ValidationError('Bonjour has not submitted any configurations to this installer')
    return json.dumps(json_cloudcfg)


# Called by Bonjour to sync webserver with cloud data
@app.route('/cloudconfigure', methods=['POST'])
def cloudconfigure():
    logging.info('Call for cloud initialization invoked')
    global json_cloudcfg
    try:
        json_cloudcfg = request.get_json()
        json_cloudcfg['cloudinstall'] = False
        json_cloudcfg['initializedfromcloud'] = True
    except Exception as str_e:
        raise ValidationError("Unable to extract json: {0}".format(str_e))
    logging.info("Received configuration data from Bonjour: {0}".format(json_cloudcfg))
    with open(os.path.join('config', 'silentinstall.json'), 'w') as fd:
        fd.write(json.dumps(json_cloudcfg))
    return json.dumps({"status": "OK"})


# Validation error handler to redirect error messages to HTML client before install starts
@app.errorhandler(ValidationError)
def handle_invalid_usage(error):
    response = jsonify(error.to_dict())
    response.status_code = error.status_code
    if error.is_cors:
        response.headers.add('Access-Control-Allow-Origin', '*')
        response.headers.add('Access-Control-Allow-Headers', 'Content-Type,Authorization')
    logging.error ("Exception encountered: {0}".format(error.message))
    logging.error ("Printing exception stack...")
    logging.error(inspect.trace())
    return response

# Generic error handler for the download files after the install starts
@app.errorhandler(Exception)
def all_exception_handler(e):
    global last_error
    is_logs = False
    msg = []

    logging.error("Unexpected error encountered - throwing error: {0}".format(last_error))
    logging.error("Printing exception stack trace:")
    logging.error(inspect.trace())
    # Keeping this in try block because exception in 
    # exception handler will not end well otherwise!
    try:
        ret = gather_logs()
        if ret:
            is_logs = False
        else:
            is_logs = True
    except:
        is_logs = False

    if is_logs == True:
        logsDict = { }
        logsDict["link"] = logs_basename + ".tar.gz"
        msg.append(logsDict)

    return render_template('error.html', error_desc="{0}".format(e), msg=msg)


# This is the first page that opens up when
# the user enters the IP 
@app.route('/', methods=['POST', 'GET'])
def index():
    logging.info ("Index page opened")
    logging.info("HyperScale node type is set to: {0}".format(HS_CONFIG["HS_TYPE"]))
    logging.info("Lab install boolean is set to: {0}".format(HS_CONFIG["IS_LAB_INSTALL"]))
    logging.info("Metallic install boolean is set to: {0}".format(HS_CONFIG["IS_METALLIC_INSTALL"]))
    return render_template('index.html')

@app.route('/favicon.ico')
def favicon():
    return send_from_directory(os.path.join(app.root_path, 'static', 'img'), 'logo.png')

if __name__ == '__main__':
    app.debug = True
    app.run(host='0.0.0.0')
