"use strict";

var MODEL_STORAGE_KEY = "model";
var ENTITY_TYPES = {
  BOND: "bond",
  VLAN: "vlan",
  NETWORK: "network"
};

var NETWORK_TYPES = {
  COMM_SERVE_REGISTRATION: "commServeRegistration",
  MANAGEMENT: "management",
  STORAGE_POOL: "storagePool",
  DATA_PROTECTION: "dataProtection",
  NONE: ""
};

var NUM_NODES = 3;
var CURRENT_VERSION = 2; // Used to update or invalidate old models loaded by the user
// Update CURRENT_VERSION if the model is changed significantly.
var VERSION_STORAGE_KEY = "version";

var syncGetFunctions = {};
var syncSetFunctions = {};

window.currentNode = 0;
window.vlanIdCount = 0;

// Get the modal
var modal = document.getElementById('id01');
var modal2 = document.getElementById('create-bond-modal');

function isAppliance(nodenumber) {
  if (window.model.advancedNodes[nodenumber].nodetype.toString() === "HyperScale")
      return true;
  return false;
}

// Enable advanced n/w config by default
// Hide CommServe toggles for Reference Architecture
// Show only existing CommServe inputs for Appliance by default
// Do not show IPMI details for Reference Architecture
function loadInputsBasedOnType() {
  applyVisibilityToggles();
  if (!isAppliance(0)) {
    $(".toggle.form-group.last-toggle-step").hide();
    $(".toggle.form-group.additionalcs-toggle-step").hide();
    $(".toggle.form-group.standbykvmcs-toggle-step").hide();
    $(".appliance-only-option").hide();
  } else {
    NETWORK_TYPES["IPMI"] = "ipmi";
    $("#different-password-toggle").hide();
  }
  applyVisibilityToggles();
}

function networkTypeToString(type) {
  if (type === NETWORK_TYPES.MANAGEMENT) {
    return "Management";
  } else if (type === NETWORK_TYPES.STORAGE_POOL) {
    return "StoragePool";
  } else if (type === NETWORK_TYPES.COMM_SERVE_REGISTRATION) {
    return "CommServe registration";
  } else if (type === NETWORK_TYPES.DATA_PROTECTION) {
    return "Data Protection";
  } else if (type === NETWORK_TYPES.IPMI) {
    return "IPMI";
  } else {
    return "None";
  }
}

var escapeSpan = document.createElement("span");
function escapeHtmlString(str) {
  escapeSpan.textContent = str;
  return escapeSpan.innerHTML;
}

/* Object to be constructed in-memory which contains all the field value */
function initializeModel() {
  var newModel = {
    useSeparateManagementNetwork: false,
    useAdvancedSettings: false,
    useNodePasswords: false,
    password: "",
    confirmPassword: "",
    advancedNodes: [],
    mirrorFirstAdvancedNode: true,
    basicNodes: [],
    managementNetworkNetmask: "",
    defaultGateway: "",
    default_gw_interface: "",
    dataProtectionNetmask: "",
    ipmiNetmask: "",
    ipmiGateway: "",
    clusterNetwork: "",
    clusterDetails: [],
    dnsList: [""],
    useBondedInterface: true,
    bondMode: 1
  };
  
  for (var j = 0; j < NUM_NODES; j++) {
    newModel.advancedNodes.push({
      serialno: "",
      password: "",
      confirmPassword: "",
      nodetype: "",
      networks: [], // List of networks available in the list
      bonds: [], // List of bonds available in the list
      allBonds: [], // List of all bonds
      vlans: [], // List of vlans available in the list
      tooltips: []
    });
    for (var i = 0; i < 6; i++) {
      newModel.advancedNodes[j].networks.push({
        name: "Eno" + i,
        type: ENTITY_TYPES.NETWORK,
        ip: "",
        netmask: "",
        capacity: "",
        macaddr: "",
        link_speed: "",
        networkType: ""
      });
    }
    newModel.basicNodes.push({
      dataProtectionNetwork: "",
      storagePool: "",
      managementNetwork: "",
      password: "",
      confirmPassword: "",
      serialno: "",
      tooltips: []
    });
  }

  newModel.clusterDetails.push({
    hostname: "",
    username: "",
    commserve_backup_network: "",
    window_product: "",
    passwd: "",
    primary_cs: false,
    // DR CS details
    additional_cs: false,
    additional_cs_host: "",
    additional_cs_user: "",
    additional_cs_passwd: "",
    // Standby KVM details
    standby_kvm_cs: false,
    standby_kvm_csname: "",
    standby_kvm_csprid: ""
});

  newModel.clusterDetails[0]['existing_cs'] = !(newModel.clusterDetails[0]['additional_cs']
                                                || newModel.clusterDetails[0]['primary_cs']
                                                || newModel.clusterDetails[0]['standby_kvm_cs']);

  return newModel;
}

// When the user clicks anywhere outside of the modal, close it
window.onclick = function (event) {
  if (event.target == modal) {
    modal.style.display = "none";
  } else if (event.target == modal2) {
    modal2.style.display = "none";
  }
}

function getNodeEntities(nodeIndex) {
  var node = window.model.advancedNodes[nodeIndex];
  return [].concat(node.bonds, node.networks, node.vlans);
}

function getCurrentNodeEntities() {
  return getNodeEntities(window.currentNode);
}

function getNodeEntityByName(nodeIndex, name) {
  var entities = getNodeEntities(nodeIndex);
  for (var i = 0; i < entities.length; i++) {
    if (name === entities[i].name) {
      return entities[i];
    }
  }
  return null;
}

function getCurrentNodeEntityByName(name) {
  return getNodeEntityByName(window.currentNode, name);
}

function deleteNodeEntity(nodeIndex, name, type) {
  var node = window.model.advancedNodes[nodeIndex];
  var entities;
  if (type === ENTITY_TYPES.BOND) {
    entities = node.bonds;
  } else if (type === ENTITY_TYPES.VLAN) {
    entities = node.vlans;
  } else {
    entities = node.networks;
  }
  var indexToDelete = -1;
  for (var i = 0; i < entities.length; i++) {
    if (name === entities[i].name) {
      indexToDelete = i;
    }
  }
  if (indexToDelete >= 0) {
    var entityToDelete = entities[indexToDelete];
    entities.splice(indexToDelete, 1);
    // If a bond is deleted, we must add the children back into their lists and fix names:
    if (type === ENTITY_TYPES.BOND) {
      for (var i = 0; i < entityToDelete.children.length; i++) {
        var child = entityToDelete.children[i];
        if (child.type === ENTITY_TYPES.BOND) {
          // Insert the child bonds into the bond list sorted by index
          var childInserted = false;
          for (var j = 0; j < node.bonds.length; j++) {
            if (child.index <= node.bonds[j].index) {
              node.bonds.splice(j, 0, child);
              childInserted = true;
              break;
            }
          }
          if (!childInserted) {
            node.bonds.push(child);
          }
        } else if (child.type === ENTITY_TYPES.VLAN) {
          node.vlans.push(child);
        } else {
          node.networks.push(child);
        }
      }
      // Remove bond from the allBonds list
      for (var i = entityToDelete.index + 1; i < node.allBonds.length; i++) {
        node.allBonds[i].index -= 1;
        node.allBonds[i].name = "bond" + (node.allBonds[i].index + 1);
      }
      node.allBonds.splice(entityToDelete.index, 1);
      // Update names for vlans:
      for (var i = 0; i < node.vlans.length; i++) {
        node.vlans[i].name = node.vlans[i].parent.name + "." + node.vlans[i].id;
      }
    }
  }
}

function deleteCurrentNodeEntity(name, type) {
  deleteNodeEntity(window.currentNode, name, type);
  // If the entity is deleted in one node, it should be deleted in the other two as well
  for (let i = 0; i < NUM_NODES; i++)
    deleteNodeEntity(i, name, type);
  saveLocalStorageModel();
  refreshCurrentNodeEntities();
}

function isLinkUp(entity) {
  if (entity.link_speed !== "No link")
    return true;
  return false;
}

function refreshCurrentNodeEntities() {
  var node = window.model.advancedNodes[window.currentNode];
  var entities = getCurrentNodeEntities().sort(function (a, b) {
    if (a.name > b.name) {
      return 1;
    } else if (a.name < b.name) {
      return -1;
    } else {
      return 0;
    }
  });
  var htmlString = "";
  // Find bonds with vlans:
  var entitiesWithVlans = {};
  node.vlans.forEach(function (vlan) {
    entitiesWithVlans[vlan.parent.name] = vlan.parent;
  });
  for (var i = 0; i < entities.length; i++) {
    var entity = entities[i];
    var entityHtml;
    let disable_entry = false;
    if (!isLinkUp(entity))
      disable_entry = true;

    if (!isAppliance(0) && entity.name === "ipmi")
      disable_entry = true;

    if (entity.type === ENTITY_TYPES.BOND) {
        var bondDeleteHtml;
        if (entitiesWithVlans[entity.name]) {
        // Disable delete if bond has vlans:
        bondDeleteHtml =
            '<a class="delete tooltip-sibling-anchor" disabled>&times;</a>' +
            '<div class="tooltip">Bonds with VLANs cannot be deleted</div>';
        } else {
        bondDeleteHtml =
            '<a class="delete" onclick=\'deleteCurrentNodeEntity("' + entity.name + '", "' + ENTITY_TYPES.BOND + '")\'>' +
            '&times;' +
            '</a>';
        }
        entityHtml = '<div class="list-item">' +
        '<input id="bond' + i + '" type="checkbox" class="bonds" value="' + entity.name + '">' +
        '<span class="node-info-title aneesh">' +
            '<label for="bond' + i + '">' + entity.name + '</label>' +
        '</span>' +
        '<div class="tooltip">' +
            '<u>' + entity.name + '</u>' +
            "<br>Networks: <em>" + entity.children.map(function (child) {
                return child.name;
            }).join(", ") + '</em>' +
            '<br>Mode: <em>' + entity.mode + '</em>' +
            '<br>IP Address: <em data-ip-tooltip></em>' +
            '<br>Netmask: <em data-netmask-tooltip></em>' +
            '<br>Link Speed: <em data-link-speed-tooltip></em>' +
            '<br>Network Type: <em data-network-type-tooltip></em>' +
        '</div>' +
        bondDeleteHtml +
        '</div>';
    } else if (entity.type === ENTITY_TYPES.VLAN) {
        entityHtml = '<div class="list-item">' + 
            '<input id="vlan' + i + '" type="checkbox" class="vlans" value="' + entity.name + '">' +
            '<span class="node-info-title aneesh">' +
                '<label for="vlan' + i + '">' + entity.name + '</label>' +
            '</span>' +
            '<div class="tooltip">' +
                '<u>' + entity.name + '</u>' +
                '<br>Network: <em>' + entity.parent.name + '</em>' +
                '<br>IP Address: <em data-ip-tooltip></em>' +
                '<br>Netmask: <em data-netmask-tooltip></em>' +
                '<br>Link Speed: <em data-link-speed-tooltip></em>' +
                '<br>Network Type: <em data-network-type-tooltip></em>' +
            '</div>' +
            '<a class="delete" onclick=\'deleteCurrentNodeEntity("' + entity.name + '", "' + ENTITY_TYPES.VLAN + '")\'>' +
                '&times;' +
            '</a>' +
            '</div>';
    } else {
        entityHtml = '<div class="list-item">';
        let macadd;
        if (disable_entry) {
            entityHtml += '<input id="' + entity.name + '" type="checkbox" class="networks" value="' + entity.name + '" disabled>';
            macadd = entity.macaddr;
        } else {
            entityHtml += '<input id="' + entity.name + '" type="checkbox" class="networks" value="' + entity.name + '">';
        }
        entityHtml += '<span class="node-info-title aneesh">' +
                        '<label for="' + entity.name + '">' + entity.name + '</label>' +
                      '</span>' +
                        '<div class="tooltip">' +
                        '<u>' + entity.name + '</u>' +
                        '<br>IP Address: <em data-ip-tooltip></em>' +
                        '<br>Netmask: <em data-netmask-tooltip></em>' +
                        '<br>Link Speed: <em data-link-speed-tooltip></em>' +
                        '<br>Link Capacity: <em data-link-capacity-tooltip></em>' +
                        '<br>Network Type: <em data-network-type-tooltip></em>';
        if (macadd) {
            entityHtml += '<br>MacAddr: <em data-macaddr-tooltip>' + macadd + '</em>';
        } else {
            entityHtml += '<br>MacAddr: <em data-macaddr-tooltip></em>';
        }
        entityHtml += '</div></div>';
    }
    htmlString += entityHtml;
  }
  document.getElementById("network-list").innerHTML = htmlString;
  refreshNetworkEntityTooltips();
}

function refreshNetworkEntityTooltips() {
  var entities = getCurrentNodeEntities();
  for (var i = 0; i < entities.length; i++) {
    var entity = entities[i];
    if (isLinkUp(entity)) {
        var tooltip = document.querySelector('input[value="' + entity.name + '"]').parentNode.getElementsByClassName("tooltip")[0];
        tooltip.querySelector("[data-ip-tooltip]").textContent = entity.ip;
        tooltip.querySelector("[data-netmask-tooltip]").textContent = entity.netmask;
        tooltip.querySelector("[data-network-type-tooltip]").textContent = networkTypeToString(entity.networkType);
        tooltip.querySelector("[data-link-speed-tooltip]").textContent = entity.link_speed;
        if (entity.type == ENTITY_TYPES.NETWORK) {
          if (entity.capacity)
            tooltip.querySelector("[data-link-capacity-tooltip]").textContent = entity.capacity;
          if (entity.macaddr)
            tooltip.querySelector("[data-macaddr-tooltip]").textContent = entity.macaddr;
        }
    }
  }
}

function refreshNetworkEntityTooltip(name) {
  var entity = getCurrentNodeEntityByName(name);
  if (entity !== null) {
    var tooltip = document.querySelector('input[value="' + entity.name + '"]').parentNode.getElementsByClassName("tooltip")[0];
    tooltip.querySelector("[data-ip-tooltip]").textContent = entity.ip;
    tooltip.querySelector("[data-netmask-tooltip]").textContent = entity.netmask;
    tooltip.querySelector("[data-network-type-tooltip]").textContent = entity.networkType;
    tooltip.querySelector("[data-link-speed-tooltip]").textContent = entity.link_speed;
    if (entity.type == ENTITY_TYPES.NETWORK) {
      tooltip.querySelector("[data-link-capacity-tooltip]").textContent = entity.capacity;
      tooltip.querySelector("[data-macaddr-tooltip]").textContent = entity.macaddr;
    }
  }
}

function _addVlans(nodeIndex) {
  var errorElement = document.getElementById("vlanIdError");
  errorElement.classList.add("hide");
  var checkbox = document.querySelector("#network-list input[type=checkbox]:checked");
  var parentName = checkbox.value;
  var entities = getNodeEntities(nodeIndex);
  var parentEntity;
  for (var i = 0; i < entities.length; i++) {
    if (entities[i].name === parentName) {
      parentEntity = entities[i];
    }
  }

  if (parentEntity) {
    // Add the vlans:
    var ids = document.getElementsByClassName('vlansid');
    // Validate ids:
    for (var i = 0; i < ids.length; i++) {
      if (!ids[i].value) {
        errorElement.textContent = "Please fill in all VLAN IDs";
        errorElement.classList.remove("hide");
        return false;
      }
      if (ids[i].value.indexOf('"') >= 0 || ids[i].value.indexOf("'") >= 0) {
        errorElement.textContent = "IDs cannot contain any quotation marks (\" or \')";
        errorElement.classList.remove("hide");
        return false;
      }
    }
    var existingVlanNames = {};
    window.model.advancedNodes[nodeIndex].vlans.forEach(function (vlan) {
      existingVlanNames[vlan.name] = true;
    });
    for (var i = 0; i < ids.length; i++) {
      var name = parentName + "." + ids[i].value;
      // If vlan with name already exists, do not add it:
      if (!existingVlanNames[name]) {
        window.model.advancedNodes[nodeIndex].vlans.push({
          name: name,
          id: ids[i].value,
          ip: parentEntity.ip || "",
          netmask: parentEntity.netmask || "",
          networkType: parentEntity.networkType || "",
          type: ENTITY_TYPES.VLAN,
          parent: parentEntity
        });
      }
    }
  }
  return true;
}

function addVlans() {
  if (!_addVlans(window.currentNode)) {
    return;
  }
  for (let i = 0; i < NUM_NODES; i++) {
    _addVlans(i);
  }
  saveLocalStorageModel();
  refreshCurrentNodeEntities();
  hideCreateVlanModal();
}

function _addBond(nodeIndex) {
  var checkboxes = document.querySelectorAll("#network-list input[type=checkbox]:checked");
  var node = window.model.advancedNodes[nodeIndex];
  // Map entities that are selected:
  var selectedEntityMap = {};
  var children = [];
  for (var i = 0; i < checkboxes.length; i++) {
    var name = checkboxes[i].value;
    selectedEntityMap[name] = true;
  }
  // Move all selected entities into the children array:
  var moveEntitiesToChildren = function (list, children) {
    for (var i = 0; i < list.length; i++) {
      if (list[i].name in selectedEntityMap) {
        children.push(list[i]);
        list.splice(i, 1);
        i -= 1;
      }
    }
    return children;
  }
  moveEntitiesToChildren(node.bonds, children);
  moveEntitiesToChildren(node.vlans, children);
  moveEntitiesToChildren(node.networks, children);

  var newBond = {
    index: node.allBonds.length,
    name: "bond" + (node.allBonds.length + 1),
    ip: children[0].ip || "",
    netmask: children[0].netmask || "",
    networkType: children[0].networkType || "",
    type: ENTITY_TYPES.BOND,
    children: children,
    mode: document.getElementById("bond-mode-select").value
  };
  node.allBonds.push(newBond);
  node.bonds.push(newBond);
}

function addBond() {
  for (let i = 0; i < NUM_NODES; i++) {
    _addBond(i);
  }
  saveLocalStorageModel();
  refreshCurrentNodeEntities();
  hideCreateBondModal();
}

function clearErrors() {
  document.getElementById("bondError").style.display = "none";
  document.getElementById("bondError2").style.display = "none";
  document.getElementById("vlansError").style.display = "none";
  document.getElementById("networkError").style.display = "none";
  document.getElementById("settingsError").style.display = "none";
  document.getElementById("advancedSettingsError").style.display = "none";
  document.getElementById("networkInfoError").style.display = "none";
  document.getElementById("clusterInfoError").style.display = "none";
}

function getNwIfaceChecked() {
  // Applies only to advanced settings toggle
  let checkboxes = document.querySelectorAll("#network-list input[type=checkbox]:checked");
  if (checkboxes.length === 1)
    return checkboxes[0];
  return false;
}

function setNode(node) {
  let entity = getNwIfaceChecked()
  setActiveTab(node);
  clearErrors();
  window.currentNode = node;
  applyModel();
  refreshCurrentNodeEntities();
  if (entity)
    document.getElementById(entity.id).checked = true;
  refreshNetworkListButtons();
}

function setActiveTab(tab) {
  var navItems = document.getElementsByClassName("nav-item");
  for (var i = 0; i < navItems.length; i++) {
    if (i === tab) {
      navItems[i].className = "nav-item active";
    } else {
      navItems[i].className = "nav-item";
    }
  }
}

function ableToCreateBond() {
  clearErrors();

  if ($('.networks:checked').length < 2) {
    document.getElementById("bondError").style.display = "inline-block";
    return false;
  } else if ($('.bonds:checked').length + $('.vlans:checked').length > 0) {
    document.getElementById('bondError2').style.display = "inline-block"
    return false;
  }
  if ($('#ipmi:checked').length === 1) {
    showError("Please ensure that IPMI interface is not part of a bond");
    return false;
  }

  let node = window.model.advancedNodes[window.currentNode].networks;
  let a_speeds = [];

  node.forEach(function (data) {
    let entities = document.querySelectorAll("#network-list input[type=checkbox]:checked");
    for (let i = 0; i < entities.length; ++i) {
      if (data.name === entities[i].value)
        a_speeds.push(data.link_speed);
    }
  });

  for (let j = 1; j < a_speeds.length; ++j) {
    if (a_speeds[j - 1] !== a_speeds[j]) {
      showError("Please ensure that the interfaces for bonding have the same capacity");
      return false;
    }
  }

  showCreateBondModal();
  return false;
}

function showCreateBondModal() {
  document.getElementById('create-bond-modal').style.display = 'flex';
}

function hideCreateBondModal() {
  document.getElementById('create-bond-modal').style.display = 'none';
}

function ableToCreateVLAN() {
  clearErrors();
  if ($('.networks:checked').length + $('.bonds:checked').length != 1) {
    document.getElementById("vlansError").style.display = "inline-block";
    return false
  }
  showCreateVlanModal();
}

function removeVlanId(index) {
  var input = document.getElementById("vlan-id-input-" + index);
  input.parentNode.removeChild(input);
}

function showCreateVlanModal() {
  var ids = document.getElementById("ids");
  ids.innerHTML =
    "<div id='vlan-id-input-0'>" +
      "<label>VLANs ID: </label>" +
      "<input type='text' class='vlansid'>" +
      "<a class='delete hidden'>&times;</a>" + // This is included for spacing
    "</div>";
  window.vlanIdCount = 1;
  document.getElementById('id01').style.display = 'flex';
}

function hideCreateVlanModal() {
  document.getElementById('id01').style.display = 'none';
}

function addVLANid() {
  var div = document.createElement("div");
  div.id = "vlan-id-input-" + window.vlanIdCount;
  div.innerHTML =
    "<label>VLANs ID: </label>" +
    "<input type='text' class='vlansid'>" +
    "<a class='delete' onclick='removeVlanId(" + window.vlanIdCount + ")'>&times;</a>";
  window.vlanIdCount += 1;
  document.getElementById("ids").appendChild(div);
}

function showError(message) {
  document.getElementById("advancedSettingsError").innerHTML = message;
  document.getElementById("advancedSettingsError").style.display = "inline-block";
  document.getElementById("settingsError").innerHTML = message;
  document.getElementById("settingsError").style.display = "inline-block";
  document.getElementById("networkInfoError").innerHTML = message;
  document.getElementById("networkInfoError").style.display = "inline-block";
  document.getElementById("clusterInfoError").innerHTML = message;
  document.getElementById("clusterInfoError").style.display = "inline-block";
}

var validateDNSEntries = function() {
   if (window.model.dnsList.length < 1) {
        showError("Please ensure that at least one DNS entry exists");
        return false;
   }
   if (window.model.dnsList.length > 3) {
        showError("Please ensure that a maximum of 3 DNS servers are entered");
        return false;
   }
   if (!window.model.dnsList.every(function(ip) { return _isValidIP(ip)})) {
        showError("Please ensure that you enter valid IP address for DNS servers");
        return false;
   }
   let s_dnsentries = new Set();
   for (let i = 0; i < window.model.dnsList.length; ++i)
     s_dnsentries.add(window.model.dnsList[i]);
   if (window.model.dnsList.length !== s_dnsentries.size) {
        showError("Please ensure that you enter unique DNS IPs");
        return false;
   }
   return true;
}

function setevent_clear_password_errors() {
      $("#root_pwd").click(function() {
          clearErrors();
      });
      $("#root_confirm_pwd").click(function() {
          clearErrors();
      });
}


function validateNodeStep() {
  if (!window.model.useNodePasswords) {
    if (!window.model.password) {
      showError("Please provide a password");
      setevent_clear_password_errors();
      return false;
    } else if (window.model.password !== window.model.confirmPassword) {
      showError("The passwords do not match");
      setevent_clear_password_errors();
      return false;
    }
    if (!_isValidOSPassword(window.model.password))
      return false;
  }
  if (!window.model.useNodePasswords && window.model.password !== window.model.confirmPassword) {
    showError("The passwords do not match");
    setevent_clear_password_errors();
    return false;
  }
  if (window.model.useAdvancedSettings) {
    // Skipping network input checks for in-lab installs
	if (window.model.is_lab_install)
	  return true;

    for (var i = 0; i < window.model.advancedNodes.length; i++) {
      var node = window.model.advancedNodes[i];
      if (window.model.useNodePasswords && !node.password) {
        setNode(i);
        showError("Please provide all network settings for each node");
        return false;
      }
      if (window.model.useNodePasswords && node.password !== node.confirmPassword) {
        setNode(i);
        showError("The passwords for Node " + (i + 1) + " do not match");
        setevent_clear_password_errors();
        return false;
      }

        let ips, str_err;
        // This validation is to check whether all nodes belong to same product
        if (!validateNodeTypeAdvanced()) {
            str_err = "Please ensure that all three nodes are of same type";
            showError(str_err);
            return false;
        }

        try {
            // This try-catch handles multiple cases
            ips = validateInputsAdvanced();
        } catch(e) {
          if (e) {
            if (e.get('config')) {
                str_err = "Please provide same network configurations on all nodes for: <br>";
                str_err += e.get('config').join("; ");
            } else if (e.get('ip')) {
                str_err = "Please provide IP addresses on all nodes for: <br>";
                str_err += e.get('ip').join("; ");
            } else if (e.get('link_up')) {
                str_err = "Please ensure that the interfaces are up and running for: <br>";
                let m_err = e.get('link_up');
                m_err.forEach(function (v, k) {
                    str_err += k + ": " + v.join("; ") + "<br>";
                });
            } else if (e.get('cs_sp')) {
                str_err = "Please ensure that each node has a CS Registration and Storage Pool network";
            } else if (e.get('mgmt_1G')) {
                str_err = "Please ensure that management network is on a 1G interface";
            } else if (e.get('csreg_10G')) {
                str_err = "Please ensure that CS Registration is on a 10G interface";
            } else if (e.get('dp_10G')) {
                str_err = "Please ensure that Data Protection is on a 10G interface";
            } else if (e.get('sp_10G')) {
                str_err = "Please ensure that Storage Pool is on a 10G interface";
            } else {
                str_err = "WARNING: Unexpected error encountered while parsing inputs";
            }
            showError(str_err);
            return false;
          }
        }
        // Validation for invalid IPv4 IP addresses
        ips = getInvalidIPv4Advanced();
        if (ips.size) {
            str_err = "Please correct these invalid IPs: <br>";
            ips.forEach( function (item, ip) {
                str_err += ip + ": " + item.join('; ') + "<br>";
            });
            showError(str_err);
            return false;
        }
        ips = getInvalidSubnetsAdvanced();
        if (ips.size) {
            str_err = "Please correct these invalid Netmasks: <br>";
            ips.forEach( function (item, subnet) {
                str_err += subnet + ": " + item.join('; ') + "<br>";
            });
            showError(str_err);
            return false;
        }
        // Validation for duplicate IPs
        ips = getDuplicatesAdvanced();
        if (ips.size) {
            str_err = "Please correct these duplicate IPs: <br>";
            ips.forEach( function (item, ip) {
                str_err += ip + ": " + item.join('; ') + "<br>";
            });
            showError(str_err);
            return false;
        }
        // Validate that only one interface is used for CS registration
        // Validate that CS IP is not on VLAN
        // Validate that Management IP is not on VLAN
        // Validate node entities:
        var entities = getNodeEntities(i);
        for (var j = 0; j < entities.length; j++) {
            var entity = entities[j];
         }
    }
  } else {
    for (var i = 0; i < window.model.basicNodes.length; i++) {
      var node = window.model.basicNodes[i];
      if (!node.dataProtectionNetwork ||
        !node.storagePool ||
        (window.model.useSeparateManagementNetwork && !node.managementNetwork) ||
        (window.model.useNodePasswords && !node.password)
      ) {
        showError("Please provide all network settings for each node");
        return false;
      }
      if (window.model.useNodePasswords && node.password !== node.confirmPassword) {
        showError("The passwords for Node " + (i + 1) + " do not match");
        setevent_clear_password_errors();
        return false;
      }
    }
  }
  return true;
}

var checkPTRResponse = function(json_resp) {
    var multi_map = {
      data_protection_ips: {
        CurrentNode: 'node1_data_protection',
        PeerNode1  : 'node2_data_protection',
        PeerNode2  : 'node3_data_protection'
      }
    }

    hideServerProcess();

    if (json_resp.status !== "OK") {
      for (var network_type in multimap){
        for (var node in multimap[network_type]) {
          if (json_resp[network_type][node] !== "OK") {
            showErrorForNode('#'+id_map[network_type][node], 'IP address does not map to a valid hostname.');
          }
        }
      }
      return false;
    } else {
      // All the validations succeeded for step1. Now go to step2.        
      stepProcess(1,2);
    }
    return true;
}

var validatePTR = function() {
  showServerProcess ("loading","Validating DNS entries");

  var query_data = {
      data_protection_ips: {
          CurrentNode: $('#node1_data_protection').val(),
          PeerNode1: $('#node2_data_protection').val(),
          PeerNode2: $('#node3_data_protection').val()
      },
      network: {
          config_mgmt: "false"
      }
  }

  var json_query = JSON.stringify(query_data);
  var parsed_data = {
      status: 'FAILED'
  }

  $.ajax({
      url: '/validate_ptr',
      data: json_query,
      type: 'POST',
      contentType: "application/json",
      success: function (response) {
          //Check whether each provided IP address has a valid PTR record
          hideServerProcess();        
          var json_resp = JSON.parse(response);
          return checkPTRResponse (json_resp);          
      },
      error: function(errormsg){
          hideServerProcess();        
          showError("PTR validation failed");    
          let str_response = JSON.parse(errormsg.responseText);
          console.log("Error response: ", str_response);
          return false;
      }
  });
}

var validateAdvancedNetConfResponse = function (json_resp) {
  if (json_resp.status !== "OK") {
    var htmlelement = "";
    for (var j in json_resp["advancedNodes"]) {
      if (j["error"] != ""){
          htmlelement += '<br><em>' + json_resp["advancedNodes"][j]["error"] + '</em>';
      }
    }
    showError(htmlelement);
    return false;
  } else {
    window.model.useSeparateManagementNetwork = json_resp.useSeparateManagementNetwork;

    if (true == window.model.useSeparateManagementNetwork) {
        $("#check_manageNetwork").removeClass("hide");
    } else {
        $("#check_manageNetwork").addClass("hide");
    }
    window.model.default_gw_interface = json_resp.default_gw_interface;
    saveLocalStorageModel();
    // All the validations succeeded for step1. Now go to step2.        
    stepProcess(1,2);    
  }
}

var validateAdvancedNetConf = function() {
  showServerProcess ("loading","Validating advanced network configuration");

  var query_data = window.model;
  query_data["net_conf_mode"] = "advanced";

  var json_query = JSON.stringify(query_data);
  var parsed_data = {
      status: 'FAILED'
  }

  $.ajax({
      url: '/validate_advanced',
      data: json_query,
      type: 'POST',
      contentType: "application/json",
      success: function (response) {
          hideServerProcess();        
          var json_resp = JSON.parse(response);
          console.log("Success Response: ", json_resp);
          validateAdvancedNetConfResponse(json_resp);      
      },
      error: function(errormsg){
          hideServerProcess();
          let str_response = JSON.parse(errormsg.responseText);
          console.log("Error response: ", str_response);
          showError(str_response.message);
          return false;
      }
  });
}


//Get a user friendly description for the reported error code
var getHardwareErrorDesc = function(desc) {
  if ('NWLINK_MISSING' == desc) {
    return 'One of the 10G links is down. Please check the connectivity.';
  } else if ('NVME_MISSING' == desc) {
    return 'NVME drive for DDB/Index cache is missing.';
  } else if ('SAS_MISSING' == desc) {
    return 'One of the SAS drives is missing.';
  } else if ('INSUF_1G_LINK' == desc) {
    return 'Insufficient 1G link.';
  } else if ('IPMI_MISSING' == desc) {
    return 'IPMI link is down, please check the connectivity.';
  } else if ('BOND10GLINKS_MISSING' === desc) {
    return 'Some 10G interfaces are down and bond cannot be created';
  } else {
    return 'Some of the hardware components are missing.';
  }
}

var BasicNetConfCheckHardwareResponse = function(json_resp) {
    
    // Hide CS 10G IP in case of basic configuration.
    // Should be enabled only in case of 1G management network
    $("#check_manageNetwork").addClass("hide");
    
    var id_map = {
      'CurrentNode': 'node1_data_protection',
      'PeerNode1'  : 'node2_data_protection',
      'PeerNode2'  : 'node3_data_protection'
    }

    hideServerProcess();        

    //Check whether each of the node has all the required hardware configuration.
    //Check whether both the Data protection network and StoragePool network links have a 10G connectivity
    //Check whether all the NVME/SAS drives are present
    if (json_resp.status !== "OK") {    
        for (var key in id_map) {
            if (json_resp[key] !== "OK") {
                showErrorForNode('#'+id_map[key], getHardwareErrorDesc(json_resp[key]));
            }
        }
        return false;
    } else {
      return validatePTR();  
    }
    return true;
}

var AdvancedNetConfCheckHardwareResponse = function (json_resp) {
  if (json_resp.status !== "OK") {
    var htmlelement = "";
    for (var j in json_resp["error_strings"]) {
      htmlelement += '<br><em>' + json_resp["error_strings"][j] + '</em>';
    }
    showError(htmlelement)
    return false;
  } else {
    return validateAdvancedNetConf();
  }
}

var checkHardwareResponse = function(json_resp) {
    /* If advanced option is selected call a different app route */
    if (!window.model.useAdvancedSettings) {
      return BasicNetConfCheckHardwareResponse(json_resp);
    } else {
      return AdvancedNetConfCheckHardwareResponse(json_resp);
    }
}

var validateHardware = function() {
    showServerProcess ("loading","Validating hardware configuration");  
    enableProgressDial ("progress_loader", "Validating hardware...");  

    var query_data = window.model;

    var json_query = JSON.stringify(query_data);
    var parsed_data = {
        status: 'FAILED'
    }

  $.ajax({
      url: '/validate_hardware',
      data: json_query,
      type: 'POST',
      contentType: "application/json",
      success: function (response) {
          hideServerProcess();
          disableProgressDial ("progress_loader");
          parsed_data = JSON.parse(response);
          return checkHardwareResponse (parsed_data);        
      },
      error: function(errormsg){
          hideServerProcess();
          disableProgressDial ("progress_loader");
          let str_response = JSON.parse(errormsg.responseText);
          console.log("Error response: ", str_response);
          showError(str_response.message);
          return false;
      }
  });
}

var getDuplicates = function() {
    let duplicates = new Set();    
    var ipaddr_map = {
                       'node1_data_protection' : $('#node1_data_protection').val(),
                       'node2_data_protection' : $('#node2_data_protection').val(),
                       'node3_data_protection' : $('#node3_data_protection').val(),
                       'node1_storage_pool'    : $('#node1_storage_pool').val(),
                       'node2_storage_pool'    : $('#node2_storage_pool').val(),
                       'node3_storage_pool'    : $('#node3_storage_pool').val(),
                       'node1_ipmi'            : $('#node1_ipmi').val(),
                       'node2_ipmi'            : $('#node2_ipmi').val(),
                       'node3_ipmi'            : $('#node3_ipmi').val()
                     };          

    for (var key1 in ipaddr_map) {
        for (var key2 in ipaddr_map) {
            if (key1 !== key2 && ipaddr_map[key1] === ipaddr_map[key2]) {
                duplicates.add(key2);
            }
        }
    }
    return duplicates;
}

var getInvalidIPv4 = function() {
    let s_invalidips = new Set();    
    let m_ipaddr = {
                       'node1_data_protection' : $('#node1_data_protection').val(),
                       'node2_data_protection' : $('#node2_data_protection').val(),
                       'node3_data_protection' : $('#node3_data_protection').val(),
                       'node1_storage_pool'    : $('#node1_storage_pool').val(),
                       'node2_storage_pool'    : $('#node2_storage_pool').val(),
                       'node3_storage_pool'    : $('#node3_storage_pool').val(),
                       'node1_ipmi'            : $('#node1_ipmi').val(),
                       'node2_ipmi'            : $('#node2_ipmi').val(),
                       'node3_ipmi'            : $('#node3_ipmi').val()
                     };          

    for (let key1 in m_ipaddr) {
        if (!_isValidIP(m_ipaddr[key1])) {
            s_invalidips.add(key1);
        }
    }
    return s_invalidips;
}

var _isValidOSPassword = function (str_passwd) {
  let patt = /(\s)|(\\)|(')/g;
  if (patt.test(str_passwd)) {
    showError("Please avoid spaces, ' and \\ in the OS password");
    return false;
  }
  return true;
}

var _isValidCSPassword = function (str_passwd) {
  let patt = /(\s)|(\\)|(")|(:)|(=)|(')/g;
  if (patt.test(str_passwd)) {
    showError("Please avoid spaces, \", \', \:, \= and \\ in the CommServe password");
    return false;
  }
  return true;
}

var _isSameSubnet = function (ip1, sub1, ip2, sub2) {
    let m_ip = new Map();
    m_ip.set(ip1, sub1);
    m_ip.set(ip2, sub2);
    let a_nwid = [];
    Object.keys(m_ip).forEach(function (key) {
        if (key && m_ip[key]) {
            let str_nwid = "";
            let ip_oc = key.split('.');
            let sn_oc = m_ip[key].split('.');
            for (let i = 0; i < ip_oc.length; ++i) {
              if (i > 0)
                str_nwid += '.';
              str_nwid += (parseInt(ip_oc[i]) & parseInt(sn_oc[i])).toString();
            }
            a_nwid.push(str_nwid);
        }
    });
    if (a_nwid[0] !==  a_nwid[1])
        return false;
    return true;
}

var _isValidIP = function(str_ip) {
    if (/^(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$/.test(str_ip))
        return true;
    return false;
}

var _isValidSubnet = function(str_ip) {
    let allowed_netmasks = ["0.0.0.0", "128.0.0.0", "192.0.0.0", "224.0.0.0", "240.0.0.0", "248.0.0.0", "252.0.0.0", "254.0.0.0", "255.0.0.0", "255.128.0.0", "255.192.0.0",
"255.224.0.0", "255.240.0.0", "255.248.0.0", "255.252.0.0", "255.254.0.0", "255.255.0.0", "255.255.128.0", "255.255.192.0", "255.255.224.0", "255.255.240.0", "255.255.248.0",
"255.255.252.0", "255.255.254.0", "255.255.255.0", "255.255.255.128", "255.255.255.192", "255.255.255.224", "255.255.255.240", "255.255.255.248", "255.255.255.252", "255.255.255.254", "255.255.255.255"];
    if (allowed_netmasks.includes(str_ip))
        return true;
    return false;
}

// Since the HTML does not have separate IDs for nodes like simple n/w config, we need to use the in-memory DS to pull out the details
var _getAllAdvancedIPs = function() {
    // Returns a map of IPs and array of node and interface name
    let m_ip = new Map();
    let data = window.model.advancedNodes;
    let a_interface_types = ["networks", "bonds", "vlans"];
    // Assumes that Link UP/DOWN validation and IP is empty validation is already done
    for (let i = 0; i<NUM_NODES; ++i) {
        for (let iftype=0; iftype < a_interface_types.length; ++iftype) {
            let nwtype = a_interface_types[iftype];
            for (let j = 0; j<data[i][nwtype].length; ++j) {
                if (data[i][nwtype][j].ip) {
                    let ipaddr = data[i][nwtype][j].ip;
                    if (!m_ip.get(ipaddr)) {
                        m_ip.set(ipaddr, []);
                    }
                    m_ip.get(ipaddr).push([ "node" + (i+1).toString(), data[i][nwtype][j].name ]);
                }
            }
        }
    }
    return m_ip;
}

var _getAllAdvancedSubnets = function() {
    // Returns a map of IPs and array of node and interface name
    let m_ip = new Map();
    let data = window.model.advancedNodes;
    let a_interface_types = ["networks", "bonds", "vlans"];
    // Assumes that Link UP/DOWN validation and IP is empty validation is already done
    for (let i = 0; i<NUM_NODES; ++i) {
        for (let iftype=0; iftype < a_interface_types.length; ++iftype) {
            let nwtype = a_interface_types[iftype];
            for (let j = 0; j<data[i][nwtype].length; ++j) {
                if (data[i][nwtype][j].ip) {
                    let subnet = data[i][nwtype][j].netmask;
                    if (!m_ip.get(subnet)) {
                        m_ip.set(subnet, []);
                    }
                    m_ip.get(subnet).push([ "node" + (i+1).toString(), data[i][nwtype][j].name ]);
                }
            }
        }
    }
    return m_ip;
}


var _getCSRegistryIPSimple = function() {
  let ips = [];
  var ipaddr_map = {
                     'node1_data_protection' : $('#node1_data_protection').val(),
                     'node2_data_protection' : $('#node2_data_protection').val(),
                     'node3_data_protection' : $('#node3_data_protection').val()
                   };          

  for (var key1 in ipaddr_map) {
    if (ipaddr_map[key1]) {
        ips.push(ipaddr_map[key1]);
    }
  }
  return ips;
}

var getCSRegistryIPs = function() {
  let is_advanced = window.model.useAdvancedSettings;
  let a_ips = [];
  if (! is_advanced) {
    a_ips = _getCSRegistryIPSimple();
    return a_ips;
  }
  let data = window.model.advancedNodes;
  let a_interface_types = ["bonds", "vlans", "networks"];
  for (let iftype = 0; iftype < a_interface_types.length; ++iftype) {
    let nwtype = a_interface_types[iftype];
    for (let i = 0; i < NUM_NODES; ++i) {
      for (let j = 0; j < data[i][nwtype].length; ++j) {
          if (data[i][nwtype][j].networkType && data[i][nwtype][j].networkType === "commServeRegistration") {
            if (data[i][nwtype][j].ip)
              a_ips.push(data[i][nwtype][j].ip)
          }
      }
    }
  }
  return a_ips;
}

var validateNodeTypeAdvanced = function() {
    let data = window.model.advancedNodes;
    // Assumes that there are only 3 nodes
    if (data[0].nodetype !== data[1].nodetype || data[1].nodetype !== data[2].nodetype
        || data[0].nodetype !== data[2].nodetype) {
        return false;
    }
    return true;
}

var validateInputsAdvanced = function() {
    let m_ip = new Map();
    let data = window.model.advancedNodes;
    let a_interface_types = ["bonds", "vlans", "networks"];
    let a_results = [];
    // Assumes # of nodes (NUM_NODES) is always 3
    for (let iftype=0; iftype < a_interface_types.length; ++iftype) {
        let nwtype = a_interface_types[iftype];
        if (data[0][nwtype].length !== data[1][nwtype].length ||
            data[0][nwtype].length !== data[2][nwtype].length ||
            data[1][nwtype].length !== data[2][nwtype].length) {
            a_results.push(nwtype);
        }
    }
    if (a_results.length) {
        m_ip.set('config', a_results);
        throw m_ip;
    }
    m_ip.clear();
    a_results = [];
    let m_node = new Map();
    // Assumes # of nodes (NUM_NODES) is always 3
	for (let iftype = 0; iftype < a_interface_types.length; ++iftype) {
        let nwtype = a_interface_types[iftype];
        // Use first node as reference to compare with node 2 and 3
        for (let j = 0; j < data[0][nwtype].length; ++j) {

            let base_iface = data[0][nwtype][j].name;
            if (data[0][nwtype][j].ip) {
              data[1][nwtype].forEach(function(item, index) {
                if (item.name === base_iface) {
                  if (!item.ip)
                    a_results.push(base_iface)
                }
              });
              data[2][nwtype].forEach(function(item, index) {
                if (item.name === base_iface) {
                  if (!item.ip)
                    a_results.push(base_iface)
                }
              });
            }
        }
    }
    if (a_results.length) {
        m_ip.set('ip', a_results);
        throw m_ip;
    }
    m_ip.clear();
    a_results = [];
    m_node.clear();
    for (let iftype = 0; iftype < a_interface_types.length; ++iftype) {
        let nwtype = a_interface_types[iftype];
        let str_node;
		for (let j = 0; j < data[0][nwtype].length; ++j) {
            let base_iface = data[0][nwtype][j].name;
            for (let i = 1; i < NUM_NODES; ++i) {
                str_node = "node" + (i + 1).toString();
                if (data[0][nwtype][j].ip) {
                  data[i][nwtype].forEach(function(item, index) {
                    if (nwtype === "networks" && item.name === base_iface) {
                      if (item.link_speed === "No link") {


                        if (!m_node.get(str_node))
                          m_node.set(str_node, []);
                        m_node.get(str_node).push(item.name);
                      }
                    }

                    if (nwtype === "vlans" && item.parent.name === base_iface) {
                      if (item.parent.link_speed === "No link") {
                        if (!m_node.get(str_node))
                          m_node.set(str_node, []);
                        m_node.get(str_node).push(item.parent.name);
                      }
                    }

                    if (nwtype === "bonds") {
                      item.children.forEach(function (childitem, childindex) {
                        if (childitem.name === base_iface) {
                          if (childitem.link_speed === "No link") {
                            if (!m_node.get(str_node))
                              m_node.set(str_node, []);
                            m_node.get(str_node).push(childitem.name);
                          }
                        }
                      });
                    }
                  });
                }
                m_ip.set('link_up', m_node);
            }
        }
    }
    if (m_ip.get('link_up').size) {
        throw m_ip;
    }
    m_ip.clear();
    let cs_nw_check = 0, sp_nw_check = 0;
    for (let iftype = 0; iftype < a_interface_types.length; ++iftype) {
        let nwtype = a_interface_types[iftype];
        for (let i = 0; i < NUM_NODES; ++i) {
            for (let j = 0; j < data[i][nwtype].length; ++j) {
                // TODO: Add IPMI requiment to this once other forms are checked in
                if (data[i][nwtype][j].networkType && data[i][nwtype][j].networkType === "commServeRegistration")
                    cs_nw_check++;
                if (data[i][nwtype][j].networkType && data[i][nwtype][j].networkType === "storagePool")
                    sp_nw_check++;
            }
        }
    }
    if (cs_nw_check !== NUM_NODES || sp_nw_check !== NUM_NODES) {
        m_ip.set('cs_sp', []);
        throw m_ip;
    }
    m_ip.clear();
    for (let iftype = 0; iftype < a_interface_types.length; ++iftype) {
        let nwtype = a_interface_types[iftype];
        for (let i = 0; i < NUM_NODES; ++i) {
            for (let j = 0; j < data[i][nwtype].length; ++j) {
                if (data[i][nwtype][j].networkType && data[i][nwtype][j].networkType === "management" && data[i][nwtype][j].link_speed !== "1G")
                    m_ip.set('mgmt_1G', []);
                if (data[i][nwtype][j].networkType && data[i][nwtype][j].networkType === "commServeRegistration" && data[i][nwtype][j].link_speed !== "10G") {
                    // CS Registration can be on naked 10G interface or bonded interfaces. All bonded interfaces bust be on 10G interfaces
                    if (nwtype === "networks" && data[i][nwtype][j].link_speed !== "10G") {
                        m_ip.set('csreg_10G', []);
                        throw m_ip;
                    }
                    if (nwtype === "bonds") {
                        for (let k = 0; k < data[i][nwtype][j].children.length; ++k) {
                            if (data[i][nwtype][j].children[k].link_speed !== "10G") {
                                m_ip.set('csreg_10G', []);
                                throw m_ip;
                            }
                        }
                    }
                }
                if (data[i][nwtype][j].networkType && data[i][nwtype][j].networkType === "dataProtection" && data[i][nwtype][j].link_speed !== "10G") {
                    if (nwtype === "networks" && data[i][nwtype][j].link_speed !== "10G") {
                        m_ip.set('dp_10G', []);
                        throw m_ip;
                    }
                    if (nwtype === "bonds") {
                        for (let k = 0; k < data[i][nwtype][j].children.length; ++k) {
                            if (data[i][nwtype][j].children[k].link_speed !== "10G") {
                                m_ip.set('dp_10G', []);
                                throw m_ip;
                            }
                        }
                    }
                }
                if (data[i][nwtype][j].networkType && data[i][nwtype][j].networkType === "storagePool") {
                    // SP network can be either on naked 10G interface or bonded interfaces. All bonded interfaces must be 10G interfaces
                    if (nwtype === "networks" && data[i][nwtype][j].link_speed !== "10G") { 
                            m_ip.set('sp_10G', []);
                            throw m_ip;
                    }
                    if (nwtype === "bonds") {
                        for (let k = 0; k < data[i][nwtype][j].children.length; ++k) {
                            if (data[i][nwtype][j].children[k].link_speed !== "10G") {
                                m_ip.set('sp_10G', []);
                                throw m_ip;
                            }
                        }
                    }
                }
            }
        }
    }
    if (m_ip.size) {
        throw m_ip;
    }
}



var getInvalidIPv4Advanced = function() {
    let m_ip = _getAllAdvancedIPs();
    m_ip.forEach(function(item, key) {
        if (_isValidIP(key)) {
            m_ip.delete(key);
        }
    });
    return m_ip;
}

var getInvalidSubnetsAdvanced = function() {
    let m_ip = _getAllAdvancedSubnets();
    m_ip.forEach(function(item, key) {
        if (_isValidSubnet(key)) {
            m_ip.delete(key);
        }
    });
    return m_ip;
}

var getDuplicatesAdvanced = function() {
    let m_ip = _getAllAdvancedIPs();
    m_ip.forEach(function(item, key) {
        if (item.length < 2) {
            m_ip.delete(key);
        }
    });
    return m_ip;
}


function performBackgroundCheck() {
  /* Check for dupliate IP addresses per node */
  if (!window.model.useAdvancedSettings) { 
    let duplicates = getDuplicates();
    if (duplicates.size) {
      showError("Found duplicate IP addresses. Please enter unique IP address for network configuration"); 
      var keys = [          
           'node1_data_protection',
           'node2_data_protection',
           'node3_data_protection',
           'node1_storage_pool',
           'node2_storage_pool',
           'node3_storage_pool',
           'node1_ipmi',
           'node2_ipmi',
           'node3_ipmi'
         ];            

      for (var key in keys) {
        if (duplicates.has(keys[key])) {  
            showErrorForNode('#'+keys[key], 'Found a duplicate IP address entry.');
        }
      }
      return false;
    }
    let s_invalid_ips = getInvalidIPv4();
    if (s_invalid_ips.size) {
      showError("Found invalid IP addresses. Please enter valid IP address for network configuration"); 
      let keys = [          
           'node1_data_protection',
           'node2_data_protection',
           'node3_data_protection',
           'node1_storage_pool',
           'node2_storage_pool',
           'node3_storage_pool',
           'node1_ipmi',
           'node2_ipmi',
           'node3_ipmi'
         ];            

      for (let key in keys) {
        if (s_invalid_ips.has(keys[key])) {  
            showErrorForNode('#'+keys[key], 'Found an invalid IP address.');
        }
      }
      return false;
    }
  }
  /* Validate hardware information */

  if (!validateHardware()) {
    return false;
  }
}

var BasicNetConfCheckConfigNwResponse = function(json_resp) {
    if (json_resp.status !== "OK") {
        // Navigate back to step1 and so that user can correct i/p errors
        stepProcess (2,1); 

        var id_map = {
          "CurrentNode": 'node1',
          "PeerNode1": 'node2',
          "PeerNode2": 'node3'
        }

        for (var key in id_map) {
            if (json_resp[key]["data_protection"] !== "") {
                showErrorForNode('#'+id_map[key]+"_data_protection",json_resp[key]["data_protection"]);
            }
            if (json_resp[key]["storage_pool"] !== "") {
                showErrorForNode('#'+id_map[key]+"_storage_pool",json_resp[key]["storage_pool"]);
            }
        }
        return false 
    } else {
        stepProcess (2,3);
    }
}

var AdvancedNetConfCheckConfigNwResponse = function(json_resp) {
  if (json_resp.status !== "OK") {
    // Navigate back to step1 and so that user can correct i/p errors
    stepProcess (2,1); 
    
    var htmlelement = "";
    for (var j in json_resp["advancedNodes"]) {
      if (j["error"] != ""){
          htmlelement += '<br><em>' + json_resp["advancedNodes"][j]["error"] + '</em>';
      }
    }
    showError(htmlelement);
    return false 
  } else {
    stepProcess (2,3);
  }
}

var checkConfigNwResponse = function(json_resp) {

    hideServerProcess ();        
    disableProgressDial ("progress_loader"); 

    if (!window.model.useAdvancedSettings) {
      BasicNetConfCheckConfigNwResponse(json_resp);
    } else {
      AdvancedNetConfCheckConfigNwResponse(json_resp);
    }    
}

var configNw = function() {
  showServerProcess ("loading","Configuring network");    
  enableProgressDial ("progress_loader", "Configuring network ...");

  if (!window.model.useAdvancedSettings) {
    var query_data = {
        global: {
            different_rootpass: false,
            root_pwd: $('#root_pwd').val(),
            enable_bonding: window.model.useBondedInterface,
            bonding_mode: window.model.bondMode 
        },
        data_protection_ips: {
            CurrentNode: $('#node1_data_protection').val(),
            PeerNode1: $('#node2_data_protection').val(),
            PeerNode2: $('#node3_data_protection').val()
        },
        storagepool_ips: {
            CurrentNode: $('#node1_storage_pool').val(),
            PeerNode1: $('#node2_storage_pool').val(),
            PeerNode2: $('#node3_storage_pool').val()
        },
        ipmi_ips: {
            CurrentNode: $('#node1_ipmi').val(),
            PeerNode1: $('#node2_ipmi').val(),
            PeerNode2: $('#node3_ipmi').val()
        },
        management_ips: {
            CurrentNode: "",
            PeerNode1: "",
            PeerNode2: ""
        },
        passwords: {
            CurrentNode: $('#root_pwd').val(),
            PeerNode1: $('#root_pwd').val(),
            PeerNode2: $('#root_pwd').val()
        },
        network: {
            default_gw : $('#gateway').val(),
            dp_netmask: $('#netmask').val(),
            ipmi_gw : $('#ipmi_gateway').val(),
            ipmi_netmask : $('#ipmi_netmask').val(),
            nameservers: window.model.dnsList,
            config_mgmt: "false", 
            sp_netmask: $('#cluster_network').val()
        }
    }
    query_data["net_conf_mode"] = "basic";
  } else {
    var query_data = window.model;
    query_data["net_conf_mode"] = "advanced";
  }

  var json_query = JSON.stringify(query_data);

  var parsed_data = {
      status: 'FAILED'
  }

  $.ajax({
    url: '/configure_network',
    data: json_query,
    type: 'POST',
    contentType: "application/json",
    success: function (response) {
      parsed_data = JSON.parse(response);
      return checkConfigNwResponse(parsed_data);
    },
    error: function(errormsg){
      disableProgressDial ("progress_loader");
      let str_response = JSON.parse(errormsg.responseText);
      console.log("Error response: ", str_response);
      showError(str_response.message);
      return false;
    }
  });
  return parsed_data;
                     
}

var validateSubnet = function() {
  showServerProcess ("loading","Validating network configuration");    

  if (!window.model.useAdvancedSettings) {
    var query_data = {
        data_protection_ips: {
            CurrentNode: $('#node1_data_protection').val(),
            PeerNode1: $('#node2_data_protection').val(),
            PeerNode2: $('#node3_data_protection').val()
        },
        storagepool_ips: {
            CurrentNode: $('#node1_storage_pool').val(),
            PeerNode1: $('#node2_storage_pool').val(),
            PeerNode2: $('#node3_storage_pool').val()
        },
        network: {
            dp_netmask: $('#netmask').val(),
            sp_netmask: $('#cluster_network').val(),
        }
    }
    query_data["net_conf_mode"] = "basic";
  } else {
    var query_data = window.model;
    query_data["net_conf_mode"] = "advanced";
  }

  var json_query = JSON.stringify(query_data);
  var parsed_data = {
      status: 'FAILED'
  }

  $.ajax({
      url: '/validate_subnet',
      data: json_query,
      type: 'POST',
      contentType: "application/json",
      success: function (response) {
          parsed_data = JSON.parse(response);
          if (parsed_data.status !== "OK") {    
            showErrorForNode('#netmask', 'Data Protection and Storage Pool IP address in same subnet.');
            return false;
          } else {
            return configNw ();                
          }
      },
      error: function(errormsg){
          let str_response = JSON.parse(errormsg.responseText);
          showError (str_response.message);
          console.log(str_response);
          return false;
      }
  });
}

var createConfig = function() {
  showServerProcess ("loading", "Starting HyperScale configuration");    
  enableProgressDial ("progress_loader", "Starting HyperScale configuration ...");

  if (!window.model.useAdvancedSettings) {    
    var query_data = {
        global: {
            different_rootpass: false,
            management_network: "false",
            enable_bonding: window.model.useBondedInterface,
            bonding_mode: window.model.bondMode, 
            root_pwd: $('#root_pwd').val()
        },
        data_protection_ips: {
            CurrentNode: $('#node1_data_protection').val(),
            PeerNode1: $('#node2_data_protection').val(),
            PeerNode2: $('#node3_data_protection').val()
        },
        storagepool_ips: {
            CurrentNode: $('#node1_storage_pool').val(),
            PeerNode1: $('#node2_storage_pool').val(),
            PeerNode2: $('#node3_storage_pool').val()
        },
        ipmi_ips: {
            CurrentNode: $('#node1_ipmi').val(),
            PeerNode1: $('#node2_ipmi').val(),
            PeerNode2: $('#node3_ipmi').val()
        },
        management_ips: {
            CurrentNode: "",
            PeerNode1: "",
            PeerNode2: ""
        },
        passwords: {
            CurrentNode: $('#root_pwd').val(),
            PeerNode1: $('#root_pwd').val(),
            PeerNode2: $('#root_pwd').val()
        },
        network: {
            default_gw : $('#gateway').val(),
            dp_netmask: $('#netmask').val(),
            ipmi_gw : $('#ipmi_gateway').val(),
            ipmi_netmask : $('#ipmi_netmask').val(),
            nameservers: window.model.dnsList,
            config_mgmt: "false", 
            sp_netmask: $('#cluster_network').val()
        },
        cluster: {
            cshname: $('#hostname').val(),
            passwd: $('#pwd').val(),
            username: $('#username').val(),
            existing_cs: isExistingCommServe(),
            primary_cs: isPrimaryCommServe(),
            additional_cs: isAdditionalCommServe(),
            additional_cs_host: $('#additional_cs_host').val(),
            additional_cs_user: $('#additional_cs_user').val(),
            additional_cs_passwd: $('#additional_cs_passwd').val(),
            productid: $('#window_product').val(),
            commserve_backup_network: $('#commserve_backup_network').val(),
            standby_kvm_cs: isStandbyKvmCommServe(),
            standby_kvm_csname: $('#standbykvmcs_host').val(),
            standby_kvm_csprid: $('#standby_window_product').val()
        }
    }
    query_data["net_conf_mode"] = "basic"
  } else {
    var query_data = window.model;
    query_data["net_conf_mode"] = "advanced";
    query_data["clusterDetails"][0]["existing_cs"] = isExistingCommServe();
  }

  var json_query = JSON.stringify(query_data);

  $.ajax({
      type: 'POST',
      url: '/validate_setup',
      contentType: "application/json",
      data: json_query,
      success: function(response){
          var parsed_data = JSON.parse(response);
          if (parsed_data.status === "OK") {
              location.href="/processList";
          } else if (parsed_data.status == "resume_setup") {
              console.log("Resuming setup locally..");
              location.href = "/execute_setup";
          } else if (parsed_data.status == "redirect_to_peer") {
            console.log("Redirecting to: ", parsed_data.hname);
            window.location.href = "http://"+parsed_data.hname+"/execute_setup";
          }
          else {
              location.href="/error_index";
          }
      },
      error: function(errormsg) {
          hideServerProcess();
          disableProgressDial ("progress_loader");
          let str_response = JSON.parse(errormsg.responseText);
          console.log("Error response: ", str_response);
          showError(str_response.message);
      }
  });
}

var BasicNetConfCheckValidateDNSResponse = function(json_resp) {
  if (json_resp.status !== "OK") {    
      var id_map = {
        'Commserve_fqdn': 'hostname',
      }

      for (var key in id_map) {
        if (json_resp[key] !== "OK") {
            showErrorForNode('#'+id_map[key], 'Hostname does not map to a valid IP address.');
        }
      }
  } else {
      createConfig()
  }
}

var AdvancedNetConfCheckValidateDNSResponse = function(json_resp) {
  if (json_resp.status !== "OK") {    
      var id_map = {
        'Commserve_fqdn': 'hostname',
      }

      for (var key in id_map) {
        if (json_resp[key] !== "OK") {
            showErrorForNode('#'+id_map[key], 'Hostname does not map to a valid IP address.');
        }
      }
  } else {
      createConfig()
  }
}

var checkValidateDNSResponse = function(json_resp) {

  if (!window.model.useAdvancedSettings) {
    BasicNetConfCheckValidateDNSResponse(json_resp);
  } else {
    AdvancedNetConfCheckValidateDNSResponse(json_resp);
  }
}

var validateDNS = function() {
  showServerProcess ("loading", "Validating DNS entries");    
  var query_data = {
       Commserve_fqdn: $('#hostname').val(),
       useSeparateManagementNetwork: window.model.useSeparateManagementNetwork
  }
  var json_query = JSON.stringify(query_data);

  var parsed_data = {
      status: 'FAILED'
  }

  $.ajax({
      url: '/validate_dns',
      data: json_query,
      type: 'POST',
      contentType: "application/json",
      success: function (response) {
          hideServerProcess();        
          parsed_data = JSON.parse(response);
          checkValidateDNSResponse (parsed_data);            
      },
      error: function(errormsg){
          hideServerProcess();        
          disableProgressDial ("progress_loader");
          let str_response = JSON.parse(errormsg.responseText);
          console.log("Error response: ", str_response);
          showError(str_response.message);
      }
  });
  return parsed_data;
}

var validateCSCredentials = function() {
  showServerProcess ("loading", "Verfying CS password entered");

  let csname, cspasswd, csuser;
  if (!isAdditionalCommServe()) {
    csname = window.model.clusterDetails[0].hostname;
    cspasswd = window.model.clusterDetails[0].passwd;
    csuser = window.model.clusterDetails[0].username;
  } else {
    csname = window.model.clusterDetails[0].additional_cs_host;
    cspasswd = window.model.clusterDetails[0].additional_cs_passwd;
    csuser = window.model.clusterDetails[0].additional_cs_user;
  }

  let query_data = {
    csname: csname,
    cspasswd: cspasswd,
    csuser: csuser    
  }

  let json_query = JSON.stringify(query_data);
  
  let ips = getCSRegistryIPs();

  // showServerProcess ("loading", "Starting HyperScale configuration");
  enableProgressDial ("progress_loader", "Verifying CommServe credentials...");
  let redirected = false;
  let l_error = [];

  ips.forEach(function(host_ip)
  {
    $.ajax({
      url: 'http://' + host_ip + '/validate_cscreds',
      crossDomain: true,
      data: json_query,
      type: 'POST',
      contentType: "application/json",
      success: function (response) 
      {
        hideServerProcess();
        if (response.status === "OK" && !redirected)
        {
          redirected = true;
          clearLocalStorageModel();
          validateDNS();
        }
      },
      error: function(errormsg)
      {
        l_error.push(false);
  	    if (l_error.length === 3)
		{
          // let str_response = JSON.parse(errormsg.responseText);
          // console.log("Error response: ", str_response);
          hideServerProcess();
          disableProgressDial ("progress_loader");
          showError("Unable to validate credentials for the CS. Please verify and retry");
		}
      }
    });
  });
}

function submitStep() {
  clearErrors();
  var curStepNum = parseInt($(".current_step").attr("data-step"));

  if (curStepNum == 1) {
    if (!validateNodeStep()) {
      return;
    }

    if (!performBackgroundCheck()) {
      return;
    }
  } else if (curStepNum == 2) {

    if (!validateDNSEntries()) {
      return;
    }

    if (!validateNetworkStep()) {
      return;
    }

    // As of now, this validation is a backend validation.
    if (!validateSubnet()) {
      return;
    }

  }

  if (curStepNum !== 3) {
    stepProcess(curStepNum, curStepNum + 1);
  }

  if (curStepNum === 3) {
    if (!validateClusterStep()) {
      return;
    }

    let existing_cs = isExistingCommServe();
    if (!existing_cs) {
      let csname = window.model.clusterDetails[0].hostname;
      let cspasswd = window.model.clusterDetails[0].passwd;
      let hname_vector = csname.split(".");       
      let hname = hname_vector[0];    
      if (hname > 15) {
        disableProgressDial ("progress_loader");
        showError("CommServe Hostname cannot be more than 15 characters long");
        return false;
      }
	  
      if (!pwdRuleCheck(cspasswd, false)) {
        disableProgressDial ("progress_loader");
        showError("CommServe password complexity check failed");
        return false;
      }

      // Validate to check if Control Host and CS are in same subnet
      if (isAdditionalCommServe()) {
        // Validate the credentials of external CS
        validateCSCredentials();
      } else {
        clearLocalStorageModel();
        validateDNS();
        return true;
      }
    } else {
      // Validate the credentials of external CS
      validateCSCredentials();
    }

  }
}


function validateClusterStep() {
  //$(".node-info .input-inline").addClass("three-inputs"); addClass("error-input");
  clearErrors();
  
  let existing_cs = isExistingCommServe();
  let standby_kvm_cs = isStandbyKvmCommServe();

  if (!window.model.clusterDetails[0].hostname ||
     (existing_cs && !window.model.clusterDetails[0].username) ||
     (!$('#pwd').val()) ||
     (!existing_cs && !$('#confirm_pwd').val()) ||
     (!existing_cs && !window.model.clusterDetails[0].window_product)) {
    showError("Please enter all cluster information");
    return false;
  }

  if (!existing_cs) {
    if ($('#pwd').val() !== $('#confirm_pwd').val()) {
      showError("Passwords don't match");
      return false;
    }
    if (window.model.clusterDetails[0].hostname.length > 62) {
        showError("Please ensure that that CommServe FQDN is less than 63 characters");
        return false;
    }
    let cshost_format = window.model.clusterDetails[0].hostname.split('.');
    /*
    if (cshost_format.length < 3 || cshost_format.length > 4) {
      showError("Please ensure that Commserve FQDN is of the form <i>host.domain.org</i> or <i>host.domain1.domain2.org</i>");
      return false;
    }
    */
    if (cshost_format[0].length > 15) {
      showError("Please ensure that Commserve hostname is less than 15 characters");
      return false;
    }
    if (_isValidIP(window.model.clusterDetails[0].hostname)) {
        showError("CommServe does not accept IP address. Please enter a valid FQDN");
        return false;
    }
    if (!_isValidCSPassword(window.model.clusterDetails[0].passwd))
      return false;
    if (!window.model.clusterDetails[0].window_product.match(/^[A-Z0-9|-]+$/)) {
      showError("Primary CommServe's product ID should not contain special characters");
      return false;
    }
    if (window.model.clusterDetails[0].window_product.length !== 29) {
      showError("Primary CommServe's product ID should contain exactly 25 characters");
      return false;
    }

    // In case of standby database
    if (standby_kvm_cs) {
      if (!window.model.clusterDetails[0].standby_kvm_csprid || !window.model.clusterDetails[0].standby_kvm_csname) {
        showError("Please enter all cluster information for Standby CommServe");
        return false;
      }
      if (_isValidIP(window.model.clusterDetails[0].standby_kvm_csname)) {
        showError("Standby CommServe does not accept IP address. Please enter a valid FQDN");
        return false;
      }
      if (window.model.clusterDetails[0].standby_kvm_csname.length > 62) {
          showError("Please ensure that that Standby CommServe FQDN is less than 63 characters");
          return false;
      }
      if (window.model.clusterDetails[0].standby_kvm_csname.split('.')[0].length > 15) {
        showError("Please ensure that Standby Commserve hostname is less than 15 characters");
        return false;
      }
      if (!window.model.clusterDetails[0].standby_kvm_csprid.match(/^[A-Z0-9|-]+$/)) {
        showError("Standby CommServe's product ID should not contain special characters");
        return false;
      }
      if (window.model.clusterDetails[0].standby_kvm_csprid.length !== 29) {
        showError("Standby CommServe's product ID should contain exactly 25 characters");
        return false;
      }
      if (window.model.clusterDetails[0].standby_kvm_csprid === window.model.clusterDetails[0].window_product) {
        showError("Primary and Standby CommServe's product ID cannot be the same");
        return false;
      }
    }
    
  } else  { // Existing CS case
    if (!_isValidCSPassword(window.model.clusterDetails[0].passwd))
      return false;
  }

  if (window.model.useAdvancedSettings) {
    if (!existing_cs && window.model.useSeparateManagementNetwork && !window.model.clusterDetails[0].commserve_backup_network) {
      showError("Please enter all cluster information");
      return false;
    }
  }

  return true;
}

function validateNetworkStep() {
  clearErrors();
  if (!window.model.defaultGateway) {
    showError("Please provide the default gateway for data protection network");
    return false;
  }
  if (!_isValidIP(window.model.defaultGateway)) {
    showError("Please provide a valid IP for default gateway");
    return false;
  }
  if (isAppliance(0)) {
    if (!window.model.ipmiGateway) {
      showError("Please provide the IPMI gateway");
      return false;
    }
  }
  if (!window.model.useAdvancedSettings) {
    if (!window.model.dataProtectionNetmask) {
        showError("Please provide the data protection netmask");
        return false;
    }
    if (!window.model.clusterNetwork) {
        showError("Please provide the cluster (storage pool) netmask");
        return false;
    }
    if (isAppliance(0)) {
      if (!window.model.ipmiNetmask) {
        showError("Please provide the IPMI netmask");
        return false;
      }
      if (!_isValidSubnet(window.model.clusterNetwork)) {
        showError("Please provide a valid cluster (storage pool) netmask");
        return false;
      }
      if (!_isValidSubnet(window.model.dataProtectionNetmask)) {
        showError("Please provide a valid data protection netmask");
        return false;
      }
    }
  }
  if (!window.model.dnsList.every(function(dns) { return !!dns; })) {
    showError("Please enter DNS servers for all inputs");
    return false;
  }
  return true;
}

function _addDnsInput(index) {
  if (index === undefined) {
    index = window.model.dnsList.length;
    window.model.dnsList.push("");
  }
  var dnsListContainer = document.getElementById("dns-list-container");
  // var dnsRow;
  // if (index % 2 === 0) {
  //   // If even, add dns row
  //   dnsRow = document.createElement("div");
  //   dnsRow.className = "input-inline padding-input-inline two-inputs dns-row";
  //   dnsListContainer.appendChild(dnsRow);
  // } else {
  //   var dnsRows = dnsListContainer.getElementsByClassName("dns-row");
  //   dnsRow = dnsRows[dnsRows.length - 1];
  // }
  var dnsRow = document.createElement("div");
  dnsRow.setAttribute("id", "dnsRow" + index);
  dnsRow.className = "input-inline padding-input-inline dns-row";
  dnsListContainer.appendChild(dnsRow);

  var dnsInput = document.createElement("div");
  dnsInput.className = "dns-input-group";
  dnsInput.setAttribute("id", "dnsInputGroup" + index);
  dnsInput.innerHTML =
    '<label>DNS ' + (index+1) + '</label>' +
    (index > 0 ? ' <a class="delete" href="#" tabindex="-1" value="' + index + '">&times;</a>' : '') +
    '<input type="text" placeholder="255.255.255.255" data-sync="model.dnsList[' + index + ']" />';
  if (index > 0) {
    dnsInput.querySelector(".delete").addEventListener("click", function (event) {
      event.preventDefault();
      removeDns(parseInt(event.target.getAttribute("value")));
    });
  }
  dnsInput.querySelector("input").value = window.model.dnsList[index] || "";
  dnsRow.appendChild(dnsInput);
}

function addDnsInput() {
  if (window.model.dnsList.length < 3) {
    _addDnsInput();
    saveLocalStorageModel();
  } else {
    showError("The installer only supports upto three DNS servers");
  }
}

function removeDns(index) {
  var dnsRow = document.getElementById("dnsRow" + index);
  dnsRow.parentNode.removeChild(dnsRow);
  model.dnsList.splice(index, 1);
  var rows = document.getElementsByClassName("dns-row");
  for (var i = index; i < rows.length; i++) {
    var row = rows[i];
    row.setAttribute("id", "dnsRow" + i);
    row.querySelector(".dns-input-group").setAttribute("id", "dnsInputGroup" + i);
    row.querySelector("label").innerHTML = "DNS " + (i + 1);
    row.querySelector("a").setAttribute("value", i);
    row.querySelector("input").setAttribute("data-sync", "model.dnsList[" + i + "]");
  }
  saveLocalStorageModel();
}

function refreshNetworkListButtons() {
  var inputs = document.querySelectorAll("#network-list input[type=checkbox]");
  var networkCount = 0;
  var bondCount = 0;
  var vlanCount = 0;
  for (var i = 0; i < inputs.length; i++) {
    if (inputs[i].checked) {
      if (inputs[i].className === "vlans") {
        vlanCount += 1;
      } else if (inputs[i].className === "bonds") {
        bondCount += 1;
      } else {
        networkCount += 1;
      }
    }
  }
  var createVlansButton = document.getElementById("createVlansButton");
  var createVlansTooltip = document.getElementById("createVlansTooltip");
  if ((networkCount + bondCount) != 1 || vlanCount > 0) {
    createVlansButton.disabled = true;
    createVlansTooltip.innerHTML = "Please select a single network or bond";
  } else {
    createVlansButton.disabled = false;
  }
  var createBondButton = document.getElementById("createBondButton");
  var createBondTooltip = document.getElementById("createBondTooltip");
  if (bondCount > 0 || vlanCount > 0 || networkCount < 2) {
    createBondButton.disabled = true;
    if (bondCount > 0 || vlanCount > 0) {
      createBondTooltip.innerHTML = "Please select networks only";
    } else {
      createBondTooltip.innerHTML = "Please select 2 or more networks";
    }
  } else {
    createBondButton.disabled = false;
  }
  refreshNetworkSettingsInputs();
}

function refreshNetworkSettingsInputs() {
  // get selected
  var checkboxes = document.querySelectorAll("#network-list input[type=checkbox]:checked");
  var tooltips = document.querySelectorAll(".network-settings-input-group .tooltip");
  if (checkboxes.length !== 1) {
    // Disable and clear inputs if no value is selected
    var inputs = document.getElementsByClassName("network-settings-input");
    for (var i = 0; i < inputs.length; i++) {
      inputs[i].value = "";
      inputs[i].disabled = true;
      inputs[i].parentNode.classList.add("disabled");
    }
    for (var j = 0; j < tooltips.length; j++) {
      tooltips[j].classList.remove("hide");
    }
    return;
  }
  for (var j = 0; j < tooltips.length; j++) {
    tooltips[j].classList.add("hide");
  }
  var name = checkboxes[0].value;
  var entity = getCurrentNodeEntityByName(name);
  var ipInput = document.getElementById("ip-input");
  ipInput.value = entity.ip.trim() || "";
  ipInput.disabled = false;
  ipInput.parentNode.classList.remove("disabled");
  var netmaskInput = document.getElementById("netmask-input");
  netmaskInput.value = entity.netmask.trim() || "";
  netmaskInput.disabled = false;
  netmaskInput.parentNode.classList.remove("disabled");
  var networkTypeInput = document.getElementById("network-type-input");
  networkTypeInput.value = entity.networkType || "";
  networkTypeInput.disabled = false;
  networkTypeInput.parentNode.classList.remove("disabled");
}

function _updateNodeEntity(entity, basenode, nodeid) {
  var ipInput = document.getElementById("ip-input");
  // entity.ip = ipInput.value || "";
  // If node 1 has IP set, increment node 2 and node 3 IP
  if (ipInput.value) {
      // If IP input is empty, sync it with the IP entered in the basenode and increment
      if (!entity.ip) {
        let newip;
        let base  = parseInt(basenode, 10);
        let node  = parseInt(nodeid, 10);
        newip = parseInt(ipInput.value.split('.')[3], 10) + node - base + 1;
        entity.ip = ipInput.value.substring(0, ipInput.value.lastIndexOf('.')) + '.' + newip;
      }
      // If user deliberately wants to change an IP on a given node
      if (nodeid === (parseInt(basenode, 10) - 1))
        entity.ip = ipInput.value;
    entity.ip = entity.ip.trim();
  } else {
    // In case user wants to unassign an IP for an interface specifically
    entity.ip = ipInput.value;
  }
  var netmaskInput = document.getElementById("netmask-input");
  entity.netmask = netmaskInput.value || "";
  var networkTypeInput = document.getElementById("network-type-input");
  entity.networkType = networkTypeInput.value;
}

function updateSelectedNodeEntity() {
  var checkboxes = document.querySelectorAll("#network-list input[type=checkbox]:checked");
  if (checkboxes.length !== 1) {
    // Do not update if none or more than 1 network is selected
    return;
  }
  // Select driver node - need this in case IP needs to be deliberately changed for a single node
  let drivernode = document.querySelectorAll(".nav-item.active")[0].id.slice(-1);
  var name = checkboxes[0].value;
  var entity = getCurrentNodeEntityByName(name);
  for (let i = parseInt(drivernode, 10) - 1; i < NUM_NODES; ++i) {
    entity = getNodeEntityByName(i, name);
    _updateNodeEntity(entity, drivernode, i);
  }
  for (let i = parseInt(drivernode, 10) - 1; i >= 0; --i) {
    entity = getNodeEntityByName(i, name);
    _updateNodeEntity(entity, drivernode, i);
  }
  saveLocalStorageModel();
  refreshNetworkEntityTooltip(name);
}

function modelToJson() {
  var copy = $.extend(true, {}, window.model);
  // Do not store passwords:
  copy.password = "";
  copy.confirmPassword = "";
  copy.advancedNodes.forEach(function (node) {
    node.password = "";
    node.confirmPassword = "";
  });
  copy.basicNodes.forEach(function (node) {
    node.password = "";
    node.confirmPassword = "";
  });
  return JSON.stringify(copy, null, 2);
}

function saveLocalStorageModel() {
  window.localStorage.setItem(MODEL_STORAGE_KEY, modelToJson());
  window.localStorage.setItem(VERSION_STORAGE_KEY, CURRENT_VERSION);
}

function clearLocalStorageModel() {
  window.localStorage.removeItem(MODEL_STORAGE_KEY);
  window.localStorage.removeItem(VERSION_STORAGE_KEY);
}

function loadLocalStorageModel() {
  var savedModel = window.localStorage.getItem(MODEL_STORAGE_KEY);
  var savedModelVersion = window.localStorage.getItem(VERSION_STORAGE_KEY);
  var savedModelInvalid = false;
  if (savedModelVersion < CURRENT_VERSION) {
    // Old models are invalid
    savedModelInvalid = true;
  }
  if (savedModel) {
    try {
      savedModel = JSON.parse(savedModel);
    } catch (e) {
      // Saved model is invalid
      savedModelInvalid = true;
    }
  }
  if (savedModelInvalid) {
    // Do not use invalid model
    clearLocalStorageModel();
    return false;
  }
  setModelReferences(savedModel);
  window.model = savedModel;
  return true;
}

function applyModel() {
  var syncInputs = $("[data-sync]");
  syncInputs.each(function (index, elem) {
    var input = $(elem);
    var property = input.attr("data-sync");
    if (!(property in syncGetFunctions)) {
      syncGetFunctions[property] = new Function("return window." + property);
    }
    var modelValue = syncGetFunctions[property]();
    if (input.attr("type") === "checkbox") {
      input.prop("checked", modelValue);
    } else {
      input.val(modelValue);
    }
  });
}

function applyVisibilityToggle(id) {
  var elem = document.getElementById(id);
  var checked = elem.checked;
  var toggleShowElements = $("[data-toggle-show=" + id + "]");
  var toggleShowAdditionalElements = $("[data-toggle-show-additional=" + id + "]");
  var toggleShowAdditionalElements2 = $("[data-toggle-show-additional2=" + id + "]");
  var toggleHideElements = $("[data-toggle-hide=" + id + "]");
  var toggleHideAdditionalElements = $("[data-toggle-hide-additional=" + id + "]");
  var toggleHideAdditionalElements2 = $("[data-toggle-hide-additional2=" + id + "]");
  var toggleShowExistingCSElements = $("[data-toggle-show=existing-cs]");
  var toggleHideExistingCSElements = $("[data-toggle-show=existing-cs]");
  // console.log("Visibility toggle: ", id)
  if (checked) {
    if (id === "primary-cs-toggle") {
      $("#additional-cs-toggle").attr("checked", false);
      $("#standbykvmcs-toggle").attr("checked", false);
      window.model.clusterDetails[0]['existing_cs'] = false;
      window.model.clusterDetails[0]['additional_cs'] = false;
      window.model.clusterDetails[0]['standby_kvm_cs'] = false;
    }
    if (id === "additional-cs-toggle") {
      $("#primary-cs-toggle").attr("checked", false);
      $("#standbykvmcs-toggle").attr("checked", false);
      window.model.clusterDetails[0]['existing_cs'] = false;
      window.model.clusterDetails[0]['primary_cs'] = false;
      window.model.clusterDetails[0]['standby_kvm_cs'] = false;
    }
    if (id === "standbykvmcs-toggle") {
      $("#primary-cs-toggle").attr("checked", false);
      $("#additional-cs-toggle").attr("checked", false);
      window.model.clusterDetails[0]['existing_cs'] = false;
      window.model.clusterDetails[0]['additional_cs'] = false;
      window.model.clusterDetails[0]['primary_cs'] = false;
    }
    toggleShowElements.show();
    toggleShowAdditionalElements.show();
    toggleShowAdditionalElements2.show();
    toggleHideElements.hide();
    toggleHideAdditionalElements.hide();
    toggleHideAdditionalElements2.hide();
    if (id === "advanced-settings-toggle")
      hideSimpleBond();
  } else {
    if (id === "primary-cs-toggle") {
      $("#primary-cs-toggle").attr("checked", false);
      window.model.clusterDetails[0]['primary_cs'] = false;
    }
    if (id === "additional-cs-toggle") {
      $("#additional-cs-toggle").attr("checked", false);
      window.model.clusterDetails[0]['additional_cs'] = false;
    }
    if (id === "standbykvmcs-toggle") {
      $("#standbykvmcs-toggle").attr("checked", false);
      window.model.clusterDetails[0]['standby_kvm_cs'] = false;
    }
    toggleShowElements.hide();
    toggleShowAdditionalElements.hide();
    toggleShowAdditionalElements2.hide();
    toggleHideElements.show();
    toggleHideAdditionalElements.show();
    toggleHideAdditionalElements2.show();
    if (id === "advanced-settings-toggle")
      unhideSimpleBond();
  }
  if (isExistingCommServe()) {
    $(".password_hint").addClass("hide");
    // Since existing CS case is not mapped to a toggle, we hide the relevant DIVs explicitly
    $("[data-toggle-show=primary-cs-toggle]").hide();
    $("[data-toggle-show=additional-cs-toggle]").hide();
    $("[data-toggle-show=standbykvmcs-toggle]").hide();
    toggleShowExistingCSElements.show();
  } else {
    toggleHideExistingCSElements.hide();
    $("[data-toggle-show=primary-cs-toggle]").show();
    $(".password_hint").removeClass("hide");
  }
}

function applyVisibilityToggles() {
  var toggles = $("[data-visibility-toggle]");
  toggles.each(function (index, elem) {
    applyVisibilityToggle(elem.id);
  });
}

function refreshInputs() {
  var dnsListContainer = document.getElementById("dns-list-container");
  dnsListContainer.innerHTML = "";
  for (var i = 0; i < window.model.dnsList.length; i++) {
    _addDnsInput(i);
  }
  applyModel();
  applyVisibilityToggles();
  refreshCurrentNodeEntities();
  refreshNetworkListButtons();
}

function fillNodeInformation() {
  var currentModel = window.model;
  var newModel = initializeModel();
  var savedInformation = window.localStorage.getItem("backEndData");
  var json = JSON.parse(savedInformation);
  for (var prop in json) {
    newModel[prop] = json[prop];
  }
  setModelReferences(newModel);

  for (var i = 0; i < NUM_NODES; i++) {
    /* Fill tool tips */
    var toolid = 'node' + (i+1) + '_tooltip';
    var htmlelement = "";
    for (var j in newModel.basicNodes[i].tooltips) {
      htmlelement += '<br><em>' + newModel.basicNodes[i].tooltips[j] + '</em>';
    }
    document.getElementById(toolid).innerHTML = htmlelement;

    /* Fill MAC ID */
    var macid = 'node' + (i+1) + '_macID';
    document.getElementById(macid).innerHTML = "#" + newModel.basicNodes[i].serialno;

    /* Fill MAC ID for advanced nodes */
    var nodeid = "node-tab-" + (i+1);
    htmlelement = '<div class="tooltip-anchor">Node ' + (i+1);
    htmlelement += "<br><em><small> #" + newModel.advancedNodes[i].serialno + "</small></em>"

    /* Fill tool tips */
    htmlelement += '<div class="tooltip node-tooltip">';
    for (var j in newModel.advancedNodes[i].tooltips) {
      htmlelement += '<br>' + newModel.advancedNodes[i].tooltips[j];
    }
    htmlelement += "</div>"
    document.getElementById(nodeid).innerHTML = htmlelement;
  }

  window.model = newModel;
  refreshInputs();
}

function fixBondChildrenReferences(model, bond, entityMap) {
  for (var i = 0; i < bond.children.length; i++) {
    var child = bond[i];
    if (child.type === ENTITY_TYPES.BOND) {
      
    } else if (child.type === ENTITY_TYPES.VLAN) {
      
    } else {
      
    }
  }
}

function _setVlanParentReference(vlan, entityMap) {
  if (vlan.parent.name in entityMap) {
    vlan.parent = entityMap[vlan.parent.name];
  } else {
    entityMap[vlan.parent.name] = vlan.parent;
    _setEntityReferences(vlan.parent, entityMap);
  }
}

function _setBondChildrenReferences(bond, entityMap) {
  if (!('index' in bond)) {
    bond.index = parseInt(bond.name.substring(4)); // Use number at end of name as index
  }
  var pendingEntities = [];
  for (var i = 0; i < bond.children.length; i++) {
    var child = bond.children[i];
    if (child.name in entityMap) {
      bond.children[i] = entityMap[child.name];
    } else {
      entityMap[child.name] = child;
      pendingEntities.push(child);
    }
  }
  for (var i = 0; i < pendingEntities.length; i++) {
    _setEntityReferences(pendingEntities[i], entityMap);
  }
}

function _setEntityReferences(entity, entityMap) {
  if (entity.type === ENTITY_TYPES.BOND) {
    _setBondChildrenReferences(entity, entityMap);
  } else if (entity.type === ENTITY_TYPES.VLAN) {
    _setVlanParentReference(entity, entityMap);
  }
  // else do nothing, no extra references to set
}

function setNodeReferences(node) {
  var entityMap = {};
  var entities = [].concat(node.networks, node.bonds, node.vlans);
  for (var i = 0; i < entities.length; i++) {
    entityMap[entities[i].name] = entities[i];
  }
  for (var i = 0; i < node.bonds.length; i++) {
    _setBondChildrenReferences(node.bonds[i], entityMap);
  }
  for (var i = 0; i < node.vlans.length; i++) {
    _setVlanParentReference(node.vlans[i], entityMap);
  }
  // Populate allBonds list:
  node.allBonds = [];
  Object.keys(entityMap).forEach(function(name) {
    var entity = entityMap[name];
    if (entity.type === ENTITY_TYPES.BOND) {
      node.allBonds[entity.index] = entity;
    }
  });
}

function setModelReferences(model) {
  for (var i = 0; i < model.advancedNodes.length; i++) {
    setNodeReferences(model.advancedNodes[i]);
  }
}

function loadFromJsonFile(file) {
  clearErrors();
  var currentModel = window.model;
  var fileReader = new FileReader();
  var onError = function() {
    showError("An error occurred while reading the settings file");
    window.model = currentModel;
    refreshInputs();
  }
  fileReader.onerror = onError;
  fileReader.onload = function(event) {
    try {
      var text = event.target.result;
      var json = JSON.parse(text);
      var newModel = initializeModel();
      for (var prop in json) {
        newModel[prop] = json[prop];
      }
      setModelReferences(newModel);
      window.model = newModel;
      refreshInputs();
    } catch (e) {
      onError();
    }
  };
  fileReader.readAsText(file);
}

function uploadModelFile() {
  $("#modelFileInput").click();
}

function saveModelToFile() {
  var file = new Blob([modelToJson()], { type: "text/plain" });
  var filename = "hyperscale_config.json";
  if (window.navigator.msSaveOrOpenBlob) {
    // IE must use this function to save files from js:
    window.navigator.msSaveBlob(file, filename);
  } else {
    // Create a download link and click it:
    var a = document.createElement("a");
    a.href = URL.createObjectURL(file);
    a.download = filename;
    var event = document.createEvent("MouseEvent");
    event.initMouseEvent("click",true,true,window,0,0,0,0,0,false,false,false,false,0,null);
    a.dispatchEvent(event);
  }
}

$(document).ready(function () {
  window.model = initializeModel();
  document.getElementById("setupForm").reset();
  loadLocalStorageModel();
  refreshInputs();
  fillNodeInformation();
  loadInputsBasedOnType();

  $("#network-list").on("change", "input[type=checkbox]", function (event) {
    refreshNetworkListButtons();
  });

  $("#root_confirm_pwd").bind('paste',function(e) {
    e.preventDefault();
    showError("Please type in the keys to confirm OS password");
  });

  $("#confirm_pwd").bind('paste',function(e) {
    e.preventDefault();
    showError("Please type in the keys to confirm CommServe password");
  });

  // Sync model with properties:
  $(".setup-form").on("change", "[data-sync]", function (event) {
    var elem = $(this);
    var property = elem.attr("data-sync");
    var type = elem.attr("type");
    var value = elem.val();
    if (type === "checkbox") {
      value = elem.prop("checked");
    }
    if (!(property in syncSetFunctions)) {
      syncSetFunctions[property] = new Function("val", "window." + property + "=val");
    }
    syncSetFunctions[property](value);
    saveLocalStorageModel();
  });

  // Elements with the data-visibility-toggle attribute are treater as visibility toggles.
  // data-toggle-show=id and data-toggle-hide=id attributes can be applied to other elemnts to
  // set their visibility based on the value of the data-visibility-toggle with the specified id
  $(".setup-form").on("change", "[data-visibility-toggle]", function (event) {
    applyVisibilityToggle(this.id);
  });

  $("#modelFileInput").on("change", function(e) {
    loadFromJsonFile(this.files[0]);
  });
});
