#!/usr/bin/env python
"""
File Name   : cvfortify.py

Version     : 1.0

Description : Module to install and check whether the STIG issues have been identified
"""
# Import python specific modules
import sys
import argparse
from time import gmtime, strftime
import ConfigParser
import base64

# Import CV specific modules
import cvutils
import wrappers
import utils

class Fortifycv(object):
    """Class to configure STIG lists"""

    # Banner string that needs to be populated
    str_banner = """************************************************************************************************************************
* You are accessing a U.S. Government (USG) Information System (IS) that is provided for USG-authorized use only.      *
* By using this IS (which includes any device attached to this IS), you consent to the following conditions:           *
*                                                                                                                      *
*  - The USG routinely intercepts and monitors communications on this IS for purposes including, but not limited to,   *
*    penetration testing, COMSEC monitoring, network operations and defense, personnel misconduct (PM),                *
*    law enforcement (LE), and counterintelligence (CI) investigations.                                                *
*                                                                                                                      *
*  - At any time, the USG may inspect and seize data stored on this IS.                                                *
*                                                                                                                      *
*  - Communications using, or data stored on, this IS are not private, are subject to routine monitoring,              *
*    interception, and search, and may be disclosed or used for any USG-authorized purpose.                            *
*                                                                                                                      *
*  - This IS includes security measures (e.g., authentication and access controls) to protect USG interests -- not for *
*    your personal benefit or privacy.                                                                                 *
*                                                                                                                      *
*  - Notwithstanding the above, using this IS does not constitute consent to PM, LE or CI investigative searching or   *
*    monitoring of the content of privileged communications, or work product, related to personal representation or    *
*    services by attorneys, psychotherapists, or clergy, and their assistants. Such communications and work product    *
*    are private and confidential. See User Agreement for details.                                                     *
************************************************************************************************************************
"""

    def __init__(self):
        self.o_cvosutil = cvutils.cvosutils()
        # Takes a backup of config files every time this object is instantiated
        self._backup_config_files()

    def _configure_banner(self):
        banner_file = '/etc/issue'
        with open(banner_file, 'w+') as fd:
            fd.write('{0}'.format(self.str_banner))
        
    def _configure_ssh(self):
        # TODO : SSH related configurations
        pass

    def _configure_logins(self):
        # TODO : Configure login profiles
        # Sets STIG RHEL-07-010230 and RHEL-07-010250
        login_file = '/etc/login.defs'
        str_cmd = r"sed -i 's/.*{0}.*/{1}/g' {2}".format(str_comm, 'PASS_MIN_DAYS\t 1', login_file)
        self.o_cvosutil.run_os_command(str_cmd)
        str_cmd = r"sed -i 's/.*{0}.*/{1}/g' {2}".format(str_comm, 'PASS_MAX_DAYS\t 60', login_file)
        self.o_cvosutil.run_os_command(str_cmd)
        # Sets STIG RHEL-07-010240. Assumes no new OS user is added
        str_cmd = r"for i in $(awk -F: '{print $1}' /etc/shadow); do chage -m 1 ${i}; done"
        self.o_cvosutil.run_os_command(str_cmd)
        # Sets STIG RHEL-07-010260
        str_cmd = r"for i in $(awk -F: '{print $1}' /etc/shadow); do chage -M 60 ${i}; done"
        self.o_cvosutil.run_os_command(str_cmd)
        # Sets STIG RHEL-07-010310
        passwd_file = '/etc/default/useradd'
        str_cmd = r"sed -i 's/.*INACTIVE.*/{0}/g' {1}".format('INACTIVE=0', passwd_file)
        self.o_cvosutil.run_os_command(str_cmd)

    def _backup_config_files(self):
        # Uses authconfig backup to back up important config files in /opt/commvault/MediaAgent
        str_time = strftime("%Y%m%d_%H%M%S", gmtime())
        str_cmd = 'authconfig --savebackup=/opt/commvault/MediaAgent/{0}_{1}.bck'.format('cv_authconfig', str_time)
        self.o_cvosutil.run_os_command(str_cmd)
        
    def _configure_passwd(self):
        # Configure password complexity implementations
        # Sets STIG RHEL-07-010270
        auth_file = '/etc/pam.d/system-auth-ac'
        str_cmd = "sed -i 's/.*password\s*sufficient.*/password    sufficient    pam_unix.so use_authtok sha512 shadow remember=5/g' {0}".format(auth_file)
        self.o_cvosutil.run_os_command(str_cmd)
        # Sets STIG RHEL-07-010160
        pwquality_file = '/etc/security/pwquality.conf'
        str_cmd = "sed -i 's/.*difok.*/difok = 8/g' {0}".format(pwquality_file)
        self.o_cvosutil.run_os_command(str_cmd)
        # Set STIG RHEL-07-010119
        str_cmd = 'password   required     pam_pwquality.so retry=3\n'
        passwd_file = '/etc/pam.d/passwd'
        with open(passwd_file, "a") as fd:
            fd.write(str_cmd)
        # Sets STIG RHEL-07-010320
        passwd_file = '/etc/pam.d/system-auth-ac'
        str_comm = "auth required pam_faillock.so preauth silent audit deny=3 even_deny_root fail_interval=900 unlock_time=604800\nauth sufficient pam_unix.so try_first_pass\nauth [default=die] pam_faillock.so authfail audit deny=3 even_deny_root fail_interval=900 unlock_time=604800 fail_interval=3\n"
        str_cmd = r"sed -i 's/^auth\s*sufficient\s*pam_unix.so.*/{0}/g' {1}".format(str_comm, passwd_file)
        self.o_cvosutil.run_os_command(str_cmd)
        passwd_file = '/etc/pam.d/password-auth-ac'
        str_comm = "auth required pam_faillock.so preauth silent audit deny=3 even_deny_root fail_interval=900 unlock_time=604800\nauth sufficient pam_unix.so try_first_pass\nauth [default=die] pam_faillock.so authfail audit deny=3 even_deny_root fail_interval=900 unlock_time=604800 fail_interval=3\n"
        str_cmd = r"sed -i 's/^auth\s*sufficient\s*pam_unix.so.*/{0}/g' {1}".format(str_comm, passwd_file)
        self.o_cvosutil.run_os_command(str_cmd)
        # Sets STIG RHEL-07-010120, 30, 40, 50, 70, 80, 90, RHEL-07-010280
        # Run the update command at the end so that sed changes are also reflected
        str_cmd = 'authconfig --passminlen=15 --passminclass=4 --passmaxrepeat=3 --passmaxclassrepeat=4 --enablerequpper --enablereqdigit --enablereqlower --enablereqother --enablepamaccess --update'
        self.o_cvosutil.run_os_command(str_cmd)

    def _configure_audit(self):
        # TODO : Auditing related changes
        pass

    def _configure_package_permissions(self):
        # Sets STIG RHEL-07-010010
        str_cmd = r"for i in $(rpm -Va | grep '^.M' | awk '{print $3}' | grep -E '^\/'); do for j in $(rpm -qf ${i}); do rpm --setperms ${j}; done; done"
        self.o_cvosutil.run_os_command(str_cmd)

    def _configure_yum_repos(self):
        # Sets STIG RHEL-07-020070
        yum_file = r'/etc/yum.conf'
        config = ConfigParser.ConfigParser()
        config.read(yum_file)
        config.set('main', 'repo_gpgcheck', '1')
        fd = open(yum_file, "w+")
        config.write(fd)

    def _configure_grub(self):
        # Sets STIG RHEL-07-021350
        str_cmd = "dracut -f"
        self.o_cvosutil.run_os_command(str_cmd)
        # Selinux is still disabled as it is work in progress feature
        grub_file = r'/etc/default/grub'
        str_repl = r'GRUB_CMDLINE_LINUX="crashkernel=auto fips=1 selinux=0"'
        str_cmd = r"sed -i 's/.*GRUB_CMDLINE_LINUX.*/{0}/g' {1}".format(str_repl, grub_file)
        self.o_cvosutil.run_os_command(str_cmd)
        # Sets STIG RHEL-07-010480 -- Follow https://access.redhat.com/solutions/979643
        str_passwd = base64.b64decode(r'MlN0cm9uZ1BAc3NXMHJk')  # Encoded password
        str_cmd = r"useradd -g wheel cvadmin"
        self.o_cvosutil.run_os_command(str_cmd)
        str_cmd = r"echo -e '{0}\n{0}\n' | passwd cvadmin".format(str_passwd)
        self.o_cvosutil.run_os_command(str_cmd)
        str_cmd = r'sed -i "/^CLASS=/s/ --unrestricted//" /etc/grub.d/10_linux'
        self.o_cvosutil.run_os_command(str_cmd)
        str_cmd = r"echo -e '{0}\n{0}\n' | grub2-mkpasswd-pbkdf2".format(str_passwd)
        # cvutils does not pull out stdout. Use utils library
        p = utils.Process()
        p.execute(str_cmd)
        # Last part of stdout has the hash. Strip it out.
        str_hash = p.getout().split()[-1]
        # Find out if the system boots using EFI to update the correct 
        boot_file = ''
        if wrappers.isdir("/sys/firmware/efi"):
            boot_file = r'/boot/efi/EFI/redhat/grub.cfg'
        else:
            boot_file = r'/boot/grub2/grub.cfg'
        boot_user_file= r'/boot/grub2/user.cfg'  # Verified on EFI systems; location is same
        str_cmd = r'GRUB2_PASSWORD={0}\n'.format(str_hash)
        with open(boot_user_file, "w+") as fd:
            fd.write(str_cmd)
        # Rebuild grub after all changes are in
        str_cmd = r'grub2-mkconfig -o {0}'.format(boot_file)
        self.o_cvosutil.run_os_command(str_cmd)
        

    def fortify(self):
        # Master method that calls the underlings
        # TODO: Fill this up and make it multi-threaded
        self._configure_banner()  # Sets DoD Banner
        self._configure_passwd()  # All password related configurations
        self._configure_logins()  # Sets all login related configurations
        self._configure_package_permissions()
        self._configure_yum_repos()
        self._configure_grub()


def install_packages():
    l_packages = ['screen', 'aide', 'ansible']
    o_cvosutil = cvutils.cvosutils()
    for pkg in l_packages:
        o_cvosutil.install_os_package(pkg)


def check_stig_recommendations(args):
    # TODO: Run unit tests on configured items to see whether they're actually configured
    pass


def run(args):
    parser = argparse.ArgumentParser(description = 'Check and install STIG recommendations')
    parser.add_argument('-i', '--install', action = "store_true",
                            required = False, help = 'Install and configure items')
    parser.add_argument('-c', '--check', action = "store_true",
                            required = False, help = 'Check for configured items')
    args = parser.parse_args()
    if args.install:
        # If install fails, config should not proceed
        try:
            install_packages()
        except Exception as str_e:
            print ('ERROR: Encountered exception on Install -- {0}'.format(str_e))
            sys.exit(1)
        try:
            o_config = Fortifycv()
            o_config.fortify()
        except Exception as str_e:
            print ('ERROR: Encountered exception on Configure -- {0}'.format(str_e))
            sys.exit(1)
    if args.check:
        check_sig_recommendations()


if __name__ == '__main__':
    run(sys.argv[1:])

