# coding: utf8

import sys
from inspect import isgeneratorfunction
import threading
from .command import CommandExecutor
from appliancelocale import getLocalizedString as _
import cvpylogger
import pdb

class WizardApplication(object):
    "Wizard App"
    current_wizard_app = None
    def __init__(self, first_controller, config=None):
        WizardApplication.current_wizard_app = self
        self.config = config or {}
        self.controller = first_controller
        self.view = None
        self.protocol = WizardProtocol()
        self.protocol.factory = self
        self.log = cvpylogger.getLog()

    def set_view(self, view):
        "Set the view for this app"
        self.view = view
        moduleName = 'ui.'+view
        if view == 'tui':
            from ui import tui
        #sys.modules[view] = module

    def make_controller_instance(self, klass):
        """Create an instance of the controller class"""
        obj = klass()
        obj.config = self.config
        init_method = getattr(obj, "init", None)
        if init_method:
            init_method()
        return obj

    def run(self):
        try:
            self.protocol.start(self.controller)
        except SilentInstallExitException as siee:
            self.stop(-1)

    def stop(self, exitVal=0):
        self.protocol.quit()
        #sys.exit(exitVal)


class WizardProtocol(object):
    """Controller protocol for wizard-like applications
    This object is responsible for processing each controller object thats part
    of the wizard.

    """
    def __init__(self):
        self.factory = None
        self.current_controller = None
        self.current_render = None
        self.commands = []
        self.main_window = None
        self.log = cvpylogger.getLog()
        self.quit_pressed = False

    def process_state(self, controller):
        "Process the current state of the wizard"
        self.current_controller = controller
        if self.factory.view == 'gui':  # needed since screen is not recontructed only widgets are added
            if not self.main_window is None:
                self.main_window.clearBodyPanel()
        self.show()

    def show(self):
        """Present the current controller to the user and setup callbacks for
        running next. If optional render method is not present, call next()
        ourselves.

        """
        while getattr(self.current_controller, 'ActionOnly', False):
            command_method = getattr(self.current_controller, "command", None)
            if command_method:
                commands = command_method()
                if isinstance(commands, list):
                    self.commands = []
                    for cmd in commands:
                        self.commands.append((cmd, tuple()))
                else:
                    self.commands = [(commands, tuple())]

                #if commands is not None:
                #    self.commands = [(commands, tuple())]
            if getattr(self.current_controller, 'showProgress', None) is not None:
                noReturn = False
                if getattr(self.current_controller, 'noReturn', False):
                    noReturn = True
                currentRender = self.current_controller.render()
                if getattr(self.current_controller, 'showProgress', True):
                    nextFlag = getattr(self.current_controller, 'AutoNext', False)
                    self.process_task_queue(currentRender.title, noReturn, getattr(self.current_controller, 'stepName', ''), showUI=True, autoNext=nextFlag)
                    return
                else:
                    self.process_task_queue(currentRender.title, noReturn, getattr(self.current_controller, 'stepName', ''), showUI=False)
                    self.current_controller = self.current_controller.next()
            else:
                process_method = getattr(self.current_controller, "process", None)
                if process_method:
                    if self.execute(process_method, tuple(), {}):
                        #if hasattr(self.current_controller, 'node'):
                        #    self.current_controller.node.set('showToUser', 'False')
                        self.current_controller = self.current_controller.next()
        if hasattr(self.current_controller, "render"):
            self.current_render = self.current_controller.render()
            if self.main_window is None:
                self.main_window = self.current_render.main_window

            if hasattr(self.current_controller, "set_alarm_in"):
                self.current_controller.callBackIn = self.main_window.callBackIn
                self.current_controller.set_alarm_in()

            self.main_window.clear_callbacks()
            self.main_window.alert('')
            self.main_window.message('')
            self.current_render.update_main_window()
            if self.current_controller.prev_controller is None:
                self.main_window.disable_prev_button()
            else:
                self.main_window.enable_prev_button()
                prev_controller = self.current_controller.prev_controller
                while getattr(prev_controller, 'ActionOnly', False) or getattr(prev_controller, 'AutoNext', False):
                    prev_controller = prev_controller.prev_controller
                    if prev_controller is None:
                        self.main_window.disable_prev_button()
                        break
                self.current_controller.prev_controller = prev_controller
            self.current_controller.alert = self.main_window.alert
            self.current_controller.message = self.main_window.message
            self.main_window.add_next_callback(self.next)
            self.main_window.add_previous_callback(self.previous)
            self.main_window.add_quit_callback(self.quit)
            if getattr(self.current_controller, 'AutoNext', False):
                self.next()
        else:
            self.current_render = None
            self.next()

        if hasattr(self.current_controller, "auto_process"):
            retval = self.current_controller.auto_process()

    def reset_widgets(self, new_render_object):
        "reset current render object with new render object to get inputs from newly added widgets"
        self.current_render = new_render_object
        self.main_window = self.current_render.main_window
        self.current_render.update_main_window()

    def run_controller(self):
        "Execute tasks specified by the controller"
        process_method = getattr(self.current_controller, "process", None)
        if self.current_render is None:
            args = tuple()
        else:
            args = self.current_render.get_user_input()
        if process_method:
            if not self.execute(process_method, args, {}):
                return False
        return True

    def next(self, *_):
        "Jump to the next state of the application"
        if self.run_controller():
            next_controller = None
            if self.current_controller is not None:
                next_controller = self.current_controller.next()
                if getattr(next_controller, 'ActionOnly', False):
                    self.current_controller = next_controller
                    command_method = getattr(self.current_controller, "command", None)
                    if command_method:
                        cmds = command_method()
                        if isinstance(cmds, list):
                            self.commands = []
                            for cmd in cmds:
                                self.commands.append((cmd, tuple()))
                        else:
                            self.commands = [(cmds, tuple())]
                    #if command_method:
                    #    self.commands = [(command_method(), tuple())]
                    if getattr(self.current_controller, 'showProgress', None) is not None:
                        noReturn = False
                        if getattr(self.current_controller, 'noReturn', False):
                            noReturn = True
                        currentRender = self.current_controller.render()
                        if getattr(self.current_controller, 'showProgress', True):
                            nextFlag = getattr(self.current_controller, 'AutoNext', False)
                            self.process_task_queue(currentRender.title, noReturn, getattr(self.current_controller, 'stepName', ''), showUI=True, autoNext=nextFlag)
                            return
                        else:
                            self.process_task_queue(currentRender.title, noReturn, getattr(self.current_controller, 'stepName', ''), showUI=False)
                            raise Exception(str(next_controller)+"\n"+str(next_controller.next()))
                            #next_controller = next_controller.next()
                    else:
                        process_method = getattr(self.current_controller, "process", None)
                        if process_method:

                            if self.execute(process_method, tuple(), {}):                                
                                '''
                                The below statements will also mark stepComlete to True for nodes which has children nodes, which is not desirable.
                                next() method, which is invoked below will take care of marking stepComplete for each node correctly if it has child nodes.                                                            
                                if hasattr(next_controller, 'node'):
                                    next_controller.node.set('stepComplete', 'True')                                    
                                '''
                                next_controller = next_controller.next()
            if next_controller is not None:
                self.process_state(next_controller)
            else:
                self.quit()

    def previous(self, *_):
        "Jump back to previous controller"
        #if self.run_controller():
        prev_controller = self.current_controller.previous()
        if prev_controller is not None:
            if getattr(prev_controller, 'ActionOnly', False):
                prev_controller = prev_controller.previous()
            elif getattr(prev_controller, 'AutoNext', False):
                prev_controller = prev_controller.previous()
            self.process_state(prev_controller)
        else:
            self.quit()
        """
        prev_controller = self.current_controller.previous()
        if prev_controller is not None:
            self.process_state(prev_controller)
        else:
            self.quit()
        """

    def quit(self, *_):
        """Quit the application"""
        self.main_window.stop()
        import ui.tui
        if len(_) == 1 and isinstance(_[0], ui.tui.customButton): #and _[0].label == 'Quit':
            self.quit_pressed = True
            

    def execute(self, a_callable, args, kwargs):
        "Execute callable"
        return a_callable(*args, **kwargs)

    def process_task_queue(self, title, noReturn, stepName='', showUI=True, autoNext=False):
        """Execute tasks scheduled in the command list.
        We will inject a last controller for showing progress and reporting
        task completion, and set callbacks for quiting.
        """
        log = cvpylogger.getLog()
        if self.main_window is not None:
            self.main_window.disable_buttons()
        queue = []
        executor = CommandExecutor()

        for func, args in self.commands:
            if isgeneratorfunction(func):
                for f, a, desc in func(*args):
                    queue.append((f, a, desc))
            else:
                queue.append((func, args, func.__doc__))

        self.commands = []
        if showUI is False:
            import traceback
            exitViaSilentInstallException = False
            for f, a, desc in queue:
                try:
                    returnTuple = f(*a)
                    if type(returnTuple) == type(tuple()) and returnTuple[0] is False:
                        log.info("Failed function call in silent mode : %s with error message %s" % (str(f.__name__), str(returnTuple[1])))
                        exitViaSilentInstallException = True
                    elif type(returnTuple) == type(bool()) and returnTuple is False:
                        log.info("Failed function call in silent mode : %s " % str(f.__name__))
                        exitViaSilentInstallException = True
                except Exception as e:
                    log.error("Exception in ui.CommandExecutor while calling %s in silent mode" % (str(f.__name__)))
                    log.error(str("\n".join(traceback.format_stack())))
                    raise SilentInstallExitException()
                if exitViaSilentInstallException:
                    raise SilentInstallExitException()
            if self.main_window is not None:
                self.main_window.enable_buttons()
            return

        def tasks_finished_callback():
            "Installation Finished"
            if stepName == '':
                self.main_window.enable_buttons()
            self.main_window.clear_callbacks()
            self.main_window.add_next_callback(self.progressBarNext)
            self.main_window.add_previous_callback(self.progressBarBack)
        self.autoNext = autoNext

        def tasks_error_callback(exceptionFunc, errorMessage='', isException=False):
            self.autoNext = False
            import traceback
            thLog = log
            self.main_window.disable_next_button()
            self.main_window.disable_prev_button()
            self.main_window.enable_quit_button()
            self.main_window.enable_buttons()
            if isinstance(errorMessage, dict):
                errorMessage = str(errorMessage)
            elif not errorMessage.strip():
                errorMessage = _('FATAL_ERROR_GENERIC')% thLog.getLogFilePath()
                thLog.info("Failed function call : %s " % str(exceptionFunc))
            self.main_window.alert(errorMessage)
            if isException:
                thLog.error("Exception in ui.CommandExecutor while calling %s" % (str(exceptionFunc)))
                thLog.error(str("\n".join(traceback.format_stack())))
        executor.setErrorCallback(tasks_error_callback)
        queue.append((tasks_finished_callback, tuple(), 'INSTALLATION_FINISH_MESSAGE' if stepName == '' else 'UPGRADE_' + stepName[7:].upper() + '_DONE'))
        Render = type(self.current_render)

        class ExecutorController(object):
            def __init__(self):
                self.log = cvpylogger.getLog()
                self.prev_controller = None
                self.next_controller = None

            def render(self):
                #from gxlibrary.ui.render import Render
                render_obj = Render(title, ("Please wait until we process the requested operations"), executor)
                render_obj.set_next_button_label("Finish")
                render_obj.disable_quit_button()
                return render_obj


            def next(self):
                return self.next_controller

            def previous(self):
                return self.prev_controller

        #controller = self.factory.make_controller_instance(ExecutorController)
        controller = ExecutorController()
        if noReturn:
            controller.prev_controller = None
        else:
            controller.prev_controller = self.current_controller
        controller.next_controller = self.current_controller.next()
        self.process_state(controller)
        #raise Exception(str(queue))
        log.info("before execute_all queue ->")
        log.info(queue)
        executor.execute_all(queue)
        executor.start()
        executor.join()
        if stepName != '':  # Only for upgrade
            if not self.main_window._disable_next:
                self.progressBarNext()
        if self.autoNext:
            def dummyNextFunc():
                self.main_window.next_button._emit('click')
            self.main_window.afterRenderProcess(dummyNextFunc)
            self.main_window.update()

    def start(self, first_controller):
        "Start the main loop"
        self.process_state(first_controller)
        #self.main_window.disable_prev_button()
        if self.main_window is not None:
            self.main_window.run()

    def stop(self, *_):
        "Exit the wizard"
        self.main_window.stop()

    def progressBarBack(self, *_):
        "Exit the wizard and reboot"
        prev_controller = self.current_controller.previous()
        if prev_controller is not None:
            self.process_state(prev_controller)
        else:
            self.quit()

    def progressBarNext(self, *_):
        "Exit the wizard and reboot"
        next_controller = self.current_controller.next()
        #raise Exception(next_controller)
        if next_controller is not None:
            next_controller.prev_controller = self.current_controller.previous()
            self.process_state(next_controller)
        else:
            self.quit()


class UpdateMessageThread(threading.Thread):
    """Class to update the message on ui in sequence"""

    def __init__(self, valList):
        """
        Update UI's message box
        @param valList: Display message either string or list of strings (refreshed every second)
        """
        from ui.render import Render
        threading.Thread.__init__(self)
        #if type(valList) == type('') or type(valList) == type(u''):
        if isinstance(valList, basestring):
            self.val = [valList]
        else:
            self.val = valList
        self.msgFunc = Render.main_window_instance.message
        self._stop = False
        self.disableButtonFunc = Render.main_window_instance.disable_buttons
        self.enableButtonFunc = Render.main_window_instance.enable_buttons

    def run(self):
        import time
        i = 0
        maxI = len(self.val)
        self.disableButtonFunc()
        while not self._stop:
            try:
                self.msgFunc(str(self.val[i]))
            except UnicodeEncodeError:
                self.msgFunc(unicode(self.val[i]))
            time.sleep(1)
            i = (i+1) % maxI
        self.enableButtonFunc()
        self.msgFunc('')

    def stopThread(self):
        """Stop the thread execution"""
        self._stop = True


class SilentInstallExitException(Exception):
    """Exception to be raised during silent install to exit gracefully"""

    def __init__(self, statusCode=None):
        """
        Exit silent install gracefully in case of any error
        @param errorCode: Error code to be sent to CS and also the return code. Default=0
        """
        log = logger.getLog()
        self.errorMessage = None
        if statusCode is not None:
            constants.PUSH_JOB_STATUS = constants.QINSTALL_CODES[statusCode]
        log.debug('error code:%s' % (constants.PUSH_JOB_STATUS))
        if constants.PUSH_JOB_STATUS == 0:
            if constants.SERVICEPACK_INSTALL:
                constants.PUSH_JOB_STATUS = constants.QINSTALL_CODES['QUPDATE_FAILED']
                self.errorMessage = 'QUPDATE_FAILED'
            else:
                constants.PUSH_JOB_STATUS = constants.QINSTALL_CODES['QINSTALL_FAILED']
                self.errorMessage = 'QINSTALL_FAILED'
        else:
            for key, val in constants.QINSTALL_CODES.iteritems():
                if key == constants.PUSH_JOB_STATUS:
                    self.errorMessage = key
        log.info("JOB_ID = %s, errorCode = %s" % (constants.JOB_ID, constants.PUSH_JOB_STATUS))

    def __str__(self):
        return "Silent exit with code: ["+str(constants.PUSH_JOB_STATUS)+"] message: ["+self.errorMessage+"]"
