#-*- coding: utf-8 -*-

"""An Urwid based view."""

from __future__ import absolute_import
from itertools import cycle
import math
import os
import signal
import urwid
from urwid.widget import Text
from ui.render import Render
from ui import core, model, command
from appliancelocale import getLocalizedString as _
from urwid.signals import connect_signal, disconnect_signal
import cvpylogger
import pdb
from threading import Lock
import wrappers

class UrwidView:
    """
    Base class for Urwid view.
    """

    # overload some methods from the interface that are not needed for this
    # view
    def show(self):
        """Toggle all the widgets visible."""
        pass

    def connect(self):
        """Connect signals."""
        pass


@core.main_window
class UrwidMainWindow(object):
    """
    Top level widget for Urwid views.
    """
    console_palette = [
        ('body',            '',       'white', 'standout'),
        ('bg 1',            'yellow',  'white', 'standout'),
        ('header',          'black',  'white', 'bold'),
        ('title',           'black',  'black', 'bold'),
        ('intro',           'black',  'white'),
        ('background main', 'white',  'black', 'standout'),
        ('body bg',         'yellow',  'dark cyan', 'standout'),
        ('edit',            'yellow',  'dark cyan'),
        ('edit active',     'white',  'black', 'bold'),
        ('active',          'yellow',  'dark cyan', 'bold'),
        ('button normal',   'black',  'white', 'standout'),
        ('button select',   'black',  'white', 'bold'),
        ('popbg',           'white',  'white', 'bold'),
        ('pg normal',       'black',  'white'),
        ('pb complete',     'white',    'black'),
        ('alert message',   'dark red', 'dark cyan', 'bold'),
        ('normal message',  'black',  'dark cyan',   'bold'),
        ('line',            'yellow',  'dark cyan', 'standout'), ]

    white_black_palette = [
        ('body',            '',       'white', 'standout'),
        ('bg 1',            'white',  'white', 'standout'),
        ('header',          'black',  'white', 'bold'),
        ('title',           'black',  'black', 'bold'),
        ('intro',           'black',  'white'),
        ('background main', 'white',  'white', 'standout'),
        ('body bg',         'black',  'white', 'standout'),
        ('edit',            'black',  'white'),
        ('edit active',     'white',  'black', 'bold'),
        ('active',          'black',  'white', 'bold'),
        ('button normal',   'black',  'white', 'standout'),
        ('button select',   'black',  'white', 'bold'),
        ('popbg',           'white',  'white', 'bold'),
        ('pg normal',       'black',  'white'),
        ('pb complete',     'white',    'black'),
        ('alert message',   'black', 'white', 'bold'),
        ('normal message',  'black',  'white',   'bold'),
        ('line',            'black',  'white', 'standout'), ]

    black_white_palette = [
        ('body',            '',       'black', 'standout'),
        ('bg 1',            'black',  'black', 'standout'),
        ('header',          'white',  'black', 'bold'),
        ('title',           'black',  'black', 'bold'),
        ('intro',           'white',  'black'),
        ('background main', 'black',  'black', 'standout'),
        ('body bg',         'white',  'black', 'standout'),
        ('edit',            'white',  'black'),
        ('edit active',     'black',  'white', 'bold'),
        ('active',          'white',  'black', 'bold'),
        ('button normal',   'white',  'black', 'standout'),
        ('button select',   'white',  'black', 'bold'),
        ('popbg',           'black',  'black', 'bold'),
        ('pg normal',       'white',  'black'),
        ('pb complete',     'black',    'white'),
        ('alert message',   'white', 'black', 'bold'),
        ('normal message',  'white',  'black',   'bold'),
        ('line',            'white',  'black', 'standout'), ]

    default_palette = [
        ('body',            'black',        'dark cyan',   'standout'),
        ('bg 1',            'light green',  'black',        'standout'),
        ('header',          'dark green',   'black',        'bold'),
        ('title',           'light green',  'black',        'bold'),
        ('intro',           'dark green',   'black'),
        ('background main', 'white',        'black',        'standout'),
        ('body bg',         'light green',  'dark gray',    'standout'),
        ('edit',            'light green',  'dark gray'),
        ('edit active',     'white',        'black',        'bold'),
        ('active',          'white',        'dark gray',    'bold'),
        ('button normal',   'dark green',   'light gray',        'standout'),
        ('button select',   'black',  'light gray',        'bold'),
        ('popbg',           'light green',  'black',        'bold'),
        ('pb normal',       'dark green',   'black'),
        ('pb complete',     'black',        'dark green'),
        ('pb complete',     'black',        'dark green'),
        ('alert message',   'light red',    'dark gray',        'bold'),
        ('normal message',  'light green',  'dark gray',        'bold'),
        ('line',            'light green',  'dark gray', 'standout'), ]

    blue_palette = [
        ('body',            'dark blue',    'light gray',   'standout'),
        ('bg 1',            'yellow',       'dark blue',    'standout'),
        ('header',          'yellow',       'dark red',     'bold'),
        ('title',           'yellow',       'dark blue',    'bold'),
        ('intro',           'yellow',       'dark blue'),
        ('background main', 'yellow',       'dark blue',    'standout'),
        ('body bg',         'yellow',       'dark gray',    'standout'),
        ('editfc','white', 'dark blue', 'bold'),
        ('editbx','light gray', 'dark blue'),
        ('edit',            'yellow',       'dark gray'),
        ('edit active',     'white',        'black',    'bold'),
        ('active',          'white',        'dark gray',    'bold'),
        ('button normal',   'yellow',       'dark blue',    'standout'),
        ('button select',   'white',        'dark blue',    'bold'),
        ('popbg',           'white',        'dark blue',    'bold'),
        ('pb normal',       'yellow',       'dark red'),
        ('pb complete',     'dark blue',    'yellow'),
        ('alert message',    'light red',    'dark gray',       'bold'),
        ('normal message',  'light green',  'dark gray',        'bold'),
        ('line',         'yellow',      'dark gray', 'standout'), ]

    nw_palette = [
        ('body',            'black',        'light gray',   'standout'),
        ('bg 1',            'yellow',       'dark blue',    'standout'),
        ('header',          'black',        'light gray',   'bold'),
        ('title',           'yellow',       'dark blue',    'bold'),
        ('intro',           'yellow',       'dark blue'),
        ('background main', 'yellow',       'dark blue',    'standout'),
        ('body bg',         'yellow',       'dark gray',    'standout'),
        ('editfc',          'white',        'dark blue',    'bold'),
        ('editbx',          'light gray',   'dark blue'),
        ('edit',            'yellow',       'dark gray'),
        ('edit active',     'white',        'black',        'bold'),
        ('active',          'white',        'dark gray',    'bold'),

        ('labelfocus',      'dark magenta', 'light gray',   'bold, underline'),
        ('label',           'dark blue',    'light gray'),
        ('fieldfocus',      'dark blue',    'light gray',   'bold, underline'),
        ('field',           'black',        'light gray'),

        ('button normal',   'dark blue',       'light gray',    'standout'),
        ('button select',   'dark magenta',    'light gray',    'bold'),
        ('pb normal',       'yellow',       'dark red'),
        ('pb complete',     'dark blue',    'yellow'),
        ('alert message',    'dark red',    'light gray',       'bold'),
        ('normal message',  'light green',  'light gray',        'bold'),
        ('line',         'yellow',      'dark gray', 'standout'), ] 

    mw_palette = [
        ('body',            'black',        'light gray',   'standout'),
        ('bg 1',            'black',        'light gray',   'standout'),
        ('header',          'black',        'light gray',   'bold'),
        ('title',           'yellow',       'dark blue',    'bold'),
        ('intro',           'dark blue',    'light gray'),
        ('background main', 'light gray',   'dark blue',    'standout'),
        ('body bg',         'dark blue',    'light gray',    'standout'),
        ('editfc',          'white',        'dark blue',    'bold'),
        ('editbx',          'light gray',   'dark blue'),
        #('edit',            'yellow',       'dark gray'),
        ('edit',            'yellow',       'light gray'),
        ('edit active',     'white',        'black',        'bold'),
        #('active',          'white',        'dark gray',    'bold'),
        ('active',          'white',        'light gray',    'bold'),

        ('labelfocus',      'dark magenta', 'light gray',   'bold, underline'),
        ('label',           'dark blue',    'light gray'),
        ('fieldfocus',      'dark blue',    'light gray',   'bold, underline'),
        ('field',           'black',        'light gray'),

        ('button normal',   'dark blue',       'light gray',    'standout'),
        ('button select',   'dark magenta',    'light gray',    'bold'),
        ('pb normal',       'dark blue',       'dark red'),
        ('pb complete',     'dark blue',    'brown'),
        ('alert message',    'dark red',    'light gray',       'bold'),
        ('normal message',  'light green',  'light gray',        'bold'),
        ('line',            'yellow',      'light gray', 'standout'), ] 


    term_default_palette = [
        ('body',            '',    '',   ''),
        ('bg 1',            '',    '',   ''),
        ('header',          '',    '',   ''),
        ('title',           '',    '',   ''),
        ('intro',           '',    ''),
        ('background main', '',    '',   ''),
        ('body bg',         '',    '',   ''),
        ('edit',            '',    ''),
        ('edit active',     '',    '',    ''),
        ('active',          '',    '',    ''),
        ('button normal',   '',    '',    ''),
        ('button select',   '',    '',    ''),
        ('popbg',           '',    '',    ''),
        ('pb normal',       '',    '',    ''),
        ('pb complete',     '',    ''),
        ('alert message',   '',    '',    ''),
        ('normal message',  '',    '',    ''),
        ('line',            '',    '',    ''), ]

    urwid.command_map["esc"] = "cursor down"
    #urwid.command_map["tab"] = "cursor down"
    #urwid.command_map["shift tab"] = "cursor up"
    urwid.command_map["ctrl c"] = "cursor up"
    urwid.command_map["j"] = "cursor down"
    urwid.command_map["k"] = "cursor up"

    def __init__(self):
        """View initialization."""
        self.log = cvpylogger.getLog()
        self.brand = ""
        self.title = "Commvault HyperScale"
        self.subtitle = ""
        self.intro = ""
        self.widgets = []
        self.showRecord = False
        self.hidePageOption = None
        self.alert_box = urwid.Text("")
        self.alert_box_wrap = urwid.AttrWrap(self.alert_box, "alert message")
        self.next_button = customButton(_("NEXT_BUTTON_TEXT"))
        self.back_button = customButton(_("BACK_BUTTON_TEXT"))
        self.quit_button = customButton(_("QUIT_BUTTON_TEXT"))
        self.next_callback = None
        self.back_callback = None
        self.quit_callback = None
        self._input_disabled = False
        self._disable_next = False
        self._disable_prev = False
        self._disable_quit = False
        self.frame = self.get_frame()
        self.loop = urwid.MainLoop(self.frame, unhandled_input=self.unhandled_input, pop_ups=True, event_loop=urwid.GLibEventLoop())
        #self.loop = urwid.MainLoop(self.frame, unhandled_input=self.unhandled_input, pop_ups=True)

        self.palettes = cycle([self.mw_palette, self.console_palette, self.default_palette, self.white_black_palette, self.black_white_palette, self.blue_palette, self.term_default_palette])
        #self.palettes = cycle([self.console_palette, self.default_palette, self.white_black_palette, self.black_white_palette, self.blue_palette, self.term_default_palette])
        self.set_palette()

    def get_title(self):
        """Return a Text instance for the section title."""

        text = urwid.Text(("body bg", self.title), align='center')
        #vline = urwid.AttrWrap(urwid.SolidFill(u'\u2504'), 'line')
        vline = urwid.AttrWrap(urwid.SolidFill('__'), 'line')
        pile = urwid.Pile([('weight', 2, text), ('fixed', 1, vline)])
        #container = urwid.LineBox(container, tlcorner='|', tline='-', lline='|', trcorner='|', blcorner='|', rline='|', bline='-', brcorner='|')
        return pile  # urwid.LineBox(pile)
        #return urwid.Text(("body bg", text), align='center')

    def set_subtitle(self):
        serialnum = wrappers.get_server_serial_number()
        for nwif in wrappers.get_nw_intfx_list():
            ipaddr = wrappers.get_ip_address_of_nw_interface(nwif)
            if len(ipaddr): break

        #Start subtitle with \n so that it is displayed below the main header string
        if (serialnum):
            self.subtitle = "\nSR:" + serialnum
        if (ipaddr):
            self.subtitle = self.subtitle + " (" + ipaddr + ")"
        self.log.info("Subtitle string set as: %s" % (self.subtitle))

    def get_intro(self):
        """Return a Text instance for the introductory paragraph."""

        text = self.intro
        return urwid.Text(('intro', text))

    def get_contents(self):
        """Return a Pile containing the central widgets."""

        widgets = []
        edit_widget_count = 0  # For TAB function: 0 mean next btn focus, 1 mean only 1 editable widget so next btn focus, >2 means dwn btn function
        widgets.append(('flow', self.get_title()))
        widgets.append(('flow', urwid.Divider()))
        widgets.append(self.alert_box_wrap)
        for i in self.widgets:
            if not isinstance(i, urwid.Text):  # and (isinstance(i._get_original_widget(),urwid.Edit))):
                edit_widget_count += 1
            if isinstance(i, urwid.Pile) and getattr(i, 'isLongList', False):
                occupiedRows = 1
                pos = self.widgets.index(i)
                for j in self.widgets:
                    if i != j:
                        occupiedRows += j.rows((urwid.raw_display.Screen().get_cols_rows()[0],))+1
                i.occupiedRows = occupiedRows
                i.longListRef.occupiedRows = i.occupiedRows
                self.magicno = occupiedRows
                i.doHandleScreenChange()
            widgets.append(('flow', i))
            widgets.append(('flow', urwid.Divider()))
        container = urwid.Pile(widgets)
        container = urwid.Padding(container, "center", 90)

        if edit_widget_count > 1:
            urwid.command_map["tab"] = "cursor down"
        elif "tab" in urwid.command_map._command:
            del urwid.command_map["tab"]
        return container

    def get_buttons(self):
        """Return a GridFlow instance containing the buttons."""

        if self._input_disabled:
            return urwid.Divider()
        buttonsList = []
        if not self._disable_prev:
            back_button = urwid.AttrWrap(self.back_button, "button normal", "button select")
            buttonsList.append(back_button)
        if not self._disable_next:
            next_button = urwid.AttrWrap(self.next_button, "button normal", "button select")
            buttonsList.append(next_button)
        if not self._disable_quit:
            quit_button = urwid.AttrWrap(self.quit_button, "button normal", "button select")
            buttonsList.append(quit_button)
        #buttonsList.append(urwid.AttrWrap(urwid.Text(" "*(len(self.quit_button.get_label()))), "bg 1", "bg 1"))
        return urwid.GridFlow(buttonsList, 10, 2, 2, align="center")

    def get_body(self):
        """
        Return a SimpleListWalker for holding the central frame including
        the title, the introductory text, the form widgets and the buttons.
        """
        myList = [
            urwid.AttrWrap(urwid.LineBox(urwid.Filler(self.get_contents(), valign='top'), tlcorner=' ', tline=' ', lline=' ', trcorner=' ', blcorner=' ', rline=' ', bline=' ', brcorner=' '), "body bg"),
            ('pack', urwid.Divider()),
            ('pack', self.get_buttons()),
            ('pack', urwid.Divider())
        ]

        pile = urwid.Pile(myList)
        return pile

    def get_frame(self):
        """
        Return a Frame object which will act as the top level widget of our application.
        """

        frame = urwid.Frame(self.get_body(), header=urwid.Columns([
            urwid.Text(("header", "  "+self.brand), align='left'),
            urwid.Text(("header", "  "+self.title+"  "), align='center'),
            urwid.Text(("header", "  "+' '+"  "), align='right')], 1))
        frame = urwid.AttrWrap(frame, "background main")
        return frame

    def callBackIn(self, seconds, callbackfun):
        self.loop.set_alarm_in(seconds, callbackfun)

    def alert(self, alertMessage):
        """Update the alert message box with alert message"""

        self.alert_box_wrap.set_attr("alert message")
        self.alert_box.set_text(alertMessage)
        self.update()

    def message(self, normalMessage):
        """Update the alert message box with normal message"""

        self.alert_box_wrap.set_attr("normal message")
        self.alert_box.set_text(normalMessage)
        self.update()

    def update(self):
        """
        Update the frame with a new body after the next or previous button is clicked.
        Used by the controller classes to refresh the UI.
        """

        body = self.get_body()
        #body = urwid.Pile([('flow',urwid.Text('aaaa'))])
        self.frame.set_body(body)
        if self.loop.screen.started:
            self.loop.screen.clear()
            self.loop.draw_screen()

    def disable_buttons(self):
        """
        Disable buttons, usefull for avoiding errors during long running tasks
        """
        self._input_disabled = True
        self.update()

    def enable_buttons(self):
        """
        Enable user input after a long running process ended
        """
        self._input_disabled = False
        self.update()

    def disable_next_button(self):
        """disable next button"""
        self._disable_next = True
        self.update()

    def enable_next_button(self):
        """disable next button"""
        self._disable_next = False
        self.update()

    def disable_prev_button(self):
        """disable prev button"""
        self._disable_prev = True
        self.update()

    def enable_prev_button(self):
        """disable prev button"""
        self._disable_prev = False
        self.update()

    def disable_quit_button(self):
        """disable quit button"""
        self._disable_quit = True
        self.update()

    def enable_quit_button(self):
        """disable quit button"""
        self._disable_quit = False
        self.update()

    def set_next_button_label(self, text):
        """Set next button label

        """
        self.next_button.set_label(text)
        self.update()

    def set_previous_button_label(self, text):
        """Set previous button label

        """
        self.back_button.set_label(text)
        self.update()

    def refresh(self):
        """Force an update of the screen

        """

    def add_next_callback(self, func, user_arg=None):
        """
        Set a callback for the next button "click" signal. This is used by
        the controller classes to process user input and setup the next state
        of the application.
        """
        self.next_callback = func
        urwid.connect_signal(self.next_button, "click", func, user_arg=user_arg)

    def add_previous_callback(self, func, user_arg=None):
        """
        Set a callback for the previous button "click" signal.
        Used by the controller classes for jumping to a previous state.
        """
        self.back_callback = func
        urwid.connect_signal(self.back_button, "click", func, user_arg=user_arg)

    def add_quit_callback(self, func, user_arg=None):
        """
        Set a callback for the quit button "click" signal. Used by the
        controller classes for performing operation before exiting application.
        """
        self.quit_callback = func
        signal.signal(signal.SIGINT, func)
        urwid.connect_signal(self.quit_button, "click", func, user_arg=user_arg)

    def clear_callbacks(self):
        """Remove every callback from the buttons."""
        if self.next_callback:
            urwid.disconnect_signal(self.next_button, "click", self.next_callback)
        if self.back_callback:
            urwid.disconnect_signal(self.back_button, "click", self.back_callback)
        if self.quit_callback:
            urwid.disconnect_signal(self.quit_button, "click", self.quit_callback)

    def set_palette(self):
        """Set the coloscheme."""
        palette = next(self.palettes)
        self.loop.screen.register_palette(palette)

    def unhandled_input(self, key):
        """Handle unmanaged keys."""
        #raise Exception(str(key))
        if key == "f2":
            self.set_palette()
            self.update()
        if key == "tab":
            self.setFocusOnNext()
        if (key == 'enter') and (not self._input_disabled) and (not self._disable_next):
            self.next_callback()
        if ((key == 'ctrl B' or key == 'ctrl b') and (not self._input_disabled) and (not self._disable_prev)):
            self.back_callback()

    def run(self):
        """Start the main loop."""
        self.loop.run()

    def stop(self):
        """Stop the main loop."""
        #self.loop.screen.set_mouse_tracking(False)

        def delayExit(loop, user_data=None):
            """delay the exit so that mouse press event is not echo to terminal"""
            raise urwid.ExitMainLoop
        
        self.loop.set_alarm_in(0.5, delayExit)
    
    def setFocusOnNext(self):
        """Set focus on next button"""
        if self._input_disabled or (self._disable_prev and self._disable_next and self._disable_quit):
            return
        self.frame.set_focus('body')
        self.frame.get_body().set_focus(2)
        if self._disable_prev:
            self.frame.get_body().contents[2][0].set_focus(0)
        else:
            self.frame.get_body().contents[2][0].set_focus(1)

    def afterRenderProcess(self, func):
        def dummyFunc(loop, user_data=None):
            func()
        self.loop.set_alarm_in(0.01, dummyFunc)

    def updateOEM(self, oemid=None):
        """Update the UI based on OEM selected"""
        if not self.subtitle:
            self.set_subtitle()
        self.frame.set_header(urwid.Columns([
            urwid.Text(("header", "  "+self.brand), align='left'),
            urwid.Text(("header", "  "+self.title+" "+self.subtitle), align='center'),
            urwid.Text(("header", "  "+' '+" "), align='right')], 1))


@core.renders(model.SimpleText)
class UrwidTextLabel(UrwidView):
    """
    A simple text label for information purposes only.
    """
    def __init__(self, label):
        """UrwidTextLabel initialization."""
        self.text = label.text
        self.urwidWidget = None

    def accept(self):
        """Return a text widget."""
        self.urwidWidget = urwid.Text(self.text)
        return self.urwidWidget

    def configure(self):
        """We dont need it in this simple widget."""
        pass

    def get_value(self):
        """This widget has no user input. Return None."""
        return None

    def change(self, newVal):
        """change the text of the urwidTextWidget"""
        self.urwidWidget.__init__(newVal)


@core.renders(model.FormattedText)
class UrwidFormattedTextLabel(UrwidView):
    """
    A formatted text label for information purposes only.
    """
    def __init__(self, label):
        """UrwidFormattedTextLabel initialization."""
        self.text = label.text
        self.fg_fmt = label.fg_fmt
        self.bg_fmt = label.bg_fmt
        self.urwidWidget = None

    def accept(self):
        """Return a text widget."""
        fmt_attr = urwid.AttrSpec(self.fg_fmt, self.bg_fmt, 256)
        self.urwidWidget =  urwid.AttrWrap(urwid.Text(self.text), fmt_attr)
        return self.urwidWidget

    def configure(self):
        """We dont need it in this simple widget."""
        pass

    def get_value(self):
        """This widget has no user input. Return None."""
        return None

    def change(self, newVal):
        """change the text of the urwidTextWidget"""
        self.urwidWidget.__init__(newVal)


@core.renders(model.TextOption)
class UrwidTextInput(UrwidView):
    """
    A text input widget for rendering the TextOption model.
    """
    def __init__(self, text_option):

        # Expose the model
        self.option_name = text_option.name
        self.default_val = text_option.defaultVal
        self.option_maxlen = text_option.maxlen
        self.option_short_desc = text_option.short_desc
        self.option_help = text_option.help_text

        # Widgets
        self.edit = urwid.Edit()
        self.wrap = None

    def configure(self):
        """Setup the elements composing the model view."""

        # Configure the Edit widgets
        self.option_name = ('active',self.option_name+":")
        self.edit.set_caption(self.option_name)
        self.edit.set_edit_text(self.default_val)
        self.edit.set_edit_pos(-1)
        self.wrap = urwid.AttrWrap(self.edit, 'edit active')
        #self.wrap = urwid.AttrWrap(self.edit, 'editfc', 'editbx')

    def get_value(self):
        """Returns the text entered by the user."""
        return self.edit.edit_text

    def accept(self):
        """Returns the configured top level container widget."""
        return self.wrap


@core.renders(model.NumericOption)
class UrwidNumericInput(UrwidView):
    """
    A widget accepting numeric input for rendering the NumericOption model.
    """
    def __init__(self, text_option):

        # Expose the model
        self.option_name = text_option.name
        self.default_Val = str(text_option.defaultVal)
        self.option_short_desc = text_option.short_desc
        self.option_help = text_option.help_text

        # Widgets
        self.edit = urwid.IntEdit()
        self.wrap = None

    def configure(self):
        """Setup the elements composing the model view."""

        # Configure the Edit widgets
        self.edit.set_caption(self.option_name + ": ")
        self.edit.set_edit_text(self.default_Val)
        self.edit.set_edit_pos(-1)
        self.wrap = urwid.AttrWrap(self.edit, 'edit active')

    def get_value(self):
        """Returns the text entered by the user."""

        return self.edit.value()

    def accept(self):
        """Returns the configured top level container widget."""

        return self.wrap


@core.renders(model.BoolOption)
class UrwidBoolInput(UrwidView):
    """
    A two-states widget for boolean options.
    """
    def __init__(self, bool_option):

        self.option_name = bool_option.name
        self.option_short_desc = bool_option.short_desc
        self.option_help = bool_option.help_text
        self.action_func = bool_option.actionListner
        # Widgets
        #self.check_box = urwid.CheckBox(self.option_name)
        self.check_box = urwid.CheckBox(self.option_name, on_state_change=self.action_func, user_data=self.option_short_desc)
        self.widget_toggleFunc = self.check_box.toggle_state
        self.widget_set_state = self.check_box.set_state
        self.wrap = None
        if bool_option.selected:
            self.check_box.set_state(True)

    def configure(self):
        """Set up the container widget."""
        self.wrap = urwid.AttrWrap(self.check_box, 'edit', 'active')

    def get_value(self):
        """Return the user input."""
        return self.check_box.state

    def accept(self):
        """Return the toplevel widget."""
        return self.wrap

    def change(self, newVal, doCallBack):
        """Change the state of checkbox"""
        if newVal:
            self.check_box.set_state(True, do_callback=doCallBack)
        else:
            self.check_box.set_state(False, do_callback=doCallBack)

    def changeActionListener(self, newHandler):
        """change the action state event listner"""
        if(self.action_func is not null):
            disconnect_signal(self.check_box, 'change', self.action_func, self.option_short_desc)
        self.action_func = newHandler
        connect_signal(self.check_box, 'change', self.action_func, self.option_short_desc)

    def disable(self, disable):
        """disable by adding event handler to change back old status"""
        def _blankFunction(*args, **kwargs):
            pass
        if disable:
            disconnect_signal(self.check_box, 'change', self.action_func, self.option_short_desc)
            self.check_box.toggle_state = _blankFunction
            self.check_box.set_state = _blankFunction
        else:
            connect_signal(self.check_box, 'change', self.action_func, self.option_short_desc)
            self.check_box.toggle_state = self.widget_toggleFunc
            self.check_box.set_state = self.widget_set_state


@core.renders(model.ExclusiveOptionList)
class UrwidExclusiveList(UrwidView):
    """
    A list of options, from which the user can pick only one.
    """
    def __init__(self, options_list):

        self.options_list = options_list.options

        # Widgets
        self.radio_button_list = []
        self.group = []
        self.wrap = None

    def configure(self):
        """Setup widgets."""
        for option in self.options_list:
            radio_button = urwid.RadioButton(self.group, option['name'])
            if option['selected']:
                radio_button.set_state(True)
            wrap = urwid.AttrWrap(radio_button, 'edit', 'active')
            self.radio_button_list.append(wrap)

        #longListModel = model.LongListBox()
        if len(self.radio_button_list) > 3:
            pile = UrwidLongListBox
            pile.window = self.window
            pile = pile(self.radio_button_list, isAlreadyUrwid=True)
            pile.configure()
            pile = pile.accept()
            self.wrap = pile
        else:
            pile = urwid.Pile(self.radio_button_list)
            self.wrap = urwid.AttrWrap(pile, "selectable")

    def get_value(self):
        """Return the user input."""
        for index, radio_button in enumerate(self.radio_button_list):
            if radio_button.state:
                return index

    def accept(self):
        """Return the top level widget."""
        return self.wrap

    def set_option(self, num):
        """ Select the option in the radio buttons list"""
        self.radio_button_list[num].set_state(True)

@core.renders(model.CheckBoxList)
class UrwidCheckBoxList(UrwidView):
    """
    A list of options, from which the user can pick only one.
    """
    def __init__(self, options_list):
        self.options_list = options_list.options
        # Widgets
        self.checkbox_list = []
        self.group = []
        self.wrap = None
        self.hashmap = {}
        for option in self.options_list:
            self.hashmap[option['name']] = option['selected']

    def item_chosen(self, checkbox, new_state):
        label = checkbox.get_label()
        self.hashmap[label] = new_state

    def configure(self):
        """Setup widgets."""
        for option in self.options_list:
            checkbox = urwid.CheckBox(option['name'], option['selected'])
            urwid.connect_signal(checkbox, 'change', self.item_chosen)
            #self.checkbox_list.append(checkbox)
            wrap = urwid.AttrWrap(checkbox, 'edit', 'active')
            self.checkbox_list.append(wrap)

        if len(self.checkbox_list) > 3:
            pile = UrwidLongListBox
            pile.window = self.window
            pile = pile(self.checkbox_list, isAlreadyUrwid=True)
            pile.configure()
            pile = pile.accept()
            self.wrap = pile
            #self.wrap = urwid.AttrWrap(pile, "selectable")
        else:
            pile = urwid.Pile(self.checkbox_list)
            self.wrap = urwid.AttrWrap(pile, "selectable")


    def get_value(self):
        """Return the user input."""
        selected_items = []
        for option in self.options_list:
            if self.hashmap[option['name']]:
                selected_items.append(option['name'])

        return selected_items

    def accept(self):
        """Return the top level widget."""
        return self.wrap

    #def set_option(self, num):
    #    """ Select the option in the radio buttons list"""
    #    self.radio_button_list[num].set_state(True)


@core.renders(command.CommandExecutor)
class UrwidCommandExecutorView(UrwidView):
    """
    Renders a Batch Processing class in a progressbar
    """

    UPDATELOCK = Lock()
    def __init__(self, ce):

        ce.delegate = self
        self.ce = ce

        # Widgets
        if not self.ce.name:
            self.caption = urwid.Text("")
        else:
            self.caption = urwid.Text(self.ce.name)

        self.progress_bar = urwid.ProgressBar('pb normal', 'pb complete')

        self.pipe = self.window.loop.watch_pipe(self.notify)

    def configure(self):
        """Setup widgets"""
        pass

    def get_value(self):
        """
        Return user input. We return None since this widget does not accept focus.
        """
        return None

    def notify(self, s):
        """Notify window about changes"""
        s = s.strip()
        messages = s.split("*")
        for m in messages:
            if not m.strip():
                continue

            if "pb" in m:
                k, v = m.split(":")
                if k == "pb":
                    v = float(v)
                    self.progress_bar.set_completion(v)
                    if v == 100.0:
                        Render.main_window_instance.update()
                        return False
            else:
                v = ":".join(m.split(':')[1:])
                try:
                    caption_text = _(v)
                except:
                    caption_text = v

                #self.caption.set_text(_(v))
                if self.ce.name:
                    caption_text = self.ce.name + ": " + caption_text

                self.caption.set_text(caption_text)

        with UrwidCommandExecutorView.UPDATELOCK:
            Render.main_window_instance.update()

        return True

    def on_execution_started(self, queue):
        """Called before starting to process the input queue"""
        self.notify("pb:1*")
        #os.write(self.pipe, "pb:1*")

    def on_execution_ended(self, queue):
        """Called when the input queue is done"""
        self.notify("pb:100*")
        #os.write(self.pipe, "pb:100*")

    def on_command_start(self, item):
        """Called before processing the item"""
        self.notify("caption:" + item.description + "*")
        #os.write(self.pipe, "caption:" + item.description + "*")

    def on_command_end(self, item, total, done):
        """Called after item is complete."""
        self.notify("pb:" + str(done * self.progress_bar.done / total) + "*")
        #os.write(self.pipe, "pb:" + str(done * self.progress_bar.done / total) + "*")

    def update_caption(self, caption_text):
        self.notify("caption:" + caption_text + "*")

    def update_progress(self, percent_complete):
        self.notify("pb:%s*" % (str(percent_complete)))

    def accept(self):
        """Return the top level widget."""
        return urwid.Pile([self.caption, urwid.Divider(), self.progress_bar])


@core.renders(model.LongListBox)
class UrwidLongListBox(UrwidView):
    """
    A simple text label for information purposes only.
    """
    def __init__(self, longListBox, isAlreadyUrwid=False):
        """UrwidTextLabel initialization."""
        self.isAlreadyUrwid = isAlreadyUrwid
        self.occupiedRows = 0
        if isAlreadyUrwid:
            self.longList = longListBox
        else:
            self.longListBox = longListBox
            self.longList = longListBox.pileList
            self.innerRenderObj = Render("", "", *self.longList)
            self.longList = self.innerRenderObj.get_widgets()
        self.displayCount = urwid.raw_display.Screen().get_cols_rows()[1] - 13 - self.occupiedRows
        if self.displayCount <= 0:
            self.displayCount = 1
        self.currentCount = 0

        self.urwidWidget = None
        #self.currentPageBox = urwid.Text('  1 / 1  ')
        total = int(math.ceil((len(self.longList) / float(self.displayCount))))
        current = int(math.ceil(((self.currentCount+1) / float(self.displayCount))))
        self.currentPageBox = urwid.Text('  '+str(current)+' / '+str(total)+'  ')

        def _sigwinch_handler2(signum, frame):
            from urwid.compat import PYTHON3, bytes, B
            self2 = self.window.loop.screen
            if not self2._resized:
                os.write(self2._resize_pipe_wr, B('R'))
            self2._resized = True
            self2.screen_buf = None
            self.doHandleScreenChange()
        self.window.loop.screen._sigwinch_handler = _sigwinch_handler2
        self.window.loop.screen.signal_init()

    def doHandleScreenChange(self, currentCount=None):
        """handle changes in screen and fit long box accordingly"""
        self.displayCount = urwid.raw_display.Screen().get_cols_rows()[1] - 13 - self.occupiedRows
        noSpaceLeftToDraw = True if self.displayCount < 0 else False
        if self.displayCount <= 0:
            self.displayCount = 1
        if currentCount != None:
            self.currentCount = currentCount
        else:
            self.currentCount = 0
        total = int(math.ceil((len(self.longList) / float(self.displayCount))))
        current = int(math.ceil(((self.currentCount+1) / float(self.displayCount))))
        self.currentPageBox.set_text('  '+str(current)+' / '+str(total)+'  ')
        self.endCount = [(self.currentCount+self.displayCount), len(self.longList)][((self.currentCount+self.displayCount) > len(self.longList))]
        self.innerPile = urwid.Pile(self.longList[self.currentCount:self.endCount])

        if len(self.longList) <= self.displayCount:
            self.urwidWidget.__init__(self.longList[self.currentCount:self.endCount])
            return
        filer = urwid.Filler(self.innerPile, valign='top')
        box = urwid.BoxAdapter(filer, self.displayCount)
        next_more_button = customButton(_("NEXT_BUTTON_TEXT"), leftSymbol='', rightSymbol='')
        prev_more_button = customButton(_("BACK_BUTTON_TEXT"), leftSymbol='', rightSymbol='')
        next_more_button_wrap = urwid.AttrWrap(next_more_button, "button normal", "button select")
        prev_more_button_wrap = urwid.AttrWrap(prev_more_button, "button normal", "button select")
        if current == 1:
            button_grid = urwid.GridFlow([next_more_button_wrap], 9, 2, 1, align="right")
        elif current == total:
            button_grid = urwid.GridFlow([prev_more_button_wrap], 9, 2, 1, align="right")
        else:
            button_grid = urwid.GridFlow([prev_more_button_wrap, next_more_button_wrap], 9, 2, 1, align="right")
        moreColumn = urwid.Columns([self.currentPageBox, button_grid])
        #moreColumn = urwid.Columns([button_grid, self.currentPageBox])
        self.urwidWidget.__init__([('weight', self.displayCount, box), urwid.Divider(), moreColumn])
        urwid.connect_signal(prev_more_button, 'click', self.prev_more_button_func)
        urwid.connect_signal(next_more_button, 'click', self.next_more_button_func)

    def accept(self):
        """Return a text widget."""
        self.urwidWidget.doHandleScreenChange = self.doHandleScreenChange
        self.urwidWidget.longListRef = self
        self.urwidWidget.isLongList = True
        return self.urwidWidget

    def next_more_button_func(self, btn):
        currentCount = self.currentCount + self.displayCount
        self.doHandleScreenChange(currentCount)

    def prev_more_button_func(self, btn):
        currentCount = self.currentCount - self.displayCount
        self.doHandleScreenChange(currentCount)

    def configure(self):
        """We dont need it in this simple widget."""
        self.endCount = [(self.currentCount+self.displayCount), len(self.longList)][((self.currentCount+self.displayCount) > len(self.longList))]
        self.innerPile = urwid.Pile(self.longList[self.currentCount:self.endCount])
        if len(self.longList) < self.displayCount:
            self.urwidWidget = self.innerPile
            return
        filer = urwid.Filler(self.innerPile, valign='top')
        box = urwid.BoxAdapter(filer, self.displayCount)
        next_more_button = customButton(_("NEXT_BUTTON_TEXT"), leftSymbol='', rightSymbol='')
        prev_more_button = customButton(_("BACK_BUTTON_TEXT"), leftSymbol='', rightSymbol='')
        next_more_button_wrap = urwid.AttrWrap(next_more_button, "button normal", "button select")
        prev_more_button_wrap = urwid.AttrWrap(prev_more_button, "button normal", "button select")
        button_grid = urwid.GridFlow([prev_more_button_wrap, next_more_button_wrap], 9, 2, 1, align="right")
        moreColumn = urwid.Columns([self.currentPageBox, button_grid])
        self.urwidWidget = urwid.Pile([('weight', self.displayCount, box), urwid.Divider(), moreColumn])
        #self.urwidWidget = self.mainPile
        urwid.connect_signal(prev_more_button, 'click', self.prev_more_button_func)
        urwid.connect_signal(next_more_button, 'click', self.next_more_button_func)

    def get_value(self):
        """This widget has no user input. Return None."""
        if self.isAlreadyUrwid:
            return [widget.get_val() for widget in self.longList]
        return self.innerRenderObj.get_user_input()

    def change(self, newVal):
        """change the text of the urwidTextWidget"""
        pass


class customButton(urwid.Button):
    """
    Customized buttons with "<" and ">" symbols able to be customized
    """
    def __init__(self, label, on_press=None, user_data=None, leftSymbol='', rightSymbol=''):
        """leftSymbol & rightSymbol replaces the default symbol used by urwid """
        self.button_left = Text(leftSymbol)
        self.button_right = Text(rightSymbol)
        super(customButton, self). __init__(label, on_press, user_data)


@core.renders(model.PasswordTextOption)
class UrwidPasswordTextInput(UrwidView):
    """
    A text input widget for rendering the TextOption model.
    """
    def __init__(self, text_option):

        # Expose the model
        self.option_name = text_option.name
        self.default_val = text_option.defaultVal
        self.option_maxlen = text_option.maxlen
        self.option_short_desc = text_option.short_desc
        self.option_help = text_option.help_text

        # Widgets
        self.edit = urwid.Edit(mask='*')
        self.wrap = None

    def configure(self):
        """Setup the elements composing the model view."""

        # Configure the Edit widgets
        self.edit.set_caption(self.option_name + ": ")
        self.edit.set_edit_text(self.default_val)
        self.edit.set_edit_pos(-1)
        self.wrap = urwid.AttrWrap(self.edit, 'edit', 'edit active')

    def get_value(self):
        """Returns the text entered by the user."""
        return self.edit.edit_text

    def accept(self):
        """Returns the configured top level container widget."""
        return self.wrap
    
@core.renders(model.IPAddress)
class UrwidIPAddress(UrwidView):
    """A widget designed for entering an IP address.

    """
    def __init__(self, ip_option):

        self.option_name = ip_option.name
        self.default_val = str(ip_option.defaultVal)
        self.short_desc = ip_option.short_desc
        self.help = ip_option.help_text

        # Widgets
        self.caption = None
        self.columns = None
        self.widgets = []
        self.edit_widgets = []

    def configure(self):
        """Setup the input widgets and a container.

        """
        self.caption = urwid.Text(self.option_name + ": ")
        self.widgets.append(self.caption)
        
        vals = ["", "", "", ""]
        if len(self.default_val):
            vals = self.default_val.split(".")

        for i in xrange(4):
            edit = urwid.Edit("")
            edit = urwid.AttrWrap(edit, "edit", "edit active")
            edit.set_edit_text(vals[i])
            edit.set_edit_pos(0)
            edit.set_align_mode("right")
            self.widgets.append(edit)
            self.edit_widgets.append(edit)
            if i != 3:
                self.widgets.append(urwid.Text("."))

        self.widgets[-1].set_edit_text(vals[3])

        self.columns = urwid.Columns(
                [("fixed", len(self.option_name) + 2, self.widgets[0]),
                 ("fixed", 4, self.widgets[1]),
                 ("fixed", 1, self.widgets[2]),
                 ("fixed", 4, self.widgets[3]),
                 ("fixed", 1, self.widgets[4]),
                 ("fixed", 4, self.widgets[5]),
                 ("fixed", 1, self.widgets[6]),
                 ("fixed", 4, self.widgets[7])],
                dividechars=1, min_width=1)

    def get_value(self):
        """Return user input.

        """
        values = [ i.get_edit_text() for i in self.edit_widgets ]
        values = [ str(value) for value in values ]
        if len("".join(values)):
            return ".".join(values)
        return ""

    def accept(self):
        """Return the top level widget.

        """
        return self.columns


#@core.renders(model.ProgressBar)
#class UrwidProgressBar(UrwidView):
#    """A widget designed for displaying progress bar
#    """
#
#    def __init__(self, options):
#        self.name = options.name
#        self.default_val = options.defaultVal
#        self.smooth = options.smooth
#        self.val = options.defaultVal
#
#        # Widgets
#        self.caption = None
#        self.progress_bar = None
#        self.widget = None
#
#
#    def configure(self):
#        """Setup the widgets 
#        """
#
#        self.caption = urwid.Text(self.name)
#        if self.smooth:
#            self.progress_bar = urwid.ProgressBar('pg normal', 'pg complete',
#                                        0, 1, 'pg smooth')
#        else:
#            self.progress_bar = urwid.ProgressBar('pg normal', 'pg complete', 0, 1)
#
#        self.progress_bar.set_completion(self.val)
#        self.widget = urwid.Pile([self.caption, urwid.Divider(), self.progress_bar])
#
#    def get_value(self):
#        """Return user input.
#        """
#        return self.val
#        
#
#    def accept(self):
#        """Return the top level widget.
#
#        """
#        return self.widget
#
