#-*- coding: utf-8 -*-

"""The render module provides a Render class, which responsibility is to
allow access to the view objects at runtime."""

from ui.core import get_main_window, get_view
from appliancelocale import getLocalizedString as _


class Render(object):
    """Given a set of model objects, find the proper view and setup window
    properties.
    """
    main_window_instance = None

    def __init__(self, title, intro, *options):
        """Render initialization
        """
        self.title = title
        self.intro = intro
        self.options = options
        self.prev_Text = (_("BACK_BUTTON_TEXT"))
        self.quit_Text = (_("QUIT_BUTTON_TEXT"))
        self.next_Text = (_("NEXT_BUTTON_TEXT"))
        #self.input_disabled = False
        self.disable_quit = False
        self.disable_prev = False
        self.disable_next = False
        self.renderers = []

    def disable_next_button(self):
        """disable next button"""
        self.disable_next = True

    def enable_next_button(self):
        """disable next button"""
        self.disable_next = False

    def disable_prev_button(self):
        """disable prev button"""
        self.disable_prev = True

    def enable_prev_button(self):
        """disable prev button"""
        self.disable_prev = False

    def disable_quit_button(self):
        """disable quit button"""
        self.disable_quit = True

    def enable_quit_button(self):
        """disable quit button"""
        self.disable_quit = False

    def set_next_button_label(self, text):
        """Set next button label
        """
        self.next_Text = text

    def set_quit_button_label(self, text):
        """Set next button label
        """
        self.quit_Text = text

    def set_previous_button_label(self, text):
        self.prev_Text = text

    def get_renderer(self, model):
        """Given a model, return its view.

        """
        ModelClass = type(model)
        renderer = get_view(ModelClass)
        renderer.window = self.main_window_instance
        return renderer

    def setup_widgets(self):
        """Initialize the widgets for each model.

        """
        for option in self.options:
            WidgetClass = self.get_renderer(option)
            widget = WidgetClass(option)
            widget.configure()
            widget.show()
            option.widgetRef = widget
            self.renderers.append(widget)

    def get_widgets(self):
        """Get the list of configured widgets.

        """
        self.setup_widgets()
        return [renderer.accept() for renderer in self.renderers]

    def get_user_input(self):
        """Get the widgets state. Used by controller instances for getting user
        input. The _process attribute in renderers are set by the core module,
        and they indicate if the widget provides a value entered by the user or
        not (for example, simple text labels do not provide any value, and they
        are not processed by the controller.process method)

        """
        values = [i.get_value() for i in self.renderers if i._process]
        return values

    @property
    def main_window(self):
        """Lazy initialization of the main window object.

        """
        if Render.main_window_instance is None:
            MainWindowClass = get_main_window()
            Render.main_window_instance = MainWindowClass()
            Render.main_window_instance.updateOEM()
        return type(self).main_window_instance

    def update_main_window(self):
        """Update the attributes of the main window instance and refresh the
        view.

        """
        self.main_window.title = self.title
        self.main_window.intro = self.intro
        self.main_window.widgets = self.get_widgets()

        self.main_window._disable_quit = self.disable_quit
        self.main_window._disable_prev = self.disable_prev
        self.main_window._disable_next = self.disable_next

        self.main_window.next_button.set_label(self.next_Text)
        self.main_window.quit_button.set_label(self.quit_Text)
        self.main_window.back_button.set_label(self.prev_Text)
        self.main_window.update()
