#-* encoding: utf8 -*-

"""Command Pattern"""

from threading import Thread, Lock
try:
    from Queue import Queue
except ImportError:
    from queue import Queue
import cvpylogger
from appliancelocale import getLocalizedString as _
import pdb


class Command(object):

    def __init__(self, func, args=None, description=None):
        """Store the callable and its arguments.

        Example:
            >>> inspect_command_class = Command(dir, Command)
            >>> inspect_command_class()
            ['__init__', '__call__', ...]

        """
        assert callable(func), "%r is not callable" % func
        self.func = func
        self.args = args or tuple()
        self.description = description or ""

    def __call__(self):
        """Call the command.
        """
        return apply(self.func, self.args)

    def __repr__(self):
        return "%s(<%s %s>)" % (self.__class__.__name__, self.func.__name__,
                                repr(self.args))


class CommandExecutor(Thread):
    """Execute commands in another thread.
    Call hooks before and after executing commands, and before and after
    processing the queue

    """
    def __init__(self, delegate=None, name=None):
        "Initialize CommandExecutor"
        super(CommandExecutor, self).__init__()
        self.executor = Executor()
        self.executor.delegate = self
        self.size = 0
        self.delegate = delegate
        self.name = name
        self._lock = Lock()
        self.setDaemon(True)

    def setErrorCallback(self, tasks_error_callback):
        "attach error callback"
        self.executor.error_callback = tasks_error_callback

    def started(self):
        "Execution started"
        if self.delegate is not None:
            with self._lock:
                self.delegate.on_execution_started(self)

    def ended(self):
        "Execution ended"
        if self.delegate is not None:
            with self._lock:
                self.delegate.on_execution_ended(self)

    def command_execution_started(self, command):
        "Command started hook"
        if self.delegate is not None:
            with self._lock:
                self.delegate.on_command_start(command)

    def command_execution_ended(self, command):
        "Command ended hook"
        if self.delegate is not None:
            with self._lock:
                self.delegate.on_command_end(command, self.size,
                                             self.executor.items_done)

    def add_command(self, func, args=None, description=None):
        """Create a Command instance and add it to the stack.

        """
        command = Command(func, args, description)
        f = self.executor.submit(command)
        return f

    def execute_all(self, commands):
        self.size = len(commands)
        for command in commands:
            self.add_command(*command)

    def update_caption(self, caption):
        if self.delegate:
            self.delegate.update_caption(caption)
        
    def update_progress(self, percent_completion):
        if self.delegate:
            self.delegate.update_progress(percent_completion)

    def update_status(self, caption, percent_completion):
        self.update_caption(caption)
        self.update_progress(percent_completion)

    def run(self):
        cvpylogger.getLog().info("Command Executor") 
        self.started()
        self.executor.start()
        self.executor.join()
        self.ended()


class Executor(Thread):

    def __init__(self, q=None):
        super(Executor, self).__init__()
        if q is None:
            q = Queue()
        self.q = q
        self.items_done = 0
        self.delegate = None
        self.setDaemon(True)

    def submit(self, item):
        self.q.put(item)

    def run(self):
            while not self.q.empty():
                item = self.q.get()

                if item.func.__name__.startswith('upgrade_'):
                    cvpylogger.getLog().info("%s ..." % _(item.description))
                if self.delegate is not None:
                    self.delegate.command_execution_started(item)
                try:
                    #pdb.set_trace()
                    returnTuple = item()
                    if type(returnTuple) == type(tuple()) and returnTuple[0] is False:
                        self.error_callback(item, errorMessage=returnTuple[1])
                        break
                    elif type(returnTuple) == type(bool()) and returnTuple is False:
                        self.error_callback(item, errorMessage="")
                        break
                except Exception as e:
                    self.error_callback(item, isException=True)
                    break
                if self.delegate is not None:
                    self.delegate.command_execution_ended(item)
                self.items_done += 1
