#!/usr/bin/python
import os
import signal
import sys
import os.path
import subprocess
import resource
import wrappers
import utils
from ui import model
from ui import command
from ui.render import Render
from ui.controller import Controller
from ui.wizard import UpdateMessageThread
from ui.command import CommandExecutor
from appliancelocale import getLocalizedString as _
import cvpylogger
import re
import codecs
import xmltodict
import time
import threading
import argparse
import pdb
import bmr

#Flag to enable/disable DAV config in UI
enable_dav = False

is_control_node = False
system_disks = []
ddb_disks = []
data_disks = []
reinit_drives = True
is_autoconf_mode = False
autoconf_action = None
input_devconfig_file = "hsblkdev_config.xml"
input_autoconf_file = None
DEFAULT_CONFIG_FILE = "hscm_config.xml"
output_file = DEFAULT_CONFIG_FILE
is_cluster_installation = False
configure_dav = False
cluster_nodes = []
discovered_nodes = []
local_ip_addresses = []
use_same_sys_meta_drives = False
forced_same_sys_meta_drives = False
the_app = None
config_file_found = False
cluster_nodes_install_status = {}
localhost_install_status = 0

ip_host_map = {}
SWIFT_CONFIG_FILENAME="/hsswiftconf.map"
appliance_swift = False

INSTALL_OPTION_FILE = "/hsinstallation.xml"
HS_IMAGING_SERVICE_NAME = "hsimaging"
HS_IMAGING_SERVICE_TYPE = "_CommVaultApplianceImaging._tcp"
HS_IMAGING_SERVICE_TXT_RECORD = "setup_in_progress=True"
HS_IMAGING_SERVICE_PORT = "11022"
REBOOT_NOTICE_FILE = "/tmp/.reboot-notice"

DBUS_SHARE_CONF_PATH="/usr/share/dbus-1/system.conf"
DBUS_ETC_CONF_PATH="/etc/dbus-1/system.conf"

def __XMLElem(elem, attribs ={}):
    attrs = ""
    seperator = " "
    for k in attribs:
        kv = "%s%s=\"%s\"" % (seperator, k, attribs[k])
        attrs = attrs + kv
    
    return elem + attrs


def XMLElem(elem, attribs={}):
    return "<%s/>" % (__XMLElem(elem, attribs))


def XMLElemOpen(elem, attribs={}):
    return "<%s>" % (__XMLElem(elem, attribs))


def XMLElemClose(elem):
    return "</%s>" % (elem)


def write_bootpartdrive_config(conf_file):
    global device_map
    XMLConsts = device_map.get_XML_consts()
    cnfmap = device_map.get_cnf_map()

    if DeviceMap.KEY_BOOTPARTDRIVE in cnfmap:
        bpd = cnfmap[DeviceMap.KEY_BOOTPARTDRIVE]
        bpdAttrs = {XMLConsts["BOOTPARTDRIVE_ATTR_DEVNAME"] : bpd[DeviceMap.KEY_BOOTPARTDRIVE_DEVNAME]}
        conf_file.write(XMLElem(XMLConsts["ELEM_BOOTPARTDRIVE"], bpdAttrs) + '\n')


def write_driveset_config(conf_file):
    global device_map
    XMLConsts = device_map.get_XML_consts()
    cnfmap = device_map.get_cnf_map()

    dsInfo = cnfmap[DeviceMap.KEY_DRIVESET]
    dsAttrs = { XMLConsts["DRIVESET_ATTR_TYPE"] : dsInfo[DeviceMap.KEY_DRIVESET_TYPE] }
    conf_file.write(XMLElem(XMLConsts["ELEM_DRIVESET"], dsAttrs) + '\n')


def write_arch_config(conf_file):
    global device_map
    cnfmap = device_map.get_cnf_map()

    if not DeviceMap.KEY_ARCH in cnfmap:
        return

    arch_info = cnfmap[DeviceMap.KEY_ARCH]
    XMLConsts = device_map.get_XML_consts()

    archAttrs = { XMLConsts["ARCH_ATTR_TYPE"] : arch_info[DeviceMap.KEY_ARCH_TYPE] }
    conf_file.write(XMLElem(XMLConsts["ELEM_ARCHITECTURE"], archAttrs) + '\n')


def write_bootmedia_config(conf_file):
    global device_map
    cnfmap = device_map.get_cnf_map()

    if not DeviceMap.KEY_BOOTMEDIA in cnfmap:
        return

    bm_info = cnfmap[DeviceMap.KEY_BOOTMEDIA]
    bmAttrs = {}

    XMLConsts = device_map.get_XML_consts()

    if DeviceMap.KEY_BOOTMEDIA_DEVNAME in bm_info:
        bmAttrs.update({XMLConsts["BOOTMEDIA_ATTR_DEVNAME"] : bm_info[DeviceMap.KEY_BOOTMEDIA_DEVNAME]})

    if DeviceMap.KEY_BOOTMEDIA_PXEBOOT in bm_info:
        bmAttrs.update({XMLConsts["BOOTMEDIA_ATTR_PXEBOOT"] : bm_info[DeviceMap.KEY_BOOTMEDIA_PXEBOOT]})

    if DeviceMap.KEY_BOOTMEDIA_NFSIP in bm_info:
        bmAttrs.update({XMLConsts["BOOTMEDIA_ATTR_NFSIP"] : bm_info[DeviceMap.KEY_BOOTMEDIA_NFSIP]})

    if DeviceMap.KEY_BOOTMEDIA_NFSDIR in bm_info:
        bmAttrs.update({XMLConsts["BOOTMEDIA_ATTR_NFSDIR"] : bm_info[DeviceMap.KEY_BOOTMEDIA_NFSDIR]})

    conf_file.write(XMLElem(XMLConsts["ELEM_BOOTMEDIA"], bmAttrs) + '\n')


def write_usb_config(conf_file):
    global device_map
    cnfmap = device_map.get_cnf_map()

    if not DeviceMap.KEY_USB in cnfmap:
        return

    XMLConsts = device_map.get_XML_consts()
    usb_info = cnfmap[DeviceMap.KEY_USB]

    usbAttrs = { XMLConsts["USB_ATTR_DEVNAME"] : usb_info[DeviceMap.KEY_USB_DEVNAME] }
    if DeviceMap.KEY_USB_MNTPATH in usb_info:
        usbAttrs.update({XMLConsts["USB_ATTR_MNTPATH"] : usb_info[DeviceMap.KEY_USB_MNTPATH]})

    conf_file.write(XMLElem(XMLConsts["ELEM_USB"], usbAttrs) + '\n')


def write_ova_config(conf_file):
    global device_map
    cnfmap = device_map.get_cnf_map()

    if not DeviceMap.KEY_OVA in cnfmap:
        return

    XMLConsts = device_map.get_XML_consts()
    ova_list = cnfmap[DeviceMap.KEY_OVA]

    for ova_info in ova_list:
        ovaAttrs = {}

        if DeviceMap.KEY_OVA_TYPE in ova_info:
            ovaAttrs.update({XMLConsts["OVA_ATTR_TYPE"] : ova_info[DeviceMap.KEY_OVA_TYPE]})

        if DeviceMap.KEY_OVA_MD5SUM in ova_info:
            ovaAttrs.update({XMLConsts["OVA_ATTR_MD5SUM"] : ova_info[DeviceMap.KEY_OVA_MD5SUM]})

        if DeviceMap.KEY_OVA_SRCPATH in ova_info:
            ovaAttrs.update({XMLConsts["OVA_ATTR_SRCPATH"] : ova_info[DeviceMap.KEY_OVA_SRCPATH]})

        if DeviceMap.KEY_OVA_DESTPATH in ova_info:
            ovaAttrs.update({XMLConsts["OVA_ATTR_DESTPATH"] : ova_info[DeviceMap.KEY_OVA_DESTPATH]})

        if DeviceMap.KEY_OVA_SKIPCHK in ova_info:
            ovaAttrs.update({XMLConsts["OVA_ATTR_SKIPCHECK"] : ova_info[DeviceMap.KEY_OVA_SKIPCHK]})

        conf_file.write(XMLElem(XMLConsts["ELEM_OVA"], ovaAttrs) + '\n')
  

def write_drive_config(conf_file, drive_list, xml_element):
    global device_map
    devmap = device_map.get_dev_map()
    XMLConsts = device_map.get_XML_consts()

    conf_file.write(XMLElemOpen(xml_element) + '\n')
    for drive in sorted(drive_list):
        attrs = { XMLConsts["DISK_ATTR_DEVALIAS"] : devmap[drive][DeviceMap.KEY_DEVALIAS],
                  XMLConsts["DISK_ATTR_DEVSIZE"] : devmap[drive][DeviceMap.KEY_DEVSIZE] }
        conf_file.write(XMLElem(XMLConsts["ELEM_SINGLE_DISK"], attrs) + '\n')

    conf_file.write(XMLElemClose(xml_element) + '\n')


def write_dav_config(conf_file):
    global device_map
    XMLConsts = device_map.get_XML_consts()

    global configure_dav
    global reinit_drives
    is_reference_arch = device_map.is_ref_arch()
    if is_reference_arch and configure_dav:
        davAttr = { XMLConsts["DAV_ATTR_CREATE"] : "1" }
        conf_file.write(XMLElem(XMLConsts["ELEM_DAV"], davAttr) + '\n')


def write_mount_config(conf_file):
    global device_map
    XMLConsts = device_map.get_XML_consts()
    mntmap = device_map.get_mnt_map()
    #<MntPath type="data" blkdev="/dev/sdd" mntpath="/ws/disk1" fstype="xfs" fsargs="noatime,inode64,nofail"/>

    for path in mntmap: 
        pathinfo = mntmap[path]
        attrs = { XMLConsts["MNTPATH_ATTR_TYPE"] : pathinfo[DeviceMap.KEY_MNTTYPE],
                    XMLConsts["MNTPATH_ATTR_BLKDEV"] : pathinfo[DeviceMap.KEY_MNTBLKDEV],
                    XMLConsts["MNTPATH_ATTR_MNTPATH"] : path,
                    XMLConsts["MNTPATH_ATTR_FSTYPE"] : pathinfo[DeviceMap.KEY_MNTFSTYPE],
                    XMLConsts["MNTPATH_ATTR_FSARGS"] : pathinfo[DeviceMap.KEY_MNTFSARGS] }

        conf_file.write(XMLElem(XMLConsts["ELEM_MNTPATH"], attrs) + '\n')


def write_installation_option_file():
    '''<?xml version="1.0" encoding="UTF-8"?>
        <HyperScaleUserConfiguration>
        <Installation type="Initialize"/>
        </HyperScaleUserConfiguration>
    '''
    XML_DECL = """<?xml version="1.0" encoding="UTF-8"?>"""
    global device_map
    XMLConsts = device_map.get_XML_consts()
    
    global INSTALL_OPTION_FILE
    with open(INSTALL_OPTION_FILE, 'w') as conf_file:
        conf_file.write(XML_DECL + '\n')
        conf_file.write(XMLElemOpen(XMLConsts["HSUSER_CONFIG_ROOT_ELEM"]) + '\n')

        global reinit_drives
        install_type = XMLConsts["STR_INITIALIZE"] if reinit_drives else XMLConsts["STR_PRESERVE"]
        install_attrs = { XMLConsts["INSTALL_ATTR_TYPE"] : install_type }
        conf_file.write(XMLElem(XMLConsts["ELEM_INSTALL"], install_attrs) + '\n')

        conf_file.write(XMLElemClose(XMLConsts["HSUSER_CONFIG_ROOT_ELEM"]) + '\n')


def write_config():
    '''<?xml version="1.0"?>
    <HyperScaleConfigurationManager>
    <Node type="Control"/>
    <OSDrives>
    <Disk devname value="/dev/sda"/>
    </OSDrives>
    <MetadataDrives>
    <Disk devname value="/dev/sdb"/>
    </MetadataDrives>
    <DataDrives>
    <Disk devname value="/dev/sde"/>
    </DataDrives>
    <MntPath type="data" blkdev="/dev/sdd" mntpath="/ws/disk1" fstype="xfs" fsargs="noatime,inode64,nofail"/>
    </HyperScaleConfigurationManager>'''
    global device_map
    XMLConsts = device_map.get_XML_consts()

    global output_file
    HSCM_CONFIG_FILE = output_file
    XML_DECL = """<?xml version="1.0" encoding="UTF-8"?>"""
    
    with open(HSCM_CONFIG_FILE, 'w') as conf_file:
        conf_file.write(XML_DECL + '\n')
        conf_file.write(XMLElemOpen(XMLConsts["HSCM_CONFIG_ROOT_ELEM"]) + '\n')

        global is_control_node
        nodeType = "Control" if is_control_node else "Data"
        node_attrs = { XMLConsts["NODE_ATTR_TYPE"] : nodeType }
        conf_file.write(XMLElem(XMLConsts["ELEM_NODE"], node_attrs) + '\n')

        global system_disks
        write_drive_config(conf_file, system_disks, XMLConsts["ELEM_OSDRIVES"])

        global data_disks
        write_drive_config(conf_file, data_disks, XMLConsts["ELEM_DATADRIVES"])

        global ddb_disks
        write_drive_config(conf_file, ddb_disks, XMLConsts["ELEM_METADRIVES"])

        write_arch_config(conf_file)
        write_bootmedia_config(conf_file)
        write_usb_config(conf_file)
        write_ova_config(conf_file)

        global reinit_drives
        global use_same_sys_meta_drives
        if not reinit_drives:
            if not device_map.get_meta_disks():
                cnfmap = device_map.get_cnf_map()
                dstype = "same" if use_same_sys_meta_drives else "separate"
                cnfmap[DeviceMap.KEY_DRIVESET] = { XMLConsts["DRIVESET_ATTR_TYPE"] : dstype }
            write_driveset_config(conf_file)
            if not device_map.get_meta_disks():
                write_dav_config(conf_file)
            write_bootpartdrive_config(conf_file)
            write_mount_config(conf_file)
        else:
            cnfmap = device_map.get_cnf_map()
            dstype = "same" if use_same_sys_meta_drives else "separate"
            cnfmap[DeviceMap.KEY_DRIVESET] = { XMLConsts["DRIVESET_ATTR_TYPE"] : dstype }
            write_driveset_config(conf_file)
            write_dav_config(conf_file)

        conf_file.write(XMLElemClose(XMLConsts["HSCM_CONFIG_ROOT_ELEM"]) + '\n')


def get_list_box(list_items):
    pile_list = []
    for li in list_items:
        item = model.SimpleText(li)
        pile_list.append(item)

    list_box = model.LongListBox(pile_list, len(pile_list))

    return list_box


def get_formatted_label(label_text, label_fg_color = model.FormattedText.Color.FG.BLACK,
            label_bg_color = model.FormattedText.Color.BG.DARK_CYAN):
    fg_fmt = '%s,bold,underline' % (label_fg_color)
    bg_fmt = label_bg_color
    
    return model.FormattedText(label_text, fg_fmt=fg_fmt, bg_fmt=bg_fmt)


def read_xml_file(devconf_xml_file):
    with open(devconf_xml_file) as fd:
        doc = xmltodict.parse(fd.read())

    return doc


#{
class Host:
    ARCH_BIN = "/usr/local/bin/arch"

    def __init__(self, host):
        self.log = cvpylogger.getLog()
        self.host = host


    def get_host(self):
        return self.host


    def validate(self):
        VALIDATION_FILE = "/tmp/%s.validation" % (self.host)
        DUMMY_DATA = "File created for validating host %s" % (self.host)

        with open(VALIDATION_FILE, 'w') as f:
                f.write(DUMMY_DATA)

        status, msg = self.upload_file(VALIDATION_FILE, VALIDATION_FILE)
        return status


    def send_shutdown_notice(self):
        global REBOOT_NOTICE_FILE
        DUMMY_DATA = "Shutdown notice for host %s" % (self.host)

        with open(REBOOT_NOTICE_FILE, 'w') as f:
                f.write(DUMMY_DATA)

        status, msg = self.upload_file(REBOOT_NOTICE_FILE, REBOOT_NOTICE_FILE)
        return status, msg


    def upload_file(self, local_file, remote_file):
        UPLOAD_CMD = "write-file"
        #write-file <remote host> <local file path> [<remote file path>]
        proc = subprocess.Popen([Host.ARCH_BIN, UPLOAD_CMD, self.host, local_file, remote_file],
                                    stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        out, err = proc.communicate()
        if proc.returncode != 0:
            return False, err

        rfile = "%s:%s" % (self.host, remote_file)
        return True, "Uploaded file %s to remote location %s successfully." % (local_file, rfile)


    def download_file(self, remote_file, local_file):
        DOWNLOAD_CMD = "read-file"

        proc = subprocess.Popen([Host.ARCH_BIN, DOWNLOAD_CMD, self.host, remote_file, local_file],
                                    stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        out, err = proc.communicate()
        if proc.returncode != 0:
            return False, err

        rfile = "%s:%s" % (self.host, remote_file)
        return True, "Downloaded remote file %s to %s successfully." % (rfile, local_file)


    def get_dev_conf_maps(self):
        host_devconf_file = "/tmp/.%s.devconf.xml" % (self.host)

        global input_devconfig_file 
        status, err_msg = self.download_file(input_devconfig_file, host_devconf_file)
        if not status:
            self.log.error("Failed to download device config file for host %s with error [%s]." % (self.host, err_msg))
            return None

        return DeviceMap.get_dev_config_maps(host_devconf_file)


    def read_devconf_file(self):
        host_devconf_file = "/tmp/.%s.devconf.xml" % (self.host)

        global input_devconfig_file 
        status, err_msg = self.download_file(input_devconfig_file, host_devconf_file)
        if not status:
            self.log.error("Failed to download device config file for host %s with error [%s]." % (self.host, err_msg))
            return None

        return read_xml_file(host_devconf_file)

    @staticmethod
    def set_hostname():
        serial_number = wrappers.get_server_serial_number()
        wrappers.set_hostname(serial_number)

    @staticmethod
    def get_hostname():
        return wrappers.get_hostname()


#} class Host

#{
class DeviceMap:
    XML_CONSTS = None
    XML_CONSTS_FILE = "xml_helper.tbl"
    DEV_XML_FILE = "hsblkdev_config.xml"

    KEY_DEVTYPE = "type"
    KEY_CVDRIVE = "cvdrive"
    KEY_DEVSIZE = "devsize"
    KEY_DEVALIAS = "devalias"
    KEY_DRIVESET = "DriveSet"
    KEY_DRIVESET_TYPE = "type"

    KEY_BOOTMEDIA = "BootMedia"
    KEY_BOOTMEDIA_DEVNAME = "devname"
    KEY_BOOTMEDIA_PXEBOOT = "pxeboot"
    KEY_BOOTMEDIA_NFSIP = "nfsip"
    KEY_BOOTMEDIA_NFSDIR = "nfsdir"

    KEY_ARCH = "Architecture"
    KEY_ARCH_TYPE = "type"

    KEY_USB  = "Usb"
    KEY_USB_DEVNAME  = "devname"
    KEY_USB_MNTPATH  = "mntpath"

    KEY_OVA  = "Ova"
    KEY_OVA_TYPE  = "type"
    KEY_OVA_SRCPATH  = "srcpath"
    KEY_OVA_DESTPATH  = "destpath"
    KEY_OVA_MD5SUM  = "md5sum"
    KEY_OVA_SKIPCHK  = "skipcheck"

    KEY_BOOTPARTDRIVE = "BootPartDrive"
    KEY_BOOTPARTDRIVE_DEVNAME = "devname"

    KEY_MNTTYPE = "type"
    KEY_MNTBLKDEV = "blkdev"
    KEY_MNTFSTYPE = "fstype"
    KEY_MNTFSARGS = "fsargs"


    def __init__(self):
        DeviceMap.__read_XML_constants(DeviceMap.XML_CONSTS_FILE)
        self.dev_map = {}
        self.mnt_map = {}
        self.cnf_map = {}
        self.__get_device_map()

        self.__check_autoconf()
        meta_mnt_paths = self.get_meta_mount_paths()
        global is_control_node
        if len(meta_mnt_paths):
            for mnt_path in meta_mnt_paths:
                if "/ws/ddb" in mnt_path:
                    is_control_node = True

        self.is_cv_disk_present = self.__are_cv_disks_present()
        global forced_same_sys_meta_drives
        forced_same_sys_meta_drives = self.__mustUseSameSystemAndMetaDisks()
        

    def __mustUseSameSystemAndMetaDisks(self):
        numDisks = len(self.get_disks())
        return True if numDisks <= 3 else False


    def cv_disks_present(self):
        return self.is_cv_disk_present


    def __are_cv_disks_present(self):
        if not self.dev_map:
            return False
    
        for disk in self.dev_map:
            if self.is_cv_disk(disk):
                return True
    
        return False


    @staticmethod
    def get_XML_consts():
        return DeviceMap.XML_CONSTS


    @staticmethod
    def __read_XML_constants(xmlConstantsFile):
        """
        Read XML constants from given file.
        """
        if DeviceMap.XML_CONSTS:
            return

        dictObj = {}
        with codecs.open(xmlConstantsFile, "r", encoding='utf-8') as rf:
            for line in rf.readlines():
                line = line.strip()
                if line.startswith('#') or line.startswith('--'):
                    continue
                elif len(line) > 2:
                    split, key, value = re.split('^(\S+)*', line)
                    dictObj[key.strip()] = value.strip()
    
        DeviceMap.XML_CONSTS = dictObj


    @staticmethod
    def is_blkdev_conf_similar(dev_map1, dev_map2):

        """
        Number of disks in both dev maps must be same
        """
        if len(dev_map1) != len(dev_map2):
            return False

        """
        Create local map of format - { u'sas1': {'type': u'sas', 'devsize': u'100G'} }
        from dev_map of format - { u'/dev/sdd': {'cvdrive': u'none', 'type': u'sas', 'devsize': u'100G', 'devalias': u'sas1'} }
        """
        local_dev_map1 = {}
        for d in dev_map1:
            dev_dict = dict(dev_map1[d])
            inner_d = {
                    DeviceMap.KEY_DEVTYPE : dev_dict[DeviceMap.KEY_DEVTYPE], 
                    DeviceMap.KEY_DEVSIZE : dev_dict[DeviceMap.KEY_DEVSIZE] }
            local_dev_map1[dev_dict[DeviceMap.KEY_DEVALIAS]] = inner_d

        local_dev_map2 = {}
        for d in dev_map2:
            dev_dict = dict(dev_map2[d])
            inner_d = {
                    DeviceMap.KEY_DEVTYPE : dev_dict[DeviceMap.KEY_DEVTYPE],
                    DeviceMap.KEY_DEVSIZE : dev_dict[DeviceMap.KEY_DEVSIZE] }
            local_dev_map2[dev_dict[DeviceMap.KEY_DEVALIAS]] = inner_d

        """
        Compare local dev maps i.e. devalias, devtype and devsize.
        All should be same.
        """
        for d in local_dev_map1:
            if not d in local_dev_map2:
                return False

            dev1 = dict(local_dev_map1[d])
            dev2 = dict(local_dev_map2[d])

            if cmp(dev1, dev2):
                return False

        return True 


    @staticmethod
    def get_dev_config_maps(dev_conf_xml_file):
        DEVINFO_XML_ROOT = DeviceMap.XML_CONSTS["HSBLK_DEVINFO_ROOT_ELEM"]
        DISK_XML_ELEMENT = DeviceMap.XML_CONSTS["ELEM_SINGLE_DISK"]
        DEVTYPE_ATTR = "@%s" % (DeviceMap.XML_CONSTS["DISK_ATTR_DEVTYPE"])
        CVDRIVE_ATTR = "@%s" % (DeviceMap.XML_CONSTS["DISK_ATTR_CVDRIVE"])
        DEVNAME_ATTR = "@%s" % (DeviceMap.XML_CONSTS["DISK_ATTR_DEVNAME"])
        DEVSIZE_ATTR = "@%s" % (DeviceMap.XML_CONSTS["DISK_ATTR_DEVSIZE"])
        DEVALIAS_ATTR = "@%s" % (DeviceMap.XML_CONSTS["DISK_ATTR_DEVALIAS"])

        MNTPATH_XML_ELEMENT = DeviceMap.XML_CONSTS["ELEM_MNTPATH"]
        MNTPATH_TYPE_ATTR = "@%s" % (DeviceMap.XML_CONSTS["MNTPATH_ATTR_TYPE"])
        MNTPATH_BLKDEV_ATTR = "@%s" % (DeviceMap.XML_CONSTS["MNTPATH_ATTR_BLKDEV"])
        MNTPATH_PATH_ATTR = "@%s" % (DeviceMap.XML_CONSTS["MNTPATH_ATTR_MNTPATH"])
        MNTPATH_FSTYPE_ATTR = "@%s" % (DeviceMap.XML_CONSTS["MNTPATH_ATTR_FSTYPE"])
        MNTPATH_FSARGS_ATTR = "@%s" % (DeviceMap.XML_CONSTS["MNTPATH_ATTR_FSARGS"])
        
        DRIVESET_XML_ELEMENT =  DeviceMap.XML_CONSTS["ELEM_DRIVESET"]
        DRIVESET_TYPE_ATTR = "@%s" % (DeviceMap.XML_CONSTS["DRIVESET_ATTR_TYPE"])

        BOOTPARTDRIVE_XML_ELEMENT = DeviceMap.XML_CONSTS["ELEM_BOOTPARTDRIVE"]
        BOOTPARTDRIVE_DEVNAME_ATTR = "@%s" % (DeviceMap.XML_CONSTS["BOOTPARTDRIVE_ATTR_DEVNAME"])

        ARCH_XML_ELEMENT = DeviceMap.XML_CONSTS["ELEM_ARCHITECTURE"]
        ARCH_TYPE_ATTR = "@%s" % (DeviceMap.XML_CONSTS["ARCH_ATTR_TYPE"])

        BOOTMEDIA_XML_ELEMENT = DeviceMap.XML_CONSTS["ELEM_BOOTMEDIA"]
        BOOTMEDIA_DEVNAME_ATTR = "@%s" % (DeviceMap.XML_CONSTS["BOOTMEDIA_ATTR_DEVNAME"])
        BOOTMEDIA_PXEBOOT_ATTR = "@%s" % (DeviceMap.XML_CONSTS["BOOTMEDIA_ATTR_PXEBOOT"])
        BOOTMEDIA_NFSIP_ATTR = "@%s" % (DeviceMap.XML_CONSTS["BOOTMEDIA_ATTR_NFSIP"])
        BOOTMEDIA_NFSDIR_ATTR = "@%s" % (DeviceMap.XML_CONSTS["BOOTMEDIA_ATTR_NFSDIR"])

        USB_XML_ELEMENT = DeviceMap.XML_CONSTS["ELEM_USB"]
        USB_DEVNAME_ATTR = "@%s" % (DeviceMap.XML_CONSTS["USB_ATTR_DEVNAME"])
        USB_MNTPATH_ATTR = "@%s" % (DeviceMap.XML_CONSTS["USB_ATTR_MNTPATH"])

        OVA_XML_ELEMENT = DeviceMap.XML_CONSTS["ELEM_OVA"]
        OVA_TYPE_ATTR = "@%s" % (DeviceMap.XML_CONSTS["OVA_ATTR_TYPE"])
        OVA_SRCPATH_ATTR = "@%s" % (DeviceMap.XML_CONSTS["OVA_ATTR_SRCPATH"])
        OVA_DESTPATH_ATTR = "@%s" % (DeviceMap.XML_CONSTS["OVA_ATTR_DESTPATH"])
        OVA_MD5SUM_ATTR = "@%s" % (DeviceMap.XML_CONSTS["OVA_ATTR_MD5SUM"])
        OVA_SKIPCHK_ATTR = "@%s" % (DeviceMap.XML_CONSTS["OVA_ATTR_SKIPCHECK"])

        doc = read_xml_file(dev_conf_xml_file)
        docRoot = doc[DEVINFO_XML_ROOT]

        #Sample Dev Map:
        #{ u'/dev/sdd': {'cvdrive': u'none', 'type': u'sas', 'devsize': u'100G'} }
        elem = docRoot[DISK_XML_ELEMENT]
        diskElementList = []
        if isinstance(elem, list):
            diskElementList.extend(elem)
        else:
            diskElementList.append(elem)

        dev_map = {}
        for d in diskElementList:
            inner_d = { DeviceMap.KEY_DEVTYPE : d[DEVTYPE_ATTR], 
                    DeviceMap.KEY_CVDRIVE : d[CVDRIVE_ATTR],
                    DeviceMap.KEY_DEVSIZE : d[DEVSIZE_ATTR],
                    DeviceMap.KEY_DEVALIAS : d[DEVALIAS_ATTR] }
            dev_map[d[DEVNAME_ATTR]] = inner_d

        #Sample Conf Map:
        #{   'Usb': {'devname': u'/dev/sdf1'}, 
        #    'BootMedia': {'nfsip':'xx.xx.xx.xx', 'nfsdir':'/xyz', 'devname':'', 'pxeboot':'1'}, 
        #    'DriveSet': {'type':'same'}, 
        #    'Ova':[{'md5sum':'b3308e67a014f28c08814bcc3349a943', 'type':'vm'}, 
        #            {'md5sum': 'b3308e67a014f28c08814bcc3349a943', 'type': 'he'}], 
        #    'Architecture': {'type':'Reference'}, 
        #    'BootPartDrive': {'devname':'/dev/sda'}
        #}
        cnf_map = {}
        if DRIVESET_XML_ELEMENT in docRoot:
            d = docRoot[DRIVESET_XML_ELEMENT]
            cnf_map[DeviceMap.KEY_DRIVESET] = { DeviceMap.KEY_DRIVESET_TYPE : d[DRIVESET_TYPE_ATTR] }
 
        if BOOTPARTDRIVE_XML_ELEMENT in docRoot:
            d = docRoot[BOOTPARTDRIVE_XML_ELEMENT]
            cnf_map[DeviceMap.KEY_BOOTPARTDRIVE] = { DeviceMap.KEY_BOOTPARTDRIVE_DEVNAME : d[BOOTPARTDRIVE_DEVNAME_ATTR] }

        if BOOTMEDIA_XML_ELEMENT in docRoot:
            d = docRoot[BOOTMEDIA_XML_ELEMENT]
            bm_attr_map = {}

            if BOOTMEDIA_PXEBOOT_ATTR in d:
                bm_attr_map.update({DeviceMap.KEY_BOOTMEDIA_PXEBOOT: d[BOOTMEDIA_PXEBOOT_ATTR]})

            if BOOTMEDIA_DEVNAME_ATTR in d:
                bm_attr_map.update({DeviceMap.KEY_BOOTMEDIA_DEVNAME: d[BOOTMEDIA_DEVNAME_ATTR]})

            if BOOTMEDIA_NFSIP_ATTR in d:
                bm_attr_map.update({DeviceMap.KEY_BOOTMEDIA_NFSIP: d[BOOTMEDIA_NFSIP_ATTR]})

            if BOOTMEDIA_NFSDIR_ATTR in d:
                bm_attr_map.update({DeviceMap.KEY_BOOTMEDIA_NFSDIR: d[BOOTMEDIA_NFSDIR_ATTR]})

            cnf_map[DeviceMap.KEY_BOOTMEDIA] = bm_attr_map

        if ARCH_XML_ELEMENT in docRoot:
            d = docRoot[ARCH_XML_ELEMENT]
            cnf_map[DeviceMap.KEY_ARCH] = {DeviceMap.KEY_ARCH_TYPE: d[ARCH_TYPE_ATTR]}

        if USB_XML_ELEMENT in docRoot:
            d = docRoot[USB_XML_ELEMENT]
            cnf_map[DeviceMap.KEY_USB] = { DeviceMap.KEY_USB_DEVNAME: d[USB_DEVNAME_ATTR] }
            if USB_MNTPATH_ATTR in d:
                cnf_map[DeviceMap.KEY_USB].update({DeviceMap.KEY_USB_MNTPATH: d[USB_ATTR_MNTPATH]})

        if OVA_XML_ELEMENT in docRoot:
            ml = []
            if isinstance(docRoot[OVA_XML_ELEMENT], list):
                ml.extend(docRoot[OVA_XML_ELEMENT])
            else:
                ml.append(docRoot[OVA_XML_ELEMENT])

            cnf_map[DeviceMap.KEY_OVA] = []
            for ova in ml:
                ova_attr = {}
                if OVA_TYPE_ATTR in ova:
                    ova_attr.update({DeviceMap.KEY_OVA_TYPE: ova[OVA_TYPE_ATTR]})

                if OVA_MD5SUM_ATTR in ova:
                    ova_attr.update({DeviceMap.KEY_OVA_MD5SUM: ova[OVA_MD5SUM_ATTR]})

                if OVA_SRCPATH_ATTR in ova:
                    ova_attr.update({DeviceMap.KEY_OVA_SRCPATH: ova[OVA_SRCPATH_ATTR]})

                if OVA_DESTPATH_ATTR in ova:
                    ova_attr.update({DeviceMap.KEY_OVA_DESTPATH: ova[OVA_DESTPATH_ATTR]})

                if OVA_SKIPCHK_ATTR in ova:
                    ova_attr.update({DeviceMap.KEY_OVA_SKIPCHK: ova[OVA_SKIPCHK_ATTR]})

                cnf_map[DeviceMap.KEY_OVA].append(ova_attr)

        #Sample Mount Map:
        #{'/ws/ddb':{'blkdev':'/dev/vg/ddb','type':'meta','fsargs':'noatime,inode64','fstype':'xfs'}}
        mnt_map = {}
        if MNTPATH_XML_ELEMENT in docRoot:
            ml = []
            if isinstance(docRoot[MNTPATH_XML_ELEMENT], list):
                ml.extend(docRoot[MNTPATH_XML_ELEMENT])
            else:
                ml.append(docRoot[MNTPATH_XML_ELEMENT])

            for m in ml:
                inner_m = { DeviceMap.KEY_MNTTYPE : m[MNTPATH_TYPE_ATTR], 
                    DeviceMap.KEY_MNTBLKDEV : m[MNTPATH_BLKDEV_ATTR], 
                    DeviceMap.KEY_MNTFSTYPE : m[MNTPATH_FSTYPE_ATTR], 
                    DeviceMap.KEY_MNTFSARGS: m[MNTPATH_FSARGS_ATTR] }

                mnt_map[m[MNTPATH_PATH_ATTR]] = inner_m

        return dev_map, cnf_map, mnt_map


    def __read_user_config(self):
        global input_autoconf_file
        return read_xml_file(input_autoconf_file)


    def __check_autoconf(self):
        global is_autoconf_mode
        if not is_autoconf_mode:
            return 

        doc = self.__read_user_config()
        HSUSER_XML_ROOT = DeviceMap.XML_CONSTS["HSUSER_CONFIG_ROOT_ELEM"]
        if not HSUSER_XML_ROOT in doc:
            return

        INSTALL_XML_ELEMENT =  DeviceMap.XML_CONSTS["ELEM_INSTALL"]
        docroot = doc[HSUSER_XML_ROOT]
        if not INSTALL_XML_ELEMENT in docroot:
            return

        global autoconf_action
        INSTALL_TYPE_ATTR = "@%s" % (DeviceMap.XML_CONSTS["INSTALL_ATTR_TYPE"])
        autoconf_action = docroot[INSTALL_XML_ELEMENT][INSTALL_TYPE_ATTR] 
        

    def __get_device_map(self):
        global input_devconfig_file
        self.dev_map, self.cnf_map, self.mnt_map = DeviceMap.get_dev_config_maps(input_devconfig_file)


    def is_ref_arch(self):
        refArchConst = DeviceMap.XML_CONSTS["STR_REFERENCE"]
        if self.cnf_map[DeviceMap.KEY_ARCH][DeviceMap.KEY_ARCH_TYPE] == refArchConst:
            return True
        return False


    def is_appliance_arch(self):
        applianceArchConst = DeviceMap.XML_CONSTS["STR_APPLIANCE"]
        if self.cnf_map[DeviceMap.KEY_ARCH][DeviceMap.KEY_ARCH_TYPE] == applianceArchConst:
            return True
        return False


    def get_dev_map(self):
        return self.dev_map


    def get_mnt_map(self):
        return self.mnt_map


    def get_cnf_map(self):
        return self.cnf_map


    def get_extended_disk_name(self, disk_name):
        """ Appends disk type to the given diskName """
        if not self.dev_map:
            return disk_name

        if not disk_name in self.dev_map:
            return disk_name
    
        return disk_name + " [type: %s | size: %s]" % (self.dev_map[disk_name][DeviceMap.KEY_DEVTYPE], 
                                        self.dev_map[disk_name][DeviceMap.KEY_DEVSIZE])

    
    def __is_data_disk(self, diskName):
        return diskName in self.dev_map and self.dev_map[diskName][DeviceMap.KEY_CVDRIVE] == "data"


    def __is_meta_disk(self, diskName):
        return diskName in self.dev_map and ("meta" in self.dev_map[diskName][DeviceMap.KEY_CVDRIVE])


    def __is_os_disk(self, diskName):
        return diskName in self.dev_map and ("os" in self.dev_map[diskName][DeviceMap.KEY_CVDRIVE])


    def is_cv_disk(self, diskName):
        return self.__is_data_disk(diskName) or self.__is_meta_disk(diskName)

    
    def get_cv_disks(self):
        if not self.dev_map:
            return []
    
        return [disk for disk in sorted(self.dev_map.keys()) if self.is_cv_disk(disk)]
   

    def get_data_disks(self):
        return [disk for disk in sorted(self.dev_map.keys()) if self.__is_data_disk(disk)]
 

    def get_meta_disks(self):
        return [disk for disk in sorted(self.dev_map.keys()) if self.__is_meta_disk(disk)]


    def get_os_disks(self):
        return [disk for disk in sorted(self.dev_map.keys()) if self.__is_os_disk(disk)]


    def is_drive_set_same(self):
        #checks if os and metadata drives are same
        if DeviceMap.KEY_DRIVESET in self.cnf_map: 
            return self.cnf_map[DeviceMap.KEY_DRIVESET][DeviceMap.KEY_DRIVESET_TYPE] == "same"

        return False


    def get_disks(self):
        return [] if not self.dev_map else sorted(self.dev_map.keys())

    def compare_disktype(self, diskName, typeStr):
        return diskName in self.dev_map and self.dev_map[diskName][DeviceMap.KEY_DEVTYPE] == typeStr

    def get_disk_type(self, diskName):
        if not self.dev_map or not diskName in self.dev_map:
            return ""

        if self.__is_meta_disk(diskName):
            if self.__is_os_disk(diskName):
                return "meta,os"
            return "meta"

        if self.__is_data_disk(diskName):
            return "data"

        return "raw "


    def get_disks_with_init_status(self, diskList = None):
        if not self.dev_map:
            return []
        
        disk_list = []
        disks = self.dev_map.keys() if not diskList else diskList
        for disk in sorted(disks):
            disk_type = self.get_disk_type(disk)
            disk_list.append(disk + " [%s]" % (disk_type))

        return disk_list


    def get_mount_path(self, disk):
        for mpath in self.mnt_map:
            if self.mnt_map[mpath][DeviceMap.KEY_MNTBLKDEV] == disk:
                return mpath

        return ""


    def get_meta_mount_paths(self):
        meta_mnt_paths = []
        meta_blk_devs = { "index" : "Index Cache", "ddb" : "DDB" }
        for mpath in self.mnt_map:
            mnt_type = self.mnt_map[mpath][DeviceMap.KEY_MNTTYPE]
            blk_dev = self.mnt_map[mpath][DeviceMap.KEY_MNTBLKDEV] 
            if mnt_type != "meta":
                continue

            for key in meta_blk_devs:
                if key in blk_dev:
                    meta_mnt_paths.append("%s: %s" % (meta_blk_devs[key], mpath))
        
        return meta_mnt_paths


    def get_extended_disk_names(self, diskNames, showMountPaths=True):
        if not self.dev_map:
            return []
        
        disk_list = []
        for disk in diskNames:
            if not disk in self.dev_map:
                continue

            minfo = ""
            mountpath = self.get_mount_path(disk)
            if len(mountpath) > 0:
                minfo = " | mount: %s" % (mountpath)

            dmap = self.dev_map[disk]
            if showMountPaths:
                disk_inf = disk + " [type: %s | size: %s%s]" % (dmap[DeviceMap.KEY_DEVTYPE], 
                   dmap[DeviceMap.KEY_DEVSIZE], minfo)
            else:
                disk_inf = disk + " [type: %s | size: %s]" % (dmap[DeviceMap.KEY_DEVTYPE], 
                   dmap[DeviceMap.KEY_DEVSIZE])

            disk_list.append(disk_inf)

        return disk_list


    def get_extended_disk_names_with_init_status(self, diskNames):
        if not self.dev_map:
            return []
        
        disk_list = []
        for disk in diskNames:
            if not disk in self.dev_map:
                continue

            minfo = ""
            mountpath = self.get_mount_path(disk)
            if len(mountpath) > 0:
                minfo = " | mount: %s" % (mountpath)

            cv_disk_info = ""
            cv_disk_type = self.get_disk_type(disk)
            if len(cv_disk_type) > 0:
                cv_disk_info = "%s | " %(cv_disk_type)

            dmap = self.dev_map[disk]
            disk_inf = disk + " [%stype: %s | size: %s%s]" % (cv_disk_info, 
                    dmap[DeviceMap.KEY_DEVTYPE], dmap[DeviceMap.KEY_DEVSIZE], minfo)
            disk_list.append(disk_inf)

        return disk_list

#} class DeviceMap

device_map = None

def getDiskNameListFormExtendedDiskNames(extDiskNameList): 
    return [ disk.split('[')[0].strip() for disk in extDiskNameList ]


def get_current_controller():
    if the_app:
        return the_app.protocol.current_controller

    return None

#{
class ApplianceSetupNavigator:
    def __init__(self):
        """Initialize the Appliance Setup navigator"""
        self.controller_list = [
            'NodeTypeSelectionController',
            'SwiftConfigController',
            'SystemDiskSelectionController',
            'DDBIndexDiskSelectionController',
            'DataDiskSelectionController',
            'SummaryController',
        ]

        self.controller_map = { 
            'NodeTypeSelectionController' : None,
            'SwiftConfigController' : None,
            'SystemDiskSelectionController' : None,
            'DDBIndexDiskSelectionController': None,
            'DataDiskSelectionController':  None,
            'SummaryController' : None,
            'PreservationController' : None,
            'ConfigurationDisplayController' : None,
            'MultiNodeConfigContorller' : None,
            'AutoConfConfirmationController' : None,
            'InstallationProgressController' : None
        }

        self.controller_list_iter = -1
        self.current_controller = None


    def __get_controller(self, controller_name):
        controller_name = controller_name.split(".")[-1]

        if self.controller_map[controller_name]:
            return self.controller_map[controller_name]
        else:
            cur_import_module = __import__(__name__, globals(), locals(), __name__)
            controller = getattr(cur_import_module, controller_name)()
            self.controller_map[controller_name] = controller
            return controller


    def get_next_controller(self, currController):
        global device_map 
        global config_file_found
 
        if isinstance(currController, ConfigurationDisplayController):
            if config_file_found:
                return self.__get_controller("InstallationProgressController")

            return self.__get_controller("PreservationController")

        if isinstance(currController, AutoConfConfirmationController):
            return self.__get_controller("InstallationProgressController")

        global reinit_drives
        if isinstance(currController, PreservationController):
            global is_autoconf_mode
            global autoconf_action
            if is_autoconf_mode:
                return self.__get_controller("InstallationProgressController")

            if reinit_drives or not device_map.get_meta_disks():
                return self.__get_controller("NodeTypeSelectionController")

            if device_map.is_drive_set_same():
                return self.__get_controller("SummaryController")

            return self.__get_controller("SystemDiskSelectionController")

        if isinstance(currController, NodeTypeSelectionController):
            if config_file_found:
                return self.__get_controller("InstallationProgressController")

            global is_cluster_installation
            if is_cluster_installation:
                return self.__get_controller("MultiNodeConfigContorller")

            return self.__get_controller("SystemDiskSelectionController")

        if isinstance(currController, SwiftConfigController):
            if config_file_found:
                return self.__get_controller("InstallationProgressController")

            return self.__get_controller("MultiNodeConfigContorller")

        if isinstance(currController, MultiNodeConfigContorller):
            return self.__get_controller("SystemDiskSelectionController")

        if isinstance(currController, SystemDiskSelectionController):

            global appliance_swift
            if appliance_swift:
                #For Swift Appliance, go directly to DataDisk screen
                return self.__get_controller("DataDiskSelectionController")

            if device_map.get_meta_disks() and not reinit_drives:
                return self.__get_controller("SummaryController")
            
            global use_same_sys_meta_drives
            if not device_map.get_meta_disks() and not reinit_drives and use_same_sys_meta_drives:
                return self.__get_controller("SummaryController")
            
            if use_same_sys_meta_drives and reinit_drives:
                return self.__get_controller("DataDiskSelectionController")

            return self.__get_controller("DDBIndexDiskSelectionController")
                
        if isinstance(currController, DDBIndexDiskSelectionController):
            if device_map.cv_disks_present() and not reinit_drives:
                return self.__get_controller("SummaryController")

            return self.__get_controller("DataDiskSelectionController")

        if isinstance(currController, DataDiskSelectionController):
            return self.__get_controller("SummaryController")

        if isinstance(currController, SummaryController):
            return self.__get_controller("InstallationProgressController")
            #return None

        if isinstance(currController, InstallationProgressController):
            return None

        return None


    def get_prev_controller(self, currController):
        if isinstance(currController, NodeTypeSelectionController):
            return None

        if isinstance(currController, SystemDiskSelectionController):
            return self.__get_controller("NodeTypeSelectionController")

        if isinstance(currController, DDBIndexDiskSelectionController):
            return self.__get_controller("SystemDiskSelectionController")

        if isinstance(currController, DataDiskSelectionController):
            return self.__get_controller("DDBIndexDiskSelectionController")

        if isinstance(currController, SummaryController):
            return self.__get_controller("DataDiskSelectionController")


    def get_next(self):
        """return the next controller"""
        if self.controller_list_iter == -1:
            return self.get_start_node()

        if self.controller_list_iter + 1 == len(self.controller_list):
            return None

        self.controller_list_iter = self.controller_list_iter + 1 
        return self.__get_controller(self.controller_list[self.controller_list_iter])


    def get_prev(self):
        """return the prev controller"""
        if self.controller_list_iter - 1 < 0:
            return None

        self.controller_list_iter = self.controller_list_iter - 1 
        return self.__get_controller(self.controller_list[self.controller_list_iter])


    def get_start_node(self):
        if self.controller_list_iter == -1:
            global is_autoconf_mode
            global autoconf_action
            global device_map

            #Create hsinstallation.xml file with Initialize option in case of fresh install
            if is_autoconf_mode and not device_map.cv_disks_present():
                write_installation_option_file()
                XMLConsts = device_map.get_XML_consts()
                autoconf_action = XMLConsts["STR_INITIALIZE"]

            if is_autoconf_mode and autoconf_action:
                return self.__get_controller("AutoConfConfirmationController")
   
            #If previous configuration exists, show page to display the config.
            global appliance_swift
            if not appliance_swift:
                #Skip this check for Swift Appliance
                if device_map.cv_disks_present():
                    return self.__get_controller("ConfigurationDisplayController")

            #Fresh install, check for smart config
            #If smart config returns true, goto page AutoConfConfirmationController
            global input_devconfig_file
            global input_autoconf_file
            if input_autoconf_file:
                smartconf=bmr.smartconfig_check(input_devconfig_file, input_autoconf_file, True, False)
                if smartconf:
                    is_autoconf_mode = True
                    XMLConsts = device_map.get_XML_consts()
                    autoconf_action = XMLConsts["STR_INITIALIZE"]
                    return self.__get_controller("AutoConfConfirmationController")

            self.controller_list_iter = self.controller_list_iter + 1
            if appliance_swift:
                #For Swift, select first node as Swift Config one i.e. index 1
                self.controller_list_iter = self.controller_list_iter + 1
            return self.__get_controller(self.controller_list[self.controller_list_iter])
        else:
            return None

#} class ApplianceSetupNavigator

appliance_setup_navigator = ApplianceSetupNavigator()


#{
class InstallationProgressController(Controller):
    """ UI class for showing hyperscale setup progress """

    def __init__(self):
        self.log = cvpylogger.getLog()
        self.prev_controller = None
        self.next_controller = None
        self.executors = {}
        self.render_obj = None
        self.reboot_requested = False


    def render(self):
        self.prev_controller = None

        global cluster_nodes
        hosts = ["localhost"] + cluster_nodes

        for host in hosts:
            self.executors[host] = CommandExecutor(name=host)

        sorted_vals = [self.executors[k] for k in sorted(self.executors.keys())]

        render_obj = Render(_("PROGRESS_TITLE"), self.__doc__, *sorted_vals)
        #render_obj = Render(_("PROGRESS_TITLE"), self.__doc__, *self.executors.values())
        render_obj.set_next_button_label(_("FINISH_BUTTON_TEXT"))
        #render_obj.disable_next_button()
        render_obj.disable_quit_button()
        render_obj.disable_prev_button()
        self.render_obj = render_obj
        return render_obj

    def set_alarm_in(self):
        REBOOT_FILE_MONITOR_INTERVAL = 2

        def monitorRebootFile(loop, user_data=None):
            global REBOOT_NOTICE_FILE
            if os.path.isfile(REBOOT_NOTICE_FILE):
                self.log.info("Found reboot file: %s. Proceeding to reboot." % (REBOOT_NOTICE_FILE))
                self.reboot_requested = True
                Render.main_window_instance.next_button._emit('click')
                return

            currController = get_current_controller()
            if currController and isinstance(currController, InstallationProgressController):
                self.callBackIn(REBOOT_FILE_MONITOR_INTERVAL, monitorRebootFile)
            else:
                self.log.info("Stopping the recurring alarm to monitor reboot file.")


        global config_file_found
        if config_file_found:
            self.log.info("Starting recurring alarm to monitor reboot file.")
            self.callBackIn(REBOOT_FILE_MONITOR_INTERVAL, monitorRebootFile)


    def process(self, *args):
        global config_file_found
        if config_file_found:
            return True

        global cluster_nodes_install_status
        self.log.info("Cluster nodes installation status: %s" % (cluster_nodes_install_status))
        global cluster_nodes
        for node in cluster_nodes:
            if node == "localhost":
                continue

            if cluster_nodes_install_status[node]:
                self.log.info("Sending reboot notice to host: %s." % (node))
                chost = Host(node)
                status, msg = chost.send_shutdown_notice()
                if not status:
                    self.log.error("Failed to send reboot notice to host [%s] with error [%s]." % (node, msg))

 
        return True


    def __install_hyperscale(self, host, progress_file):
        #return 0
        PYTHON_INTERPRETER = "/bin/python"
        PYTHON_PATH = "/"
        INSTALL_SCRIPT = "/cvhyperscale_install.py"

        if host == "localhost":
            cmd_env = dict(os.environ, **{'PYTHONPATH': PYTHON_PATH})
            cmd_params = [PYTHON_INTERPRETER, INSTALL_SCRIPT, progress_file]
            global is_autoconf_mode
            if is_autoconf_mode:
                global input_autoconf_file
                cmd_params.extend([input_autoconf_file])
                if not autoconf_action:
                    global INSTALL_OPTION_FILE
                    cmd_params.extend([INSTALL_OPTION_FILE])

            p1 = subprocess.Popen(cmd_params, env=cmd_env)
            out, err = p1.communicate()
            return p1.returncode

        return 0


    def __pull_install_progress_file(self, host, progress_file):
        if host == "localhost":
            #nothing to be done for localhost
            return True, "Successfully downloaded progress file."
       
        HOST_PROGRESS_FILE = "/tmp/.localhost.install.progress"
        chost = Host(host)
        status, msg = chost.download_file(HOST_PROGRESS_FILE, progress_file)
        return status, msg


    def __monitor_hyperscale_install(self, thr, host, progress_file):
        #time.sleep(2)
        #return True, "Success!!!"

        last_status = ""
        last_progress = ""
        total_wait_time = 0
        start_time = int(time.time())
        MAX_WAIT_TIME = 720 * 60
        MAX_PROGRESS_DOWNLOAD_ATTEMPTS = 500;
        
        failed_progress_pull_attempts = 0;
    
        thLog = cvpylogger.getLog()
        host_executor = self.executors[host]
        global cluster_nodes_install_status
        while True:
            sleeptime = 1
            time.sleep(sleeptime)

            status, msg = self.__pull_install_progress_file(host, progress_file)
            if not status:
                thLog.error("Failed to download install progress file [%s] for host [%s] with error [%s]" % (progress_file, host, msg))
                failed_progress_pull_attempts += 1
                if failed_progress_pull_attempts >= MAX_PROGRESS_DOWNLOAD_ATTEMPTS:
                    cluster_nodes_install_status[host] = False
                    install_status = "Failed to retrieve installation progress status"
                    host_executor.update_status(install_status, 100)
                    return False, "Maximum number of attempts to download progress file for host [%s] exceeded" % (host)
            else:
                failed_progress_pull_attempts = 0

            total_wait_time = int(time.time()) - start_time
            if total_wait_time >= MAX_WAIT_TIME:
                cluster_nodes_install_status[host] = False
                install_status = "Maximum installation time exceeded"
                host_executor.update_status(install_status, 100)
                return False, "Maximum installation time exceeded"
    
            if not os.path.isfile(progress_file):     
                continue
    
            lines = []
            try: 
                with open(progress_file, 'r+') as pf:
                    lines = pf.readlines()
            except:
                thLog.error("Exception while reading progress file for host [%s]" % (host))
        
    
            if len(lines) == 0 or len(lines) < 2:
                continue
    
            lines = [line.strip() for line in lines]
            status = lines[0].split(":",1)[1].strip()
            if status != last_status:
                host_executor.update_caption(status)
    
            progress = lines[1].split(":",1)[1].strip()
            if progress != last_progress: 
                #print "Current Progress: %s%s" % (progress, "%")
                host_executor.update_progress(progress)
    
            if progress == "100":
                if not "FAIL" in status:
                    host_executor.update_caption("Installation completed successfully.")
                    cluster_nodes_install_status[host] = True
                    return True, "Success!!!"
                else:
                    cluster_nodes_install_status[host] = False
                    return False, "Installation Failed"

            last_status = status
            last_progress = progress
    
        return True, "Success!!!"

    
    def __run_hyperscale_install(self, host):
        PROGRESS_FILE = '/tmp/.%s.install.progress' % (host)
        thr = threading.Thread(target=self.__install_hyperscale, args=(host,PROGRESS_FILE,))
        thr.start()
    
        ret, msg = self.__monitor_hyperscale_install(thr, host, PROGRESS_FILE)
        self.log.info("Host [%s] Installation Monitor Status: %s" % (host, msg))
        if not ret and host == "localhost":
            global localhost_install_status
            localhost_install_status = 1

        thr.join()
        return True


    def auto_process(self, args=None):
        self.render_obj.main_window.disable_buttons()

        def put_file(host, configFile):
            "Copy config file to host : "
            thLog = cvpylogger.getLog()
            if host != "localhost" :
                thLog.info("Copying %s config file to host: %s" % (configFile, host))
                chost = Host(host)
                status, msg = chost.upload_file(configFile, configFile)
                if not status:
                    host_executor = self.executors[host]
                    host_executor.delegate.update_caption(host + ": " + "Failed to copy config file")
                    thLog.error("Failed to copy config file [%s] to host [%s] with error [%s]" % (configFile, host, msg))
                    return False

                return True

            time.sleep(1)
            return True


        def monitor_progress(host, *args):
            "Monitor installation progress on host: "
            cvpylogger.getLog().info("%s: Monitoring installation progress" % (host))
            return self.__run_hyperscale_install(host)


        def tasks_error_callback(exceptionFunc, errorMessage='', isException=False):
            import traceback
            thLog = cvpylogger.getLog()
            self.render_obj.main_window.disable_next_button()
            self.render_obj.main_window.disable_prev_button()
            self.render_obj.main_window.enable_quit_button()
            self.render_obj.main_window.enable_buttons()
            if isinstance(errorMessage, dict):
                errorMessage = str(errorMessage)
            elif not errorMessage.strip():
                errorMessage = "Fatal Error"
                thLog.info("Failed function call : %s " % str(exceptionFunc))
            self.render_obj.main_window.alert(errorMessage)
            self.render_obj.main_window.update()

            if isException:
                thLog.error("Exception in ui.CommandExecutor while calling %s" % (str(exceptionFunc)))
                thLog.error(str("\n".join(traceback.format_stack())))

 
        hosts = sorted(self.executors.keys())
        global output_file
        for host in hosts:
            cmds = []
            global appliance_swift
            if appliance_swift:
                #For Swift Appliance, include transfer of swift config file
                if os.path.isfile(SWIFT_CONFIG_FILENAME):
                    cmds.append((put_file, (host, SWIFT_CONFIG_FILENAME), put_file.__doc__ + host))
            cmds.append((put_file, (host, output_file), put_file.__doc__ + host))
            cmds.append((monitor_progress, (host, ""), monitor_progress.__doc__ + host))

            self.executors[host].setErrorCallback(tasks_error_callback)
            self.executors[host].execute_all(cmds)
            self.executors[host].start()

        for host in hosts:
            self.executors[host].join()

        self.render_obj.main_window.enable_buttons()
        self.render_obj.main_window.update()
        if self.reboot_requested:
            self.render_obj.main_window.next_button._emit('click')
        # No need to wait for user to click Finish in case of Appliance
        # as Appliance imaging should be done without user intervention
        global device_map
        if device_map.is_appliance_arch():
            self.render_obj.main_window.next_button._emit('click')

        return True
    

    def next(self):
        self.next_controller = appliance_setup_navigator.get_next_controller(self)
        if self.next_controller:
            self.next_controller.prev_controller = self
        return self.next_controller


    def previous(self):
        return None

#} class InstallationProgressController

#{
class MultiNodeConfigContorller(Controller):
    """ UI class for collecting hyperscale cluster nodes information """
    def __init__(self):
        self.log = cvpylogger.getLog()
        self.prev_controller = None
        self.next_controller = None
        self.ActionOnly = False
        self.MAX_CLUSUTER_NODES = 5


    def render(self):
        multinode_setup_info = model.SimpleText(_("MULTI_NODE_INFO"))

        nodes_label = model.SimpleText(_("DISCOVERED_HOSTS"))
        node_list = model.CheckBoxList()
        global discovered_nodes
        for node in discovered_nodes:
            is_checked =  (node in cluster_nodes) if len(cluster_nodes) else True
            node_list.add_option(node, is_checked)
 
        render_obj = Render(_("MULTI_NODE_TITLE"), self.__doc__, multinode_setup_info, 
                        nodes_label, node_list)
        render_obj.disable_quit_button()
        return render_obj

    def _render(self):
        multinode_setup_info = model.SimpleText(_("MULTI_NODE_INFO"))

        global cluster_nodes
        divider = model.SimpleText("")
        def_ip_vals = cluster_nodes + [""]*(self.MAX_CLUSUTER_NODES-len(cluster_nodes))

        #node1_ip = model.IPAddress(_("MULTI_NODE_HOST1") + " ", defaultVal=def_ip_vals[0])
        #node2_ip = model.IPAddress(_("MULTI_NODE_HOST2") + " ", defaultVal=def_ip_vals[1])
        #node3_ip = model.IPAddress(_("MULTI_NODE_HOST3") + " ", defaultVal=def_ip_vals[2])
        #node4_ip = model.IPAddress(_("MULTI_NODE_HOST4") + " ", defaultVal=def_ip_vals[3])
        #node5_ip = model.IPAddress(_("MULTI_NODE_HOST5") + " ", defaultVal=def_ip_vals[4])

        node1_ip = model.TextOption(_("MULTI_NODE_HOST1")+" ", defaultVal=def_ip_vals[0])
        node2_ip = model.TextOption(_("MULTI_NODE_HOST2")+" ", defaultVal=def_ip_vals[1])
        node3_ip = model.TextOption(_("MULTI_NODE_HOST3")+" ", defaultVal=def_ip_vals[2])
        node4_ip = model.TextOption(_("MULTI_NODE_HOST4")+" ", defaultVal=def_ip_vals[3])
        node5_ip = model.TextOption(_("MULTI_NODE_HOST5")+" ", defaultVal=def_ip_vals[4])
    
        return Render(_("MULTI_NODE_TITLE"), self.__doc__, multinode_setup_info, 
                divider, node1_ip, node2_ip, node3_ip, node4_ip, node5_ip)


    def __validate_node(self, hostip):
        host = Host(hostip)

        if not host.validate():
            return False, "Failed to connect to cluster node %s" % (hostip)

        self.log.info("Successully validated connection to cluster node: " + hostip)
        global device_map
        local_devmap = device_map.get_dev_map()

        dev_map, cnf_map, mnt_map =  host.get_dev_conf_maps()
        if not dev_map:
            return False, "Failed to read block device config on cluster node %s" % (hostip)

        if not DeviceMap.is_blkdev_conf_similar(local_devmap, dev_map):
            return False, "Local block device config does not match with block device config on remote cluster node %s" % (hostip)
 
        return True, "Successfuly validated node %s" % (hostip)


    def process(self, selected_node_list=None):
        if selected_node_list == None or len(selected_node_list) == 0:
            error_message(self, "No nodes selected for installation")
            return False 

        self.log.info("Validating nodes: %s" % (",".join(selected_node_list)))
        for n in selected_node_list:
            valid_node, msg = self.__validate_node(n)
            if not valid_node:
                error_message(self, msg)
                return False

        global cluster_nodes
        cluster_nodes = selected_node_list
        self.log.info("Cluster nodes: %s" % (",".join(cluster_nodes)))

        global appliance_swift
        if appliance_swift:
            #Validate that number of nodes selected is more than or equal to replica count
            global ip_host_map
            global swiftmap
            if len(cluster_nodes) < (swiftmap.get_replica_count() - 1):
                self.log.info("Number of nodes selected [%d] is less than replica count [%d]" % (len(cluster_nodes), swiftmap.get_replica_count()))
                error_message(self, "Total number of nodes for installation should be more than or equal to Replica count [%d]" % (swiftmap.get_replica_count()))
                return False
            #set the hostnames of the IPs selected for installation
            swiftmap.set_hostname_list(cluster_nodes, ip_host_map)

        return True


    def _process(self, n1 = None, n2 = None, n3 = None, n4 = None, n5 = None):
        if not n1 and not n2 and not n3 and not n4 and not n5:
            error_message(self, "IP address is not provided for any of cluster nodes.")
            return False

        inp_nodes = [n1, n2, n3, n4, n5]
        valid_cluster_nodes = True
        invalid_addrs = []
        for n in inp_nodes:
            if n and len(n) > 0 and not wrappers.is_valid_ipv4(n):
                valid_cluster_nodes = False
                invalid_addrs.append(n) 

        if not valid_cluster_nodes:
            error_message(self, "Invalid IPv4 addresses: %s" %(",".join(invalid_addrs)))
            return False

        nodes = [n for n in inp_nodes if n != None and len(n)] 
        sorted_uniq_nodes = sorted(set(nodes))
        if len(sorted_uniq_nodes) != len(nodes):
            error_message(self, "Provided Node IP addresses are not distinct.")
            return False

        self.log.info("Validating nodes: %s" % (",".join(nodes)))

        for n in nodes:
            valid_node, msg = self.__validate_node(n)
            if not valid_node:
                error_message(self, msg)
                return False

        global cluster_nodes
        cluster_nodes = nodes
        self.log.info("Cluster nodes: %s" % (",".join(cluster_nodes)))
        return True


    def next(self):
        self.next_controller = appliance_setup_navigator.get_next_controller(self)
        if self.next_controller:
            self.next_controller.prev_controller = self
        return self.next_controller


    def previous(self):
        return self.prev_controller

#} class MultiNodeConfigContorller

#{
class ConfigurationDisplayController(Controller):
    """ UI class for displaying detected hyperscale appliance configuration """
    def __init__(self):
        self.log = cvpylogger.getLog()
        self.prev_controller = None
        self.next_controller = None
        self.ActionOnly = False


    def set_alarm_in(self):
        CONFIG_FILE_MONITOR_INTERVAL = 1

        def monitorConfigFile(loop, user_data=None):
            global output_file
            if os.path.isfile(output_file):
                global config_file_found
                config_file_found = True
                self.log.info("Found config file: %s. Proceeding to installation." % (output_file))
                Render.main_window_instance.next_button._emit('click')
                return

            currController = get_current_controller()
            if currController and isinstance(currController, ConfigurationDisplayController):
                self.callBackIn(CONFIG_FILE_MONITOR_INTERVAL, monitorConfigFile)
            else:
                self.log.info("Stopping the recurring alarm to monitor config file.")

        self.log.info("Starting recurring alarm to monitor config file.")
        self.callBackIn(CONFIG_FILE_MONITOR_INTERVAL, monitorConfigFile)


    def render(self):
        config_disp_info = model.SimpleText(_("CONFIG_DISP_INFO"))

        global device_map
        divider = model.SimpleText("")

        data_disks_label = get_formatted_label(_("DATA_DISKS")) 
        data_disks = get_list_box(device_map.get_extended_disk_names(device_map.get_data_disks()))

        meta_disks_label = get_formatted_label(_("DDB_IDXCACHE_DISKS")) 
        meta_disks = get_list_box(device_map.get_extended_disk_names(device_map.get_meta_disks()))

        mountpaths_label = get_formatted_label(_("META_MOUNT_PATHS")) 
        mount_paths = get_list_box(device_map.get_meta_mount_paths())

        render_obj = Render(_("CONFIG_DISP_TITLE"), self.__doc__, config_disp_info,
                data_disks_label, data_disks, meta_disks_label, meta_disks,
                mountpaths_label, mount_paths)
        render_obj.disable_quit_button()
        return render_obj


    def process(self, data_disks = None, meta_disks = None, mount_paths = None):
        return True


    def next(self):
        self.next_controller = appliance_setup_navigator.get_next_controller(self)
        if self.next_controller:
            self.next_controller.prev_controller = self
        return self.next_controller


    def previous(self):
        return self.prev_controller

#} class ConfigurationDisplayController


#{
class NodeTypeSelectionController(Controller):
    """ UI class for selecting node type data/control """
    def __init__(self):
        self.log = cvpylogger.getLog()
        self.prev_controller = None
        self.next_controller = None
        self.ActionOnly = False
        self.node_type = 0
        self.driveset_arg = 0
        self.dav_arg = 0


    def set_alarm_in(self):
        CONFIG_FILE_MONITOR_INTERVAL = 1

        def monitorConfigFile(loop, user_data=None):
            global output_file
            if os.path.isfile(output_file):
                global config_file_found
                config_file_found = True
                self.log.info("Found config file: %s. Proceeding to installation." % (output_file))
                Render.main_window_instance.next_button._emit('click')
                return

            currController = get_current_controller()
            if currController and isinstance(currController, NodeTypeSelectionController):
                self.callBackIn(CONFIG_FILE_MONITOR_INTERVAL, monitorConfigFile)
            else:
                self.log.info("Stopping the recurring alarm to monitor config file.")

        self.log.info("Starting recurring alarm to monitor config file.")
        self.callBackIn(CONFIG_FILE_MONITOR_INTERVAL, monitorConfigFile)


    def render(self):
        divider = model.SimpleText("")
        node_select_info_msg = model.SimpleText(_("NODECONFIG_SELECT_INFO"))

        render_params = [_("NODETYPE_SELECT_TILE"), self.__doc__,
                            node_select_info_msg]

        argcnt = 0;
        self.driveset_arg = 0
        self.dav_arg = 0
        global reinit_drives
        if reinit_drives:                
            cluster_info_msg = model.SimpleText(_("CLUSTER_INSTALL_INFO"))
            global is_cluster_installation
            cluster_check = model.BoolOption(_("MULTI_NODE_INSTALL"),
                                            selected=is_cluster_installation)

            nw_if_label = model.SimpleText(_("NETWORK_INTERFACE") + ":")
            nw_if_listbox_items = []
            ip_addr_list = []
            for nwif in wrappers.get_nw_intfx_list():
                ipaddr = wrappers.get_ip_address_of_nw_interface(nwif)
                if not len(ipaddr): continue
                ip_addr_list.append(ipaddr)
                macaddr = wrappers.get_hw_address(nwif)
                list_item = "%s [IP - %s | MAC - %s]" %(nwif, ipaddr, macaddr)
                nw_if_listbox_items.append(list_item)

            global local_ip_addresses
            local_ip_addresses = sorted(set(ip_addr_list))
            nw_ifs_listbox = get_list_box(nw_if_listbox_items)

            render_params.extend([cluster_info_msg, cluster_check, 
                                    nw_if_label, nw_ifs_listbox]) 

            argcnt = 2;

        global device_map
        numDisksFound = len(device_map.get_disks())
 
        global use_same_sys_meta_drives
        global forced_same_sys_meta_drives
        if not forced_same_sys_meta_drives:
            same_disk_info_msg = model.SimpleText(_("SAME_SYS_META_DISK_INFO"))
            same_disk_check = model.BoolOption(_("SAME_SYS_META_DISK"), 
                                                selected=use_same_sys_meta_drives) 
            render_params.extend([same_disk_info_msg, same_disk_check])
            argcnt += 1
            self.driveset_arg = argcnt;

        num_disks = "%d" % (numDisksFound)
        numdisks_msg = model.SimpleText(_("NUM_DISKS_DETECTED") + ": " + num_disks)
        render_params.extend([numdisks_msg])

        is_reference_arch = device_map.is_ref_arch()
        if is_reference_arch:
            global configure_dav
            config_dav_info_msg = model.SimpleText(_("DAV_CONF_INFO"))
            dav_check = model.BoolOption(_("DAV_CONF_MSG"),
                                                selected=configure_dav)
            if enable_dav:
                render_params.extend([config_dav_info_msg, dav_check]) 
                argcnt += 1
                self.dav_arg = argcnt;

        self.log.info("Driveset_arg: %s" % (str(self.driveset_arg)))
        self.log.info("dav_arg: %s" % (str(self.dav_arg)))

        render_obj = Render(*render_params)
        render_obj.disable_quit_button()
        return render_obj


    def process(self, *args):
        global is_control_node
        is_control_node = True

        self.log.info("Num of args: %s" % (str(len(args))))

        global reinit_drives
        if reinit_drives:
            global is_cluster_installation
            is_cluster_installation = args[0]
            self.log.info("Multi Node Install: %s" % (str(is_cluster_installation)))

            if is_cluster_installation:
                rc, hosts = discover_services()
                if not rc and len(hosts) > 0:
                    self.log.info("Discovered Hosts: %s" % (",".join(hosts)))
                    global discovered_nodes
                    discovered_nodes = hosts
                else:
                    error_message(self, "No hosts discovered for multinode installation.")
                    return False

        global use_same_sys_meta_drives
        if self.driveset_arg != 0 and isinstance(args[self.driveset_arg-1], bool):
            use_same_sys_meta_drives = args[self.driveset_arg-1]

        global forced_same_sys_meta_drives
        if forced_same_sys_meta_drives:
            use_same_sys_meta_drives = forced_same_sys_meta_drives
            
        self.log.info("Use same drive for system & metadata: %s" % (str(use_same_sys_meta_drives)))

        global configure_dav
        global device_map
        is_reference_arch = device_map.is_ref_arch()
        if is_reference_arch:
            if self.dav_arg != 0 and isinstance(args[self.dav_arg-1], bool):
                configure_dav = args[self.dav_arg-1]

        self.log.info("Configure Direct Access Volume: %s" % (str(configure_dav)))
        return True


    def next(self):
        self.next_controller = appliance_setup_navigator.get_next_controller(self)
        if self.next_controller:
            self.next_controller.prev_controller = self
        return self.next_controller


    def previous(self):
        return self.prev_controller

#} class NodeTypeSelectionController

#{
class SwiftConfigController(Controller):
    """ UI class for configuring Swift Appliance """
    def __init__(self):
        self.log = cvpylogger.getLog()
        self.prev_controller = None
        self.next_controller = None
        self.ActionOnly = False

    def set_alarm_in(self):
        CONFIG_FILE_MONITOR_INTERVAL = 1

        def monitorConfigFile(loop, user_data=None):
            global output_file
            if os.path.isfile(output_file):
                global config_file_found
                config_file_found = True
                self.log.info("Found config file: %s. Proceeding to installation." % (output_file))
                Render.main_window_instance.next_button._emit('click')
                return

            currController = get_current_controller()
            if currController and isinstance(currController, SwiftConfigController):
                self.callBackIn(CONFIG_FILE_MONITOR_INTERVAL, monitorConfigFile)
            else:
                self.log.info("Stopping the recurring alarm to monitor config file.")

        self.log.info("Starting recurring alarm to monitor config file.")
        self.callBackIn(CONFIG_FILE_MONITOR_INTERVAL, monitorConfigFile)


    def render(self):
        divider = model.SimpleText("")
        replica_select_info_msg = model.SimpleText(_("SWIFT_REPLICA_CNT_INFO"))

        node_group = model.NumericOption(_("SWIFT_REPLICA_CNT"))

        render_params = [_("NODETYPE_SELECT_TILE"), self.__doc__,
                            replica_select_info_msg, node_group]

        global swiftmap
        dpnw_if, spnw_if = swiftmap.get_nwif()
        nw_if_info_msg = model.SimpleText(_("NETWORK_INTERFACE"))
        dpnw_info_msg = model.SimpleText(_("DPNW_INTERFACE_INFO") + ": " + dpnw_if)
        spnw_info_msg = model.SimpleText(_("SPNW_INTERFACE_INFO") + ": " + spnw_if)

        render_params.extend([nw_if_info_msg, dpnw_info_msg, spnw_info_msg]) 

        nw_if_label = model.SimpleText(_("NETWORK_INTERFACE") + ":")
        nw_if_listbox_items = []
        ip_addr_list = []
        for nwif in wrappers.get_nw_intfx_list():
            ipaddr = wrappers.get_ip_address_of_nw_interface(nwif)
            if not len(ipaddr): continue
            ip_addr_list.append(ipaddr)
            macaddr = wrappers.get_hw_address(nwif)
            list_item = "%s [IP - %s | MAC - %s]" %(nwif, ipaddr, macaddr)
            nw_if_listbox_items.append(list_item)

        global local_ip_addresses
        local_ip_addresses = sorted(set(ip_addr_list))
        nw_ifs_listbox = get_list_box(nw_if_listbox_items)

        render_params.extend([nw_if_label, nw_ifs_listbox]) 


        render_obj = Render(*render_params)
        render_obj.disable_quit_button()
        return render_obj


    def process(self, replica_cnt=0, *args):
        self.log.info("Replica Count: %s" % (str(replica_cnt)))
        global swiftmap
        global config_file_found
        if not config_file_found:
            if replica_cnt < swiftmap.get_min_replica_count():
                self.log.info("Minimum replica count should be %d" % swiftmap.get_min_replica_count())
                error_message(self, "Minimum replica count should be %d" % swiftmap.get_min_replica_count())
                return False

        swiftmap.set_replica_count(replica_cnt)

        global reinit_drives
        if reinit_drives:
            global is_cluster_installation
            is_cluster_installation = True
            self.log.info("Multi Node Install: %s" % (str(is_cluster_installation)))

            if is_cluster_installation:
                rc, hosts = discover_services()
                if not rc and len(hosts) > 0:
                    self.log.info("Discovered Hosts: %s" % (",".join(hosts)))
                    global discovered_nodes
                    discovered_nodes = hosts
                else:
                    error_message(self, "No hosts discovered for multinode installation.")
                    return False

        global is_control_node
        self.log.info("Node Type: Data node")
        is_control_node = False

        global use_same_sys_meta_drives
        use_same_sys_meta_drives = False
        self.log.info("Use same drive for system & metadata: %s" % (str(use_same_sys_meta_drives)))

        global configure_dav
        configure_dav = False
        self.log.info("Configure Direct Access Volume: %s" % (str(configure_dav)))

        return True


    def next(self):
        self.next_controller = appliance_setup_navigator.get_next_controller(self)
        if self.next_controller:
            self.next_controller.prev_controller = self
        return self.next_controller


    def previous(self):
        return self.prev_controller

#} class SwiftConfigController


#{
class SystemDiskSelectionController(Controller):
    """ UI class for selecting disks to be used for system installation """
    def __init__(self):
        self.log = cvpylogger.getLog()
        self.prev_controller = None
        self.next_controller = None


    def render(self):
        title = _("SYSDISK_SELECT_TILE")
        select_req = _("SYSDISK_SELECT_REQ")
        info_msg = _("SYSDISK_SELECT_INFO")

        global use_same_sys_meta_drives
        if use_same_sys_meta_drives:
            title = _("SYS_META_DISK_SELECT_TILE")
            select_req = _("SYS_META_DISK_SELECT_REQ")
            info_msg = _("SYSDISK_SELECT_INFO")
            global is_control_node
            if is_control_node:
                info_msg += (" " + _("DDB_INDEX_DISK_SELECT_INFO"))
            else:
                info_msg += (" " + _("INDEX_DISK_SELECT_INFO"))
            
        render_params = [ title, self.__doc__, model.SimpleText(info_msg),
                            model.SimpleText(select_req) ]

        global device_map
        global reinit_drives
        disk_list = model.CheckBoxList()
        #list disks in following order
        disktype_order_list = ["ssd","nvme","sas"]
        for dtype in disktype_order_list:
            for disk in sorted(device_map.get_dev_map()):
                #if type of disk is not one that is to be considered as per order, then continue to next
                if device_map.compare_disktype(disk, dtype) == False:
                    continue

                #if cv_disks_present and not reinit_drives and is_cv_disk(disk):
                if device_map.cv_disks_present() and device_map.is_cv_disk(disk) and not reinit_drives:
                    continue

                is_checked = disk in system_disks
                disk = device_map.get_extended_disk_name(disk)
                disk_list.add_option(disk, is_checked)
        
        render_params.extend([disk_list])
        render_obj = Render(*render_params)
        render_obj.disable_quit_button()
        return render_obj


    def process(self, selected_disk_list=None):
        global use_same_sys_meta_drives
        drive_type = "system" if not use_same_sys_meta_drives else "system/metadata"

        if selected_disk_list == None or len(selected_disk_list) == 0:
            error_message(self, "No %s drives selected" % (drive_type))
            return False 

        global appliance_swift
        min_disks_req = 2 if use_same_sys_meta_drives or appliance_swift == True else 3
        global device_map
        total_num_disks = len(device_map.get_disks())
        disks_selected = len(selected_disk_list)
        disks_left = total_num_disks - disks_selected
        if disks_left < min_disks_req:
            if min_disks_req == 2:
                error_message(self, 
                    "Number of %s drives selected here leave just %d drives available for Data Drive selection. Minimum of %d drive(s) must be available for Data Drive selection." % (drive_type, disks_left, min_disks_req))
                return False 

            if min_disks_req == 3:
                error_message(self, 
                    "Number of %s drives selected here leave just %d drive(s) available for Metadata & Data Drive selection. Minimum of %d drives must be available for Metadata & Data Drive selection." % (drive_type, disks_left, min_disks_req))
                return False 
                

        self.log.info("Selected %s drives: [%s]" % (drive_type, ",".join(selected_disk_list)))
        global system_disks
        system_disks = getDiskNameListFormExtendedDiskNames(selected_disk_list)

        if use_same_sys_meta_drives:
            global ddb_disks
            ddb_disks = system_disks

        return True


    def next(self):
        self.next_controller = appliance_setup_navigator.get_next_controller(self)
        if self.next_controller:
            self.next_controller.prev_controller = self
        return self.next_controller


    def previous(self):
        return self.prev_controller

#} class SystemDiskSelectionController


#{
class DDBIndexDiskSelectionController(Controller):
    """ UI class for selecting disks to be used for system installation """
    def __init__(self):
        self.log = cvpylogger.getLog()
        self.prev_controller = None
        self.next_controller = None


    def render(self):
        ddbdisks_select_request = model.SimpleText(_("DDB_INDEX_DISK_SELECT_REQ"))
        global is_control_node
        if is_control_node:
            ddbdisks_select_info_msg = model.SimpleText(_("DDB_INDEX_DISK_SELECT_INFO"))
        else:
            ddbdisks_select_info_msg = model.SimpleText(_("INDEX_DISK_SELECT_INFO"))

        global ddb_disks
        global system_disks
        global device_map
        disk_list = model.CheckBoxList()
        #list disks in following order
        disktype_order_list = ["ssd","nvme","sas"]
        for dtype in disktype_order_list:
            for disk in sorted(device_map.get_dev_map()):
                if not disk in system_disks:
                    #if type of disk is not one that is to be considered as per order, then continue to next
                    if device_map.compare_disktype(disk, dtype) == False:
                        continue

                    #if cv_disks_present and not reinit_drives and is_cv_disk(disk):
                    if device_map.cv_disks_present() and device_map.is_cv_disk(disk) and not reinit_drives:
                        continue

                    is_checked = disk in ddb_disks
                    disk = device_map.get_extended_disk_name(disk)
                    disk_list.add_option(disk, is_checked)

        render_obj = Render(_("DDB_INDEX_DISK_SELECT_TILE"), self.__doc__,
                      ddbdisks_select_info_msg, ddbdisks_select_request, disk_list)
        render_obj.disable_quit_button()
        return render_obj


    def process(self, selected_disk_list=None):
        if selected_disk_list == None or len(selected_disk_list) == 0:
            error_message(self, "No metadata drives selected")
            return False

        min_disks_req = 2
        global device_map
        total_num_disks = len(device_map.get_disks())
        global system_disks
        disks_selected = len(system_disks) + len(selected_disk_list)
        disks_left = total_num_disks - disks_selected
        if disks_left < min_disks_req:
            if min_disks_req == 2:
                error_message(self, 
                    "Number of System Drives & the drives selected here leave just %d drives available for Data Drive selection. Minimum of %d drives must be available for Data Drive selection." % (disks_left, min_disks_req))
                return False 

        global ddb_disks
        self.log.info("Selected ddb/index cache disks: [%s]" % (",".join(selected_disk_list)))
        ddb_disks = getDiskNameListFormExtendedDiskNames(selected_disk_list)
        return True


    def next(self):
        self.next_controller = appliance_setup_navigator.get_next_controller(self)
        if self.next_controller:
            self.next_controller.prev_controller = self
        return self.next_controller


    def previous(self):
        return self.prev_controller

#} class DDBIndexDiskSelectionController

#{
class DataDiskSelectionController(Controller):
    """UI class for selecting disks to be used as data disks for backup"""
    def __init__(self):
        self.log = cvpylogger.getLog()
        self.prev_controller = None
        self.next_controller = None


    def render(self):
        disks_select_request = model.SimpleText(_("DATADISK_SELECT_REQ"))
        disks_select_info_msg = model.SimpleText(_("DATADISK_SELECT_INFO"))

        global system_disks
        global ddb_disks
        global data_disks
        global device_map
        disk_list = model.CheckBoxList()
        #list disks in following order
        disktype_order_list = ["ssd","nvme","sas"]
        for dtype in disktype_order_list:
            for disk in sorted(device_map.get_dev_map()):
                if not disk in system_disks and not disk in ddb_disks:
                    #if type of disk is not one that is to be considered as per order, then continue to next
                    if device_map.compare_disktype(disk, dtype) == False:
                        continue

                    if data_disks:
                        is_checked = disk in data_disks
                    else:
                        is_checked = True
                    disk = device_map.get_extended_disk_name(disk)
                    disk_list.add_option(disk, is_checked)

        render_obj = Render(_("DATADISK_SELECT_TILE"), self.__doc__,
                      disks_select_info_msg, disks_select_request, disk_list)
        render_obj.disable_quit_button()
        return render_obj

    def process(self, selected_disk_list=None):
        global data_disks

        if selected_disk_list == None or len(selected_disk_list) == 0:
            error_message(self, "No data drives selected")
            return False

        if len(selected_disk_list) < 2:
            error_message(self, "Minimum two data drives are not selected")
            return False

        self.log.info("Selected data disks: [%s]" % (",".join(selected_disk_list)))
        data_disks = getDiskNameListFormExtendedDiskNames(selected_disk_list)

        global appliance_swift
        if appliance_swift:
            #set data disk count
            global swiftmap
            swiftmap.set_datadisk_count(len(selected_disk_list))

        return True


    def next(self):
        self.next_controller = appliance_setup_navigator.get_next_controller(self)
        if self.next_controller:
            self.next_controller.prev_controller = self
        return self.next_controller


    def previous(self):
        return self.prev_controller

#} class DataDiskSelectionController

#{
class PreservationController(Controller):
    """ UI class to show preserve/reinitialize menu """

    def __init__(self):
        self.log = cvpylogger.getLog()
        self.prev_controller = None
        self.next_controller = None
        self.node_type = 0


    def render(self):
        preserve_request = model.SimpleText(_("PRESERVE_REQ"))

        fg_fmt = '%s,bold' % (model.FormattedText.Color.FG.LIGHT_BLUE)
        bg_color = model.FormattedText.Color.BG.DARK_CYAN
        preserve_info_msg1 = model.FormattedText(_("PRESERVE_INFO_MSG1"), fg_fmt=fg_fmt, bg_fmt=bg_color)
        #preserve_info_msg2 = model.FormattedText(_("PRESERVE_INFO_MSG2"), fg_fmt=fg_fmt, bg_fmt=bg_color)

        preserve_group = model.ExclusiveOptionList()
        preserve_group.add_option(_("PRESERVE_DRIVES"))
        preserve_group.add_option(_("REINIT_DRIVES"))
        preserve_group.select_option(0)

        divider = model.SimpleText("")

        render_params = [_("PRESERVE_TILE"), self.__doc__,
                preserve_request, preserve_info_msg1, preserve_group]

        disks_label = model.SimpleText(_("DISKS_STATUS"))
        disks_info = get_list_box(device_map.get_extended_disk_names_with_init_status(device_map.get_disks()))

        render_params.extend([divider, disks_label, disks_info])

        render_obj = Render(*render_params)
        render_obj.disable_quit_button()
        return render_obj


    def process(self, preserve_option=0, disk_list = None):
        global reinit_drives
        reinit_drives = True if preserve_option == 1 else False
        self.log.info("Reinitialize: %s" % (reinit_drives))

        global device_map
        global ddb_disks
        global data_disks
        global system_disks
        if not reinit_drives:
            data_disks = device_map.get_data_disks()
            ddb_disks = device_map.get_meta_disks()
            if device_map.is_drive_set_same():
                system_disks = ddb_disks

            #force same driveset if only one unused drive is remaining
            #for preserve case and no metadata drives
            if not device_map.get_meta_disks():
                global forced_same_sys_meta_drives
                total_num_disks = len(device_map.get_disks())
                used_disks = len(device_map.get_data_disks())
                if (total_num_disks - used_disks) < 2:
                    forced_same_sys_meta_drives = True

        global is_autoconf_mode
        global autoconf_action
        if is_autoconf_mode and not autoconf_action:
            write_installation_option_file()


        #Check for smart config
        global input_devconfig_file
        global input_autoconf_file
        if input_autoconf_file:
            smartconf=bmr.smartconfig_check(input_devconfig_file, input_autoconf_file, reinit_drives, False)
            if smartconf:
                is_autoconf_mode = True
                XMLConsts = device_map.get_XML_consts()
                if reinit_drives:
                    autoconf_action = XMLConsts["STR_INITIALIZE"]
                else:
                    autoconf_action = XMLConsts["STR_PRESERVE"]

        return True


    def next(self):
        self.next_controller = appliance_setup_navigator.get_next_controller(self)
        if self.next_controller:
            self.next_controller.prev_controller = self
        return self.next_controller

    def previous(self):
        return self.prev_controller

#} class PreservationController

#{
class SummaryController(Controller):
    """ UI class to show status of the setup """
    def __init__(self):
        self.log = cvpylogger.getLog()
        self.prev_controller = None
        self.next_controller = None
        self.ActionOnly = False


    def render(self):
        config_finish_msg = model.SimpleText(_("CONFIG_FINISH_MESSAGE"))
        divider = model.SimpleText("")

        global device_map
        global is_control_node
        node_type = _("NODETYPE_CONTROL") if is_control_node else _("NODETYPE_DATA")
        nodetype_label = get_formatted_label(_("LABEL_NODETYPE"))
        node_type_disks = node_type + " [with %d drives]" % (len(device_map.get_disks()))
        nodetype_msg = model.SimpleText(node_type_disks)
        #nodetype_msg = model.SimpleText(_("LABEL_NODETYPE") + " " + node_type_disks)

        global system_disks
        global ddb_disks
        global data_disks

        sys_disks_label = get_formatted_label(_("SYSTEM_DISKS"))
        sys_disks_msg = get_list_box(device_map.get_extended_disk_names(system_disks, False))

        render_params = [_("SUMMARY_TILE"), self.__doc__,
                            nodetype_label, nodetype_msg ]

        global reinit_drives
        global configure_dav
        is_reference_arch = device_map.is_ref_arch()
        dav_selected = "Yes" if configure_dav else "No"
        dav_label = get_formatted_label(_("DAV_CONF_MSG"))
        dav_msg = model.SimpleText(dav_selected)
        if enable_dav and reinit_drives and is_reference_arch:
            render_params.extend([dav_label, dav_msg])

        #if preserve disks
        if device_map.cv_disks_present() and not reinit_drives:
            preserve_status_label = get_formatted_label(_("PRESERVE_DRIVES")+":")
            preserve_status_msg = model.SimpleText("YES")

            #display metadata drives selected if medatata paths were not preserved
            if not device_map.get_meta_disks():
                ddb_disks_label = get_formatted_label(_("DDB_IDXCACHE_DISKS"))
                ddb_disks_msg = get_list_box(device_map.get_extended_disk_names(ddb_disks, False))

            action_disks = device_map.get_extended_disk_names_with_init_status(device_map.get_cv_disks())

            preserve_action_label = get_formatted_label("Preserving Drives:")
            preserve_action_msg = get_list_box(action_disks)

            Render.main_window_instance.enable_buttons()
            if enable_dav and not device_map.get_meta_disks() and is_reference_arch:
                render_params.extend([dav_label, dav_msg])
            render_params.extend([preserve_status_label, preserve_status_msg, sys_disks_label, sys_disks_msg])
            if not device_map.get_meta_disks():
                render_params.extend([ddb_disks_label, ddb_disks_msg])
            render_params.extend([preserve_action_label, preserve_action_msg])

            render_obj = Render(*render_params)

            render_obj.disable_quit_button()
            render_obj.set_next_button_label(_("APPLY_BUTTON_TEXT"))
            return render_obj

        Render.main_window_instance.enable_buttons()

        data_disks_label = get_formatted_label(_("DATA_DISKS"))
        data_disks_msg = get_list_box(device_map.get_extended_disk_names(data_disks, False))

        ddb_disks_label = get_formatted_label(_("DDB_IDXCACHE_DISKS"))
        ddb_disks_msg = get_list_box(device_map.get_extended_disk_names(ddb_disks, False))

        render_params.extend([sys_disks_label, sys_disks_msg, ddb_disks_label, ddb_disks_msg, 
                                data_disks_label, data_disks_msg])

        global is_cluster_installation
        if is_cluster_installation:
            multinode_label = get_formatted_label(_("MULTI_NODE_INSTALL") + ":")
            global cluster_nodes
            cluster_nodes_listbox = get_list_box(cluster_nodes)
            render_params.extend([multinode_label, cluster_nodes_listbox])
       
        render_obj = Render(*render_params)
        render_obj.disable_quit_button()
        render_obj.set_next_button_label(_("APPLY_BUTTON_TEXT"))
        return render_obj


    def process(self, sysdisks = None, ddbdisks = None, datadisks = None, *args):
        self.log.info("############################")
        global is_control_node
        node_type = _("NODETYPE_CONTROL") if is_control_node else _("NODETYPE_DATA")
        self.log.info("Node Type: [%s]" %(node_type))

        global system_disks
        self.log.info("System Drives: [%s]" %(", ".join(system_disks)))

        global ddb_disks
        self.log.info("Metadata Drives: [%s]" %(", ".join(ddb_disks)))

        global data_disks
        self.log.info("Data Drives: [%s]" %(", ".join(data_disks)))

        global cluster_nodes
        if len(cluster_nodes):
            self.log.info("Cluster Nodes: [%s]" %(", ".join(cluster_nodes)))

        self.log.info("############################")

        self.log.info("Configuration completed successfully")
        write_config()

        global appliance_swift
        if appliance_swift:
            #For Swift Appliance, write key-value pair config file
            global swiftmap
            swiftmap.write_config()

        return True


    def next(self):
        self.next_controller = appliance_setup_navigator.get_next_controller(self)
        if self.next_controller:
            self.next_controller.prev_controller = self
        return self.next_controller


    def previous(self):
        return self.prev_controller

#} class SummaryController

#{
class AutoConfConfirmationController(Controller):
    """ UI class to show auto conf confirmation dialog """

    def __init__(self):
        self.log = cvpylogger.getLog()
        self.prev_controller = None
        self.next_controller = None

    def set_alarm_in(self):
        AUTO_NEXT_INTERVAL = 1

        def autoNextTimerExpired(loop, user_data=None):
           Render.main_window_instance.next_button._emit('click')
           return

        self.log.info("Starting auto next timer of 1 second.")
        self.callBackIn(AUTO_NEXT_INTERVAL, autoNextTimerExpired)


    def render(self):
        autoconf_msg = model.SimpleText(_("AUTOCONF_MSG"))

        render_obj = Render(_("AUTOCONF_TILE"), self.__doc__, autoconf_msg)
        render_obj.disable_quit_button()
        render_obj.disable_prev_button()

        return render_obj


    def process(self):
        return True


    def next(self):
        self.next_controller = appliance_setup_navigator.get_next_controller(self)
        if self.next_controller:
            self.next_controller.prev_controller = self
        return self.next_controller

    def previous(self):
        return None

#} class AutoConfConfirmationController

#{
class SwiftMap:

    def __init__(self):
        self.log = cvpylogger.getLog()
        self.replica = 0
        self.min_replica_cnt = 3
        self.partition_power = 0
        self.password = "cvadmin"
        self.nonroot_username = "demo"
        self.nonroot_password = "demo123"
        self.nodetype = "proxy, storage"
        self.dpnw_if = ""
        self.spnw_if = ""
        self.__get_nw_if()
        self.datadisk_count = 0
        self.datadisk_path = "/ws/disk*"
        self.host_list = []
        self.config_filename = SWIFT_CONFIG_FILENAME
        self.log.info("Swift Appliance")

    def __get_nw_if(self):
        self.dpnw_if, self.spnw_if = SwiftMap.select_nw_if()

    def get_nwif(self):
        return self.dpnw_if, self.spnw_if

    #to determine the number of partitions, estimate the maximum number of disks,
    #multiply that number by 100, and then round up to the nearest power of 2.
    def set_partition_power(self):
        part_power = len(self.host_list) * self.datadisk_count * 100
        part_power = 1 << (part_power.bit_length())
        self.partition_power = part_power

    @staticmethod
    def select_nw_if():
        linkspeed_list = []
        platform=utils.Platform()
        intfx_list = platform.get_nwintfx_list()
        if len(intfx_list) == 0:
            return "", ""
        if len(intfx_list) == 1:
            return intfx_list[0], intfx_list[0]
        for n in intfx_list:
            link_speed = platform.get_link_speed(n)
            linkspeed_list.append((n, link_speed))
        #Sort list based on link speed
        linkspeed_list.sort(key=lambda tup: tup[1], reverse=True)
        return intfx_list[0], intfx_list[1]
        

    def set_hostname_list(self, cluster_nodes, ip_host_map):
        #Include local hostname
        self.host_list.append(Host.get_hostname())
        for n in cluster_nodes:
            self.log.info("Selected host [%s] with IP [%s]" % (ip_host_map[n], n))
            self.host_list.append(ip_host_map[n])

    def set_replica_count(self, count):
        self.replica = count

    def get_replica_count(self):
        return self.replica

    def get_min_replica_count(self):
        return self.min_replica_cnt

    def set_datadisk_count(self, count):
        self.datadisk_count = count
        self.set_partition_power()

    def __write_label(self, conf_file, label):
        conf_file.write('[' + label + ']\n')

    def __write_kvpair(self, conf_file, key, value):
        conf_file.write(key + ' = ' + value + '\n')

    def write_config(self):
        with open(self.config_filename, 'w') as conf_file:
            self.__write_label(conf_file, "swift")
            self.__write_kvpair(conf_file, "replica", str(self.replica))
            self.__write_kvpair(conf_file, "partition_power", str(self.partition_power))
            self.__write_kvpair(conf_file, "sql_root_password", self.password)
            self.__write_kvpair(conf_file, "keystone_db_password", self.password)
            self.__write_kvpair(conf_file, "user", self.nonroot_username)
            self.__write_kvpair(conf_file, "password", self.nonroot_password)
            conf_file.write('\n' + '\n')

            for index, host in enumerate(self.host_list, start=1):
                self.__write_label(conf_file, "node" + str(index))
                self.__write_kvpair(conf_file, "host", host)
                self.__write_kvpair(conf_file, "type", self.nodetype)
                self.__write_kvpair(conf_file, "storage_if", self.dpnw_if)
                self.__write_kvpair(conf_file, "replication_if", self.spnw_if)
                self.__write_kvpair(conf_file, "user", "root")
                self.__write_kvpair(conf_file, "password", self.password)
                self.__write_kvpair(conf_file, "disks", self.datadisk_path)
                conf_file.write('\n' + '\n')


#} class SwiftMap

swiftmap = None

def error_message(node, message):
    """ get localized error message """
    from appliancelocale import getEnglishStringForMsg
    log = cvpylogger.getLog()
    msg = getEnglishStringForMsg(message)
    log.info(msg)
    node.alert(message)


def stop_process(proc_id, proc_name):
    try:
        os.kill(proc_id, signal.SIGKILL)
        return True
    except:
        cvpylogger.getLog().error("Failed to stop %s, pid = %d" % (proc_name, proc_id))
        return False


def stop_process_pid_file(pid_file, proc_name):
    if os.path.exists(pid_file):
        try:
            proc_id = wrappers.readline(pid_file)
            proc_id = int(proc_id.strip())
            return stop_process(proc_id, proc_name)
        except:
            cvpylogger.getLog().error("Failed to stop %s, with pidfile = %s" % (proc_name, pid_file))
            return False

    return False 

def stop_avahi_dbus_daemon():
    conf_file_path=""
    if os.path.exists(DBUS_SHARE_CONF_PATH):
        conf_file_path=DBUS_SHARE_CONF_PATH
    else:
        if os.path.exists(DBUS_ETC_CONF_PATH):
            conf_file_path=DBUS_ETC_CONF_PATH

    search=utils.Grep()
    result=search.lgrep(conf_file_path, "<pidfile>")
    filename = re.search('<pidfile>(.*)</pidfile>', result[0]).group(1)

    AVAHI_DAEMON_PID_FILE = "/var/run/avahi-daemon/pid"
    DBUS_DAEMON_PID_FILE = filename

    stop_process_pid_file(AVAHI_DAEMON_PID_FILE, "avahi daemon")
    stop_process_pid_file(DBUS_DAEMON_PID_FILE, "dbus daemon")


def launch_installui():
    """ invokes TUI """
    from ui.wizard import WizardApplication
    log = cvpylogger.getLog()
    global localhost_install_status
    try:
        next_controller = appliance_setup_navigator.get_next()
        app = WizardApplication(next_controller)
        app.set_view("tui")

        global the_app
        the_app = app

        app.run()
    except Exception as excp:
        log.error("Exception in appliance setup:%s" % (excp))
        localhost_install_status = 1
    finally:
        exit_status = 1 if app.protocol.quit_pressed else 0
        if not exit_status:
            exit_status = localhost_install_status
        stop_avahi_dbus_daemon()
        sys.exit(exit_status)


def USAGE():
    print "Usage: %s <input xml file> <output file>" % (sys.argv[0])
    sys.exit(0)



def set_open_files_limit(softlimit=65535, hardlimit=65535):
    resource.setrlimit(resource.RLIMIT_NOFILE, (softlimit, hardlimit))


def run_dbus_daemon():
    conf_file_path=""
    if os.path.exists(DBUS_SHARE_CONF_PATH):
        conf_file_path=DBUS_SHARE_CONF_PATH
    else:
        if os.path.exists(DBUS_ETC_CONF_PATH):
            conf_file_path=DBUS_ETC_CONF_PATH

    search=utils.Grep()
    result=search.lgrep(conf_file_path, "<listen>")
    filename = re.search('<listen>unix:path=(.*)</listen>', result[0]).group(1)
    sockdir = os.path.dirname(filename)

    DBUS_DAEMON = "/bin/dbus-daemon"
    DBUS_SOCK_DIR = sockdir
    DBUS_SOCK_FILE = "system_bus_socket"
    DBUS_SOCK_FILE_PATH = DBUS_SOCK_DIR + "/" + DBUS_SOCK_FILE
    
    wrappers.mkdirs(DBUS_SOCK_DIR)
    import socket as s
    sock = s.socket(s.AF_UNIX); 
    sock.bind(DBUS_SOCK_FILE_PATH)

    #/bin/dbus-daemon --system   --nopidfile 
    proc = subprocess.Popen([DBUS_DAEMON, "--system"],
                                stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    out, err = proc.communicate()
    return proc.returncode


def is_ipv4_interface_available(MAX_RETRY = 3, SLEEP_INTERVAL = 1):
    retry = 0
    while retry < MAX_RETRY:
        for nwif in wrappers.get_nw_intfx_list():
            ipaddr = wrappers.get_ip_address_of_nw_interface(nwif)
            if not len(ipaddr): continue
            if wrappers.is_valid_ipv4(ipaddr):
                return True
        
        time.sleep(SLEEP_INTERVAL)
        retry += 1
        
    cvpylogger.getLog().error("No IPv4 interface discovered on host.")
    return False


def run_avahi_daemon():
    rc = run_dbus_daemon()
    if rc:
        cvpylogger.getLog().error("Failed to start dbus daemon")
        return 

    AVAHI_DAEMON = "/usr/sbin/avahi-daemon"
    AVAHI_DAEMON_CONF = "/avahi-daemon.conf"

    if not is_ipv4_interface_available():
        cvpylogger.getLog().error("Failed to find an IPv4 interface to run avahi service.")

    proc = subprocess.Popen([AVAHI_DAEMON, "-f", AVAHI_DAEMON_CONF, "--no-chroot", "-D"],
                                stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    out, err = proc.communicate()
    return proc.returncode



def publish_avahi_service(service_name = HS_IMAGING_SERVICE_NAME, service_type = HS_IMAGING_SERVICE_TYPE, 
             port= HS_IMAGING_SERVICE_PORT, service_desc = HS_IMAGING_SERVICE_TXT_RECORD):
    AVAHI_PUBLISH = "/usr/bin/avahi-publish"


    proc = subprocess.Popen([AVAHI_PUBLISH, "-s", service_name, service_type, port, service_desc],
                                stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    out, err = proc.communicate()
    return proc.returncode


def discover_services(service_name = HS_IMAGING_SERVICE_NAME, service_type = HS_IMAGING_SERVICE_TYPE, 
             port = HS_IMAGING_SERVICE_PORT, service_desc= HS_IMAGING_SERVICE_TXT_RECORD):
    AVAHI_BROWSE = "/usr/bin/avahi-browse"

    proc = subprocess.Popen([AVAHI_BROWSE, "-rtp", service_type], 
                        stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    out, err = proc.communicate()

    cvpylogger.getLog().info("Service Discovery -> stdout = [%s], err = [%s]" %(out, err))
    #stdout
    #out = '''+;eth0;IPv4;hsimaging\032\0352;_cv._udp;local
    #+;eth0;IPv4;hsimaging;_cv._udp;local
    #=;eth0;IPv4;hsimaging;_cv._udp;local;none.local;172.16.224.151;11022;"Hyperscale Imaging"
    #=;eth0;IPv4;hsimaging\032\0352;_cv._udp;local;none-2.local;172.16.224.152;11022;"Hyperscale Imaging"'''
    hosts = []
    global local_ip_addresses
    if not proc.returncode:
        pattern = '^=.*IPv4.*"%s"$' % (service_desc)
        matched_lines = re.findall(pattern, out, re.MULTILINE)
        hosts_found = []

        service_names = []
        for line in matched_lines:
            fields = line.split(';')
            service_name = fields[3]
            if not service_name in service_names:
                ip = fields[7]
                if not ip in local_ip_addresses and wrappers.is_valid_ipv4(ip):
                    hosts_found.append(ip)

                    #Create map of IP and hostname. Hostname is 6th field as <hostname.local>
                    global ip_host_map
                    hostname_fields = fields[6].split('.')
                    ip_host_map[fields[7]] = hostname_fields[0]

                    service_names.append(service_name)
                else:
                    cvpylogger.getLog().info("Discarding discovered host [%s] because either its local host IP or IPv6" %(ip))

        hosts = sorted(set(hosts_found))

    return proc.returncode, hosts


def __setup_avahi_service_discovery():
    rc = run_avahi_daemon()
    if rc:
        cvpylogger.getLog().error("Failed to run avahi daemon")
        return

    rc = publish_avahi_service()
    cvpylogger.getLog().error("avahi-publish returned %s" % (str(rc)))
     

def setup_avahi_service_discovery():
    thr = threading.Thread(target=__setup_avahi_service_discovery)
    thr.setDaemon(True)
    thr.start()
    

def config_arg_parser():
    cmdLineParser = argparse.ArgumentParser(description='Spawns Python UI for hyperscale imaging.')

    cmdLineParser.add_argument('-o', '--output', help="Output file to be generated", type=str, required=True, metavar="<Output File>")
    cmdLineParser.add_argument('-c', '--config', help="Block device configuration file for hyperscale reference architecture imaging.", 
                                        type=str, required=True, metavar="<Block device config XML file>")
    cmdLineParser.add_argument('-a', '--autoconfig', help="Auto configuration file for hyperscale appliance imaging.", 
                                        type=str, default=None, metavar="<User config XML file>")
    return cmdLineParser


def validate_cmdline_args(cmdLineParser):
    cmdLineArgs = cmdLineParser.parse_args()

    global input_devconfig_file
    in_file = cmdLineArgs.config 
    if not os.path.isfile(in_file):
        print "No such file exists [%s]" % (in_file)
        sys.exit(1)

    input_devconfig_file = in_file

    out_file = cmdLineArgs.output.strip()
    if len(out_file) <= 0:
        print "No such file exists [%s]" % (in_file)
        sys.exit(1)

    global output_file
    output_file = out_file

    #print "inputdev = %s output = %s" % (input_devconfig_file, output_file)
    if not cmdLineArgs.autoconfig:
        return

    in_file = cmdLineArgs.autoconfig
    global input_autoconf_file
    input_autoconf_file = in_file.strip()
    if os.path.isfile(in_file):
        global is_autoconf_mode
        is_autoconf_mode = True 


def swift_appliance_check():
    apptype_file="/swift_appliance.cfg"
    global appliance_swift
    if os.path.exists(apptype_file):
        is_swift=wrappers.readline(apptype_file)
        is_swift=is_swift.strip()
        if (is_swift=="true"):
            appliance_swift = True


if __name__ == '__main__':

    try:
        cmdLineParser = config_arg_parser()
        validate_cmdline_args(cmdLineParser)
        swift_appliance_check()
        if appliance_swift:
            print "Swift Appliance Installation"
            swiftmap = SwiftMap()
        device_map = DeviceMap()
        numDisks = len(device_map.get_disks())
        if numDisks < 3:
            errMsg = "Minimum 3 disks are required for hyperscale appliance setup." \
                    "Found only %d disks." % (numDisks)

            cvpylogger.getLog().error(errMsg)
            print >> sys.stderr, errMsg
            sys.exit(1)    

        Host.set_hostname()
        set_open_files_limit()
        setup_avahi_service_discovery()

    except:
        cvpylogger.getLog().error("Exception before launching UI")
        sys.exit(1)

    launch_installui()

