import datetime
import platform
import logging
import logging.handlers
import os
import shutil
import socket
import traceback
import sys

log = None

# --------------------------------
# Logging levels
# --------------------------------
LOG_LEVEL_DEBUG = "DEBUG"
LOG_LEVEL_INFO = "INFO"
LOG_LEVEL_WARNING = "WARNING"
LOG_LEVEL_ERROR = "ERROR"
LOG_LEVEL_CRITICAL = "CRITICAL"
LOG_LEVEL_EXCEPTION = "EXCEPTION"

dLogLevels = {
    LOG_LEVEL_DEBUG: logging.DEBUG,
    LOG_LEVEL_INFO: logging.INFO,
    LOG_LEVEL_WARNING: logging.WARNING,
    LOG_LEVEL_ERROR: logging.ERROR,
    LOG_LEVEL_CRITICAL: logging.CRITICAL,
    LOG_LEVEL_EXCEPTION: logging.exception
}
HEADER_STRING = """
=== %s (%s) @ %s ===
"""


class CVLogger:

    def __init__(self, sLogFile=None, sLogLocation=None, sDefaultLogLevel=None):
        """
        Constructor
        """
        defaultLogLocation = "/var/log"
        self.sLogLocation = sLogLocation
        self.sLogFile = sLogFile
        self.sDefaultLogLevel = sDefaultLogLevel
        if self.sDefaultLogLevel is None:
            self.sDefaultLogLevel = LOG_LEVEL_DEBUG
        self.sHandler = None
        self.sLogger = None
        self.sMessage = None

        # Do this only if the Log Location is not explicitly specified.
        if not sLogLocation:
            self.sLogLocation = os.path.join(defaultLogLocation)

        if not os.path.exists(self.sLogLocation):
            os.makedirs(self.sLogLocation)

    def __str__(self):
        return self.sMessage

    def setLog(self, setHeader=True):
        """
        Creates the logfile at the log location and sets the default log level.
        """

        self.sLogger = logging.getLogger(os.path.join(self.sLogLocation, self.sLogFile))
        if not self.sLogger.handlers:
            # Add a handler only if it isn't configured
            self.sLogger.setLevel(dLogLevels[LOG_LEVEL_DEBUG])
            self.sHandler = logging.handlers.RotatingFileHandler(os.path.join(
                self.sLogLocation, self.sLogFile), maxBytes=5 * 1024 * 1024, backupCount=10)
            self.sHandler.setLevel(dLogLevels[self.sDefaultLogLevel])
            # sFormat = logging.Formatter("%(process)-8s %(asctime)s\t%(parentpid)-8s %(levelname)-8s: %(message)s", "%Y-%m-%d %H:%M:%S")
            sFormat = logging.Formatter(
                "%(process)d %(thread)d %(asctime)-4s %(jobid) -4s %(message)s", "%m/%d %H:%M:%S")
            self.sHandler.setFormatter(sFormat)
            self.sLogger.addHandler(self.sHandler)
            if setHeader:
                self.writeHeader()

    def writeLog(self, sLogLevel, sLogLine):
        """
        Writes lines to the log file with the appropriate log level.
        """

        if not self.sLogger:
            if self.sLogLocation and self.sLogFile and self.sDefaultLogLevel:
                self.setLog()

        self.sMessage = sLogLine
        jobId = '###'

        if sLogLevel.upper() == LOG_LEVEL_DEBUG:
            self.sLogger.debug(sLogLine, extra={"jobid": jobId})
        elif sLogLevel.upper() == LOG_LEVEL_INFO:
            self.sLogger.info(sLogLine, extra={"jobid": jobId})
        elif sLogLevel.upper() == LOG_LEVEL_WARNING:
            self.sLogger.warn(sLogLine, extra={"jobid": jobId})
        elif sLogLevel.upper() == LOG_LEVEL_ERROR:
            self.sLogger.error(sLogLine, extra={"jobid": jobId}, exc_info=True)
        elif sLogLevel.upper() == LOG_LEVEL_CRITICAL:
            self.sLogger.critical(sLogLine, extra={"jobid": jobId}, exc_info=True)
        else:
            self.sLogger.warn("Not a valid log level: %s" % (sLogLine), extra={"jobid": jobId})

    def getLogFilePath(self):
        return os.path.join(self.sLogLocation, self.sLogFile)

    def writeHeader(self):
        """
        Write header for log files.
        """
        with open(os.path.join(self.sLogLocation, self.sLogFile), "a") as fh:
            fh.write("*******************************************************************************\n")
            fh.write("* Machine name : %s\n" % (socket.getfqdn()))
            fh.write("*     platform : %s\n" % (platform.platform(aliased=True)))
            fh.write("*              : %s\n" % (platform.platform(terse=True)))
            fh.write("*           os : %s\n" % (' '.join(platform.uname())))
            fh.write("*         arch : %s\n" % (' '.join(platform.architecture())))
            fh.write("*         Date : %s\n" % (datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")))
            fh.write("*******************************************************************************\n")

    def debug(self, sMessage):
        self.writeLog(LOG_LEVEL_DEBUG, sMessage)

    def info(self, sMessage):
        self.writeLog(LOG_LEVEL_INFO, sMessage)

    def warning(self, sMessage):
        self.writeLog(LOG_LEVEL_WARNING, sMessage)

    def error(self, sMessage):
        self.writeLog(LOG_LEVEL_ERROR, sMessage)

    def critical(self, sMessage):
        self.writeLog(LOG_LEVEL_CRITICAL, sMessage)


def getLog():
    """Returns the instance of logger"""
    global log
    return log


def setLog(logv):
    """Returns the instance of logger"""
    global log
    log = logv

logfileName = "/var/log/cvsetuphca.log"
log = CVLogger(sLogFile=logfileName, sDefaultLogLevel=LOG_LEVEL_DEBUG)

if __name__ == '__main__':
    log = CVLogger('cvsetuphca.log', '/var/log')
    print("log path:%s" % (log.getLogFilePath()))
    log.info("test")
