"""
All the APIs necessary to use the Custom Report Viewer
"""

from Web.Common.page_object import (
    WebAction,
    PageService
)

from Web.WebConsole.Reports.Custom.builder import BaseReportPage
from Web.WebConsole.Reports.Custom.inputs import HTMLController
from Web.WebConsole.Reports.cte import CustomSecurity
from Web.WebConsole.Reports.report import FileMenu
from Web.WebConsole.Reports.report import Report

from ._components import base
from ._components.table import (
    DataTableViewer,
    ColumnInViewer,
    ButtonInViewer,
    TableViewer
)
from ._components.form import (
    DateRangeViewer,
    SearchBarViewer
)
from ._components.chart import (
    HorizontalBarViewer,
    LineChartViewer,
    TimelineChartViewer,
    RectangularChartViewer,
    CircularChartViewer
)
from ._components.other import (
    HtmlComponentViewer,
    RComponentViewer
)



class CustomReportViewer(BaseReportPage, Report):

    """All the custom report viewer specific actions go here"""

    @property
    def report_type(self):
        return "CustomReport"

    @WebAction()
    def __read_report_name(self):
        """Read report name"""
        report_title = self._driver.find_element_by_xpath(
            "//*[@model='customReport.report.customReportName']"
        )
        return report_title.text

    @WebAction()
    def __click_edit(self):
        """Click edit on file menu"""
        edit_btn = self._driver.find_element_by_xpath(
            "//*[@id='editLink']"
        )
        edit_btn.click()

    @WebAction()
    def __read_report_description(self):
        """ Read report description. """
        report_description = self._driver.find_element_by_xpath("//div[@id='crdescription']")
        return report_description.text.strip()

    @WebAction()
    def __click_delete(self):
        """Click delete on file menu"""
        delete_btn = self._driver.find_element_by_xpath(
            "//*[@id='deleteButton']"
        )
        delete_btn.click()

    @WebAction()
    def __fetch_all_page_title_names(self):
        """Returns all page titles"""
        list_ = self._driver.find_elements_by_xpath("//li/div[@class = 'ng-binding']")
        return [title.get_attribute("title") for title in list_]

    @PageService()
    def get_report_name(self):
        """Get report name"""
        return self.__read_report_name()

    @PageService()
    def get_report_description(self):
        """Get report description. """
        return self.__read_report_description()

    @PageService()
    def associate_component(self, component, page="Page0"):
        """Associate component to Viewer

        Args:
            component (CRComponentViewer): Any component which
                has a viewer implementation

            page (str): Page in which component is residing

        """
        if not isinstance(component, base.CRComponentViewer):
            raise ValueError("invalid component type")
        component.configure_viewer_component(self._webconsole, page)

    @PageService()
    def associate_input(self, input_):
        """Associate input to viewer"""
        if not isinstance(input_, HTMLController):
            raise ValueError("Invalid input component")
        input_.configure(self._webconsole, _builder=False)

    @PageService()
    def edit_report(self):
        """Edit custom report"""
        file = FileMenu(self._webconsole)
        file.click_file_menu()
        self.__click_edit()

    @PageService()
    def delete_report(self):
        """Delete custom report"""
        file = FileMenu(self._webconsole)
        file.click_file_menu()
        self.__click_delete()
        self._click_yes_on_confirmation_popup()
        self._webconsole.wait_till_load_complete()

    @PageService()
    def get_all_page_title_names(self):
        """Returns all page titles."""
        return self.__fetch_all_page_title_names()

    @PageService()
    def open_security(self):
        """
        Opens Security panel

        Returns:
             security panel object
        """
        file = FileMenu(self._webconsole)
        file.click_file_menu()
        file.click_security()
        self._webconsole.wait_till_load_complete()
        return CustomSecurity(self._webconsole)


class DataTable(DataTableViewer):

    """
    Dummy class to expose all the private DataTableViewer APIs as public

    For the builder and properties panel specific actions, refer builder.py file

    Example::

        table = DataTable("tableName")
        rpt_viewer.associate_component(table)
        print(table.get_table_data())  # gets all the table data as JSON

    """
    Button = ButtonInViewer
    Column = ColumnInViewer


class PivotTable(TableViewer):
    """
    Dummy class to expose all the private Pivot Table APIs as public

    For the builder and properties panel specific actions, refer builder.py file

    """
    @property
    def type(self):
        """Returns:Category type as 'Table'"""
        return "PIVOT_TABLE"


class DateRange(DateRangeViewer):
    """
    Dummy class to expose all the private DateRange Viewer APIs as public

    For the builder and properties panel specific actions, refer builder.py file

    """


class SearchBar(SearchBarViewer):
    """
    Dummy class to expose all the private Search Viewer APIs as public

    For the builder and properties panel specific actions, refer builder.py file

    """


class VerticalBar(RectangularChartViewer):
    """
    Dummy class to expose all the private VerticalBar APIs as public

    For the builder and properties panel specific actions, refer builder.py file

    """


class HorizontalBar(HorizontalBarViewer):
    """
    Dummy class to expose all the private HorizontalBar APIs as public

    For the builder and properties panel specific actions, refer builder.py file

    """


class PieChart(CircularChartViewer):
    """
    Dummy class to expose all the private PieChart APIs as public

    For the builder and properties panel specific actions, refer builder.py file

    """


class DonutChart(CircularChartViewer):
    """
    Dummy class to expose all the private DonutChart APIs as public

    For the builder and properties panel specific actions, refer builder.py file

    """


class LineChart(LineChartViewer):
    """
    Dummy class to expose all the private LineChart APIs as public

    For the builder and properties panel specific actions, refer builder.py file

    """


class TimelineChart(TimelineChartViewer):
    """
    Dummy class to expose all the private TimelineChart APIs as public

    For the builder and properties panel specific actions, refer builder.py file

    """


class HtmlComponent(HtmlComponentViewer):
    """
    Dummy class to expose all the private HTML component APIs as public

    For the builder and properties panel specific actions, refer builder.py file

    """


class RComponent(RComponentViewer):
    """
    Dummy class to expose all the private R component APIs as public

    For the builder and properties panel specific actions, refer builder.py file
    """
