# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
""" Operations related to companies page.


Companies:

    __init__()                          --  initialize instance of the Companies class,
                                             and the class attributes.

    filter_by_company_name()           --  filter monitoring page by company name.


"""
from time import sleep
from selenium.common.exceptions import NoSuchElementException
from selenium.webdriver import ActionChains

from Web.WebConsole.webconsole import WebConsole
from Web.WebConsole.Reports.Custom import viewer
from Web.Common.page_object import WebAction, PageService
from Web.WebConsole.Reports.report import FileMenu


class RegisteredCompanies:
    """Operations on companies page"""

    def __init__(self, web_console: WebConsole):
        self._driver = web_console.browser.driver
        self._webconsole = web_console
        self.viewer = viewer.CustomReportViewer(self._webconsole)
        self.__companies_table = None

    @property
    def _companies_table(self):
        if self.__companies_table is None:
            self.__companies_table = viewer.DataTable("")
            self.viewer.associate_component(self.__companies_table)
        return self.__companies_table

    @WebAction()
    def _is_company_name_filtered(self, company_name):
        """
        Verify company page is already filtered by company name using url
        Args:
            company_name        (String)     --         name of the company

        Returns(Boolean) : True if page is filtered with company name else return false
        """
        _str = "filter.Name=%s" % company_name
        return _str in self._driver.current_url

    @WebAction()
    def _click_company_name(self, company_name):
        """
        Click on specified company name
        Args:
            company_name        (String)     --         name of the company
        """
        xpath = "//a[@title='%s']" % company_name
        self._driver.find_element_by_xpath(xpath).click()

    @WebAction()
    def _click_delete(self):
        """Click on specified company name"""
        self._driver.find_element_by_xpath("//button[text()='Delete']").click()

    @PageService()
    def filter_by_company_name(self, company_name):
        """
        Filter by specified company name on column 'Company Name'
        Args:
            company_name        (String)     --         name of the company
        """
        if not self._is_company_name_filtered(company_name):
            self._companies_table.set_filter('Name', company_name)
        self._webconsole.wait_till_load_complete()

    @PageService()
    def access_company(self, company_name):
        """Access specified company"""
        self.filter_by_company_name(company_name)
        self._click_company_name(company_name)
        self._webconsole.wait_till_load_complete()

    @PageService()
    def get_companies_list(self):
        """Get companies list"""
        return self._companies_table.get_table_data()['Name']

    @PageService()
    def get_commcell_count(self):
        """Get commcell count"""
        return self._companies_table.get_table_data()['Number Of CommCells']

    @PageService()
    def edit_company(self, company_name):
        """Edit the subcompany
        Returns:
            object of Subcompany class
        """
        self._companies_table.access_action_item(company_name, 'Edit')
        sleep(5)
        return SubCompany(self._webconsole)

    @PageService()
    def delete_company(self, company_name):
        """Deletes the subcompany
        Returns:
            object of Subcompany class
        """
        self._companies_table.access_action_item(company_name, 'Delete')
        sleep(3)
        self._click_delete()
        sleep(5)

    @PageService()
    def create_company(self):
        """
        Open create subcompany menu
        Returns:
            object of Subcompany class
        """
        menu = Menu(self._webconsole)
        menu.click_file_menu()
        menu.click_new_company()
        self._webconsole.wait_till_load_complete()
        return SubCompany(self._webconsole)


class Menu(FileMenu):
    """
        FileMenu on Company listing page
    """
    @WebAction()
    def click_new_company(self):
        """
        clicks the security option in file menu
        """
        self._driver.find_element_by_id('createCompany').click()


class SubCompany:
    """Sub Company menu"""
    def __init__(self, web_console: WebConsole):
        self._driver = web_console.browser.driver
        self._webconsole = web_console

    @WebAction()
    def _enter_company_name(self, name):
        """enters company name"""
        cc_name = self._driver.find_element_by_id('companyName')
        cc_name.clear()
        cc_name.send_keys(name)

    @WebAction()
    def _click_prefix(self):
        """click prefix"""
        self._driver.find_element_by_xpath("//input[@id='ccPrefix']").click()

    @WebAction()
    def _select_prefix(self, prefix):
        """enters company name"""
        cc_name = self._driver.find_element_by_xpath(
            f"//li[@class='prefix-item' and text()='{prefix}']"
        )
        cc_name.click()

    @WebAction()
    def _click_commcells(self):
        """click Commcell dropdown"""
        self._driver.find_element_by_xpath("//input[@id='ccSearch']").click()

    @WebAction()
    def _select_commcells(self, commcells):
        """
        select commcells
        Args:
            commcells (list): list of commcell name
        """
        for cc in commcells:
            source_cc = ' - ' + cc + ' - '
            sleep(3)
            dd_xp = f"//ul[@id='companyCCSearchList']//span[contains(text(),'{source_cc}')]"
            try:
                self._driver.find_element_by_xpath(dd_xp).click()
                self._driver.find_element_by_xpath("//input [@class='ccSearchOkBtn']").click()
            except NoSuchElementException as excep:
                raise NoSuchElementException(f"Commcell {cc} not found in drop down") from excep

    @WebAction()
    def _remove_commcells(self, commcells):
        """
        Removes commcells
        Args:
            commcells (list): list of commcell name
        """
        for cc in commcells:
            dd_xp = f"//li[contains(@data-ccname, '{cc}')]/span[@title='Remove CommCell']"
            try:
                commcell_xp = f"//li[@class= 'commcellEntry' and contains(@data-ccname, '{cc}')]"
                hover = ActionChains(self._driver).move_to_element(self._driver.find_element_by_xpath(commcell_xp))
                hover.perform()
                self._driver.find_element_by_xpath(dd_xp).click()
            except NoSuchElementException as excep:
                raise NoSuchElementException(f"Commcell {cc} not found in the list") from excep

    @WebAction()
    def _click_add_user(self):
        """clicks add user"""
        self._driver.find_element_by_id('addCompanyUser').click()

    @WebAction()
    def _enter_user_name(self, username):
        """enters user name"""
        xp = "//div[@class='form-group userEntry newUser'][1]/div[@class='rowItem userName']/input"
        new_user = self._driver.find_element_by_xpath(xp)
        new_user.clear()
        new_user.send_keys(username)

    @WebAction()
    def _enter_email(self, email_id):
        """Fill the email field"""
        xp = "//*[@class='form-group userEntry newUser'][1]/div[@class='rowItem userEmail']/input"
        email = self._driver.find_element_by_xpath(xp)
        email.clear()
        email.send_keys(email_id)

    @WebAction()
    def _click_save(self):
        """clicks save button"""
        self._driver.find_element_by_xpath("//button[text()='Save']").click()

    @WebAction()
    def _click_update(self):
        """clicks update button"""
        self._driver.find_element_by_xpath("//button[text()='Update']").click()

    @WebAction()
    def _click_cancel(self):
        """clicks cancel button"""
        self._driver.find_element_by_xpath("//div[@class='ui-dialog-buttonset']//button[text()='Cancel']").click()

    @WebAction()
    def _click_delete_user(self, email_id):
        """Click on delete icon for users"""
        self._driver.find_element_by_xpath(
            f"//input[@data-user-email='{email_id}']/../../span[@title='Delete User']"
        ).click()

    @WebAction()
    def _read_saved_commcells(self):
        """read saved commcells"""
        xp = "//li[@class='commcellEntry']"
        commcells = self._driver.find_elements_by_xpath(xp)
        return [each_cc.text.split(' - ')[1] for each_cc in commcells if each_cc.text]

    @WebAction()
    def _read_emails(self):
        """gets the assoicated users email id"""
        emails = []
        xpath = "//div[@class='rowItem userEmail']/input"
        email_obj = self._driver.find_elements_by_xpath(xpath)
        for email in email_obj:
            emails.append(email.get_attribute('data-user-email'))
        return emails

    @PageService()
    def get_commcells(self):
        """Gets commcells present in subcompany
        Returns: Commcell names list in the format displayed in webpage"""
        return self._read_saved_commcells()

    @PageService()
    def add_commcells(self, commcell_name):
        """Adds more comcells in subcompany"""
        self._click_commcells()
        self._select_commcells(commcell_name)

    @PageService()
    def delete_commcells(self, commcell_name):
        """Deletes comcells from subcompany"""
        self._remove_commcells(commcell_name)

    @PageService()
    def delete_user(self, user_emailid):
        """Deletes the user"""
        self._click_delete_user(user_emailid)

    @PageService()
    def add_user(self, user_name, user_email_id):
        """Adds the user"""
        self._click_add_user()
        self._enter_user_name(user_name)
        self._enter_email(user_email_id)

    @PageService()
    def update(self):
        """Clicks on Update Button"""
        self._click_update()
        self._webconsole.wait_till_load_complete()
        sleep(5)  # wait for process to be submitted and to receive notification

    @PageService()
    def cancel(self):
        """Clicks on Cancel Button"""
        self._click_cancel()
        self._webconsole.wait_till_load_complete()

    @PageService()
    def create(self, prefix, name, commcells, users):
        """
        creates new subcompany
        Args:
            prefix      (str): prefix for subcompany
            name        (str): name of subcompany
            commcells  (list): list of commcell names
            users      (list): list of dictionary containing username and email

            ex.: [{
            'user_name': 'user1',
            'email': 'email@email.com'
            }
            ]
        """
        self._click_prefix()
        self._select_prefix(prefix)
        self._enter_company_name(name)
        self._click_commcells()
        self._select_commcells(commcells)
        for each_user in users:
            self._click_add_user()
            self._enter_user_name(each_user['user_name'])
            self._enter_email(each_user['email'])
        self._click_save()
        self._webconsole.wait_till_load_complete()
        sleep(5)   # wait for process to be submitted and to receive notification

    @PageService()
    def get_associated_users(self):
        """returns associated users email id"""
        return self._read_emails()
