# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Add all the actions on the forms landing page go here

Forms:

    _get_opened_form_title                  --      Get the title of opened form

    _click_ok                               --      Click ok

    _get_form_submit_message                --      Get the message of submit form

    _open_search_results                    --      Opens the SearchResults page for the
                                                    Searchable input based on label

    _search_submit_dropdown                 --      Search the value in SearchResults page
                                                    and select the value

    _search_workflow                        --      Search the workflow in search box

    _open_workflow                          --      Open the workflow window

    _goto_Actions                           --      Click the Actions tab in forms application

    _goto_Open_Actions                      --      Click the Open tab

    _goto_Completed_Actions                 --      Click the Completed tab

    close_form                             --      Close the opened form

    select_file                             --      Select list of files for file type input

    select_dropdown                         --      Select drop down element with label

    set_textbox_value                       --      Set value to textbox element based on label

    set_textbox_value_for_v1_form           --      Set value to textbox element based on label for v1 form

    set_multiline_value                     --      Set value to multiline input based on label

    select_radio_value                      --      Select value for radiobox based on label

    select_checkbox_value                   --      Select value for checkbox based on label

    select_listbox_value                    --      Select value for listbox type input
                                                    based on input label

    click_cancel                            --      Cancel form submission

    set_time                                --      Set the time value for time control
                                                    type input based on label

    informational_message                   --      Message displayed in informational window

    click_action_button                     --      Click the custom action button

    click_action_button_in_v1_form          --      Click the custom action button in v1 form

    set_boolean                             --      Set true or false for a boolean checkbox

    select_dropdown_list_value              --      Select list of values in dropdown input type

    get_listbox_value                       --      Show Drop down value

    get_radiobox_value                      --      Show radiobox value

    get_checkbox_value                      --      Show Checkbox value

    get_dropdown_list_value                 --      Retrieves the value list for
                                                    drop-down multi select type

    close_full_page                         --      Close the full Page form

    click_forms_full_page                   --      Click Forms link in the opened
                                                    full Page Form

    click_customize_link                    --      Click CustimizeThisForm link
                                                    in the open form

    get_action_button_labels                --      Retrieves action button

    click_ok_on_v1_form                     --      Switch driver frame for V1 form and clicks ok

    is_form_open                            --      Check if form is already open

    is_v1_form_open                         --      Check if v1 form is already open

    submit                                  --      submit the form by clicking on ok

    submit_v1_form                          --      submit the v1 form by clicking on ok

    is_form_submitted                       --      Check if form submitted successfully

    select_searchable_dropdown_value        --      Select a value for input type
                                                    of searchable,Dropdown

    open_workflow                           --      Opens the workflow

    submit_interaction                      --      Submit interaction

    is_full_page_open                       --      Check if opened form is fullPage
                                                    form or not

"""

import time

from selenium.common.exceptions import NoSuchElementException
from selenium.webdriver.common.keys import Keys

from Web.Common.page_object import (
    WebAction,
    PageService
)


class Forms:
    """Class for representing methods of forms application"""

    def __init__(self, webconsole):
        self._webconsole = webconsole
        self._browser = webconsole.browser
        self._driver = self._browser.driver

    @WebAction()
    def _get_opened_form_title(self):
        """Get the title of opened form"""
        title = self._driver.find_element_by_xpath(
            "//*[@id='modalHeaderArea']/h4"
        )
        return title.get_attribute("title")

    @WebAction()
    def close_form(self):
        """Close the opened form"""
        close_button = self._driver.find_element_by_xpath(
            "//*[@id='wfFormsPopup']//*[@class='modal-header']//*[.='x']"
        )
        close_button.click()

    @WebAction()
    def _click_ok(self):
        """Click ok"""
        self._driver.find_element_by_xpath("//button[@data-action='OK']").click()

    @WebAction()
    def _get_form_submit_message(self):
        """Get the message of submit form"""
        submit_message = self._driver.find_element_by_xpath(
            "//*[@id='inputFormArea']"
        ).text
        return submit_message

    @WebAction()
    def _open_search_results(self, label):
        """Opens the SearchResults page for the Searchable input based on label"""
        xpath = f"//label[text()='{label}']/..//div[@class='dropDown wfSearchable ']"
        self._driver.find_element_by_xpath(xpath).click()

    @WebAction()
    def _search_submit_dropdown(self, value):
        """Search the value in SearchResults page and select the value"""
        self._driver.find_element_by_id('SearchresultSearch').send_keys(value)
        self._driver.find_element_by_id('SearchresultSearch').send_keys(Keys.ENTER)
        self._webconsole.wait_till_load_complete()
        input_xpath = f"//div[@id='searchList']/..//input[@title='{value}']"
        input_element = self._driver.find_element_by_xpath(input_xpath)
        input_element.click()
        submit_selected = "//*[@id='wfSearchablePopUp']//*[@class='modal-footer']" \
                          "//*[@id='searchableOkButton']"
        self._driver.find_element_by_xpath(submit_selected).click()

    @WebAction()
    def _search_workflow(self, workflow):
        """Search the workflow in search box"""
        search_xpath = self._driver.find_element_by_xpath(
            "//input[@placeholder='Search' and @type='text']")
        search_xpath.clear()
        search_xpath.send_keys(workflow)
        search_xpath.send_keys(Keys.ENTER)

    @WebAction()
    def _open_workflow(self, workflow):
        """Open the workflow window"""
        xpath = f"//a[@title='{workflow}']"
        select_workflow = self._driver.find_element_by_xpath(xpath)
        select_workflow.click()

    @WebAction()
    def _open_Action(self, action):
        """Open the Action"""
        link = self._driver.find_element_by_link_text(action)
        link.click()

    @WebAction()
    def _goto_Actions(self):
        """Click the Actions tab in forms application"""
        self._driver.find_element_by_xpath("//div[contains(@class,'activeActionsCount ')]/ancestor::li").click()

    @WebAction()
    def _goto_Open_Actions(self):
        """Click the Open tab"""
        self._driver.find_element_by_xpath("//div[@title='Open']").click()

    @WebAction()
    def _goto_Completed_Actions(self):
        """Click the Completed tab"""
        self._driver.find_element_by_xpath("//div[@title='Completed']").click()
        self._driver.refresh()

    @WebAction()
    def select_file(self, label, file):
        """
        Select list of files for file type input
        Args :
            label               (String)        --      file input label
            file                (List)          --      list of files
        Example :
            select_file('Source File', ['C:\\TestXML\\1.xml', 'C:\\TestXML\\2.xml'])
        """
        xpath = f"//label[@title='{label}']/..//input[@type='file']"
        for option in file:
            self._driver.find_element_by_xpath(xpath).send_keys(option)

    @WebAction()
    def select_dropdown(self, label, value):
        """
        Select drop down element with label
        Args:
            label                   (String)       --     dropdown label
            value                   (String)       --     value to be selected
        """
        xpath = f"//label[@title='{label}']/..//div[@class='Select-control']"
        self._driver.find_element_by_xpath(xpath).click()
        xpath1 = f"//label[@title='{label}']/..//div[@class='Select-input']/input"
        self._driver.find_element_by_xpath(xpath1).send_keys(
            value)
        self._driver.find_element_by_xpath(xpath1).send_keys(Keys.RETURN)

    @WebAction()
    def set_textbox_value(self, label, value):
        """Set value to textbox element based on label
        Args:
            label               (String)        --      textbox field label
            value               (String)        --      value to be set
        """
        xpath = f"//label[@title='{label}']/..//*[@label='{label}']"
        self._driver.find_element_by_xpath(xpath).clear()
        self._driver.find_element_by_xpath(xpath).send_keys(value)

    @WebAction()
    def set_textbox_value_for_v1_form(self, label, value):
        """Set value to textbox element based on label for v1 form
        Args:
            label               (String)        --      textbox field label
            value               (String)        --      value to be set
        """
        frame = self._driver.find_elements_by_xpath("//iframe")
        self._driver.switch_to.frame(frame[0])
        self._driver.find_element_by_xpath("//*[@id='{0}']".format(label)).send_keys(value)
        self._driver.switch_to.default_content()

    @WebAction()
    def set_multiline_value(self, label, value):
        """Set value to multiline input based on label
        Args:
            label               (String)        --      multiline field label
            value               (String)        --      value to be set
        """
        xpath = f"//label[@title='{label}']/..//textarea"
        self._driver.find_element_by_xpath(xpath).clear()
        self._driver.find_element_by_xpath(xpath).send_keys(value)

    @WebAction()
    def select_radio_value(self, label, value):
        """Select value for radiobox based on label
        Args:
            label               (String)        --      radiobox label
            value               (String)        --      value to click
        """
        xpath = f"//label[@title='{label}']/..//input[@type='radio']/..//span[text()='{value}']"
        self._driver.find_element_by_xpath(xpath).click()

    @WebAction()
    def select_checkbox_value(self, label, value):
        """Select value for checkbox based on label
        Args:
            label               (String)        --      Checkbox label
            value               (String)        --      value to click
        """
        xpath = f"//label[@title='{label}']/..//span[text()='{value}']/../input[@type='checkbox']"
        self._driver.find_element_by_xpath(xpath).click()

    @WebAction()
    def select_listbox_value(self, label, value):
        """Select value for listbox type input based on input label
        Args:
            label               (String)        --      listbox input label
            value               (String)        --      value to select
        """
        xpath = f"//label[@title='{label}']/..//option[text()='{value}']"
        self._driver.find_element_by_xpath(xpath).click()

    @WebAction()
    def click_cancel(self):
        """Cancel form submission"""
        self._driver.find_element_by_xpath("//button[@data-action='Cancel']").click()

    @WebAction()
    def set_time(self, label, value):
        """Set the time value for time control type input based on label
        Args:
            label               (String)        --      Time input field label
            value               (String)        --      time value
        """
        xpath = f"//label[contains(@title,'{label}')]/..//*[@='{label}']"
        self._driver.find_element_by_xpath(xpath).send_keys(value)

    @WebAction()
    def click_action_button(self, button):
        """Click the custom action button
        button              (String)        --      Custom action name
        Eg:
        Popup Input, UserInput can have custom action like Approve,Deny
        """
        xpath = f"//button[@data-action='{button}']"
        self._driver.find_element_by_xpath(xpath).click()

    @WebAction()
    def click_action_button_in_v1_form(self, button):
        """Click the custom action button in v1 form
        button              (String)        --      Custom action name
        Eg:
        Popup Input, UserInput can have custom action like Approve,Deny
        """
        frame = self._driver.find_elements_by_xpath("//iframe")
        self._driver.switch_to.frame(frame[0])
        self._driver.find_element_by_xpath("//a[@id='{0}']".format(button)).click()
        self._driver.switch_to.default_content()

    @WebAction()
    def set_boolean(self, label, value):
        """Set true or false for a boolean checkbox
        Args:
            label               (String)        --      Input name
            value               (String)        --      true or false
        """
        xpath = f"//span[text()='{label}']/../input[@type='checkbox']"
        check = self._driver.find_element_by_xpath(xpath)
        value_set = check.get_attribute('data-value')
        if value_set != value:
            check.click()

    @WebAction()
    def select_dropdown_list_value(self, label, value):
        """Select list of values in dropdown input type"""
        arrow = f"//label[@title='{label}']/..//button"
        self._driver.find_element_by_xpath(arrow).click()
        for option in value:
            xpath = f"//label[@title='{label}']/..//label[text()=' {option}']"
            self._driver.find_element_by_xpath(xpath).click()
        self._driver.find_element_by_xpath(arrow).click()

    @WebAction()
    def get_listbox_value(self, label):
        """Show Drop down value"""
        xpath = f"//label[@title='{label}']/..//option"
        drop_values = []
        for option in self._driver.find_elements_by_xpath(xpath):
            drop_values.append(option.get_attribute('innerHTML'))
        return drop_values

    @WebAction()
    def get_radiobox_value(self, label):
        """Show radiobox value"""
        xpath = f"//label[@title='{label}']/..//*[@type='radio']/..//span"
        radio_values = []
        for option in self._driver.find_elements_by_xpath(xpath):
            radio_values.append(option.get_attribute('innerHTML'))
        return radio_values

    @WebAction()
    def get_checkbox_value(self, label):
        """Show Checkbox value"""
        xpath = f"//label[@title='{label}']/..//*[@type='checkbox']/..//span"
        checkbox_values = []
        for option in self._driver.find_elements_by_xpath(xpath):
            checkbox_values.append(option.get_attribute('innerHTML'))
        return checkbox_values

    @WebAction()
    def get_dropdown_list_value(self, label):
        """Retrieves the value list for drop-down multi select type"""
        value = []
        xpath = f"//label[@title='{label}']/..//option"
        for option in self._driver.find_elements_by_xpath(xpath):
            value.append(option.get_attribute('innerHTML'))
        return value

    @WebAction()
    def informational_message(self):
        """
        Message displayed in informational window
        Returns:
            (String)        --      Informational message
        """
        msg = self._driver.find_element_by_xpath("//div[@class='wfInfoArea']").text
        return msg

    @WebAction()
    def close_full_page(self):
        """Clicks the close button in fullPage form"""
        self._driver.find_element_by_xpath(
            "//*[@class='wfFullPageForm']//../*[@id='wfFullPageCloseBtn']"
        ).click()

    @WebAction()
    def click_forms_full_page(self):
        """Clicks the Forms link in the opened full page form"""
        self._driver.find_element_by_xpath(
            "//*[@class='wfFullPageForm']//../*[@id='wfBaseUrlTitle']/a"
        ).click()

    @WebAction()
    def click_customize_link(self):
        """Clicks the CustomizeThisForm link in the open form"""
        self._driver.find_element_by_xpath(
            "//*[@id='wfCustomizeThisForm']/a"
        ).click()

    @WebAction()
    def is_full_page_open(self, form_name):
        """Checks whether the opened form is fullpage or not"""
        self._webconsole.wait_till_load_complete()
        time.sleep(2)  # For the Title to fade into context
        xpath = f"//*[@class='wfFullPageForm']//../span[text()='{form_name}']"
        if self._driver.find_element_by_xpath(xpath):
            return True

    @WebAction()
    def get_action_button_labels(self):
        """Returns the Action button
        Returns
            List of Action button labels
        """
        xpath = "//div[@id='wfModalFooter']/.//button"
        return [buttons.text for buttons in self._driver.find_elements_by_xpath(xpath) if buttons]

    @WebAction()
    def click_ok_on_v1_form(self):
        """Switch driver frame for V1 form and clicks ok"""
        frame = self._driver.find_elements_by_xpath("//iframe")
        self._driver.switch_to.frame(frame[0])
        self._driver.find_element_by_xpath("//a[@id='okButton']").click()
        self._driver.switch_to.default_content()

    @PageService()
    def is_form_open(self, form_name):
        """Check if form is alrsearceady open"""
        self._webconsole.wait_till_load_complete()
        time.sleep(2)  # For the Title to fade into context
        form_title = self._get_opened_form_title()
        return form_title.lower() == form_name.lower()

    @PageService()
    def is_v1_form_open(self, form_name):
        """Check if v1 form is already open"""
        self._webconsole.wait_till_load_complete()
        try:
            self._driver.find_element_by_xpath("//span[text() = '{0}']".format(form_name))
        except NoSuchElementException:
            return False
        return True

    @PageService()
    def submit(self):
        """submit the form by clicking on ok"""
        self._click_ok()

    @PageService()
    def submit_v1_form(self):
        """submit the v1 form by clicking on ok"""
        self.click_ok_on_v1_form()

    @PageService()
    def is_form_submitted(self):
        """Check if form submitted successfully"""
        self._webconsole.wait_till_load_complete()
        time.sleep(2)
        submit_title = self._get_opened_form_title()
        if submit_title == "Message":
            submit_message = self._get_form_submit_message()
            return submit_message == "Your form has been submitted"
        else:
            return False

    @PageService()
    def select_searchable_dropdown_value(self, label, value):
        """Select a value for input type of searchable,Dropdown"""
        self._open_search_results(label)
        self._webconsole.wait_till_load_complete()
        self._search_submit_dropdown(value)

    @PageService()
    def open_workflow(self, workflow):
        """Opens the workflow"""
        self._search_workflow(workflow)
        self._open_workflow(workflow)
        self._webconsole.wait_till_load_complete()

    @PageService()
    def submit_interaction(self, label):
        """Submit interaction"""
        self._goto_Actions()
        self._goto_Open_Actions()
        xpath = f"//a[text()='{label}']"
        self._driver.find_element_by_xpath(xpath).click()
