# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""
This page adds support for all the identity providers required during SAML authentication

IdentityProviderSupport is the main class

IdentityProviderSupport:

login_to_adfs()     --Logs in to the ADFS site in a new tab

logout_from_adfs()  --Logs out from ADFS site

identity_provider_initiated_login() --Initiates a SAML login from IDP

service_provider_initiated_login()      --Initiates a SAML login from IDP

check_single_sign_on_status()--checks the single sign on status with SAML logged in page

identity_provider_initiated_logout()    --Initiates a SAML logout from IDP

service_provider_initiated_logout() --Initiates a SAML logout from SP

switch_to_tab()            --Switches to the given page in browser

check_if_entity_is_displayed()--Check if a particular element is displayed or not

check_logged_in_state_adfs()--Checks the current logged in state of the AD user

open_url_in_new_tab()   --Opens the given url in a new tab and change focus on the new tab opened

wait_for_completion()           --  wait for the page to load

get_user_name()  --Parses the given string and gets the username from mail id or AD username format
"""
import time
from selenium.common.exceptions import NoSuchElementException
from selenium.webdriver.common.action_chains import ActionChains
from selenium.webdriver.support.ui import Select
from Web.AdminConsole.Components.panel import DropDown
from Web.Common.page_object import WebAction
from AutomationUtils import logger


class IDPSupport:
    """
     this class is used to create the Objects for the classes present in this File
    """
    def __new__(cls, admin_console, options_type):
        options = {
            'ADFS': IdentityProviderSupport,
            'OKTA': OktaSupport
        }
        if cls is not __class__:
            return super().__new__(cls)
        return options[options_type](admin_console, options_type)

    def __init__(self, admin_console, options_type):
        self.options_type = options_type
        self.__admin_console = admin_console
        self.__driver = admin_console.driver
        self.log = logger.get_log()


class IdentityProviderSupport(IDPSupport):
    """
     this class provides the support for all the operations done on a IDP
    """

    def __init__(self, admin_console, options_type):
        super().__init__(admin_console, options_type)
        self.__admin_console = admin_console
        self.__driver = admin_console.driver

    @WebAction()
    def login_to_adfs(self, saml_user_name, saml_password, adfs_app_name=None):
        """
        Logs in to the ADFS site in a new tab

        Args:

            saml_user_name  (str)-- user name to log in

            saml_password   (str)-- password of the user

            adfs_app_name   (str)-- the  AD FS app name to sign in

       Returns:
            True    --  If the login is successful

            False   --  If the login is not successful

            loaded_url  (str)   -- URL loaded in the browser

        """
        if "/adfs/ls/?SAMLRequest=" not in self.__driver.current_url:
            if adfs_app_name:
                self.__driver.find_element_by_id("idp_OtherRpRadioButton").click()
                select = Select(self.__driver.find_element_by_id('idp_RelyingPartyDropDownList'))
                select.select_by_visible_text(adfs_app_name)
                self.__driver.find_element_by_name('SignInSubmit').click()
                self.__admin_console.wait_for_completion()
        self.log.info("Page loaded %s", self.__driver.current_url)
        self.__driver.find_element_by_id('userNameInput').clear()
        self.__driver.find_element_by_id('userNameInput').send_keys(saml_user_name)
        self.__driver.find_element_by_id('passwordInput').send_keys(saml_password)
        self.__driver.find_element_by_id('submitButton').click()
        self.__admin_console.wait_for_completion()
        if self.__admin_console.check_if_entity_exists('id', 'errorText'):
            error_msg = self.__driver.find_element_by_id('errorText').text
            raise Exception("Error::" + error_msg)
        loaded_url = self.__driver.current_url
        self.log.info("Page loaded %s", loaded_url)
        return True, loaded_url

    @WebAction()
    def logout_from_adfs(self, ad_name, local_sign_out=False):
        """
        Logs out from the ADFS site

        Args:
            ad_name         (str)-- AD machine name IP that resolved

            local_sign_out  (bool)--True when we need to sign out only from  AD site

        Returns:
            True    --  If the logout is successful

            False   --  If the logout is not successful
        """
        idp_url = "https://" + ad_name + "/adfs/ls/idpinitiatedsignon.aspx"
        if self.__driver.current_url != idp_url:
            self.__admin_console.browser.open_url_in_new_tab(idp_url)
            self.__admin_console.wait_for_completion()

        if self.check_logged_in_state_adfs():
            if local_sign_out:
                self.log.info("Logging out from this site")
                self.__driver.find_element_by_id("idp_LocalSignOutRadioButton").click()
            else:
                self.log.info("Logging out from all the sites")
                self.__driver.find_element_by_id("idp_SingleSignOutRadioButton").click()
            self.__driver.find_element_by_id('idp_SignOutButton').click()
            self.__admin_console.wait_for_completion()
            if self.check_logged_in_state_adfs():
                return False
        else:
            self.log.info("No user is logged in to the site")
        return True

    @WebAction()
    def check_logged_in_state_adfs(self):
        """
        checks the current logged in state of the AD user

        Returns:
            True    --  If the user is logged in

            False   --  If the user is not logged in

        """
        sign_on_state = self.__driver.find_element_by_xpath("//div"
                                                            "[@id='idp_SignInThisSiteStatusLabel']"
                                                            "/span").text
        return sign_on_state == "You are signed in."

    @WebAction()
    def identity_provider_initiated_login(self, ad_name, web_console_url, adfs_app_name, user,
                                          password, verify_sso):
        """
        Initiates a SAML login from IDP

        Args:
            ad_name         (str)-- AD machine name/ IP that resolved

            web_console_url (str)-- webconsole URL of the client

            adfs_app_name   (str)-- ADFS app name

            user            (str)-- AD user name

            password        (str)-- AD user password

            verify_sso      (bool)--verifies if SSO is successful

        Returns:
            True    --  If the login is successful

            False   --  If the login is not successful

        """
        self.log.info("Do a IDP initiated SAML login and verify Single sign on for")
        status = False
        idp_url = "https://" + ad_name + "/adfs/ls/idpinitiatedsignon.aspx"
        self.__admin_console.browser.open_url_in_new_tab(idp_url)
        self.__admin_console.wait_for_completion()
        self.logout_from_adfs(ad_name, local_sign_out=False)
        self.__admin_console.wait_for_completion()
        flag, loaded_url = self.login_to_adfs(user, password, adfs_app_name)
        if loaded_url == web_console_url + "/applications/":
            self.log.info("Logged in to webconsole successfully")
            status = True
            if verify_sso:
                status = self.check_single_sign_on_status(ad_name, user, web_console_url)
        if loaded_url == web_console_url + "/samlAcsIdpInitCallback.do?samlAppKey=":
            self.log.info("Error during SAML login")
            error_seen = self.__driver.find_element_by_xpath("//div[@class='detailsDiv']/p").text
            self.log.info("FAILURE REASON::" + error_seen)
            status = False
        return status

    @WebAction()
    def service_provider_initiated_login(self, ad_name, web_console_url, user, password,
                                         tab_off_approach, verify_sso, sp_initiated_link=None):
        """
        Initiates a SAML login from SP
        Args:
            ad_name             (str)-- AD machine name /IP that resolved

            web_console_url     (str)-- web cosnole URl of the client

            user                (str)-- AD user name

            password            (str)-- AD user password

            tab_off_approach    (bool)--True , if tab will be pressed during saml login

            verify_sso          (bool)-- verifies if SSO is successful

            sp_initiated_link   (str)--link to initiate SAML login.Either supply tab off or SP link

        Returns:
            True    --  If the login is successful

            False   --  If the login is not successful
        """
        self.log.info("Do a SP initiated SAML login")
        status = False
        if sp_initiated_link:
            self.__admin_console.browser.open_url_in_new_tab(sp_initiated_link)
            self.__admin_console.wait_for_completion()

        if tab_off_approach and not sp_initiated_link:
            self.__admin_console.browser.open_url_in_new_tab(web_console_url)
            self.__admin_console.wait_for_completion()
            self.__driver.find_element_by_id('username').send_keys(user)
            self.__driver.find_element_by_id('continuebtn').click()
            self.__admin_console.wait_for_completion()
        flag, loaded_url = self.login_to_adfs(user, password)
        if loaded_url == web_console_url + "/applications/":
            self.log.info("Logged in to webconsole successfully")
            status = True
            if verify_sso:
                status = self.check_single_sign_on_status(ad_name, user, web_console_url)
        if loaded_url == web_console_url + "/samlAcsIdpInitCallback.do?samlAppKey=":
            self.log.info("Error during SAML login")
            error_seen = self.__driver.find_element_by_xpath("//div[@class='detailsDiv']/p").text
            self.log.info("FAILURE REASON::" + error_seen)
            status = False
        return status

    @WebAction()
    def check_single_sign_on_status(self, ad_name, user_name, web_console_url):
        """
        checks the single sign on status with SAML logged in page
        Args:
            ad_name        (str)-- name of the AD machine

            user_name       (str)-- complete user name with the company

            web_console_url(str)-- web console URL of the client

        Returns:
            True    --  If the login is successful

            False   --  If the login is not successful

        """
        idp_url = "https://" + ad_name + "/adfs/ls/idpinitiatedsignon.aspx"
        applications_page = web_console_url + "/applications/"
        if self.__admin_console.browser.open_url_in_new_tab(applications_page):
            logged_in_user = self.__driver.find_element_by_xpath('//li[@class='
                                                                 '"rightMenu hideOnExportFriendly'
                                                                 ' userNameValue '
                                                                 'vw-contains-drop-down"]'
                                                                 '/span').text
            if self.get_user_name(logged_in_user) == self.get_user_name(user_name):
                found_app = False
                apps_listed = self.__driver.find_elements_by_xpath("//div[@class='displayText "
                                                                   "vw-app-text']/span")
                for app in apps_listed:
                    if app.text == "CommCell Console GUI":
                        found_app = True
                        self.log.info("Commcell console GUI listed for SAML user")
                        break
                if not found_app:
                    self.log.info("Commcell console GUI tile not listed for SAML user")
                    return False
            else:
                self.log.info("Logged in user is not the given user")
                return False

            self.log.info("Loaded page is webconsole applicaitons page, so loading IDP in "
                          "new tab")
            self.__admin_console.browser.open_url_in_new_tab(idp_url)
            self.__admin_console.wait_for_completion()

            if self.check_logged_in_state_adfs():
                self.log.info("SSO successful at IDP")
                return True
            self.log.info("SSO failed at IDP")
            return False

    @WebAction()
    def identity_provider_initiated_logout(self, ad_name, web_console_url, verify_single_logout):
        """
        Initiates a SAML logout from IDP

            Args:

                ad_name             (str)-- AD machine name/ IP that resolved

                web_console_url     (str)-- webconsole URL of the client

                verify_single_logout(bool): verifies if single logout is successful

            Returns:

            True    --  If the logout is successful

            False   --  If the logout is not successful

        """
        self.log.info("Do a SP initiated SAML logout and verify Single logout")
        status = False
        if self.logout_from_adfs(ad_name, local_sign_out=False):
            status = True
            if verify_single_logout:
                self.__admin_console.browser.open_url_in_new_tab(web_console_url)
                self.__admin_console.wait_for_completion()
                if self.__driver.current_url == web_console_url + "/login/index.jsp":
                    self.log.info("Login screen loaded after log out at IDP")
                else:
                    self.log.info("Logout unsuccessful at SP")
                    status = False
        return status

    @WebAction()
    def service_provider_initiated_logout(self, ad_name, web_console_url, verify_single_logout):
        """
        Initiates a SAML logout from SP

            Args:
                ad_name             (str)-- AD machine name/ IP that resolved

                web_console_url     (str)-- webconsole URL of the client

                verify_single_logout(bool)-- verifies if single logout is successful

            Returns:
                True    --  If the login is successful

                False   --  If the login is not successful

        """
        self.log.info("Do a SP initiated SAML logout and verify Single logout")
        idp_url = "https://" + ad_name + "/adfs/ls/idpinitiatedsignon.aspx"
        applications_page = web_console_url + "/applications/"
        if not self.__admin_console.browser.switch_to_tab(applications_page):
            self.__admin_console.browser.open_url_in_new_tab(web_console_url)
            self.__admin_console.wait_for_completion()

        if self.__admin_console.check_if_entity_exists('xpath', '//span[@class="vw-sprite-arrow '
                                                      'vw-sprite-arrow-down"]'):
            self.log.info("Applications page found, logging out from the page")
            self.__driver.find_element_by_xpath('//span[@class="vw-sprite-arrow '
                                                'vw-sprite-arrow-down"]').click()
            logout = self.__driver.find_element_by_xpath("//li[@class='logout']/a")
            logout.click()
            self.__admin_console.wait_for_completion()

            if self.__admin_console.check_if_entity_exists('id', 'logout-message'):
                self.log.info("Logout from SP successful")
                status = True
                if verify_single_logout:
                    self.__admin_console.browser.open_url_in_new_tab(idp_url)

                    if not self.check_logged_in_state_adfs():
                        self.log.info("Logged out at IDP too")
                    else:
                        status = False
                        self.log.info("Logging out at IDP failed")
            else:
                raise Exception("No option to log out from webconsole")
        else:
            raise Exception("No option to logout from SP")
        return status

    def get_user_name(self, given_string):
        """
        Parses the given string and gets the username from mail id or AD username format
        Args:
            given_string:   (str)   user name in mail address or domain\\username format

        Returns:  (str)    user name
        """

        if '\\' in given_string:
            return given_string.split('\\')[1]
        return given_string.split('@')[0]


class OktaSupport(IDPSupport):
    """
     this class provides the support for all the operations done on OKTA IDP
    """

    def __init__(self, admin_console, options_type):
        super().__init__(admin_console, options_type)
        self.__admin_console = admin_console
        self.__driver = admin_console.driver
        self.__drop_down = DropDown(admin_console)

    @WebAction()
    def login(self, username, pwd, hostname=None):
        """
        Login to OKTA site

        Args:

            username  (str)-- user name to log in

            pwd   (str)-- password of the user

            hostname (str) -- web console hostname

       Returns:

            loaded_url  (str)   -- URL loaded in the browser

        """
        try:
            self.__driver.find_element_by_id('okta-signin-username').clear()
            self.__driver.find_element_by_id('okta-signin-username').send_keys(username)
            self.__driver.find_element_by_id('okta-signin-password').send_keys(pwd)
            self.__driver.find_element_by_id('okta-signin-submit').click()
            self.__admin_console.wait_for_completion()
            loaded_url = self.__driver.current_url
            return loaded_url
        except NoSuchElementException:
            if self.__driver.current_url == "https://" + hostname + "/webconsole/login/index.jsp#":
                self.log.info("SAML user login failed")
            return None

    @WebAction()
    def check_if_login_successful(self):
        """
        Checks if login to OKTA site is successful or not
        """
        if self.__admin_console.check_if_entity_exists("id", "logout-link"):
            return True
        else:
            raise Exception("Login Failed")

    @WebAction()
    def edit_general_settings(self, app_name, sso_url, sp_entity_id, name_id_format, attributes, group_attribute,
                              slo=False, single_logout_url=None, sp_issuer=None, certificate=None):
        """
        Edits the OKTA app details

        app_name (str)          : Name of the app whose details are to be edited
        sso_url (str)           : Single sign on url
        sp_entity_id (str)      : Service Provider's entity ID
        name_id_format (str)    : Name Id value
        attributes (dict)       : Attribute Mappings
        group_attribute(dict)   : Group Attribute Mappings
        """
        if self.__admin_console.check_if_entity_exists("id", "developer-console-dropdown"):
            e = self.__driver.find_element_by_id('developer-console-dropdown')
            hover = ActionChains(self.__driver).move_to_element(e)
            hover.perform()
            time.sleep(5)
            self.__driver.find_element_by_id('it-console-link').click()
            self.__admin_console.wait_for_completion()

        self.__driver.find_element_by_id('nav-admin-apps').click()
        self.__admin_console.wait_for_completion()
        self.__driver.find_element_by_xpath('//*[@id="app-instances"]/div/div[3]/div[1]/div/span/input').send_keys(
            app_name)
        self.__admin_console.wait_for_completion()
        self.__driver.find_element_by_xpath(
            '//*[@id="app-instances"]/div/div[3]/div[3]/div/table/tbody/tr/td[2]/div/a').click()
        self.__admin_console.wait_for_completion()
        self.__driver.find_element_by_xpath('//*[@id="tabs"]/ul/li[1]/a').click()
        self.__admin_console.wait_for_completion()
        self.__driver.find_element_by_xpath('//*[@id="edit-saml-app"]').click()
        self.__admin_console.wait_for_completion()
        time.sleep(5)
        self.__driver.find_element_by_xpath('//*[@value="Next"]').click()
        self.__admin_console.wait_for_completion()
        self.__driver.find_element_by_xpath('//*[@name="postBackURL"]').clear()
        time.sleep(2)
        self.__driver.find_element_by_xpath('//*[@name="postBackURL"]').send_keys(sso_url)
        self.__admin_console.wait_for_completion()
        self.__driver.find_element_by_xpath('//*[@name="audienceRestriction"]').clear()
        time.sleep(2)
        self.__driver.find_element_by_xpath('//*[@name="audienceRestriction"]').send_keys(sp_entity_id)
        self.__admin_console.wait_for_completion()
        if name_id_format:
            self.__drop_down.select_drop_down_values(0, name_id_format)

        if attributes:
            saml_attr, user_attr = None, None
            for x in attributes:
                user_attr = attributes[x]
                saml_attr = x

            self.__driver.find_element_by_xpath('/html/body/div[1]/div[2]/div[2]/div/div[2]/div[1]'
                                                '/form[1]/div/div[2]/div[26]/div[2]/div/div/span/span/input').clear()
            self.__driver.find_element_by_xpath(
                '/html/body/div[1]/div[2]/div[2]/div/div[2]/div[1]'
                '/form[1]/div/div[2]/div[26]/div[2]/div/div/span/span/input').send_keys(saml_attr)
            self.__driver.find_element_by_xpath('//*[@name="values"]').clear()
            self.__driver.find_element_by_xpath('//*[@name="values"]').send_keys(user_attr)
            self.__admin_console.wait_for_completion()

        if group_attribute:
            saml_attr, user_attr = None, None
            for x in group_attribute:
                user_attr = group_attribute[x]
                saml_attr = x

            self.__driver.find_element_by_xpath(
                '/html/body/div[1]/div[2]/div[2]/div/div[2]/div[1]'
                '/form[1]/div/div[2]/div[27]/div[2]/div/div/span[1]/span/input').clear()
            self.__driver.find_element_by_xpath(
                '/html/body/div[1]/div[2]/div[2]/div/div[2]/div[1]'
                '/form[1]/div/div[2]/div[27]/div[2]/div/div/span[1]/span/input').send_keys(saml_attr)
            self.__driver.find_element_by_xpath('//*[@name="filterValue"]').clear()
            self.__driver.find_element_by_xpath('//*[@name="filterValue"]').send_keys(user_attr)
            self.__admin_console.wait_for_completion()

        if slo:
            self.__driver.find_element_by_xpath('//*[@class="advanced-link float-r"]').click()
            time.sleep(2)
            self.__driver.find_element_by_name('logoutUrl').clear()
            self.__driver.find_element_by_name('logoutUrl').send_keys(single_logout_url)
            time.sleep(2)
            self.__driver.find_element_by_name('samlSpIssuer').clear()
            self.__driver.find_element_by_name('samlSpIssuer').send_keys(sp_issuer)
            time.sleep(2)
            self.__driver.find_element_by_name('file').send_keys(certificate)
            time.sleep(2)
            self.__driver.find_element_by_xpath('//*[@value="Upload Certificate"]').click()
            time.sleep(3)

        self.__driver.find_element_by_xpath('//*[@value="Next"]').click()
        self.__admin_console.wait_for_completion()
        self.__driver.find_element_by_xpath('//*[@value="Finish"]').click()
        self.__admin_console.wait_for_completion()

    @WebAction()
    def logout_from_okta(self):
        """
        Logout from OKTA site
        """
        try:
            self.__driver.find_element_by_id('logout-link').click()

        except Exception as exp:
            raise Exception("No option found to logout")

    @WebAction()
    def sp_initiated_login(self, webconsole_url, hostname, okta_url, username, pwd, tab_off_approach, verify_sso,
                           sp_initiated_link=None):
        """
        Service provider initiated login

        webconsole_url (str)        : Webconsole URL
        hostname (str)              : Commcell hostname
        okta_url (str)              : OKTA web URL
        username (str)              : name of the user to login
        pwd (str)                   : password
        tab_off_approach (bool)     : redirects to other site on click of an element
        verify_sso (bool)           : verify if SSO is acheived
        sp_initiated_link (str)     : link to initiate SAML login
        """
        status = False
        self.log.info("Do a SP initiated SAML login")
        if sp_initiated_link:
            self.__admin_console.browser.open_url_in_new_tab(sp_initiated_link)
        if tab_off_approach:
            self.__admin_console.browser.open_url_in_new_tab(webconsole_url)
        self.__admin_console.wait_for_completion()
        self.__driver.find_element_by_id('username').send_keys(username)
        self.__driver.find_element_by_id('continuebtn').click()
        self.__admin_console.wait_for_completion()
        loaded_url = self.login(username, pwd, hostname)
        if loaded_url == "https://" + hostname + "/webconsole/applications/":
            self.log.info("Logged in to webconsole successfully")
            status = True
            if verify_sso:
                self.check_single_sign_on_status(okta_url)
        if loaded_url == webconsole_url + "/samlAcsIdpInitCallback.do?samlAppKey=":
            self.log.info("Error during SAML login")
            error_seen = self.__driver.find_element_by_xpath("//div[@class='detailsDiv']/p").text
            self.log.info("FAILURE REASON::" + error_seen)
            status = False
        if loaded_url is None:
            status = False
        return status

    @WebAction()
    def check_single_sign_on_status(self, okta_url):
        """
        checks single sign on status

        okta_url (str)   : OKTA web URL

        returns : True/False
        """
        self.__admin_console.browser.open_url_in_new_tab(okta_url)
        if self.__driver.find_element_by_id('logout-link'):
            return True

    def idp_initiated_login(self, hostname, okta_url, app_name, username, pwd, verify_sso):
        """
        Identity Provider initiated login

        hostname (str)      : webconsole hostname
        okta_url (str)      : OKTA web URL
        app_name (str)      : SAML app name in OKTA
        username (str)      : Username to login
        pwd (str)           : Password
        verify_sso (bool)   : True/False

        return: current URL
        """
        status = False
        self.log.info("Do IDP initiated login")
        self.__admin_console.browser.open_url_in_new_tab(okta_url)
        self.__admin_console.wait_for_completion()
        self.login(username, pwd)
        self.__admin_console.wait_for_completion()
        self.__driver.find_element_by_xpath('//*[@id="container"]/div/header/div[1]/div/span/input').send_keys(app_name)
        self.__driver.find_element_by_xpath('//*[@id="ui-id-1"]').click()
        self.__admin_console.wait_for_completion()
        applications_page = "https://" + hostname + "/webconsole/applications/"
        self.__admin_console.browser.switch_to_tab(applications_page)
        self.__admin_console.wait_for_completion()
        if applications_page == self.__driver.current_url:
            status = True
        return status

    @WebAction()
    def saml_logout(self, hostname):
        """
        SAML logout

        loaded_url (str)    : current web URL
        hostname (str)      : Webconsole hostname
        """
        self.log.info("Do a SAML logout")
        if self.__driver.current_url == "https://" + hostname + "/webconsole/applications/":
            self.__driver.find_element_by_xpath('//*[@id="navbar-ex"]/ul/li[1]/span').click()
            self.__admin_console.wait_for_completion()
            self.__driver.find_element_by_xpath('//*[@id="header-username-menu"]/li[2]').click()
            self.__admin_console.wait_for_completion()

        elif self.__driver.current_url.startswith('https://' + hostname + '/adminconsole'):
            self.__driver.find_element_by_xpath('//*[@id="user-account-toggle"]/a/span[3]').click()
            self.__admin_console.wait_for_completion()
            self.__driver.find_element_by_xpath('//*[@id="user-header-logout"]').click()
            self.__admin_console.wait_for_completion()

        elif self.__driver.current_url.startswith('https://' + hostname +
                                                  '/webconsole/samlAcsIdpInitCallback.do?samlAppKey='):
            self.log.info("Login Failed")

        else:
            raise Exception("No option found to logout")

    @WebAction()
    def single_logout(self):
        """
        performs single logout
        """
        self.__driver.find_element_by_xpath('//*[@id="container"]/div/header/div[1]/div/ul/li[3]/div/a').click()
        time.sleep(3)
        self.__driver.find_element_by_xpath(
            '//*[@id="container"]/div/header/div[1]/div/ul/li[3]/div/div/ul/li[2]/a').click()

    @WebAction()
    def check_slo_status(self, okta_url):
        """
        checks single logout status
        """
        self.log.info(self.__driver.current_url)
        self.log.info(okta_url)
        if self.__driver.current_url == okta_url:
            return True
        else:
            return False
