# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
All the classes needed to manage the browser
"""

import os
import urllib.request
import zipfile
from abc import ABC
from abc import abstractmethod
from enum import Enum

from selenium import webdriver
from selenium.webdriver.remote.webelement import WebElement
from selenium.webdriver.firefox.options import Options
from selenium.common.exceptions import SessionNotCreatedException


from AutomationUtils import (
    config,
    logger,
    constants as automation_constants
)

_CONF = config.get_config()
_CONSTANTS = _CONF.BrowserConstants


def _get_default_browser():
    default_browser_str = _CONSTANTS.DEFAULT_BROWSER
    for browser_type in Browser.Types:
        if browser_type.value == default_browser_str:
            return browser_type
    else:
        raise ValueError(f"[{default_browser_str}] browser not found")

class Browser(ABC):
    """
    Browser has the necessary interfaces that you can use to work with the
    selenium browser

    Since browser class is an abstract class it can't be used to create any
    objects, BrowserFactory is the class that will have to be used to
    manage the creation of Browsers.

    Please look into BrowserFactory to configure the Type of browser created

    > Creating Browser Object::

        from browser import BrowserFactory

        factory = BrowserFactory()
        browser = factory.create_browser_object()

    > Opening Browser:
    Please note that the browser would not have opened after you have
    created the browser object. You need to explicitly call the open()
    method to open browser like below::

        browser.open()

    > Handling Configurations:
    By default all the configurations are implicitly handled when you
    directly call browser.open(), if at all you need to do additional
    configurations you can look into corresponding methods inside
    Browser class

    > Example Usage:

     * Below is the recommended way create the browser, by importing
     the BrowserFactory class::

        from browser import BrowserFactory

        factory = BrowserFactory()

        browser = factory.create_browser_object()
        browser.configure_proxy("machine_name", port)  # use if you want to
                                                       # config the proxy
        browser.open()  # Only at this line the browser window would be open

        txt_box = browser.driver.find_element_by_xpath("//input['username']")
        txt_box.send_keys("admin")
        browser.close()

     * We also support method chaining to create the browser object
     in one line::

        from browser import BrowserFactory

        browser = BrowserFactory().create_browser_object().set_implicit_wait(30).open()
        browser.close()

     * Using ``with`` statement: Browser has a context manager implementation, which
      automatically calls the open and close methods inside the browser. Please note
      that stacktrace would be swallowed by the Browser's close method if any exception
      is raised while closing browser::

        from browser import BrowserFactory

        factory = BrowserFactory()

        # First browser instance
        with factory.create_browser_object() as browser1:
            input = browser1.driver.find_element_by_xpath("//input['username']")
            input.send_keys("admin")

        # Second browser instance
        with factory.create_browser_object() as browser2:
            input = browser2.driver.find_element_by_xpath("//input['username']")
            input.send_keys("admin")

        # Another approach
        browser = factory.create_browser_object()
        with browser:
            input = browser2.driver.find_element_by_xpath("//input['username']")
            input.send_keys("admin")

    """

    class Types(Enum):
        """Available browser types supported"""
        CHROME = "_ChromeBrowser"
        FIREFOX = "_FirefoxBrowser"
        IE = "_IEBrowser"
        PHANTOMJS = "_PhantomJS"
        BRAVE = "_BraveBrowser"

    def __init__(self, browser_name):
        """
        Args:
             browser_name (str): Can be any string which will be used to name the
             browser internally. When working with multiple browsers the browser
             name can be used to identify the browser on those modules where browser
             object was passed as an argument
        """
        self._driver: webdriver.Chrome = None
        self._http_proxy = {}
        self._browser_name = browser_name
        self._implicit_wait = _CONSTANTS.IMPLICIT_WAIT_TIME

        self._is_proxy_configured = False
        self._is_defaults_configured = False
        self._is_grid_configured = False
        self._downloads_dir = automation_constants.TEMP_DIR
        self._drag_utils = None
        self._LOG = logger.get_log()
        self._bmp_server = None
        self._bmp_proxy_server = None

    def __enter__(self):
        self.open()
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        Browser.close_silently(self)
        return False

    @property
    def driver(self):
        """
        Returns the selenium driver inside the browser

        IMPORTANT: Python 3.6 has a bug with Properties, do not
        replace the method call with definition

        Examples:
            browser.driver.find_elements_by_xpath("//your_xpath")
        """
        return self._get_driver()

    @property
    def name(self):
        """
        Returns the name of the browser used to create the browser
        """
        return self._browser_name

    @property
    def bmp_proxy_server(self):
        """
        Returns the BrowserMobProxy server object used to record Network calls
        """
        return self._bmp_proxy_server

    @abstractmethod
    def _get_new_webdriver_object(self):
        """
        Get the browser's driver object
        """
        raise NotImplementedError

    @abstractmethod
    def _configure_default_options(self):
        """
        Loads all the default configs for the browser to work.
        """
        raise NotImplementedError

    def _get_driver(self):
        if self._driver is None:
            raise ValueError("Driver does not exist, browser might not be open")
        return self._driver

    def _get_drag_util(self):
        """Read the JS utils file for drag and drop"""
        if self._drag_utils is None:
            fpath = os.path.join(
                automation_constants.AUTOMATION_DIRECTORY,
                "Web",
                "Common",
                "DragAndDropUtil.js"
            )
            self._drag_utils = open(fpath).read()
        return self._drag_utils

    @abstractmethod
    def configure_proxy(self,
                        machine=_CONF.HttpProxy.MACHINE_NAME,
                        port=_CONF.HttpProxy.PORT):
        """
        Configures the browser to route its requests via the specified machine's
        proxy socket

        Args:
            machine (str): The machine name to use
            port (str): port number to use
        """
        raise NotImplementedError

    @abstractmethod
    def configure_grid(self, machine):
        """
        This method is used to configure Grid options for the browser to open
        on some other specified machine.

        Args:
            machine (str): name of the machine where Grid has to run.

        Currently this method is not implemented for any browser, Please
        implement if need be.
        """
        raise NotImplementedError

    @abstractmethod
    def get_browser_type(self):
        """
        Returns the type of browser currently in use.

        Returns:
            Browser.Types: Enum containing the type of browser
        """
        raise NotImplementedError

    @abstractmethod
    def get_webdriver_options(self):
        """
        Returns the options used to configure the browser before its opened.

        Chrome is configured using the webdriver.ChromeOptions class and
        firefox is configured using the webdriver.FirefoxProfile class.

        By default only one options instance is created for, this method can
        be used to read the browser's configurations.
        """
        raise NotImplementedError

    def get_downloads_dir(self):
        """Returns the downloads directory currently in use"""
        return self._downloads_dir

    def get_implicit_wait_time(self):
        """Get the currently used wait time"""
        return self._implicit_wait

    def set_downloads_dir(self, dir_path):
        """Set the download directory to be used by the browser

        Please note that this will work only if the browser if not
        already opened.
        """
        if self._driver is not None:
            raise ValueError(
                "Can't set download dir when browser is already opened")
        self._downloads_dir = dir_path

    @abstractmethod
    def set_webdriver_options(self, options):
        """
        Use this option to set the options that browser has to be configured
        with. Please note that these options would have to be configured before
        the browser is opened

        Its recommenced you check the browser type and add the necessary
        options to configure the browser

        Example::

            chrome_options = webdriver.ChromeOptions()
            if browser.get_browser_type() == Browser.TYPE_CHROME:
                browser.set_webdriver_options(chrome_options)

            firefox_profile = webdriver.FirefoxProfile()
            if browser.get_browser_type() == Browser.TYPE_FIREFOX:
                browser.set_webdriver_options(firefox_options)

        As an alternative to the above recommended procedure you can also use
        the get_webdriver_options() and configure the options on it
        """
        raise NotImplementedError

    def set_implicit_wait_time(self, time=_CONSTANTS.IMPLICIT_WAIT_TIME):
        """
        Set the implicit wait time for the driver

        At time of writing this function, selenium did not have an elegant way
        to retrieve the wait time set, so if you set driver.implicitly_wait directly
        there is no way to retrieve the currently set wait time, we save the wait
        time to the _implicit_wait variable when its set via browser. This helps
        us to temporarily override the wait time for any specific action and set
        it back to the old value
        """
        self._implicit_wait = int(time)
        return self

    def maximize_window(self):
        """
        Maximize the browser object
        """
        self.driver.maximize_window()
        return self

    def open(self, maximize=True):
        """
        Opens the browser with the default configuration.

        By default opens with the default configuration, unless explicitly
        configuration methods are called to configure the browser

        Args:
            maximize                  (bool)   :  maximizes browser
                                                  default : True

        Returns: Browser object

        """
        self._LOG.debug("Opening browser [%s]", self.name)
        if not self._is_proxy_configured:
            self.configure_proxy()
        if _CONSTANTS.ENABLE_NETWORK_MONITORING:
            import time
            from browsermobproxy import Server
            self._LOG.debug("Creating BrowserMobProxy to record Network calls")
            if self._bmp_server is None:
                self._bmp_server = Server(
                    os.path.join(automation_constants.AUTOMATION_DIRECTORY, "CompiledBins",
                                 "bmp", "bin", "browsermob-proxy.bat")
                )
                self._bmp_server.start()
                time.sleep(3)
            self._bmp_proxy_server = self._bmp_server.create_proxy()
            time.sleep(3)
            proxy_port = self._bmp_proxy_server.proxy.split(":")[1]
            self._LOG.debug(f"BrowserMobProxy started successfully on localhost:{proxy_port}")
            self.configure_proxy("localhost", proxy_port)
            self._bmp_proxy_server.new_har("cvbrowser_network_calls")
        if not self._is_defaults_configured:
            self._configure_default_options()
        self._driver = self._get_new_webdriver_object()
        self._driver.implicitly_wait(self._implicit_wait)
        if _CONSTANTS.HEADLESS_MODE:  # In headless mode maximize might not given bigger screen
            self._driver.set_window_size(_CONSTANTS.SCREEN_WIDTH, _CONSTANTS.SCREEN_HEIGHT)
        elif maximize:
            self.maximize_window()
        return self

    def goto_file(self, file_path):
        """
        Access file using browser

        Args:
            file_path               (String) : file path to access in browser

        """
        self.driver.get(file_path)

    def close(self):
        """
        Closes the Web Browser. If you have multiple tabs open in the browser
        you would need to handle the tabs yourself, this method closes the whole
        browser.

        Its recommended not to use the driver.close method directly, instead
        call this method to keep the browser object in sync with driver object
        """
        if _CONSTANTS.ENABLE_NETWORK_MONITORING:
            self._LOG.debug("Stopping BrowserMobProxy server")
            self._bmp_proxy_server.close()
            self._bmp_server.stop()
            self._bmp_proxy_server = None
            self._bmp_server = None
        self._LOG.debug("Closing browser [%s]", self.name)
        if self.driver is not None:
            self.driver.quit()
            del self._driver

    @staticmethod
    def close_silently(browser):
        """
        Use this method for webdriver cleanup inside finally statement
        where you don't want the testcase to fail because browser could
        not be closed

        Args:
             browser (Browser): Instance of Browser implementation
        """
        try:
            if browser is not None:
                if browser.driver is not None:
                    browser.close()
        except Exception as e:
            logger.get_log().warning(
                "Exception received while closing browser; " + str(e)
            )

    def click_web_element(self, web_element):
        """
        Use this to click on any UI component if selenium click fails
        with the following error even when the element is visible on the page

        `selenium.common.exceptions.WebDriverException: Message: unknown error:
        Element is not clickable at point (-5, 254)`

        Args:
            web_element: WebElement object
        """
        if not isinstance(web_element, WebElement):
            raise ValueError("argument is not an instance of WebElement")
        self._driver.execute_script("arguments[0].click();", web_element)

    def drag_and_drop_by_xpath(self, source_xpath, target_xpath):
        """
        Drag the source component identified by source xpath, and drop it on
        the target component identified by target_xpath.

        ActionChains was not working on all the pages consistently, use this as
        an alternative to it.

        Args:
            source_xpath (str): XPath of the source element
            target_xpath (str): XPath of the target element
        """
        if source_xpath.find("\"") != -1 or target_xpath.find("\"") != -1:
            raise ValueError("Double not supported in source_xpath or target_xpath")
        self.driver.find_element_by_xpath(source_xpath)  # Validate for correct XPath
        self.driver.find_element_by_xpath(target_xpath)  # Validate for correct XPath
        js = """
             function getElementByXpath(path) {
               return document.evaluate(
                    path, document, null, 
                    XPathResult.FIRST_ORDERED_NODE_TYPE, 
                    null).singleNodeValue;
             }
             source_object = getElementByXpath("%s")
             target_object = getElementByXpath("%s")
             $(source_object).simulateDragDrop({ dropTarget: $(target_object)});
             """ % (source_xpath, target_xpath)
        self.driver.execute_script(self._get_drag_util() + js)

    def open_new_tab(self):
        """
        To open a new tab in the current browser and open the URL
        """
        # To open a new tab
        self.driver.execute_script("window.open('');")

    def switch_to_latest_tab(self):
        """
        To switch to the latest tab
        """
        windows = self.driver.window_handles
        self.driver.switch_to.window(windows[-1])

    def close_current_tab(self):
        """
        To close the current tab
        """
        self.driver.execute_script("window.close('');")

    def open_url_in_new_tab(self, url):
        """
        Opens the given url in a new tab and change focus on the new tab opened

        Args:
            url     (str)--the url to be opened

        Returns:None
        """
        self.driver.execute_script("window.open('" + url + "');")
        window_list = self.driver.window_handles
        for window_id in window_list:
            self.driver.switch_to.window(window_id)
            if self.driver.current_url == url:
                return

    def switch_to_tab(self, page_to_load):
        """

        Switches to the given page in browser

        Args:

            page_to_load (str)   --  Page to be loaded in the browser

        Returns: True if the page is found otherwise False
        """
        windows_list = self.driver.window_handles
        for window_id in range(len(windows_list)):
            self.driver.switch_to.window(windows_list[window_id])
            if self.driver.current_url == page_to_load:
                return True
        return False


class _ChromeBrowser(Browser):

    def __init__(self, browser_name):
        super().__init__(browser_name)
        self.__profile = None

    def _get_new_webdriver_object(self):
        if self._driver is None:
            try:
                self._driver = webdriver.Chrome(
                os.path.join(automation_constants.AUTOMATION_DIRECTORY,
                             "CompiledBins", "chromedriver.exe"),
                options=self.get_webdriver_options())
            except SessionNotCreatedException:
                self._LOG.info(
                    'Received session not created exception for chrome, '
                    'downloading latest chromedriver'
                )
                self._download_latest_driver()
                self._driver = webdriver.Chrome(
                    os.path.join(automation_constants.AUTOMATION_DIRECTORY,
                                 "CompiledBins", "chromedriver.exe"),
                    options=self.get_webdriver_options())

        return self._driver

    def _download_latest_driver(self):
        """Download latest chromedriver"""
        path = os.path.join(automation_constants.AUTOMATION_DIRECTORY, "CompiledBins")
        fpath = os.path.join(path, 'chromedriver.zip')
        latest_release_url = 'https://chromedriver.storage.googleapis.com/LATEST_RELEASE'
        self._LOG.info('Finding latest chromedriver version')
        file = urllib.request.urlopen(latest_release_url)
        version = file.read()
        self._LOG.info(f'Downloading chromedriver version {version.decode()} to : ' + fpath)
        url = (
            f'https://chromedriver.storage.googleapis.com/{version.decode()}/'
            f'chromedriver_win32.zip'
        )
        file = urllib.request.urlopen(url)
        # save file to system
        with open(fpath, 'wb') as output:
            output.write(file.read())
        self._LOG.info(f'unzipping chromedriver to path {path}')
        with zipfile.ZipFile(fpath, 'r') as zip_ref:
            zip_ref.extractall(path)

    def _configure_default_options(self):
        profile = self.get_webdriver_options()
        profile.add_experimental_option(
            "prefs",
            {"download.default_directory": self.get_downloads_dir(),
             "download.prompt_for_download": False,
             "download.directory_upgrade": True,
             "plugins.plugins_disabled": ["Chrome PDF Viewer"],
             "safebrowsing.enabled": True})
        profile.add_argument('--ignore-certificate-errors')
        if _CONSTANTS.HEADLESS_MODE:
            profile.add_argument('--headless')
            profile.add_argument('--no-sandbox')
            profile.add_argument('--disable-gpu')
            profile.add_argument(
                f'--window-size={_CONSTANTS.SCREEN_WIDTH},{_CONSTANTS.SCREEN_HEIGHT}'
            )
            profile.add_experimental_option('excludeSwitches', ['enable-logging'])


    def get_browser_type(self):
        return Browser.Types.CHROME

    def get_webdriver_options(self):
        if self.__profile is None:
            self.__profile = webdriver.ChromeOptions()
        return self.__profile

    def configure_proxy(self,
                        machine=_CONF.HttpProxy.MACHINE_NAME,
                        port=_CONF.HttpProxy.PORT):
        self._is_proxy_configured = True
        if machine != "" and port != "":
            profile = self.get_webdriver_options()
            profile.add_argument("--proxy-server=http://%s:%s" % (machine, str(port)))

    def set_webdriver_options(self, options):
        self._is_defaults_configured = True
        self.__profile = options

    def configure_grid(self, machine):
        raise NotImplementedError(
            "Method not implemented for Chrome, please implement if need be")


class _FirefoxBrowser(Browser):

    def __init__(self, browser_name):
        super().__init__(browser_name)
        self.__profile = None
        self.__options = None
        self._allowed_mime_types = (
            "application/pdf, text/csv,text/html, text/plain, application/xml, "
            "text/xml, application/octet-stream, application/vnd.ms-word,"
            "application/vnd.openxmlformats-officedocument.wordprocessingml.document, "
            "application/vnd.openxmlformats-officedocument.presentationml.presentation, "
            "application/vnd.ms-excel, application/exe, application/vnd.ms-htmlhelp")

    def _get_new_webdriver_object(self):
        if self._driver is None:
            self._driver = webdriver.Firefox(
                executable_path=os.path.join(
                    automation_constants.AUTOMATION_DIRECTORY,
                    "CompiledBins", "geckodriver.exe"),
                firefox_profile=self.get_webdriver_options(),
                options=self.get_options(),
                service_log_path=None
            )
        return self._driver

    def _configure_default_options(self):
        profile = self.get_webdriver_options()
        profile.set_preference("browser.download.folderList", 2)
        profile.set_preference("browser.helperApps.alwaysAsk.force", False)
        profile.set_preference("browser.download.manager.showWhenStarting",
                               False)
        profile.set_preference("browser.download.dir", self.get_downloads_dir())
        profile.set_preference("plugin.disable_full_page_plugin_for_types",
                               "application/pdf")
        profile.set_preference("pdfjs.disabled", True)
        profile.set_preference("browser.download.manager.showAlertOnComplete",
                               False)
        profile.set_preference("browser.download.viewableInternally.enabledTypes", "")
        profile.set_preference("browser.download.panel.shown", True)
        profile.set_preference("browser.helperApps.neverAsk.saveToDisk",
                               self._allowed_mime_types)
        if _CONSTANTS.HEADLESS_MODE:
            options = self.get_options()
            options.add_argument('--headless')
        profile.update_preferences()

    def get_browser_type(self):
        return Browser.Types.FIREFOX

    def get_webdriver_options(self):
        if self.__profile is None:
            self.__profile = webdriver.FirefoxProfile()
        return self.__profile

    def get_options(self):
        if self.__options is None:
            self.__options = Options()
        return self.__options

    def configure_proxy(self,
                        machine=_CONF.HttpProxy.MACHINE_NAME,
                        port=_CONF.HttpProxy.PORT):
        self._is_proxy_configured = True
        if machine != "" and port != "":
            profile = self.get_webdriver_options()
            profile.set_preference("network.proxy.type", 1)
            profile.set_preference("network.proxy.http", str(machine))
            profile.set_preference("network.proxy.http_port", str(port))
            profile.update_preferences()

    def set_webdriver_options(self, options):
        self._is_defaults_configured = True
        self.__profile = options

    def configure_grid(self, machine):
        raise NotImplementedError(
            "Method not implemented for Firefox, please implement if need be")


class _IEBrowser(Browser):

    def __init__(self, browser_name):
        super().__init__(browser_name)
        self.__capabilities = None
        self._implicit_wait = 10
        self._is_proxy_configured = True  # PTo skip proxy setting

    def _get_new_webdriver_object(self):
        driver_path = os.path.join(
            automation_constants.AUTOMATION_DIRECTORY,
            "CompiledBins", "IEDriverServer.exe")
        return webdriver.Ie(driver_path)

    def _configure_default_options(self):
        self.__capabilities = self.get_webdriver_options()
        self.__capabilities["ignoreZoomSetting"] = True
        self.__capabilities["nativeEvents"] = True
        self.__capabilities["ignoreProtectedModeSettings"] = True

    def get_browser_type(self):
        return Browser.Types.IE

    def get_webdriver_options(self):
        if self.__capabilities is None:
            self.__capabilities = webdriver.DesiredCapabilities.INTERNETEXPLORER
        return self.__capabilities

    def configure_proxy(self,
                        machine=_CONF.HttpProxy.MACHINE_NAME,
                        port=_CONF.HttpProxy.PORT):
        raise NotImplementedError

    def set_webdriver_options(self, options):
        self._is_defaults_configured = True
        self.__capabilities = options

    def configure_grid(self, machine):
        raise NotImplementedError(
            "Method not implemented for IE, please implement if need be")


class _PhantomJS(Browser):

    def __init__(self, browser_name):
        super().__init__(browser_name)
        self.__service_args = []

    def _get_new_webdriver_object(self):
        return webdriver.PhantomJS(
            os.path.join(automation_constants.AUTOMATION_DIRECTORY,
                         "CompiledBins", "phantomjs.exe"),
            service_args=self.get_webdriver_options()
        )

    def _configure_default_options(self):
        self.__service_args.append('--ignore-ssl-errors=true')

    def get_browser_type(self):
        return Browser.Types.PHANTOMJS

    def get_webdriver_options(self):
        return self.__service_args

    def configure_proxy(self,
                        machine=_CONF.HttpProxy.MACHINE_NAME,
                        port=_CONF.HttpProxy.PORT):
        self._is_proxy_configured = True
        if machine != "" and port != "":
            self.__service_args.append('--proxy={%s}:{%s}' % (machine, port))
            self.__service_args.append('--proxy-type=http')

    def set_webdriver_options(self, options):
        self._is_defaults_configured = True
        self.__service_args = options

    def configure_grid(self, machine):
        raise NotImplementedError(
            "Method not implemented for PhantomJS, please implement if need be")


class _BraveBrowser(_ChromeBrowser):
    def __init__(self, browser_name):
        super().__init__(browser_name)
        self.__profile = None

    def get_webdriver_options(self):
        if self.__profile is None:
            self.__profile = webdriver.ChromeOptions()
            self.__profile.binary_location = _CONSTANTS.BRAVE_BROWSER_PATH
        return self.__profile


class BrowserFactory(object):
    """
    This class is used to control and manage the creation of the browsers
    used by the TestCases.

    Since this class is a singleton, all the instances would return the same
    instance, so calling the below code would always return True::

        factory1 = BrowserFactory()
        factory2 = BrowserFactory()
        return factory1 is factory2

    To create single instance of all browser types, use the Browser.Types enum,
    Example::

        factory = BrowserFactory()
        for _type in Browser.Types:
            browser = factory.create_browser_object(browser_type=_type)
            browser.open()

    For info on creating and configuring browsers, please refer Browser class
    """

    _instance = {}

    def __init__(self):
        self._browsers = {}
        self._LOG = logger.get_log()

    def __new__(cls, *args, **kwargs):
        if cls not in cls._instance:
            instance = super(BrowserFactory, cls).__new__(cls)
            cls._instance[cls] = instance
        return cls._instance[cls]

    def create_browser_object(self,
                              browser_type=_get_default_browser(),
                              name="defaultBrowser") -> Browser:
        """Creates and configures the type of browser to be created

        Args:
            browser_type (Browser.Type): Use the enum to create any
                specific browser other than the default one set on
                config.json file
            name (str): Name of the browser
        """
        self._LOG.debug(
            "Creating [%s] browser object with name [%s]",
            browser_type.value,
            name
        )
        if browser_type not in Browser.Types:
            raise KeyError(f"Unsupported browser type [{browser_type.value}] received")
        browser = globals()[browser_type.value]
        self._browsers[name] = browser(name)
        return self._browsers[name]

    def get_all_created_browsers(self):
        """Returns all the browser objects created by the BrowserFactory
         This also includes the browsers which are already closed
        """
        return self._browsers
