# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""All the actions common to Adminconsole application go here"""

import time

from AutomationUtils import (
    logger,
    config
)
from Web.Common.cvbrowser import Browser
from Web.Common.exceptions import (
    CVTimeOutException,
    CVWebAutomationException
)
from Web.Common.page_object import (
    WebAction,
    PageService
)

from Web.AdminConsole.Helper.AdminConsoleBase import AdminConsoleBase
from Web.AdminConsole.AdminConsolePages.LoginPage import LoginPage
from Web.AdminConsole.AdminConsolePages.AdminPage import AdminPage


_CONSTANTS = config.get_config()


class AdminConsole(AdminConsoleBase):

    """
    Handle's the operations on the Adminconsole

    Examples:

     Recommended way to use the Adminconsole class is as below::

        factory = BrowserFactory()
        browser = factory.create_browser_object()
        browser.open()

        adminconsole1 = Adminconsole(browser, "machine_name")
        adminconsole1.login()  # pass the creds as args to override config file

        # your code goes here

        adminconsole.logout()
        browser.close()

     Adminconsole has a context manager implementation which has to be used as below ::

        factory = BrowserFactory()
        with factory.create_browser_object() as browser:

            # Adminconsole login with default creds from config file
            with Adminconsole(browser, "machine_name") as adminconsole:
                # since 'with' block is used, you are already on the apps page
                # when you hit this line
                adminconsole.your_operations()
                # logout is automatically called by the end of this block

            #Adminconsole login with creds when explicitly specified
            with Adminconsole(browser, "machine", username="ViewAll", password="view") as ac:
               ac.your_operations()

        # browser's close method would be automatically invoked

    """
    def __init__(self,
                 web_browser,
                 machine,
                 username=_CONSTANTS.ADMIN_USERNAME,
                 password=_CONSTANTS.ADMIN_PASSWORD,
                 enable_ssl=False,
                 global_console=None):
        """
        Credentials supplied during creation on Adminconsole class is only
        used when login is done using `with` statement. Its not using by
        login function in any way.

        Args:
            web_browser (Browser): The browser object to use.
            machine (str): Name of the webconsole machine,
                eg. cloud.commvault.com
            username (str): username to use when ``with`` resource block is
                used for login, by default taken from the config file if
                not supplied
            password (str): password to use when ``with`` resource block is
                used for logout operation, by default taken from the config
                file if not supplied
            enable_ssl (boolean): True if 'https' has to be used, False has
                to be set for 'http'
            global_console (Boolean): if True: global console base URL will be formed
                                      else: adminconsole base URL will be formed
        """
        self.browser = web_browser
        self.driver = web_browser.driver
        self.machine_name = machine
        self.username = username
        self.password = password
        if global_console:
            self.base_url = "%s://%s/global/"
        else:
            self.base_url = "%s://%s/adminconsole/"
        if enable_ssl:
            self.base_url = self.base_url % ("https", self.machine_name)
        else:
            self.base_url = self.base_url % ("http", self.machine_name)
        self._LOG = logger.get_log()
        super().__init__(self.driver)
        self.__navigator = None

    def __enter__(self):
        self.login(self.username, self.password)
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        AdminConsole.logout_silently(self)
        return False

    def __str__(self):
        return f"<Adminconsole host=[{self.machine_name}] id=[{id(self)}]>"

    @property
    def navigator(self):
        if self.__navigator is None:
            self.__navigator = AdminPage(self.driver)
            if self.props:
                self.props.update(self.__navigator.props)
            else:
                self.props = self.__navigator.props
            self.ext = self.__navigator.ext
        return self.__navigator

    @WebAction()
    def _open_sso_disabled_login_page(self):
        """Open SSO Disabled login page"""
        self.driver.get(self.base_url + "login?skipSSO=true")
        self.wait_for_completion()

    @WebAction()
    def _open_sso_enabled_login_page(self):
        """Open SSO Enabled login page"""
        self.driver.get(self.base_url)
        self.wait_for_completion()

    @WebAction()
    def _click_username_dropdown(self):
        """ Method to expand user settings drop down """
        user_settings_drop_down = self.driver.find_element_by_xpath(
            "//div[@data-ng-if='showUserSettingsDropdown']")
        user_settings_drop_down.click()

    @WebAction()
    def _click_logout(self):
        """ Method to click on logout option in user settings drop down"""
        logout = self.driver.find_element_by_xpath("//a[@id='user-header-logout']")
        logout.click()

    @WebAction()
    def __get_login_name(self):
        """Get name of the user logged in"""
        username = self.driver.find_element_by_xpath(
            "//div[@id='user-account-toggle']//span[@class='header-menu-no-small-size']"
        )
        return username.text

    @WebAction(log=False)
    def _is_logout_page(self):
        """Check if current page is logout page"""
        return self.driver.title == "Command Center"

    @WebAction()
    def _click_service_commcell_dropdown(self):
        """ Method to expand multicommcell drop down """
        service_commcells_drop_down = self.driver.find_element_by_xpath(
            "//div[@class='uib-dropdown commcell-selection header-menu ng-scope dropdown']")
        service_commcells_drop_down.click()
        time.sleep(1)

    @WebAction()
    def _select_service_commcell(self, value):
        """ Method to select service commcell"""
        service_commcells = self.driver.find_element_by_xpath(
            "//div[@class='uib-dropdown commcell-selection header-menu ng-scope dropdown open']"
            "/.//span[contains(text(),'" + value + "')]")
        service_commcells.click()
        time.sleep(1)

    @PageService(hide_args=True)
    def login(
            self,
            username=_CONSTANTS.ADMIN_USERNAME,
            password=_CONSTANTS.ADMIN_PASSWORD,
            enable_sso=_CONSTANTS.WebConsole.SSO_LOGIN,
            stay_logged_in=_CONSTANTS.WebConsole.STAY_LOGGED_IN,
            max_tries=_CONSTANTS.WebConsole.LOGIN_MAX_TRIES,
            service_commcell=None
    ):
        """Login to Adminconsole

        Credentials supplied during creation on Adminconsole class is only
        used when login is done using ``with`` statement. Its not using by
        login function in any way.

        Args:
            username (str): username to login with, if its not supplied the
                default username saved on the config file is used.
            password (str): password to login with, if not supplied it would
                use the default password using on the config file
            enable_sso (bool): If enabled login won't enter username or password,
                and Login button will not be clicked
            stay_logged_in (boolean): Checks the 'Stay Logged In' when 'True',
                leaves it as it is if 'False'. Default value is set from the
                config file.
            max_tries (int): Maximum number of login attempts, if case of any
                webdriver exceptions
            service_commcell(str): Performs router login at provided service commcell if configured/available
        """
        stay_logged_in = stay_logged_in or _CONSTANTS.SECURITY_TEST
        for i_try in range(max_tries):
            try:
                if enable_sso is False:
                    self._open_sso_disabled_login_page()
                    LoginPage(self).login(username, password, stay_logged_in, service_commcell=service_commcell)
                else:
                    self._open_sso_enabled_login_page()
                    self.wait_for_completion()
                time.sleep(2)  # Button load screen disappears earlier
                break
            except CVTimeOutException as ex:  # Timeout will be retried
                if max_tries == i_try + 1:
                    raise ex
                self._LOG.error(
                    "FH Unable to login, retrying again, received "
                    "error [%s]" % str(ex)
                )
        self.close_popup()

    @PageService()
    def logout(self):
        """Logout from Adminconsole"""
        if not _CONSTANTS.SECURITY_TEST:
            if not self._is_logout_page():
                self.wait_for_completion()
                self._click_username_dropdown()
                time.sleep(1)
                self._click_logout()
                time.sleep(1)
                self.wait_for_completion()
            else:
                raise CVWebAutomationException(
                    "Unable to logout from [%s] page" % self.driver.current_url)

    @PageService()
    def select_service_commcell(self, service_commcell_name):
        """Selects the specified service commcell """

        self._click_service_commcell_dropdown()
        self._select_service_commcell(value=service_commcell_name)
        self.wait_for_completion()

    @staticmethod
    def logout_silently(adminconsole):
        """
        Use this logout for resource cleanup inside finally statement
        when you don't want any exception to be raised causing a testcase
        failure.

        This logout is never to be used to check the working of
        Adminconsole logout

        Args:
             adminconsole (AdminConsole): Adminconsole object
        """
        try:
            if adminconsole is not None:
                adminconsole.logout()
        except Exception as err:
            _LOG = logger.get_log()
            err = ";".join(str(err).split("\n"))
            _LOG.warning(f"Silent logout received exception; {err}")

    @PageService()
    def get_login_name(self):
        """Get name of the user currently logged in

        To get the username passed during login, use the self.username variable
        """
        return self.__get_login_name()

    @PageService()
    def goto_adminconsole(self):
        """ To go to adminconsole home page"""
        self._open_sso_disabled_login_page()
