# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides methods for all the actions that can be done of the Subclient Details page.


Classes:

    VsaSubclientDetails() ---> SubclientDetails() --->  AdminConsoleBase() ---> object()


VsaSubclientDetails  --  This class contains all the methods for action in a particular
                        subclient's page

Functions:

    subclient_storage_pool_info()   --  Gets the list of all storage pools

    schedule_info()     --  Gets the list of all schedules associated with the subclient

    manage_content()    --  Manages the collection content by adding or removing VMs and rules

    subclient_settings()-- Edits the no of readers assigned to the collection and the backup type
                            of the virtual machine

    subclient_summary() --  Gets the last backup time, last backup size, next backup time, etc

    snap_mount_esx_host() -- Sets the given ESX host as the snap mount ESX

    set_backup_validation() --  Enables and sets the credentials to run backup validation

    schedule_backup_validation()  --   Schedules Backup validation

    run_validate_backup()   --  Runs backup validation job

"""

import re
import time
from Web.AdminConsole.Components.panel import Backup
from Web.AdminConsole.Components.panel import PanelInfo, ModalPanel, DropDown
from Web.AdminConsole.Components.dialog import ModalDialog
from Web.Common.page_object import (
    PageService
)


class VsaSubclientDetails:
    """
    This class contains all the methods for action in a particular subclient's page
    """

    def __init__(self, admin_console):
        """ """
        self.__admin_console = admin_console
        self.__admin_console._load_properties(self, unique=True)
        self.__driver = admin_console.driver
        self.__panel_info_obj = PanelInfo(admin_console)

    @PageService()
    def subclient_storage_pool_info(self):
        """
        Displays the storage pool info of the subclient

        Returns:
            pool_info   (list):  lists the storage pool info

        """
        self.__admin_console.select_configuration_tab()
        elements = self.__driver.find_elements_by_xpath("//cv-tile-component[@data-title='"
                                                        "Storage pool']/div/div[3]/div/div[2]/ul/li")
        pool_info = {}
        for element in elements:
            pool_info[element.find_element_by_xpath(
                "./span[1]").text] = element.find_element_by_xpath("./span[2]").text
        self.__admin_console.log.info("The storage pool associated with plan of the subclient is %s",
                                      str(pool_info))
        return pool_info

    @PageService()
    def schedule_info(self):
        """
        Displays all the schedules associated with the subclient

        Returns:
            schedules  (list):  list of all schedules associated to the subclient

        """
        self.__admin_console.select_configuration_tab()
        elements = self.__driver.find_elements_by_xpath("//cv-tile-component[@data-title="
                                                        "'Schedules']/div/div[3]/div/div[2]/ul")
        schedules = []
        for element in elements:
            schedules.append(element.find_element_by_xpath("./li").text)
        self.__admin_console.log.info("The schedules associated with the subclient are %s", str(schedules))
        return schedules

    @PageService()
    def restore(self, recovery_time=None):
        """
        Opens the select restore page

        Args:
            recovery_time   (basestring):   the backup date in 01-September-1960 format

        Raises:
            Exception:
                if there is no option to restore

        """
        self.__admin_console.select_overview_tab()
        if recovery_time:
            calender = {}
            calender['date'], calender['month'], calender['year'] = recovery_time.split("-")
            self.__admin_console.date_picker(calender)
        panel_info = PanelInfo(self.__admin_console, self.__admin_console.props['label.recoveryPoint'])
        panel_info.open_hyperlink_on_tile(self.__admin_console.props['header.restore'])
        self.__admin_console.wait_for_completion()

    @PageService()
    def backup_now(self, bkp_type):
        """
        Starts a backup job for the collection

        Args:

            bkp_type (BackupType): the backup level, among the type in Backup.BackupType enum

        Returns:
            Job ID of backup job

        """
        self.__admin_console.select_overview_tab()
        self.__driver.execute_script("window.scrollTo(0,0)")
        self.__admin_console.access_menu(self.__admin_console.props['action.backup'])
        backup = Backup(self.__admin_console)
        return backup.submit_backup(bkp_type)

    @PageService()
    def manage_content(self):
        """
        Manages the collection content by adding or removing VMs and rules

        Raises:
            Exception:
                if there is no link to Manage the subclient content

        """
        self.__admin_console.select_overview_tab()
        panel_info = PanelInfo(self.__admin_console, "Content")
        panel_info.open_hyperlink_on_tile(self.__admin_console.props['label.nav.manage'])
        self.__admin_console.wait_for_completion()

    @PageService()
    def subclient_settings(self, reader=2, collect_file_details=False, bkp_type=None, cbt=True,
                           auto_detect_owner=False, transport_mode=None,
                           free_space=None, ds_free_space_check=True):
        """
        Edits the no of readers assigned to the collection and the backup type
        of the virtual machine

        Args:
            reader                   (int):         the number of readers

            collect_file_details     (bool):        to enable or disable metadata collection

            bkp_type                 (basestring):  app / crash consistent backup

            cbt                      (bool):        enable / disable CBT

            auto_detect_owner        (bool):        to auto detect the owner of the VM

            transport_mode           (basestring):  the transport mode to be used for backup

            free_space               (int):         the amount of free space to be set

            ds_free_space_check      (bool):        to enable / disable datastore freespace check

        Raises:
            Exception:
                when the subclient setting could not be edited or
                when there is an error while updating the settings

        """
        self.__admin_console.select_configuration_tab()
        if cbt:
            self.__admin_console.toggle_enable("Use changed block tracking")
        else:
            self.__admin_console.toggle_disable("Use changed block tracking")

        if auto_detect_owner:
            self.__admin_console.toggle_enable("Auto detect VM Owner")
        else:
            self.__admin_console.toggle_disable("Auto detect VM Owner")

        if collect_file_details:
            self.__admin_console.toggle_enable("Collect file details")
        else:
            self.__admin_console.toggle_disable("Collect file details")

        if self.__panel_info_obj.check_if_hyperlink_exists_on_tile("Edit"):
            self.__admin_console.tile_select_hyperlink("Settings", "Edit")
            self.__admin_console.fill_form_by_id("noOfReaders", reader)

            if bkp_type:
                self.__driver.find_element_by_xpath(
                    "//div[@class='frequency']//span[contains("
                    "text(),'" + bkp_type + "')]").click()

            if transport_mode:
                self.__admin_console.select_value_from_dropdown("transportMode", transport_mode)

            if self.__admin_console.check_if_entity_exists("xpath", "//input[@id='datastoreFreespaceCheckCB']"):
                if not ds_free_space_check:
                    self.__admin_console.checkbox_deselect("datastoreFreespaceCheckCB")
                else:
                    self.__admin_console.checkbox_select("datastoreFreespaceCheckCB")
                    if free_space:
                        self.__admin_console.fill_form_by_id("datastoreFreespaceRequired", free_space)

            self.__admin_console.submit_form()
            self.__admin_console.check_error_message()
        else:
            raise Exception("There is no option to edit the subclient properties")

    @PageService()
    def subclient_summary(self):
        """
        Provides the summary info of the subclient like last backup time, next backup time

        Returns:
            summary_info    (dict): info about the subclient

        """
        panel_info = PanelInfo(self.__admin_console, self.__admin_console.props['label.summary'])
        return panel_info.get_details()

    @PageService()
    def snap_mount_esx_host(self, esx_host):
        """
        Sets the given ESX host as the snap mount ESX

        Args:
            esx_host (basestring):      the esx host to set as snap mount esx

        """
        self.__admin_console.select_configuration_tab()
        self.__admin_console.tile_select_hyperlink("Snap mount esx", "Edit")
        self.__admin_console.select_destination_host(esx_host)
        self.__admin_console.submit_form()

    @PageService()
    def get_storage_policy(self):
        """
        Gets the storage policy or plan associated with the subclient

        Returns:
            storage_policy  (basestring):   the storage policy associated with the subclient

        """
        policies_tile = "//cv-entity-plan-policy-tiles//span[contains(., 'Policies')]/../.."
        plan_tile = "//cv-entity-plan-policy-tiles//span[contains(., '" + \
                    self.__admin_console.props['label.plan'] + "')]/../.."
        summary_plan_tile = "//cv-tile-component[@data-title='" + \
                            self.__admin_console.props['label.summary'] + "']//cv-plan-select//a"
        if self.__admin_console.check_if_entity_exists("xpath", policies_tile):
            policy = self.__driver.find_element_by_xpath(policies_tile + "//ul/li[1]/a").text
        elif self.__admin_console.check_if_entity_exists("xpath", plan_tile):
            policy = self.__driver.find_element_by_xpath(plan_tile + "/div/div[2]//a").text
            policy = re.split(" \( Server \)", policy, flags=re.IGNORECASE)[0].strip()
        else:
            if not self.__admin_console.check_if_entity_exists("xpath", summary_plan_tile):
                raise Exception("The Plan tile or Storage policy or plan in summary tile is not present")
            policy = self.__driver.find_element_by_xpath(summary_plan_tile).text
        return policy

    @PageService()
    def run_file_indexing(self):
        """
        Runs the file indexing job for the subclient

        Returns:
            job_id  (basestring):   the job id of the content indexing job

        """
        self.__admin_console.select_configuration_tab()
        self.__admin_console.tile_select_hyperlink("File indexing", "Run file indexing")
        self.__admin_console.click_button("Yes")
        return self.__admin_console.get_jobid_from_popup()

    @PageService()
    def set_backup_validation(self, backup_validation_options):
        """
        Sets backup Validation details

        Args:
            backup_validation_options    (dict):     Credential details of the user

        Returns:

        """
        self.__admin_console.select_configuration_tab()
        panel_details = PanelInfo(self.__admin_console, self.__admin_console.props['label.backupValidation'])
        panel_details.enable_toggle(self.__admin_console.props['label.enableBackupValidation'])
        if panel_details.check_if_hyperlink_exists_on_tile('Edit'):
            panel_details.edit_tile()
        my_modal = ModalPanel(self.__admin_console)
        if backup_validation_options.get('recovery_target'):
            self.__admin_console.disable_toggle(toggle_id='liveMountUsingSourceEsx')
            drop_details = DropDown(self.__admin_console)
            drop_details.select_drop_down_values(values=[backup_validation_options['recovery_target']],
                                                 drop_down_id='lifeCyclePolicyModal_isteven-multi-select_#7626')
        else:
            self.__admin_console.enable_toggle(toggle_id='liveMountUsingSourceEsx')
        if backup_validation_options.get('schedule', False):
            self.schedule_backup_validation()
        else:
            self.__admin_console.disable_toggle(toggle_id='toggleBackupValidationSchedule')
        # custom validation
        self.__driver.find_element_by_xpath(f"//span[contains(text(),"
                                            f"'{self.__admin_console.props['label.customApplication']}')]").click()
        # self.expand_accordin(self.props['label.customApplication'])
        if backup_validation_options.get('path_win') or backup_validation_options.get('path_unix'):
            if backup_validation_options.get('path_win'):
                if self.__admin_console.check_if_entity_exists('id', 'cv_windowsCustomScript_add'):
                    self.__admin_console.access_tile('cv_windowsCustomScript_add')
                else:
                    if self.__admin_console.check_if_entity_exists('id', 'cv_windowsCustomScript_edit'):
                        self.__admin_console.access_tab_from_dropdown('Edit', 4)
                    else:
                        raise Exception("Not able to add windows custom script")
                self.__admin_console.fill_form_by_id("uncNetworkPath", backup_validation_options.get('path_win'))
                if backup_validation_options.get('script_argument_win'):
                    self.__admin_console.fill_form_by_id("scriptArguments",
                                                         backup_validation_options.get('script_argument_win'))
                my_modal = ModalPanel(self.__admin_console)
                my_modal.submit()
            if backup_validation_options.get('path_unix'):
                if self.__admin_console.check_if_entity_exists('id', 'cv_unixCustomScript_add'):
                    self.__admin_console.access_tile('cv_unixCustomScript_add')
                else:
                    if self.__admin_console.check_if_entity_exists('id', 'cv_windowsCustomScript_edit'):
                        self.__admin_console.access_tab_from_dropdown('Edit', 6)
                    else:
                        raise Exception("Not able to add unix custom script")
                self.__admin_console.fill_form_by_id("uncNetworkPath", backup_validation_options.get('path_unix'))
                if backup_validation_options.get('script_argument_unix'):
                    self.__admin_console.fill_form_by_id("scriptArguments",
                                                         backup_validation_options.get('script_argument_unix'))
                my_modal = ModalPanel(self.__admin_console)
                my_modal.submit()
        else:
            self.__admin_console.log.info("Clearing custom scripts, if any")
            if self.__admin_console.check_if_entity_exists('id', 'cv_windowsCustomScript_edit'):
                self.__admin_console.access_tab_from_dropdown('Delete', 4)
            if self.__admin_console.check_if_entity_exists('id', 'cv_unixCustomScript_edit'):
                self.__admin_console.access_tab_from_dropdown('Delete', 6)
        # Guest credentials
        self.__driver.find_element_by_xpath(f"//span[contains(text(),"
                        f"'{self.__admin_console.props['VsaSubclientDetails']['label.guestCredentials']}')]").click()
        # self.expand_accordin(self.props['label.guestCredentials'])
        self.__admin_console.select_radio('user')
        self.__admin_console.fill_form_by_id("uname", backup_validation_options.get('user', ''))
        self.__admin_console.fill_form_by_id("pass", backup_validation_options.get('password', ''))
        my_modal.submit()

    def schedule_backup_validation(self, only_schedule=False, after_time=None):
        """
        Scheduling backup validation
        Args:
            only_schedule           (bool): If we are only editing the schedule of the backup validation

            after_time:             (integer): set the schedule after number of seconds

        """
        # can be used schedules.py to enhance this
        if only_schedule:
            self.__admin_console.select_configuration_tab()
            panel_details = PanelInfo(self.__admin_console, self.__admin_console.props['label.backupValidation'])
            panel_details.enable_toggle(self.__admin_console.props['label.enableBackupValidation'])
            if panel_details.check_if_hyperlink_exists_on_tile('Edit'):
                panel_details.open_hyperlink_on_tile('Edit')
        self.__admin_console.enable_toggle(toggle_id='toggleBackupValidationSchedule')
        if self.__admin_console.check_if_entity_exists(
                "xpath",
                '//label[@class="col-xs-12 col-md-6 col-lg-6 padding-top-5"]'):
            _text = self.__driver.find_element_by_xpath(
                '//label[@class="col-xs-12 col-md-6 col-lg-6 padding-top-5"]').text
            self.__driver.find_element_by_xpath("//a[contains(text(),'" + _text + "')]").click()
        self.__admin_console.fill_form_by_id('name', self.__driver.session_id)
        drop_details = DropDown(self.__admin_console)
        drop_details.select_drop_down_values(values=['Daily'],
                                             drop_down_id='addScheduleModal_isteven-multi-select_#5119')

        current_time = time.time()
        # adding 3 hours in the schedule
        if not after_time:
            after_time = 3 * 60 * 60
        schedule_time = current_time + after_time
        schedule_time = {
            'hour': time.strftime("%I", time.localtime(schedule_time)),
            'min': time.strftime("%M", time.localtime(schedule_time)),
            'meridiem': time.strftime("%p", time.localtime(schedule_time))
        }
        input_val = self.__driver.find_element_by_xpath('//div[@id="scheduleTime"]//input[@placeholder="HH"]')
        input_val.clear()
        input_val.send_keys(schedule_time['hour'])
        input_val = self.__driver.find_element_by_xpath('//div[@id="scheduleTime"]//input[@placeholder="MM"]')
        input_val.clear()
        input_val.send_keys(schedule_time['min'])
        if self.__admin_console.check_if_entity_exists(
                'xpath', '//button[@class="btn btn-default text-center ng-binding"][contains(text(),"AM")]'):
            input_val = 'AM'
        else:
            input_val = 'PM'
        if schedule_time['meridiem'] != input_val:
            self.__driver.find_element_by_xpath(
                '//button[@class="btn btn-default text-center ng-binding"]'
                '[contains(text(),input_val)]').click()
        my_modal = ModalPanel(self.__admin_console)
        my_modal.submit()
        if only_schedule:
            my_modal = ModalPanel(self.__admin_console)
            my_modal.submit()

    @PageService()
    def run_validate_backup(self):
        """
        Runs Backup Validation jobs

        Returns:
            job_id  (basestring):   the job id of the backup validation
        """
        self.__admin_console.select_overview_tab()
        panel_details = PanelInfo(self.__admin_console, self.__admin_console.props['label.recoveryPoint'])
        panel_details.open_hyperlink_on_tile(self.__admin_console.props['label.validateBackup'])
        my_dialog = ModalDialog(self.__admin_console)
        my_dialog.click_submit()
        return self.__admin_console.get_jobid_from_popup()
