# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides all the methods that can be done on the VM groups page.


Classes:

    VMGroups() ---> AdminPage() ---> AdminConsoleBase() ---> object()


VMGroups  --  This class contains all the methods for action in VM groups page and is inherited
                  by other classes to perform VSA related actions

Functions:

    select_vm_group()			--	Opens the VM group with the given name

    add_vm_group()          	--  Creates a VM group for the given hypervisor

    action_backup_vm_groups()   --  Backs up the given VM group

    action_delete_vm_groups()   --  Deletes the given VM groups

    action_restore_vm_group()   --  Restores the given VM group

    action_jobs_vm_group()      --  Opens the jobs page of the VM group

    select_vm_group_hypervisor()--  Opens the hypervisor client of the VM group

    list_vm_groups()            --  Lists all the VM groups

"""
from Web.AdminConsole.Components.panel import Backup
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.Components.panel import DropDown
from Web.AdminConsole.Components.panel import PanelInfo
from Web.Common.handlers import adminconsole_handler
from Web.AdminConsole.VSAPages.hypervisor_details import HypervisorDetails
from Web.Common.page_object import PageService
from Web.Common.exceptions import CVWebAutomationException


class VMGroups:
    """
    Class for the VM groups page
    """

    def __init__(self, admin_console):
        """
        Init method to create objects of classes used in the file.

        Args:
            driver      (object)   :  the browser object
        """
        self.__admin_console = admin_console
        self.__admin_console._load_properties(self)
        self.__driver = admin_console.driver
        self.__hyp_det_obj = HypervisorDetails(admin_console)
        self.__panel_dropdown_obj = DropDown(admin_console)
        self.__panel_info_obj = PanelInfo(admin_console)
        self.__table = Table(admin_console)

    @PageService()
    def select_vm_group(self, vm_group_name):
        """
        Opens the VM group with the given name

        Args:
            vm_group_name (basestring) :  the name of the VM group to open

        """
        self.__table.access_link(vm_group_name)

    @PageService()
    def add_vm_group(self, hypervisor_name, vm_group_name, vm_content, plan=None):
        """
        Adds a new VM group

        Args:
            hypervisor_name (basestring)     : the hypervisor whose VMs should be grouped

            vm_group_name (basestring)       : the name of the vm group to create

            vm_content      (dict)           :   the content to be added to the subclient

                Sample value:   {'Datastores and datastore clusters':[ds1,ds2],
                                 'Hosts and clusters':[h1,h2],
                                 'Tags and categories':[tag1,category1],
                                 'VMs and templates':[vm1,vm2],
                                 'Storage':[strg1, strg2]
                                }

            plan            (basestring)    :   the plan to be associated with the subclient

        """
        self.__admin_console.access_menu("Add VM group")
        self.__panel_dropdown_obj.select_drop_down_values(0, [hypervisor_name])
        self.__admin_console.fill_form_by_id("name", vm_group_name)
        self.__driver.find_element_by_xpath('//*[@id="ADD_VIRTUALMACHINES"]').click()
        self.__admin_console.wait_for_completion()
        self.__hyp_det_obj.select_vm_from_browse_tree(vm_content)
        self.__admin_console.cv_single_select("Plan", plan)
        self.__admin_console.submit_form()
        self.__admin_console.check_error_message()

    @PageService()
    def action_backup_vm_groups(self, vm_group_name, backup_type):
        """
        Starts a backup of the given type for the specified collection

        Args:
           vm_group_name (basestring)  : the name of the VM group to backup
           backup_type (BackupType)    : the type of backup to run, among the type in Backup.BackupType enum

        Returns:
            the backup job ID
        """
        self.__table.access_action_item(vm_group_name, self.__admin_console.props['action.commonAction.backup'])
        backup = Backup(self.__admin_console)
        return backup.submit_backup(backup_type)

    @PageService()
    def action_delete_vm_groups(self, vm_group_name):
        """
        Deletes a vm group with the given name

        Args:
            vm_group_name (basestring) : the VM group to delete
        """

        self.__table.access_action_item(vm_group_name, self.__admin_console.props['action.delete'])
        self.__admin_console.click_button('Yes')

    @PageService()
    def action_restore_vm_groups(self, vm_group_name):
        """
        Opens the restore page of the vm group from the server details page

        Args:
            vm_group_name (basestring):  the VM group to restore

        """
        self.__table.access_action_item(vm_group_name, self.__admin_console.props['action.commonAction.restore'])

    @PageService()
    def action_jobs_vm_groups(self, vm_group_name):
        """
        Lists all the jobs of the specific subclient

        Args:
            vm_group_name (basestring): the VM group whose jobs should be opened
        """
        self.__table.access_action_item(vm_group_name, self.__admin_console.props['action.commonAction.jobs'])

    @PageService()
    def select_vm_group_hypervisor(self, vm_group_name):
        """
        Opens the hypervisor of the VM group provided

        Args:
            vm_group_name (basestring): name of the VM group whose hypervisor to open
        """
        vmgroups = self.__table.get_column_data("Name")
        hypersiors = self.__table.get_column_data("Hypervisor")
        index = vmgroups.index(vm_group_name)
        self.__admin_console.select_hyperlink(hypersiors[index])

    @PageService()
    def list_vm_groups(self):
        """
        Lists all the VM groups

        Returns:
            list of all VM groups
        """
        return self.__table.get_column_data("Name")

    @PageService()
    def has_vm_group(self, vm_group):
        """
        Check if vm group exists
        Args:
            vm_group               (str):   vm group name
        Returns                    (bool): True if vm group exists or False otherwise
        """
        return (self.__table.is_entity_present_in_column('Name', vm_group) or
                self.__table.is_entity_present_in_column('Name', f"{vm_group}_Administrator"))

    @PageService()
    def get_details_by_vm_group(self, vm_group):
        """
        Get table content filtered by vm group
        Args:
            vm_group               (str):  vm group name
        Returns:                   (Dict): table content of specified vm group
        """
        if self.has_vm_group(vm_group):
            return self.__table.get_table_data()
        raise CVWebAutomationException("VM group [%s] not found in vm groups page" % vm_group)
