# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides methods to get the details about a vm.


Classes:

    VMDetails() --> VMsOwned() ---> AdminPage() ---> login_page --->
        AdminConsoleBase() ---> object()


VMDetails  --  This class contains methods to get details about a virtual machine like last backup,
               vmjobs, restore, etc.

    Functions:

    last_backup()        --  Opens the last backup job details

    vm_jobs()            --  Opens the jobs of the vm

    vm_restore()         --  Opens the restore options page for the vm

    do_not_backup()      --  Removes the VM from the subclient content

    vm_backup_now()      --  Backs up the VM

    vm_summary()         --  Returns the dictionary of all the  vm summary information listed

    vm_backup_details()  --  Returns the backup details of the VM in a dictionary

    recovery_point_restore() -- Restores the VM from the date and time selected from recovery point

    vm_search_content()  --  Searches the VM for files and folders

    backup_validation_results() --  Return the backup validation details of the vm

"""
from selenium.webdriver.common.keys import Keys
from Web.Common.handlers import adminconsole_handler
from Web.AdminConsole.Components.panel import PanelInfo
from Web.AdminConsole.Components.panel import Backup


class VMDetails:
    """
    This class contains methods to get details about a virtual machine like last backup,
    vm jobs, restore, etc.
    """

    def __init__(self, admin_console):
        self.__admin_console = admin_console
        self.__admin_console._load_properties(self)
        self.__driver = admin_console.driver

    @adminconsole_handler()
    def last_backup(self):
        """
        Opens the last backup job details
        """
        self.__admin_console.select_configuration_tab()
        self.__admin_console.select_hyperlink("Last backup")

    @adminconsole_handler()
    def vm_jobs(self):
        """
        Opens the jobs of the vm
        """
        self.__admin_console.access_menu("Jobs")

    @adminconsole_handler()
    def vm_restore(self, recovery_time=None):
        """
        Opens the restore options page for the vm

        Args:
            recovery_time (basestring): the backup date in 01-September-1960 format

        """
        self.__admin_console.recovery_point_restore(recovery_time)

    @adminconsole_handler()
    def do_not_backup(self):
        """
        Removes the VM from the backed up list

        Raises:
            Exception:
                if the VM is on the only content in the subclient

        """
        self.__admin_console.access_menu_from_dropdown("Do not back up")
        self.__admin_console.click_button('Yes')
        if self.__admin_console.check_if_entity_exists(
                "xpath",
                "//div[@class='modal-body text-danger ng-binding ng-scope']"):
            exp = self.__driver.find_element_by_xpath("//div[@class='"
                                                      "modal-body text-danger ng-binding "
                                                      "ng-scope']").text
            self.__admin_console.click_button("Close")
            raise Exception(exp)

    @adminconsole_handler()
    def vm_backup_now(self, bkp_type):
        """
        Backs up the VM

        Args:
            bkp_type    (BackupType):    the backup level, among the type in Backup.BackupType enum

        Returns:
            job_id  (basestring):   the backup job ID

        """
        self.__admin_console.select_configuration_tab()
        self.__admin_console.access_menu("Back up")
        backup = Backup(self.__admin_console)
        return backup.submit_backup(bkp_type)

    @adminconsole_handler()
    def vm_summary(self):
        """Returns the dictionary of all the  vm summary information listed

            Ex: {
                'Plan': '',
                'RPO status': '',
                'Last recovery point': '',
                'Oldest recovery point': '',
                'Server': '',
                'VM size': '',
                'Host': '',
                'OS': ''
            }
        """
        panel_details = PanelInfo(self.__admin_console, self.__admin_console.props['label.summary'])
        return panel_details.get_details()

    @adminconsole_handler()
    def vm_backup_details(self):
        """Returns the dictionary of all the  vm summary information listed

            Ex: {
                'VM Size (GB)': '',
                'Backup status': '',
                'Backup size (GB)': '',
                'Last backup time': '',
                'Server': '',
                'Guest size': '',
                'Total backup time': ''
            }
        """
        panel_details = PanelInfo(self.__admin_console)
        return panel_details.get_details()

    @adminconsole_handler()
    def vm_search_content(self, search_value):
        """
        Searches the VM for files and folders and content

        Args:
            search_value    (basestring):   the folder or file to search for in the VM

        Raises:
            Exception:
                if the search box is not present

        """
        if self.__admin_console.check_if_entity_exists("id", "fileAndFolderSearch"):
            self.__admin_console.fill_form_by_id("fileAndFolderSearch", search_value)
            self.__driver.find_element_by_id("fileAndFolderSearch").send_keys(Keys.ENTER)
            self.__admin_console.wait_for_completion()
        else:
            exp = "The search option is not present for this Virtual Machine. Please ensure " \
                  "that the server has file indexing enabled and the data analytics job ran for" \
                  "the VM"
            raise Exception(exp)

    @adminconsole_handler()
    def backup_validation_results(self):
        """Returns the dictionary of The backup validation results

        Ex: {
                'Boot status': '',
                'Last validation job ID': '',
                'Backup validated': '',
                'Backup completion date': ''
            }
        """
        panel_details = PanelInfo(self.__admin_console, self.__admin_console.props['label.backupValidationStats'])
        return panel_details.get_details()

    @adminconsole_handler()
    def backup_validation_scripts(self):
        """Returns the Column name and script status of the scripts run"""
        if self.__admin_console.check_if_entity_exists(
                "id",
                "disoveredAppsList"):
            _table = []
            table_id = self.__admin_console.driver.find_element_by_id('disoveredAppsList')
            for row in table_id.find_elements_by_xpath(".//tbody"):
                _table.append([td.text for td in row.find_elements_by_xpath(".//td")])
            return _table
        else:
            return []
