# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides all the methods to perform actions on vm.


Classes:

    VMsOwned() ---> AdminPage() ---> login_page ---> AdminConsoleBase() ---> object()


Hypervisors  --  This class contains methods to perform actions on vm like opening a vm, opening a
                  server of vm, listing backup vms etc.

    Functions:

    open_vm()               --  Opens VM with the given name

    open_hypervisor_of_vm() --  Opens the server corresponding to the VM name

    list_backup_vms()       --  Lists all the backup VM details

    action_remove_vm()      --  Removes the VM from the list of backed up VMs

    action_vm_jobs()        --    Displays all the jobs of the given VM

    action_vm_backup()      -- Backups the given VM

    action_vm_restore()     -- Restores the given VM

    run_validate_backup()   --  Runs backup validation job

"""
from Web.AdminConsole.Components.panel import Backup
from Web.AdminConsole.Components.table import Table, CVTable
from Web.Common.handlers import adminconsole_handler
from Web.AdminConsole.Components.panel import PanelInfo
from Web.AdminConsole.Components.dialog import ModalDialog
from Web.Common.page_object import PageService


class VirtualMachines:
    """
     This class contains methods to perform actions on vm like opening a vm, opening a server of \
     vm, listing backup vms etc.
    """
    def __init__(self, admin_console):
        self.__admin_console = admin_console
        self.__admin_console._load_properties(self)
        self.__table = Table(admin_console)
        self.__cvtable = CVTable(admin_console)

    @adminconsole_handler()
    def open_vm(self, vm_name):
        """
        Opens VM with the given name

        Args:
            vm_name     (basestring):   name of the VM to open

        """
        self.__table.access_link(vm_name)

    @adminconsole_handler()
    def open_hypervisor_of_vm(self, vm_name):
        """Opens the hypervisor corresponding to the VM name

        Args:
            vm_name  (basestring):  the name of the VM whose hypervisor has to be opened

        """
        all_vm_names = self.__table.get_column_data("Name")
        all_hypervisor_names = self.__table.get_column_data("Hypervisor")
        index = all_vm_names.index(vm_name)
        self.__admin_console.select_hyperlink(all_hypervisor_names[index])

    @adminconsole_handler()
    def list_backup_vms(self):
        """
        Lists all the backup VM details

        Returns:
            vms     (list):     list of all VMs owned by the user

        """
        vms = {}
        while True:
            elements = self.__admin_console.driver.find_elements_by_xpath("//div[@class='ui-grid-canvas']/div")
            for elem in elements:
                vm_details = []
                key = elem.find_element_by_xpath("./div/div[1]/a").text
                for index in range(2, 10):
                    if index == 2:
                        vm_details.append(elem.find_element_by_xpath(
                            "./div/div["+str(index)+"]/a").text)
                        continue
                    elif index == 6:
                        vm_details.append(elem.find_element_by_xpath(
                            "./div/div["+str(index)+"]/div").text)
                        continue
                    vm_details.append(elem.find_element_by_xpath(
                        "./div/div["+str(index)+"]/span").text)
                vms[key] = vm_details
            if self.__admin_console.cv_table_next_button_exists():
                if self.__admin_console.driver.find_element_by_xpath(
                        "//button[@ng-disabled='cantPageForward()']").is_enabled():
                    self.__admin_console.cv_table_click_next_button()
                    self.__admin_console.wait_for_completion()
                    continue
                else:
                    break
            else:
                break
        self.__admin_console.log.info("The list of backed up VMs owned by the user is %s", str(vms))
        return vms

    @adminconsole_handler()
    def action_remove_vm(self, vm_name):
        """
        Removes the VM from the list of backed up VMs

        Args:
            vm_name  (basestring):  the name of the VM to remove

        """
        self.__table.access_action_item(vm_name, "Do not back up")
        self.__admin_console.click_button('No')

    @adminconsole_handler()
    def action_vm_jobs(self, vm_name):
        """
        Displays all the jobs of the given VM

        Args:
            vm_name  (basestring):  the name of the VM whose jobs to open

        """
        self.__table.access_action_item(vm_name, "View jobs")

    @adminconsole_handler()
    def action_vm_backup(self, vm_name):
        """
        Backups the given VM

        Args:
            vm_name  (basestring):      the name of the VM to backup

        Returns:
            job_id  (int):  the backup job ID

        """
        self.__table.access_action_item(vm_name, "Back up")
        backup = Backup(self.__admin_console)
        return backup.submit_backup(backup.BackupType.INCR)

    @adminconsole_handler()
    def action_vm_restore(self, vm_name):
        """
        Restores the given VM

        Args:
            vm_name  (basestring):  the name of the VM to restore

        """
        self.__table.access_action_item(vm_name, "Restore")

    @adminconsole_handler()
    def action_list_mounts(self, vm_name):
        """
        Opens Active mounts page for vm

        Args:
            vm_name  (basestring):  the name of the VM to restore

        """
        self.__table.access_action_item(vm_name, "View active mounts")

    @PageService()
    def delete_active_live_mount(self, mount_name):
        """
        Deletes active live mount from Active Mounts Page
        Args:
             mount_name: (str) VM Mount Name to be deleted
        :returns
            bool: True/False

        """
        mounts = self.__cvtable.get_column_data('Name')
        self.__admin_console.log.info(f"Active Mounts: {mounts}")
        for i in mounts:
            if mount_name in i:
                self.__cvtable.access_action_item(i, 'Delete')
                my_dialog = ModalDialog(self.__admin_console)
                my_dialog.click_submit()
                self.__admin_console.wait_for_completion()
                self.__admin_console.log.info(f"Successfully deleted {i}")
                return True
        self.__admin_console.log.info("Unable to find active mount")
        return False

    @PageService()
    def run_validate_backup(self):
        """
        Runs Backup Validation jobs

        Returns:
            job_id  (basestring):   the job id of the backup validation
        """
        self.__admin_console.select_overview_tab()
        panel_details = PanelInfo(self.__admin_console, self.__admin_console.props['label.recoveryPoint'])
        panel_details.open_hyperlink_on_tile(self.__admin_console.props['label.validateBackup'])
        my_dialog = ModalDialog(self.__admin_console)
        my_dialog.click_submit()
        return self.__admin_console.get_jobid_from_popup()
