# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides methods to select any type of restore to be done.


Classes:

    SelectRestore() --> AdminPage() --->  AdminConsoleBase() ---> Object()


SelectRestore --  This class contains all the methods to select any type of restore to be done.
    Functions:

    select_guest_files() -- Opens the guest files restore page
    select_vm_files() -- Opens the virtual machine files restore page
    select_full_vm_restore() -- Opens the full VM restore page
    select_disk_restore() -- Opens the disk level restore page
"""
from Web.AdminConsole.Components.browse import Browse
from Web.Common.page_object import (
    WebAction,
    PageService
)


class SelectRestore:
    """
    This class contains all the methods to select any type of restore to be done.
    """

    def __init__(self, admin_console):
        """ Init """
        self.__admin_console = admin_console
        self.__admin_console._load_properties(self)
        self.__driver = admin_console.driver

    @WebAction()
    def __select_restore_type(self, restore_type):
        """
        Selects the Restore type"""
        for res in restore_type:
            if self.__admin_console.check_if_entity_exists("xpath", f"//div[contains(text(),'{res}')]"):
                self.__driver.find_element_by_xpath(f"//div[contains(text(),'{res}')]").click()
                break

    @PageService()
    def select_guest_files(self):
        """
        Opens the guest files restore page
        """
        self.__select_restore_type([self.__admin_console.props['label.guestFile']])
        self.__admin_console.wait_for_completion()

    @PageService()
    def select_vm_files(self):
        """
        Opens the virtual machine files restore page
        """
        self.__select_restore_type([self.__admin_console.props['label.fileRestore']])
        self.__admin_console.wait_for_completion()

    @PageService()
    def select_live_mount(self):
        """
        opens the live mount page
        """
        self.__select_restore_type([self.__admin_console.props['label.liveMount']])
        self.__admin_console.wait_for_completion()

    @PageService()
    def select_full_vm_restore(self):
        """
        Opens the full VM restore page
        """
        full_vm_list = [self.__admin_console.props['label.fullVM'], self.__admin_console.props['label.fullVMAMAZON']]

        self.__select_restore_type(full_vm_list)
        self.__admin_console.wait_for_completion()

    @PageService()
    def select_disk_restore(self):
        """
        Opens the disk level restore page
        """
        self.__select_restore_type([self.__admin_console.props['label.diskLevelRestoreAMAZON']])
        self.__admin_console.wait_for_completion()

    @PageService()
    def select_source_copy(self, copy_name):
        """
        Selects the copy from which browse and restore has to happen

        Args:
            copy_name   (basestring):   the name of the copy to browse and restore from

        Raises:
            Exception:
                if the copy could not be selected

        """
        if self.__admin_console.check_if_entity_exists("link", "Select source"):
            self.__admin_console.select_hyperlink("Select source")
            elements = self.__driver.find_elements_by_xpath("//a[.='Select source']/../ul/li")
            for element in elements:
                copy_elem = element.find_element_by_xpath("./a")
                if copy_elem.text.lower() == copy_name.lower():
                    copy_elem.click()
                    self.__admin_console.wait_for_completion()
                    break
            element = self.__driver.find_element_by_xpath("//a[@class='uib-dropdown-toggle "
                                                          "ng-binding dropdown-toggle']")
            if element.text.lower() != copy_name.lower():
                raise Exception("The copy could not be selected.")
        else:
            browse_panel = Browse(self.__admin_console)
            browse_panel.select_adv_options_submit_restore(storage_copy_name=copy_name)

    @PageService()
    def latest_backups(self):
        """Shows the latest backup"""
        browse_panel = Browse(self.__admin_console)
        browse_panel.show_latest_backups()
