# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides all the methods that can be done on the Replication
monitor page.


Class:

    ReplicationMonitor() ---> AdminPage() ---> LoginPage --->
    AdminConsoleBase() ---> object()

ReplicationMonitor  --  This class contains all the methods for action in
                        Replication monitor page and is inherited by other
                        classes to perform VSA related actions

    Functions:

    select_vm()                        --  Opens the vm with the given name

    replication_enable_disable()       --  selects the checkbox of the vm
                                           specified for enable or disable
                                           replication

    replicate_now()                    --  This method marks the VM for full
                                           replication

    mark_for_full_replication()        --  This method marks the VM for full
                                           replication

    test_boot_vm()                     --  This method test the boot for VM
"""
from Web.AdminConsole.Components.table import Table
from Web.Common.handlers import adminconsole_handler
from Web.Common.page_object import PageService


class ReplicationMonitor:
    """
     This class contains all the methods for action in replication monitor page
    """

    def __init__(self, admin_console):
        """"""
        self.__admin_console = admin_console
        self.__admin_console._load_properties(self)
        self.__driver = admin_console.driver
        self.__table = Table(admin_console)

    @PageService()
    def select_vm(self, vm_name):
        """
        Opens the vm with the given name

        Args:
            vm_name     (basestring)    : name of vm

        Returns:
            None

        Raises:
            Exception:
                If there is no vm with given name
        """
        self.__table.access_link(vm_name)

    @adminconsole_handler()
    def replication_enable_disable(self, vm_name, status):
        """
        Selects the checkbox of the vm specified for enable or disable

        Args:
            vm_name     (basestring)    : vm whose replication status has to be enabled or disabled
            status      (basestring)    : wanted status of the VM

        Returns:
            None

        Raises:
            Exception:
                Replication action item is not present

                Wrong job text is displayed
        """
        self.__table.access_action_item(vm_name, 'Replication')
        list_elements = self.__driver.find_elements_by_xpath("//li/a[text()='Replication']/../ul/li")
        for list_element in list_elements:
            status_element = list_element.find_element_by_xpath("./a")
            if status_element.text in ['Enable', 'Disable']:
                if status_element.text == status:
                    status_element.click()
                    replication_text = self.__admin_console.get_notification()
                    if status == 'Disable' and \
                            'Replication disabled successfully' not in replication_text:
                        raise Exception("The message shown is wrong")
                    elif status == 'Enable' and \
                            'Replication enabled successfully' not in replication_text:
                        raise Exception("The message shown is wrong")
                else:
                    self.__admin_console.log.info("The status of replication is already in the desired state")
            break

    @adminconsole_handler()
    def replicate_now(self, vm_name):
        """
        This method starts the replication

        Args:
            vm_name     (basestring)    : vm whose replication is to be started immediately

        Returns:
            None

        Raises:
            Exception:
                Replication action item is not present

                Wrong job text is displayed
        """
        sync_status = self.__sync_status(vm_name)
        if sync_status in ['Sync Enabled', 'Needs Sync']:
            self.__table.access_action_item(vm_name, 'Replication')
            self.__admin_console.select_hyperlink('Replicate now')
            job_text = self.__admin_console.get_notification()
            if job_text != 'Replication started':
                raise Exception("The message shown is wrong")

    @adminconsole_handler()
    def mark_for_full_replication(self, vm_name):
        """
         This method marks the VM for full replication

        Args:
            vm_name     (basestring)    : vm whose replication is to be started immediately

        Returns:
            None

        Raises:
            Exception:
                Replication action item is not present

                Wrong job text is displayed
        """
        sync_status = self.__sync_status(vm_name)
        if sync_status in ['Sync Enabled', 'In Sync', 'Needs Sync']:
            row_no = self.table_row_no(vm_name)
            col_no = self.table_column_no("Actions")
            self.__table.access_action_item(vm_name, 'Replication')

            if self.__admin_console.check_if_entity_exists("xpath", "//div[@class='ui-grid-canvas']/div["
                                                                    + str(row_no) + "]/div/div[" + str(col_no) + "]//a["
                                                                    "contains(.,'Mark for full replication')]"):
                self.__driver.find_element_by_xpath(
                    "//div[@class='ui-grid-canvas']/div[" + str(row_no) + "]/div/div["
                    + str(col_no) + "]//a[contains(.,'Mark for full replication')]").click()
            job_text = self.__admin_console.get_notification()
            if job_text != 'Replication started successfully':
                raise Exception("The message shown is wrong")

    @adminconsole_handler()
    def test_boot_vm(self, vm_name):
        """
        This method test the boot for VM

        Args:
            vm_name     (basestring)    :  vm whose replication is to be started immediately

        Returns:
            None

        Raises:
            Exception:
                Test boot VM action item is not present

                Wrong job text is displayed

        """
        sync_status = self.__sync_status(vm_name)
        if sync_status in ['In Sync', 'Needs Sync']:
            self.__table.access_action_item(vm_name, 'Test boot VM')
            self.__admin_console.click_button_using_text('Yes')
            job_text = self.__admin_console.get_notification()
            replication_text = job_text.split("-")
            if replication_text[0] != 'Test boot requested ':
                raise Exception("The message shown is wrong")

    def __sync_status(self, vm_name):
        """
        Returns the sync status of vm

        Args:
            vm_name     (basetring) : VM name whose sync status has to be checked

        Returns:
            Sync Status of the VM

        Raises:
            Exception:
                Not able to find the staus for VM

        """
        row_no = self.table_row_no(vm_name)
        col_no = self.table_column_no('Sync status')
        sync_status = self.__driver.find_element_by_xpath("//div[@class='ui-grid-contents-wrapper']/"
                                                          "div[@role = 'presentation']/div[2]/div/"
                                                          "div[" + str(row_no) + "]/div/div[" +
                                                          str(col_no) + "]//span").text
        self.__admin_console.log.info("Sync Status : %s", sync_status)
        return sync_status
