# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides methods to manage content on the subclient.

Classes:

    ManageContent() ---> AdminPage() ---> AdminConsoleBase() ---> Object()

    ManageContent() -- This class provides methods to manage content on a subclient like add vm and
                        remove content
    Functions:

    add_vm()          -- Adds VMs to the collection content
    remove_content()  -- Removes content from the collection content
    edit_rule()       -- Edits the rule
    add_rule()        -- Add collection content based on rules
    preview()         --  Preview collection content
    filters()         -- Adds VM filters to the subclient content
    disk_filters()    --  Adds disk filters to the subclient content

"""

from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as EC
from selenium.webdriver.common.by import By
from selenium.webdriver.support.ui import Select
from Web.Common.handlers import adminconsole_handler
from Web.AdminConsole.VSAPages.vsa_subclient_details import VsaSubclientDetails
from Web.AdminConsole.VSAPages.hypervisor_details import HypervisorDetails
from Web.AdminConsole.Components.panel import ModalPanel, DropDown
from Web.Common.page_object import WebAction


class ManageContent:
    """
    This class provides methods to manage content on a subclient like add vm and remove content
    """

    def __init__(self, admin_console):
        """
        Init method to create objects of classes used in the file.

        Args:
            driver      (object)   :  the browser object
        """
        self.__admin_console = admin_console
        self.__admin_console._load_properties(self, unique=True)
        self.__driver = admin_console.driver
        self.vsa_sc_obj = VsaSubclientDetails(admin_console)
        self.hypdet_obj = HypervisorDetails(admin_console)
        self.__model_panel_obj = ModalPanel(admin_console)
        self.__panel_dropdown_obj = DropDown(admin_console)

    @WebAction()
    def add_vm(self, vm_content):
        """
        Adds VMs to the collection content

        Args:
            vm_content      (dict)  --  the content to be added to the subclient
            Sample value:   {'datastore':[ds1,ds2],
                                 'host':[h1,h2],
                                 'tags':[tag1,category1],
                                 'vms':[vm1,vm2]
                                }
        """
        self.__driver.find_element_by_link_text("Add virtual machines").click()
        self.__admin_console.wait_for_completion()

        self.hypdet_obj.select_vm_from_browse_tree(vm_content)
        self.__driver.find_element_by_xpath(
            "//form/cv-browse-collection-content/following-sibling::div/button[2]").click()
        self.__admin_console.wait_for_completion()
        self.__model_panel_obj.submit()
        self.__admin_console.wait_for_completion()
        self.__admin_console.check_error_message()

    @WebAction()
    def remove_content(self, content_list, vm_filter=False, disk_filter=False):
        """
        Removes content from the collection content

        Args:
            content_list (list): List of content to remove
            vm_filter (bool)   : if or not filter vm
            disk_filter (bool) : if or not filter disk
        """
        self.__admin_console.log.info("Removing VMs and filters from the collection content")
        self.vsa_sc_obj.manage_content()
        items = []
        if vm_filter is True:
            self.__driver.find_element_by_xpath(
                "//form[@name='addCollectionForm']/cv-tabset-component/ul/li[2]/a").click()
        elif disk_filter is True:
            self.__driver.find_element_by_xpath(
                "//form[@name='addCollectionForm']/cv-tabset-component/ul/li[3]/a").click()
        for content in content_list:
            if self.__admin_console.cv_table_next_button_exists():
                if self.__driver.find_element_by_xpath(
                        "//button[@ng-click='paginationApi.seek(1)']").is_enabled():
                    self.__admin_console.cv_table_click_first_button()
                    self.__admin_console.wait_for_completion()
            while True:
                if self.__admin_console.check_if_entity_exists("xpath", "//span[@title='" + content + "']"):
                    self.__driver.find_element_by_xpath(
                        "//span[@title='" + content + "']/../../div[3]/div/a/span").click()
                    if self.__admin_console.check_if_entity_exists("link", "Remove"):
                        self.__driver.find_element_by_xpath(
                            "//span[@title='" + content + "']/../../div[3]/div/ul//a[ \
                            contains(text(),'Remove')]").click()
                        break
                elif disk_filter:
                    if self.__admin_console.check_if_entity_exists("xpath", "//span[contains(text(), '" +
                                                                            content + "')]"):
                        self.__driver.find_element_by_xpath("//span[contains(text(),'" +
                                                            content + "')]/../../div[3]/div/a"
                                                                      "/span").click()
                        if self.__admin_console.check_if_entity_exists("link", "Remove"):
                            self.__driver.find_element_by_xpath(
                                "//span[contains(text(),'" +
                                content + "')]/../../div[3]/div/ul//a[contains(text(),"
                                          "'Remove')]").click()
                            break
                else:
                    if self.__admin_console.cv_table_next_button_exists():
                        if self.__driver.find_element_by_xpath(
                                "//button[@ng-disabled='cantPageForward()']").is_enabled():
                            self.__driver.find_element_by_xpath(
                                "//button[@ng-disabled='cantPageForward()']/div").click()
                            self.__admin_console.wait_for_completion()
                            continue
                        else:
                            items.append(content)
                            break
                    else:
                        items.append(content)
                        break
        if items != []:
            raise Exception("There is no content with the name " + str(items))
        self.__driver.find_element_by_xpath(
            "//form[@name='addCollectionForm']/div/button[3]").click()
        self.__admin_console.wait_for_completion()
        if self.__admin_console.check_if_entity_exists(
                "xpath", "//div/div[2]/span[@class='serverMessage error']"):
            raise Exception(self.__driver.find_element_by_xpath(
                "//div/div[2]/span[@class='serverMessage error']").text)
        else:
            content = []
            ret = self.vsa_sc_obj.content_info()
            for vms in content_list:
                if vms in set(ret[0]):
                    content.append(vms)
            if content != []:
                raise Exception("The VMs " + str(content) + " could not be deleted.")

    @WebAction()
    def edit_rule(self, content_type, rule_name, rule_type, rule_expressions, new_rule_name):
        """
        Edits the rule

        Args:
            content_type (basestring)    : the type of content
            rule_name (basestring)       : name of the rule
            rule_type (basestring)       : select rule for
                Ex: Virtual machine name/pattern, Datastore, Guest DNS Hostname, Guest OS, Host,
                    Notes, Power state, Template
            rule_expressions (basestring): condition which matches with the rule type
                Ex: Equals, Contains, Ends with, Starts with
            new_rule_name (basestring)   : new name of the rule
        """
        self.__admin_console.log.info("Editing the rule")
        self.vsa_sc_obj.manage_content()
        if content_type == "Filters":
            self.__driver.find_element_by_xpath(
                "//form[@name='addCollectionForm']/cv-tabset-component/ul/li[2]/a").click()
        elif content_type == "Disk Filters":
            self.__driver.find_element_by_xpath(
                "//form[@name='addCollectionForm']/cv-tabset-component/ul/li[3]/a").click()
        while True:
            if self.__admin_console.check_if_entity_exists(
                    "xpath", "//span[@title='" + rule_name + "']"):
                self.__driver.find_element_by_xpath(
                    "//span[@title='" + rule_name + "']/../../div[3]/div/a/span").click()
                if self.__admin_console.check_if_entity_exists("link", "Edit rule"):
                    self.__driver.find_element_by_xpath(
                        "//span[@title='" + rule_name + "']/../../div[3]/div/ul//a[contains("
                                                        "text(),'Edit rule')]").click()
                    self.__admin_console.wait_for_completion()
                    Select(self.__driver.find_element_by_xpath(
                        "//form/label[1]/select")).select_by_visible_text(rule_type)
                    Select(self.__driver.find_element_by_xpath(
                        "//form/label[2]/select")).select_by_visible_text(rule_expressions)
                    if rule_type in ["Power state", "Template"]:
                        Select(self.__driver.find_element_by_xpath(
                            "//form/label[3]/select")).select_by_visible_text(new_rule_name)
                    else:
                        self.__admin_console.fill_form_by_id("ruleString", new_rule_name)

                    self.__driver.find_element_by_xpath(
                        "//form[@name='addRuleForm']/div/button[2]").click()
                    self.__admin_console.wait_for_completion()
                    break
            else:
                if self.__admin_console.cv_table_next_button_exists():
                    if self.__driver.find_element_by_xpath(
                            "//button[@ng-disabled='cantPageForward()']").is_enabled():
                        self.__driver.find_element_by_xpath(
                            "//button[@ng-disabled='cantPageForward()']/div").click()
                        self.__admin_console.wait_for_completion()
                        continue
                    else:
                        raise Exception("There is no rule with the name " +
                                        rule_name + " in the collection content")
                else:
                    raise Exception("There is no rule with the name " +
                                    rule_name + " in the collection content")
        self.__driver.find_element_by_xpath(
            "//form[@name='addCollectionForm']/div/button[3]").click()
        self.__admin_console.wait_for_completion()
        if self.__admin_console.check_if_entity_exists(
                "xpath", "//div/div[2]/span[@class='serverMessage error']"):
            raise Exception(self.__driver.find_element_by_xpath(
                "//div/div[2]/span[@class='serverMessage error']").text)
        else:
            ret = self.vsa_sc_obj.content_info()
            if not any(new_rule_name in x for x in ret[0]):
                raise Exception("The rule " + new_rule_name + " could not be modified.")

    @WebAction()
    def add_rule(self, rule_type, rule_expressions, rule_name):
        """
        Add subclient content based on rules

        args:
            rule_type           (basestring)   --  select rule for
                Ex: Virtual machine name/pattern, Datastore, Guest DNS Hostname, Guest OS, Host,
                    Notes, Power state, Template
            rule_expressions    (basestring)   --  condition which matches with the rule type
                Ex: Equals, Contains, Ends with, Starts with
            rule_name           (basestring)   --  the name of the rule
        """
        self.__admin_console.log.info("Creating a rule %s of type %s for discovery of content", rule_expressions,
                                      rule_type)
        self.__driver.find_element_by_partial_link_text("rule").click()
        self.__admin_console.wait_for_completion()
        Select(self.__driver.find_element_by_xpath(
            '// *[ @ id = "ruleType"]')).select_by_visible_text(rule_type)
        Select(self.__driver.find_element_by_xpath(
            '// *[ @ id = "matchesrule"]')).select_by_visible_text(rule_expressions)
        if rule_type in ["Power state", "Template"]:
            Select(self.__driver.find_element_by_xpath(
                '//*[@id="rulename3"]')).select_by_visible_text(rule_name)
        else:
            self.__driver.find_element_by_name("ruleString").send_keys(rule_name)
        self.__admin_console.click_button("OK")
        self.__admin_console.wait_for_completion()
        self.__admin_console.click_button("OK")

    @WebAction()
    def preview(self, preview_dict={}):
        """
        Preview collection content

        Args:
            preview_dict (dict): dict of VMs in the content

        Returns:
                Returns a dict of VMs in the content
        """
        self.__admin_console.log.info("Preview of collection content")
        self.__admin_console.click_button(self.__admin_console.props['ManageContent']['label.preview'])
        WebDriverWait(self.__driver, 60).until(EC.invisibility_of_element_located(
            (By.XPATH, "//div[2]/cv-grid/div/div[1]/div/div/span[2]")
        ))
        while True:
            if self.__admin_console.check_if_entity_exists("xpath",
                                                           "//div[2]/div[2]/cv-grid/div/div[3]/div/div[1]/div["
                                                           "4]/p/span[1]"):
                error = self.__driver.find_element_by_xpath("//div[2]/div[2]/cv-grid/div/div[3]/div/div[1]/div[4]"
                                                            "/p/span[1]").text
                raise Exception(error)

            element_list = self.__driver.find_elements_by_xpath(
                "//div[@class='cv-grid-modal-container']/cv-grid/div/div[3]/div/div[1]/div[1]"
                "/div[2]/div[@class='ui-grid-canvas']/div")
            for elem in element_list:
                key = elem.find_element_by_xpath("./div/div[1]/span").text
                val = elem.find_element_by_xpath("./div/div[2]/span").text
                preview_dict[key] = val
            if self.__admin_console.cv_table_next_button_exists():
                if self.__driver.find_element_by_xpath(
                        "//button[@ng-disabled='cantPageForward()']").is_enabled():
                    self.__driver.find_element_by_xpath(
                        "//button[@ng-disabled='cantPageForward()']/div").click()
                    self.__admin_console.wait_for_completion()
                    continue
                else:
                    break
            else:
                break
        self.__admin_console.log.info(preview_dict)
        self.__admin_console.cancel_form()
        self.__admin_console.wait_for_completion()
        self.__admin_console.submit_form()
        return preview_dict

    @WebAction()
    def filters(self, filter_string):
        """
        Adds VM filters to the subclient content
        Args:
            filter_string   (dict)  --  the dictionary containing the filter for subclient content
                Ex: {'vm': {'datastore':[ds1,ds2],
                                 'host':[h1,h2],
                                 'tags':[tag1,category1],
                                 'vms':[vm1,vm2]
                                },
                     'rule': [rule1,rule2]}
                        rule1 Ex:    Datastore:Equals:datastore1
        """
        self.__admin_console.log.info("Adding filters to the subclient content")
        if "vm" in filter_string.keys():
            self.vsa_sc_obj.manage_content()
            self.__driver.find_element_by_xpath(
                "//form[@name='addCollectionForm']/div/div[1]/cv-tabset-component/ul/li[2]/a").click()
            self.add_vm(filter_string["vm"])
        if "rule" in filter_string.keys():
            self.vsa_sc_obj.manage_content()
            self.__driver.find_element_by_xpath(
                "//form[@name='addCollectionForm']/div/div[1]/cv-tabset-component/ul/li[2]/a").click()
            rules_list = filter_string["rule"]
            for rule in rules_list:
                rule_type, rule_expressions, rule_name = rule.split(":")
                self.add_rule(rule_type, rule_expressions, rule_name)
        self.__admin_console.click_button("OK")

    @WebAction()
    def disk_filters(self, filter_type, filter_value, disk_label_to=None):
        """
        Adds disk filters to the subclient content
        Args:
            filter_type     (basestring)   --  the disk filter type
                Possible values :   Datastore, Hard disk label, Virtual device node, VMDK name/pattern
            filter_value    (basestring)   --  the value of disk filter
            disk_label_to   (basestring)   --  the hard disk label count till which the disks
                                                should be filtered. this option is available
                                                only if the disk filter type is 'hard disk label'
                default value   :   None
        """
        self.__admin_console.log.info("Adding disk filters to the subclient content")
        self.vsa_sc_obj.manage_content()
        self.__driver.find_element_by_xpath(
            "//form[@name='addCollectionForm']/div/div[1]/cv-tabset-component/ul/li[3]/a").click()
        self.__driver.find_element_by_link_text("Add disk filter").click()
        self.__admin_console.wait_for_completion()
        Select(self.__driver.find_element_by_xpath(
            "//form/label/select")).select_by_visible_text(filter_type)
        self.__admin_console.wait_for_completion()
        if filter_type == "VMDK name/pattern":
            self.__driver.find_element_by_name("diskPattern").send_keys(filter_value)
        elif filter_type == "Virtual Device Node":
            Select(self.__driver.find_element_by_id(
                "diskVirtualDeviceNode")).select_by_visible_text(filter_value)
        elif filter_type == "Hard disk label":
            self.__driver.find_element_by_name("diskLabelFromInput").send_keys(filter_value)
            self.__driver.find_element_by_name("diskLabelToInput").send_keys(disk_label_to)
        else:
            ds_filter = None
            while True:
                collapsed_elems = self.__driver.find_elements_by_xpath(
                    "//button[@class='collapsed']")
                if not collapsed_elems:
                    exp = "Datastores could not be loaded"
                    self.__admin_console.log.error(exp)
                    raise exp
                for element in collapsed_elems:
                    element.click()
                    self.__admin_console.wait_for_completion()
                    if self.__admin_console.check_if_entity_exists("xpath", "//form[@name='addDiskFilterForm'"
                                                                            "]//span[@title='" +
                                                                            filter_value + "']"):
                        if not self.__driver.find_element_by_xpath(
                                "//form[@name='addDiskFilterForm']//span[@title='" +
                                filter_value + "']").is_displayed():
                            continue
                        self.__driver.find_element_by_xpath(
                            "//span[@title='" + filter_value + "']").click()
                        self.__admin_console.log.info("Adding disk filter %s to the collection content",
                                                      filter_value)
                        ds_filter = filter_value
                        break
                if ds_filter is not None:
                    break
        self.__driver.find_element_by_xpath(
            "//form[@name='addDiskFilterForm']/div/button[2]").click()
        self.__admin_console.wait_for_completion()
        self.__admin_console.click_button("OK")
        self.__admin_console.wait_for_completion()
