# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides methods for submitting Live Mount for VMware

Classes:

    LiveMount() --- > AdminPage() ---> AdminConsoleBase() ---> Object()

LiveMount --  This class contains methods for submitting Live Mount.

Functions:

    submit_LiveMount()       --  Submits a VMware Live Mount

"""

from Web.AdminConsole.Components.panel import DropDown
from Web.AdminConsole.Components.dialog import ModalDialog
from Web.Common.page_object import (
    WebAction,
    PageService
)


class LiveMount:
    """
    This class contains methods for submitting Live Mount.
    """

    def __init__(self, admin_console):
        """ Init for LiveMount class"""
        self.__admin_console = admin_console
        self.__admin_console._load_properties(self)
        self.__driver = admin_console.driver
        self.__panel_dropdown_obj = DropDown(admin_console)
        self.__modal_dialog_obj = ModalDialog(admin_console)

    @WebAction()
    def __select_live_mount_policy(self, live_mount_policy_name):
        """
        Args:
            live_mount_policy_name:  (str) name of the live mount policy to be selected
        """
        self.__admin_console.select_value_from_dropdown("policyType", live_mount_policy_name)

    @WebAction()
    def __select_network(self, network):
        """
        Args:
            network : (str) name of the network
        """
        self.__panel_dropdown_obj.select_drop_down_values(drop_down_id='network', values=[network])
        if network.lower() == "original network":
            self.__modal_dialog_obj.click_submit()

    @WebAction()
    def __select_copy_precedence(self, copy_precedence):
        """
        Args:
            copy_precedence : (str) copy precedence to be selected
        """
        self.__admin_console.select_value_from_dropdown("copyPrecedence",
                                                        copy_precedence,
                                                        search=True)

    @PageService()
    def submit_live_mount(self,
                          live_mount_policy_name,
                          network,
                          copy_precedence):
        """
        Args:
            live_mount_policy_name: (str) name of the live mount policy

            network :              (str) name of the network

            copy_precedence:        (str) copy precedence to be selected
        raises:
            Exception if error in submitting live mount
        """
        try:
            self.__select_live_mount_policy(live_mount_policy_name)
            self.__select_network(network)
            self.__select_copy_precedence(copy_precedence)
            self.__admin_console.submit_form()
            self.__admin_console.log.info('live mount job submitted successfully')
            live_mount_job_id = self.__admin_console.get_jobid_from_popup()
            return live_mount_job_id
        except Exception as exp:
            self.__admin_console.log.info('Failed with error: ' + str(exp))
