# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides all the methods that can be done of the Hypervisors page.


Classes:

    Hypervisors() ---> AdminPage() ---> login_page --->
    AdminConsoleBase() ---> object()


Hypervisors  --  This class contains all the methods for action in Hypervisors page and is inherited
                  by other classes to perform VSA realted actions

Functions:

    select_hypervisor()          --  Opens the server with the given name

    add_hypervisor()             --  Adds a new server with the specified vendors and proxies

    action_jobs()                --  Opens the job details page for the chosen server

    action_delete()              --  delete a server with the specified name

    action_send_logs()           --   Send logs of a server with the specified name

    hypervisor_restore()        --  Restores a subclient from the hypervisor

    list_company_hypervisor()   --  Lists the hypervisor from the given company

"""
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.Components.panel import DropDown
from Web.Common.page_object import (
    WebAction,
    PageService
)


class Hypervisors:
    """
     This class contains all the methods for action in Hypervisors page
    """

    def __init__(self, admin_console):
        """
        Method to initiate Hypervisors class

        Args:
            admin_console   (Object) :   Admin Console Base Class object
        """
        self.__admin_console = admin_console
        self.__admin_console._load_properties(self)
        self.__table_obj = Table(admin_console)
        self.__panel_dropdown_obj = DropDown(admin_console)
        self.__driver = admin_console.driver
        self.__navigator = admin_console.navigator

    @WebAction()
    def __select_add_hypervisor(self):
        """
        Selects Add Hypervisor
        """
        self.__admin_console.access_menu(self.__admin_console.props['pageHeader.addServer'])

    @WebAction()
    def __set_hostname(self, hostname):
        """
        Sets Hostname
        """
        self.__admin_console.fill_form_by_id("hostname", hostname)

    @WebAction()
    def __set_servername(self, server_name):
        """
        Sets Servername
        """
        self.__admin_console.fill_form_by_id("serverName", server_name)

    @WebAction()
    def __set_username(self, username):
        """
        Sets username
        """
        if self.__admin_console.check_if_entity_exists("id", "vsUserName"):
            self.__admin_console.fill_form_by_id("vsUserName ", username)
        else:
            self.__admin_console.fill_form_by_id("uname", username)

    @WebAction()
    def __set_password(self, pwd):
        """
        Sets password
        """
        if self.__admin_console.check_if_entity_exists("id", "vsPassword"):
            self.__admin_console.fill_form_by_id("vsPassword", pwd)
        else:
            self.__admin_console.fill_form_by_id("pass", pwd)

    @WebAction
    def __set_domain(self, domain):
        """
        Sets domain
        """
        self.__admin_console.fill_form_by_id("domainName", domain)

    @WebAction
    def __set_access_key(self, access_key):
        """
        Sets access key
        """
        self.__admin_console.fill_form_by_id("accessKey", access_key)

    @WebAction
    def __set_secret_key(self, secret_key):
        """
        Sets secret key
        """
        self.__admin_console.fill_form_by_id("secretKey", secret_key)

    @WebAction
    def __set_client_name(self, client_name):
        """
        Sets client name
        """
        self.__admin_console.fill_form_by_id("clientName", client_name)

    @WebAction
    def __set_service_account_id(self, service_account_id):
        """
        Sets service account id
        """
        self.__admin_console.fill_form_by_id("serviceAccountID", service_account_id)

    @WebAction
    def __set_p12_keyname(self, p12_keyname):
        """
        Sets p12 keyname
        """
        self.__admin_console.fill_form_by_id("p12KeyFileName", p12_keyname)

    @WebAction
    def __set_private_key_password(self, pvt_key_pass):
        """
        Sets private key password
        """
        self.__admin_console.fill_form_by_id("privateKeysPassword", pvt_key_pass)

    @WebAction
    def __set_subscription_id(self, subs_id):
        """
        Sets subscription id
        """
        self.__admin_console.fill_form_by_id("subscriptionId", subs_id)

    @WebAction
    def __set_tenant_id(self, tenant_id):
        """
        Sets tenant id
        """
        self.__admin_console.fill_form_by_id("tenantId", tenant_id)

    @WebAction
    def __set_app_id(self, app_id):
        """
        Sets app id
        """
        self.__admin_console.fill_form_by_id("applicationId", app_id)

    @WebAction
    def __set_application_pwd(self, application_password):
        """
        Sets application pwd
        """
        self.__admin_console.fill_form_by_id("applicationPassword", application_password)

    @WebAction()
    def __skip_for_now(self):
        """
        Selects skip for now
        """
        if self.__driver.find_element_by_link_text(self.__admin_console.props['action.skip']):
            self.__driver.find_element_by_link_text(self.__admin_console.props['action.skip']).click()

    @WebAction()
    def __select_subclient(self, subclient):
        """
        Selects subclient
        """
        elems = self.__driver.find_elements_by_xpath("//ul[@class='ivh-treeview ng-scope']/li")
        for elem in elems:
            if elem.find_element_by_xpath(".//span/label").text == subclient:
                elem.find_element_by_xpath(".//span/label").click()

    @PageService()
    def select_hypervisor(self, server_name):
        """
        Opens the server with the given name

        Args:
            server_name  (basestring):  the name of the server to be opened

        Raises:
            Exception:
                if there is no server with the given name

        """
        self.__table_obj.access_link(server_name)

    @PageService()
    def add_hypervisor(
            self,
            vendor,
            server_name,
            proxy_list,
            host_name=None,
            vs_password=None,
            vs_username=None,
            access_key=None,
            secret_key=None,
            domain=None,
            project_id=None,
            service_account_id=None,
            p12_key=None,
            private_key_password=None,
            subscription_id=None,
            thumbprint=None,
            tenant_id=None,
            application_id=None,
            application_password=None
    ):
        """
        Adds a new server with the specified vendors and proxies.

        Args:
            project_id           (basestring):  the project ID of google cloud

            service_account_id   (basestring):  the service account ID of google cloud

            p12_key              (basestring):  the P12 Key of Google Cloud

            private_key_password (basestring):  the private key password of Google Cloud

            subscription_id      (basestring):  the subscription ID of Azure account

            thumbprint           (basestring):  the thumbprint of the Azure account

            tenant_id            (basestring):  the tenant ID of the Azure account

            application_id       (basestring):  the application ID of the Azure account

            application_password (basestring):  the application password of the Azure account

            domain               (basestring):  the domain name of the openstack server

            access_key           (basestring):  the access key required for Amazon client

            secret_key           (basestring):  the secret key required for Amazon client

            vendor               (basestring):  the vendor type of server to be added

            server_name          (basestring):  the name of the server to be added

            host_name            (basestring):  the hostname of the server to be added

            vs_username          (basestring):  the username of the server to be added

            proxy_list           (list):        the list of proxies to be associated with server

            vs_password          (basestring):  the password of the server to be added

        Raises:
            Exception:
                if the hypervisor could not be created

        """
        self.__admin_console.log.info("Adding a %s server %s as %s", vendor, host_name, server_name)
        self.__select_add_hypervisor()
        self.__panel_dropdown_obj.select_drop_down_values(index=0, values=[vendor], partial_selection=True)
        if vendor in ["VMware", "Nutanix AHV", "OpenStack", "Oracle Cloud Classic",
                      "Oracle VM"]:
            self.__set_hostname(host_name)
            self. __set_servername(server_name)
            self.__set_username(vs_username)
            self.__set_password(vs_password)
            if vendor == "OpenStack":
                if domain:
                    self.__set_domain(domain)
        elif vendor == "Hyper-V":
            self.__set_hostname(host_name)
            self. __set_servername(server_name)
            self.__set_username(vs_username)
            self.__set_password(vs_password)
            self.__admin_console.click_button(self.__admin_console.props['button.discover.nodes'])
            # check if at all the discovery is successful.
            self.__admin_console.check_error_message()
        elif vendor in ["Amazon", "Alibaba Cloud"]:
            self. __set_servername(server_name)
            self.__set_access_key(access_key)
            self.__set_secret_key(secret_key)
        elif vendor == "Google Cloud Platform":
            self.__set_client_name(server_name)
            self.__set_service_account_id(service_account_id)
            self.__set_p12_keyname(p12_key)
            self.__set_private_key_password(private_key_password)
        elif vendor in ['Azure - Resource Manager', 'Azure - stack']:
            self.__set_servername(server_name)
            self.__set_subscription_id(subscription_id)
            self.__set_tenant_id(tenant_id)
            self.__set_app_id(application_id)
            self.__set_application_pwd(application_password)
        self.__panel_dropdown_obj.select_drop_down_values(1, proxy_list)
        self.__admin_console.submit_form()
        self.__admin_console.check_error_message()
        self.__skip_for_now()
        self.__admin_console.wait_for_completion()
        self.__admin_console.log.info("Done with Add hypervisor")

    @PageService()
    def action_jobs(self, server_name):
        """
        Opens the job details page for the chosen server

        Args:
            server_name  (basestring):  name of the hypervisor whose jobs needs to be opened

        """
        self.__table_obj.access_action_item(server_name, self.__admin_console.props['action.jobs'])

    @PageService()
    def action_retire(self, server_name):
        """
        delete a server with the specified name

        Args:
            server_name  (basestring):  name of the hypervisor server to be deleted

        """
        self.__table_obj.access_action_item(server_name, self.__admin_console.props['action.commonAction.retire'])
        self.__admin_console.click_button("Retire")

    @PageService()
    def action_send_logs(self, server_name):
        """
        Send logs of a server with the specified name

        Args:
            server_name  (basestring):  name of the server whose logs needs to be sent

        """
        self.__table_obj.access_action_item(server_name, self.__admin_console.props['action.sendLogs'])

    @PageService()
    def hypervisor_restore(self, hypervisor_name, subclient_name=None):
        """
        Restores the given subclient in the hypervisor

        Args:
            hypervisor_name      (basestring):  the name of the hypervisor whose subclient is to
                                                be restored

            subclient_name       (basestring):  the name of the subclient to be restored

        Raises:
            Exception:
                if there is no subclient with the given name
                if restore could not be selected
                if the hypervisor was not backed up

        """
        self.__admin_console.log.info("Restoring subclient %s in hypervisor %s", subclient_name, hypervisor_name)
        self.__table_obj.access_link(hypervisor_name)
        url = self.__admin_console.current_url()
        self.__table_obj.access_action_item(subclient_name, self.__admin_console.props['pageTitle.restore'])
        if self.__admin_console.current_url() == url:
            self.__select_subclient(subclient_name)
            self.__admin_console.wait_for_completion()
            self.__admin_console.submit_form()
        else:
            self.__admin_console.log.info("There was only backed up subclient in the hypervisor")

    @PageService()
    def list_company_hypervisors(self, company_name):
        """
        Lists the companies whose hypervisors needs to be displayed

        Args:
            company_name (basestring):  the name of the company

        """
        self.__navigator.switch_company_as_operator(company_name)
