# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides methods for all the actions that can be done of the Hypervisors Details page.


Classes:

    HypervisorDetails() ---> AdminPage() --->  AdminConsoleBase() ---> object()


HypervisorDetails  --  This class contains all the methods for action in a particular server's page

Functions:

    open_hypervisor_alert()         --  Opens the given raised alert of the hypervisor

    edit_proxy()                    -- Adds / Removes a proxy to the given server

    reorder_proxy()                 -- Reorders the proxies for the server

    proxy_info()                    -- Lists all the proxy of the server

    action_backup()                 -- Starts a backup of the given type for the specified
                                            collection

    add_subclient()                 --  Adds a new subclient with the given VMs as content

    select_vm_from_browse_tree()    -- browse_and_select_vms can take one these 1.Group by host
                                        2.VMs 3.Group by datastore

    action_delete_subclient()       -- Deletes a subclient with the given name

    delete_server()                 -- Deletes the entire server with the given name

    data_management()               -- Enables and disables the data backup capability
                                        of the server

    data_recovery()                 -- Enables and disables the data restore capability of
                                        the server

    jobs()                          --  Opens the jobs page with all the running jobs for
                                        the server

    open_subclient()                --  Opens a subclient with the given name

    edit_hypervisor_details()       -- This definition edits the server credentials

    select_total_vm_graph()         -- Opens the VM details page with all the VMs present
                                        in the server

    select_protected_vm_graph()     -- Opens the VM details page with all the protected VMs
                                        present in the server

    select_non_protected_vm_graph() -- Opens the VM details page with all the unprotected VMs
                                        present in the server

    select_backedup_with_error_vm_graph() -- Opens the VM details page with all the VMs that were
                                                backed up with errors

    action_subclient_restore()      -- Opens the restore page of the subclient from the
                                        server details page

    action_subclient_jobs()         -- Lists all the jobs of the specific subclient

    server_settings()               --  Sets properties like FBR and vCloud

    server_analytics()              --  Enables / Disables analytics on the server

    hyperv_monitor()                --  Enables / Disables the hyper-v monitor

"""
from selenium.webdriver.common.action_chains import ActionChains
from Web.AdminConsole.Components.panel import Backup
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.Components.panel import PanelInfo
from Web.AdminConsole.Components.panel import DropDown
from Web.Common.page_object import (
    PageService
)


class HypervisorDetails:
    """
    This class contains all the methods for action in a particular server's page
    """

    def __init__(self, admin_console):
        """ """
        self.__admin_console = admin_console
        self.__admin_console._load_properties(self)
        self.__driver = admin_console.driver
        self.__table_obj = Table(admin_console)
        self.__panel_dropdown_obj = DropDown(admin_console)

    @PageService()
    def open_hypervisor_alert(self, alert_name):
        """
        Opens the alert with the given name
        Args:
            alert_name  (basestring):   the name of the alert to open

        """
        self.__admin_console.select_configuration_tab()
        panel_info = PanelInfo(self.__admin_console, self.__admin_console.props['label.alerts.text'])
        panel_info.open_hyperlink_on_tile(alert_name)

    @PageService()
    def edit_proxy(self, proxy_dict):
        """
        Adds / Removes a proxy to the given server

        Args:
            proxy_dict   (dict):    dictionary with proxy groups and proxies to be selected
                                        The other existing proxies will be removed.
                Example :   {'proxyGroup' : [proxy1,proxy2], 'proxy' : [proxy3,proxy4]}

        Raises:
            Exception:
                if the proxy list could not be edited

        """
        self.__admin_console.select_configuration_tab()
        panel_info = PanelInfo(self.__admin_console, title=self.__admin_console.props['header.vmAgent'])
        if self.__admin_console.check_if_entity_exists("link", "Edit"):
            panel_info.open_hyperlink_on_tile("Edit")
            self.__admin_console.log.info("Clicked on Edit Proxy")
            self.__admin_console.wait_for_completion()
            proxy_list = []
            # checking for proxy group
            if not self.__driver.find_element_by_xpath(
                    "//div[@class='panel-group']/div[1]/div[2]").is_displayed():
                self.__driver.find_element_by_xpath(
                    "//div[@class='panel-group']/div[1]//span[contains(text(),'Access node groups')]").click()
            proxy_list.extend(self.__driver.find_elements_by_xpath(
                "//div[@class='panel-group']/div[1]//ul"))
            # checking for proxies
            if not self.__driver.find_element_by_xpath(
                    "//div[@class='panel-group']/div[2]/div[2]").is_displayed():
                self.__driver.find_element_by_xpath(
                    "//div[@class='panel-group']/div[2]//span[contains(text(),'Access nodes')]").click()
            proxy_list.extend(self.__driver.find_elements_by_xpath(
                "//div[@class='panel-group']/div[2]//ul"))
            # unchecking all the elements
            for item in proxy_list:
                self.__driver.execute_script("arguments[0].scrollIntoView();", item)
                if item.find_element_by_xpath("./li/input").is_selected():
                    item.find_element_by_xpath("./li/label[@class='ng-binding']").click()
                    self.__admin_console.wait_for_completion()
            for key in proxy_dict.keys():
                proxies = proxy_dict[key]
                for proxy in proxies:
                    if key == "proxyGroup":
                        if not self.__driver.find_element_by_xpath(
                                "//label[contains(text(),'" + proxy + "')]/span[contains(text(), \
                                '(group)')]/../preceding-sibling::input").is_selected():
                            self.__driver.find_element_by_xpath(
                                "//label[contains(text(),'" + proxy + "')]").click()
                            self.__admin_console.wait_for_completion()
                    elif key == "proxy":
                        elements = self.__driver.find_elements_by_xpath(
                            "//label[text()='" + proxy + "']")
                        for elem in elements:
                            try:
                                elem.find_element_by_xpath("./span")
                            except:
                                if not elem.find_element_by_xpath(
                                        "./preceding-sibling::input").is_selected():
                                    elem.click()
                                    self.__admin_console.wait_for_completion()
                                    break
            self.__admin_console.submit_form()
        else:
            raise Exception("There is no option to edit proxy")

    @PageService()
    def reorder_proxy(self, proxy_order):
        """
        Reorders the proxies for the server

        Args:
            proxy_order  (dict):    the order in which the proxies have to be arranged
                Sample value:   {'proxy1': 3, 'proxy2': 1, 'proxy3': 2}

        Raises:
            Exception:
                if the proxies could not be re-ordered

        """
        self.__admin_console.select_configuration_tab()
        panel_info = PanelInfo(self.__admin_console, title=self.__admin_console.props['header.vmAgent'])
        # panel_info.edit_tile()
        if self.__admin_console.check_if_entity_exists("link", "Reorder"):
            proxies = self.__driver.find_elements_by_xpath(
                "//div[@id='scroll-box']/ul/li")
            index = 1
            old_proxy_order = {}
            for proxy in proxies:
                old_proxy_order[proxy.find_element_by_xpath("./span/span").text] = index
                index += 1
            self.__driver.find_element_by_link_text("Reorder").click()
            for key in proxy_order.keys():
                source = self.__driver.find_element_by_xpath("//span[contains(text(),'" + key +
                                                             "')]")
                destination = self.__driver.find_element_by_xpath("//div[@id='scroll-box']/ul/li"
                                                                  "[" + str(proxy_order[key]) +
                                                                  "]/span/span")
                ActionChains(self.__driver).drag_and_drop(source, destination).perform()
                self.__admin_console.wait_for_completion()
            # self.submit_form()
            panel_info.open_hyperlink_on_tile("Save")
            self.__admin_console.wait_for_completion()
            index = 1
            old_proxy_order = {}
            proxies = self.__driver.find_elements_by_xpath(
                "//div[@id='scroll-box']/ul/li")
            for proxy in proxies:
                old_proxy_order[proxy.find_element_by_xpath("./span/span").text] = index
                index += 1
            if old_proxy_order == proxy_order:
                self.__admin_console.log.info("Proxy reorder was successful")
            else:
                self.__admin_console.log.info("The proxy was not reordered properly.")

    @PageService()
    def proxy_info(self):
        """
        Lists all the proxy of the server

        Returns:
            proxy   (list): list of all proxies that has been assigned to server

        """
        self.__admin_console.select_configuration_tab()
        elements = self.__driver.find_elements_by_xpath(
            "//div[@id='scroll-box']/ul/li")
        proxy = []
        for elem in elements:
            try:
                if elem.find_element_by_xpath("./span/span"):
                    proxy.append(elem.find_element_by_xpath("./span/span").text)
            except:
                break
        self.__admin_console.log.info(proxy)
        return proxy

    @PageService()
    def action_backup(self, bkp_type, subclient):
        """
        Starts a backup of the given type for the specified collection

        Args:
            bkp_type    (BackupType):    the backup level, among the type in Backup.BackupType enum

            subclient   (basestring):   the name of the subclient to be backed up

        Returns:
            job_id  (int):  the backup job ID

        """
        self.__admin_console.select_overview_tab()
        self.__table_obj.access_action_item(subclient, self.__admin_console.props['action.commonAction.backup'])
        backup = Backup(self.__admin_console)
        return backup.submit_backup(bkp_type)

    @PageService()
    def add_subclient(
            self,
            subclient_name,
            vm_content,
            plan=None
    ):
        """
        Adds a new subclient with the given VMs as content

        Args:
            subclient_name  (basestring)    :   the name of the subclient to be created

            vm_content      (dict)          :   the content to be added to the subclient

                Sample value:   {'Datastores and datastore clusters':[ds1,ds2],
                                 'Hosts and clusters':[h1,h2],
                                 'Tags and categories':[tag1,category1],
                                 'VMs and templates':[vm1,vm2],
                                 'Storage':[strg1, strg2]
                                }

            plan            (basestring)    :   the plan to be associated with the subclient

        """
        self.__admin_console.select_overview_tab()
        self.__admin_console.access_menu(self.__admin_console.props['action.showAddVMGroup'])
        self.__admin_console.fill_form_by_id("name", subclient_name)
        self.select_vm_from_browse_tree(vm_content)
        self.__panel_dropdown_obj.select_drop_down_values(2, [plan])
        self.__admin_console.submit_form()
        self.__admin_console.check_error_message()

    @PageService()
    def select_vm_from_browse_tree(self, vm_content):
        """
        Select content for subclient from the browse tree

        Args:
            vm_content  (dict):     the content to be selected
                Sample value:   {'Datastores and datastore clusters':[ds1,ds2],
                                 'Hosts and clusters':[h1,h2],
                                 'Tags and categories':[tag1,category1],
                                 'VMs and templates':[vm1,vm2],
                                 'Storage':[strg1, strg2],
                                 'By region': [region1, region2],
                                 'By zone': [zone1, zone2]
                                }

        Raises:
            Exception:
                if the content could not be selected properly

        """
        selected = []
        all_content = []
        for key, value in vm_content.items():
            all_content += value
            self.__panel_dropdown_obj.select_drop_down_values(0, [key])
            for _vm in value:
                self.__admin_console.search_vm(_vm)
                selected.append(_vm)
            difference = list(set(all_content) - set(selected))
            if difference:
                raise Exception("Exception during selection of content. Some of the content "
                                "could not be selected")

    @PageService()
    def action_delete_subclient(self, subclient):
        """
        Deletes a subclient with the given name

        Args:
            subclient    (basestring):  the name of the subclient to be deleted

        """
        self.__admin_console.select_overview_tab()
        self.__table_obj.access_action_item(subclient, self.__admin_console.props['action.delete'])
        self.__admin_console.click_button(self.__admin_console.props['button.yes'])
        self.__admin_console.check_error_message()

    @PageService()
    def delete_server(self):
        """
        Deletes the entire server with the given name
        """
        self.__admin_console.access_menu_from_dropdown(self.__admin_console.props['action.commonAction.retire'])
        self.__admin_console.click_button(self.__admin_console.props['action.commonAction.retire'])

    @PageService()
    def data_backup(self, enabled=True):
        """
        Enables and disables the data backup capability of the server

        Args:
            enabled    (bool):  the enabled state [True / False] for data backup

        """
        self.__admin_console.select_configuration_tab()
        panel_info = PanelInfo(self.__admin_console, self.__admin_console.props['heading.clientActivityControl'])
        if enabled:
            panel_info.enable_toggle(self.__admin_console.props['label.dataBackup'])
        else:
            panel_info.disable_toggle(self.__admin_console.props['label.dataBackup'])

    @PageService()
    def data_restore(self, enabled=True):
        """
        Enables and disables the data restore capability of the server

        Args:
            enabled     (bool): the enabled state [True / False] for data restore
        """
        self.__admin_console.select_configuration_tab()
        panel_info = PanelInfo(self.__admin_console, title=self.__admin_console.props['heading.clientActivityControl'])
        if enabled:
            panel_info.enable_toggle(self.__admin_console.props['Data_Restore'])
        else:
            panel_info.disable_toggle(self.__admin_console.props['Data_Restore'])

    @PageService()
    def jobs(self):
        """
        Opens the jobs page with all the running jobs for the server

        Raises:
            Exception:
                if the jobs page could not be opened

        """
        self.__admin_console.access_menu(self.__admin_console.props['action.commonAction.jobs'])

    @PageService()
    def open_subclient(self, subclient_name):
        """
        Opens a subclient with the given name

        Args:
            subclient_name  (basestring):   the name of the subclient to be opened

        Raises:
            Exception:
                if the subclient is not found

        """
        self.__admin_console.select_overview_tab()
        self.__table_obj.access_link(subclient_name)

    @PageService()
    # Written only for VMWare hypervisor
    def edit_hypervisor_details(self, vs_hostname, vs_username, vs_password,
                                vs_old_password=None):
        """
        This definition edits the server credentials only for VMWare hypervisor

        Args:
            vs_hostname          (basestring):  the hostname of the server

            vs_username          (basestring):  the username / access key of the server

            vs_password          (basestring):  the password / secret key of the server

            vs_old_password      (basestring):  the old password of the server

        Raises:
            Exception:
                if the hypervisor details could not be edited

        """
        self.__admin_console.select_overview_tab()
        # self.access_menu_from_dropdown(self.props['action.editHypervisorDetails'])
        panel_info = PanelInfo(self.__admin_console, self.__admin_console.props['label.account'])
        panel_info.edit_tile()
        if vs_hostname:
            self.__admin_console.fill_form_by_name("hostname", vs_hostname)

        if vs_username:
            if self.__admin_console.check_if_entity_exists("id", "uname"):
                self.__admin_console.fill_form_by_id("uname", vs_username)
            elif self.__admin_console.check_if_entity_exists("name", "vsAccessKey"):
                self.__admin_console.fill_form_by_id("vsAccessKey", vs_username)

        if vs_old_password:
            if self.__admin_console.check_if_entity_exists("id", "vsCurrPassword"):
                self.__admin_console.fill_form_by_id("vsCurrPassword", vs_old_password)

        if vs_password:
            self.__admin_console.fill_form_by_id("pass", vs_password)
            if self.__admin_console.check_if_entity_exists("id", "confirmServerPassword"):
                self.__admin_console.fill_form_by_id("confirmServerPassword", vs_password)

        self.__admin_console.submit_form()
        self.__admin_console.check_error_message()

    @PageService()
    def action_subclient_restore(self, subclient_name):
        """
        Opens the restore page of the subclient from the server details page

        Args:
            subclient_name  (basestring):   the name of the subclient to be restored

        """
        self.__admin_console.select_overview_tab()
        self.__table_obj.access_action_item(subclient_name, self.__admin_console.props['action.commonAction.restore'])

    @PageService()
    def action_subclient_jobs(self, subclient_name):
        """
        Lists all the jobs of the specific subclient

        Args:
            subclient_name  (basestring):   the subclient whose jobs need to be opened

        """
        self.__admin_console.select_overview_tab()
        self.__table_obj.access_action_item(subclient_name, self.__admin_console.props['action.commonAction.jobs'])

    @PageService()
    def server_settings(self, fbr=None, vcloud_hostname=None,
                        vcloud_username=None, vcloud_password=None):
        """
        Sets the FBR MA and other settings like vCloud

        Args:
            fbr                  (basestring):  the name of the FBR MA to be set

            vcloud_hostname      (basestring):  the host name of the vcloud director

            vcloud_username      (basestring):  the vcloud user name

            vcloud_password      (basestring):  the vcloud password

        """
        self.__admin_console.select_configuration_tab()
        panel_info = PanelInfo(self.__admin_console, title=self.__admin_console.props['label.nav.settings'])
        if fbr:
            panel_info.edit_tile_entity(self.__admin_console.props['label.fbrUnixMA'])
            self.__panel_dropdown_obj.select_drop_down_values(0, [fbr])
            self.__admin_console.submit_form()

        if vcloud_hostname:
            panel_info.edit_tile_entity(self.__admin_console.props['label.vCloudHostName'])
            self.__admin_console.fill_form_by_id("vcloudHostName", vcloud_hostname)
            self.__admin_console.fill_form_by_id("vcloudUserName", vcloud_username)
            self.__admin_console.fill_form_by_id("vcloudPassword", vcloud_password)
            self.__admin_console.fill_form_by_id("confirmVcloudPassword", vcloud_password)
            self.__admin_console.submit_form()
