# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides method for guest files restore.

Classes:

    GuestFilesRestoreSelectFolder() ---> AdminPage() ---> AdminConsoleBase() ---> Object()

    GuestFilesRestoreSelectFolder() -- This class provides methods to do various types of guest
                                        agent files restore.
Functions:

    __select_restore_or_download()           -- Selects either restore or download

    submit_this_vm_restore()     -- Submits a VMware guest files restore with the backed up VM as
                                    the destination server

    submit_other_vm_restore()    -- Submits a VMware guest files restore to a different VM.

    submit_guest_agent_restore() -- Restores the content to the guest agents.

    download_content()           -- Downloads the files and folders

"""

import re
from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as EC
from selenium.webdriver.common.by import By
from selenium.common.exceptions import NoSuchElementException
from Web.Common.page_object import WebAction, PageService
from Web.AdminConsole.Components.panel import PanelInfo


class GuestFilesRestoreSelectFolder:
    """
    This class provides methods to do various types of guest agent files restore
    """
    def __init__(self, admin_console):
        self.__admin_console = admin_console
        self.__admin_console._load_properties(self)
        self.__driver = admin_console.driver
        self.__panel_obj = PanelInfo(admin_console)

    @WebAction()
    def __select_restore_or_download(self, action="Restore"):
        """
        Selects restore / download button

        Args:
            action  (basestring):   To Select restore or download button

        """
        self.__driver.execute_script("window.scrollTo(0,0)")
        self.__driver.find_element_by_xpath(
            f"//*[@id='browseActions']//*[contains(.,'{action}')]").click()
        self.__admin_console.wait_for_completion()

    @WebAction()
    def __fill_form_by_name(self, element_name, value):
        """
        Fill the value in a text field with element name.

        Args:
            element_name (basestring) -- the name attribute of the element to be filled
            value (basestring)      -- the value to be filled in the element

        Raises:
            Exception:
                If there is no element with given name/id

        """
        element = self.__driver.find_element_by_name(element_name)
        element.clear()
        element.send_keys(value)
        self.__admin_console.wait_for_completion()

    @WebAction()
    def __verify_vm(self, vm_name):
        """
        Verify backup vm is selected by default under myVM

        Args:
        vm_name (base string) -- name of the backup vm

        Raises:
            Exception:
                If the VM name is different from the VM selected under myVM

        """

        if vm_name not in self.__driver.find_element_by_xpath("//button[@id='vm']/div").text:
            raise Exception("The source backup VM was not selected by default.")

    @PageService()
    def submit_this_vm_restore(
            self,
            files,
            vm_name,
            proxy,
            vm_username,
            vm_password,
            path,
            over_write=True,
            all_files=False):
        """
        Submits a VMware guest files restore with the backed up VM as the destination server

        Args:
            files            (list)         :  the files to be restored

            vm_name          (basestring)   :  the VM to which the files are to be restored

            proxy            (basestring)   :  the proxy to be used for restore

            vm_username      (basestring)   :  the username of the destination VM

            vm_password      (basestring)   :  the password of the destination VM

            path             (basestring)   :  the path in the destination VM where files are
                                                to be restored

            over_write       (bool)         :  if files are to be overwritten during restore

            all_files        (bool)         :  if all the files are to be selected for restore

        Returns:
            job_id           (basestring)   :  the ID of the restore job submitted

        """
        self.__admin_console.log.info("Guest Files Restore to the same VM")
        self.__admin_console.select_for_restore(files, all_files)
        self.__admin_console.log.info("Files selected. Submitting restore")
        self.__select_restore_or_download()
        self.__panel_obj.access_tab(self.__admin_console.props['label.select.client'])
        try:
            self.__admin_console.cv_single_select("Access node", proxy)
        except:
            self.__admin_console.log.info("Access node not present")

        try:
            self.__admin_console.search_vm(vm_name)
        except:
            self.__admin_console.log.info("Search vm option not present")

        # if self.__driver.find_element_by_id("vm"):
        #     self.__verify_vm(vm_name)
        if self.__admin_console.check_if_entity_exists("name", "vmLoginName"):
            self.__admin_console.log.info("Filling the vm creds")
            self.__admin_console.fill_form_by_name("vmLoginName", vm_username)
            self.__admin_console.fill_form_by_name("vmPassword", vm_password)

        self.__admin_console.log.info("Filling in the restore path")
        self.__admin_console.fill_form_by_id("restorePath", path)

        if over_write:
            self.__admin_console.enable_toggle(toggle_id='fileLevelRestore_toggle_unconditionallyOverwrite')

        self.__admin_console.log.info("Submitting the Guest files restore to the same VM")
        self.__admin_console.submit_form()

        return self.__admin_console.get_jobid_from_popup()

    @PageService()
    def submit_other_vm_restore(
            self,
            files,
            proxy,
            destination_server,
            vm_name,
            username,
            password,
            path,
            over_write=True):
        """
        Submits a VMware guest files restore to a different VM. Agentless restore

        Args:
            files                (list)         :  list of files to be restored

            proxy                (basestring)   :  the proxy to be used for restore

            destination_server   (basestring)   :  the hypervisor where the restore to VM is

            vm_name              (basestring)   :  the VM to which the files are to be restored

            username             (basestring)   :  the username of the destination VM

            password             (basestring)   :  the password of the destination VM

            path                 (basestring)   :  the path in the destination VM where files are
                                                    to be restored

            over_write           (bool)         :  if the files are to be overwritten

        Returns:
            job_id              (basestring)   :  the ID of the restore job submitted

        """
        self.__admin_console.log.debug("Going to perform other VM restore to %s", vm_name)
        self.__admin_console.select_for_restore(files)
        self.__admin_console.wait_for_completion()
        self.__admin_console.log.info("Files selected. Submitting restore")
        self.__select_restore_or_download()
        self.__panel_obj.access_tab(self.__admin_console.props['label.otherVm'])
        self.__admin_console.log.info("Selecting destination proxy")
        self.__admin_console.cv_single_select("Access node", proxy)
        self.__admin_console.log.info("Selected the destination proxy")
        self.__admin_console.cv_single_select("Destination hypervisor", destination_server)
        self.__admin_console.wait_for_completion()
        self.__admin_console.log.info("Selected the destination server")

        try:
            self.__admin_console.search_vm(vm_name)
        except NoSuchElementException:
            self.__admin_console.select_destination_host(vm_name)

        if self.__admin_console.check_if_entity_exists("name", "vmLoginName"):
            self.__admin_console.log.info("Filling the vm creds")
            self.__fill_form_by_name("vmLoginName", username)
            self.__fill_form_by_name("vmPassword", password)

        self.__admin_console.log.info("Filling in the restore path")
        self.__admin_console.fill_form_by_id("restorePath", path)
        if over_write:
            self.__admin_console.enable_toggle(toggle_id='fileLevelRestore_toggle_unconditionallyOverwrite')

        self.__admin_console.log.info("Going to submit restore to a different VM")
        self.__admin_console.wait_for_completion()
        self.__admin_console.submit_form()

        return self.__admin_console.get_jobid_from_popup()

    @PageService()
    def submit_guest_agent_restore(self, files, path, proxy, over_write=True):
        """
        Restores the content to the guest agent (clients)

        Args:
            files                (list)         :  the files to be restored

            path                 (basestring)   :  the path where files are to be restored

            proxy                (basestring)   :  the client where the files are to be restored

            over_write           (bool)         :  if the files are to be overwritten

        Returns:
            job_id              (basestring)   :  the ID of the restore job submitted

        """
        self.__admin_console.log.info("Guest File Restore to the guest agent")
        self.__admin_console.select_for_restore(files)
        self.__admin_console.log.info("Files selected. Submitting restore")
        self.__select_restore_or_download()
        self.__panel_obj.access_tab(self.__admin_console.props['label.select.client'])
        self.__admin_console.log.info("Selecting destination client")
        # self.__admin_console.cv_single_select("Destination client", proxy)
        dropdown_path = "//span[contains(@id, 'select2-chosen-')]"
        if self.__admin_console.check_if_entity_exists('xpath', dropdown_path):
            for ele in self.__driver.find_elements_by_xpath(dropdown_path):
                try:
                    num = re.findall('\d+', ele.get_attribute('id'))[-1]
                    ele.click()
                    self.__driver.find_element_by_xpath("//*[@id='s2id_autogen" + num + "_search']").clear()
                    self.__driver.find_element_by_xpath("//*[@id='s2id_autogen" + num + "_search']").send_keys(proxy)
                    self.__driver.find_element_by_xpath(
                        "//*[@id='select2-results-" + num + "']//span[contains(text(),'" + proxy + "')]").click()
                    break
                except:
                    print("Destination client couldn't be selected")
        else:
            self.__admin_console.cv_single_select("Destination client", proxy)
        self.__admin_console.log.info("Destination client is selected. Filling in the restore path")

        self.__admin_console.fill_form_by_id("restorePath", path)

        if over_write:
            self.__admin_console.enable_toggle(toggle_id='fileLevelRestore_toggle_unconditionallyOverwrite')

        self.__admin_console.log.info("Submitting guest files restore to Guest Agent")
        self.__admin_console.wait_for_completion()
        self.__admin_console.submit_form()

        return self.__admin_console.get_jobid_from_popup()

    @PageService()
    def download_content(self, files, select_one=False):
        """
        Downloads the files and folders

        Args:
            files   (list):     list of all files and folders to download

            select_one (Boolean) : download only single file.

        Raises:
            Exception:
                if the download option is not present or
                if the download job did not trigger

        """
        if select_one:
            self.__admin_console.select_for_restore(files, select_one=True)
        else:
            self.__admin_console.select_for_restore(files)
        self.__admin_console.log.info("Selected content for download. Clicking on download")

        self.__select_restore_or_download(action="Download")

        try:
            WebDriverWait(self.__driver, 120).until(EC.presence_of_element_located((
                By.ID, "download-tracker")))
            self.__admin_console.wait_for_completion()
        except Exception as exp:
            raise Exception("Download job did not start in 5 minutes." + str(exp))

        self.__admin_console.log.info("Download job started successfully. Please wait for the job to complete.")

    @PageService()
    def submit_google_cloud_vm_restore(
            self,
            files,
            vm_name,
            proxy,
            vm_username,
            vm_password,
            path,
            over_write=True,
            all_files=False):
        """
        Submits a Google Cloud guest files restore with the backed up VM as the destination server

        Args:
            files            (list)         :  the files to be restored

            vm_name          (basestring)   :  the VM to which the files are to be restored

            proxy            (basestring)   :  the proxy to be used for restore

            vm_username      (basestring)   :  the username of the destination VM

            vm_password      (basestring)   :  the password of the destination VM

            path             (basestring)   :  the path in the destination VM where files are
                                                to be restored

            over_write       (bool)         :  if files are to be overwritten during restore

            all_files        (bool)         :  if all the files are to be selected for restore

        Returns:
            job_id           (basestring)   :  the ID of the restore job submitted

        """
        self.__admin_console.log.info("Guest Files Restore to the same VM")
        self.__admin_console.select_for_restore(files, all_files)
        self.__admin_console.log.info("Files selected. Submitting restore")
        self.__select_restore_or_download()
        self.__admin_console.log.info("Selecting destination proxy")
        self.__admin_console.cv_single_select("Destination", proxy)
        self.__admin_console.log.info("Selected destination proxy. Filling in the VM credentials")

        if self.__admin_console.check_if_entity_exists("name", "fsLoginName"):
            self.__admin_console.log.info("Filling the vm creds")
            self.__fill_form_by_name("fsLoginName", vm_username)
            self.__fill_form_by_name("fsPassword", vm_password)

        self.__admin_console.log.info("Filling in the restore path")
        self.__admin_console.fill_form_by_id("restorePath", path)

        if over_write:
            self.__admin_console.enable_toggle(0)

        self.__admin_console.log.info("Submitting the Guest files restore to the same VM")
        self.__admin_console.submit_form(False)

        return self.__admin_console.get_jobid_from_popup()
