# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides methods for submitting full vm restore for all hypervisors

Classes:

    FullVMRestore() --- > AdminPage() ---> AdminConsoleBase() ---> Object()

FullVMRestore --  This class contains methods for submitting full vm restore.

Functions:

    full_vm_restore()       --  Submits a VMware full VM restore to a specified
                                destination server

    hv_full_vm_restore()    --  Submits a hyper V full VM restore to a hyperV server

    opc_full_vm_restore()   --  Submits a full VM restore of an Oracle Cloud instance

    ali_cloud_full_vm_restore() --  Submits a full VM restore of an Alibaba Cloud instance

"""

import re
from selenium.webdriver.support.ui import Select
from Web.AdminConsole.Components.panel import DropDown
from Web.AdminConsole.Components.browse import Browse
from Web.Common.page_object import (
    WebAction,
    PageService
)


class FullVMRestore:
    """
    This class contains methods for submitting full vm restore.
    """

    def __init__(self, admin_console):
        """ Init for FullVMRestore class"""
        self.__admin_console = admin_console
        self.__admin_console._load_properties(self)
        self.__driver = admin_console.driver
        self.__panel_dropdown_obj = DropDown(admin_console)
        self.__browse_obj = Browse(admin_console)

    @WebAction()
    def __select_subnet(self, subnet_id):
        """
        Selects the subnet based on OCID of subnet
        :return: None
        """
        subnet_index = 1
        sn_xpath = f'//*[@id="subnetSettings"]/option[contains(@value,"{subnet_id}")]'
        while True:
            xpath_select = f'//*[@id="networkSettings"]/option[{subnet_index}]'
            if self.__admin_console.check_if_entity_exists("xpath", xpath_select):
                self.__driver.find_element_by_xpath(xpath_select).click()
                self.__admin_console.wait_for_completion()
                if self.__admin_console.check_if_entity_exists("xpath", sn_xpath):
                    self.__driver.find_element_by_xpath(sn_xpath).click()
                    break
            else:
                break
            subnet_index += 1

    @WebAction()
    def __check_entity_exists(self):
        """Checks if entity exists for restore"""
        return self.__admin_console.check_if_entity_exists("xpath",
                                                           "//div[@class='modal fade ng-isolate-scope in']")

    @WebAction()
    def __click_restore(self):
        """Initiates restore process by clicking on restore"""
        self.__driver.find_element_by_xpath(
            f"//*[@id='browseActions']//*[contains(.,'{self.__admin_console.props['header.restore']}')]").click()

    @WebAction()
    def __send_vm_name(self, vm_restore_name):
        """Enters the VM Restore Name in the input element
        Args:
                        vm_restore_name:     the name of the VM to restore
        Returns:
        """
        _xp = "//*[@id='perVmOptions']//label[@for='displayName']/input"
        self.__driver.find_element_by_xpath(_xp).clear()
        self.__driver.find_element_by_xpath(_xp).send_keys(vm_restore_name)

    @WebAction()
    def __select_vm_options(self, vm_name):
        """Selects VM by vm_name
                Args:
                                vm_name:     the name of the VM to restore
                Returns:
        """
        xpath = f"//a[text()='{vm_name}']"
        self.__driver.find_element_by_xpath(xpath).click()

    @WebAction()
    def __select_vm_from_list(self, vm_name):
        """Selects VM by vm_name
                Args:
                                vm_name:     the name of the VM to restore
                Returns:
        """
        xpath = f"//ul[@class='vm-full-restore-list']//a[text()='" + vm_name + "']"
        self.__driver.find_element_by_xpath(xpath).click()

    @PageService()
    def full_vm_restore(
            self,
            vms_to_restore,
            inplace=False,
            restore_as="VMware vCenter",
            proxy=None,
            destination_server=None,
            vm_info=None,
            different_vcenter=False,
            different_vcenter_info={},
            disk_prov=None,
            transport_mode=None,
            power_on=True,
            over_write=True,
            live_recovery=False,
            live_recovery_options={},
            hv_restore_options={},
            dest_target=False,
            copy_precedence='default'
    ):
        """
        Submits a VMware full VM restore to a specified destination server

        Args:
            vms_to_restore           (list)  --  list of all Vms to restore

            inplace                  (bool)  --  if the VM needs to be restored in place

            restore_as               (basestring)   --  the hypervisor to be restored to
                    default :   VMware vCenter

            proxy                    (basestring)   --  the proxy to be used for restore

            destination_server       (basestring)   --  the name of the destination hypervisor

            vm_info                  (dict)  --  dict containing each restore VM details
                        vm_info = { 'VM1':  {   'host':'abc', 'datastore':'ds1', 'respool':'rp1',
                                                'network': {'source': 'network1',
                                                            'destination':'nw2'},
                                                'IP': { 'sourceIP':'','sourceSubnet':'',
                                                        'sourceGateway':'',
                                                        'DHCP':False,
                                                        'destinationIP':'', 'desitnaionSubnet':'',
                                                        'destinationGateway':'',
                                                        'destinationPreDNS':'',
                                                        'destinationAltDNS':'',
                                                        'destinationPrefWins':'',
                                                        'destinationAltWins':''}

            different_vcenter        (bool)  --  True / False to create a new vcenter client

            different_vcenter_info   (dict)  --  dict containing the new vcenter info
                        different_vcenter_info = {  'vcenter_hostname':'newvCenter',
                                                    'vcenter_username':'username',
                                                    'vcenter_password':'password'   }

            disk_prov                (basestring) --  the disk provisioning of the restore vm disks

            transport_mode           (basestring)   --  the transport mode to be used for restore

            power_on                 (bool)  --  if the restored VM needs to be powered on

            over_write               (bool)  --  if the restored VM needs to be overwritten

            live_recovery            (bool)  --  if the restore needs to use live recovery

            live_recovery_options    (dict)  --  live recovery options
                        live_recovery_options = {   'redirect_datastore':'ds1',
                                                    'delay_migration':'2'   }

            hv_restore_options       (dict)  --  all the restore options needed to restore
                                                    VMware VM as Hyper-V VM

            dest_target              (bool) -- if restore has to be done to restore target

        Returns:
            job_id      (basestring)   --  the restore job ID

        """
        if different_vcenter_info is None:
            different_vcenter_info = {}
        self.__admin_console.log.info("Selecting VMs for performing Full VM restore")
        vm_list = vms_to_restore

        if not self.__admin_console.check_if_entity_exists("xpath",
                                                           "//div[@class='modal fade ng-isolate-scope in']"):
            if copy_precedence.lower() == 'primary':
                self.__browse_obj.restore_from_primary_copy()
            self.__browse_obj.select_for_restore(vm_list)
            self.__driver.execute_script("window.scrollTo(0,0)")
            self.__driver.find_element_by_xpath(
                "//*[@id='browseActions']//*[contains(.,'" + self.__admin_console.props[
                    'header.restore'] + "')]").click()
            self.__admin_console.wait_for_completion()

        if not inplace:
            self.__admin_console.select_radio('outOfPlace')
            if self.__admin_console.check_if_entity_exists("xapth", "//label[contains(text(),'Restore as')]"):
                if 'Oracle Cloud Infrastructure' in restore_as:
                    self.__panel_dropdown_obj.select_drop_down_values(
                        values=[restore_as], drop_down_id='restoreVM_isteven-multi-select_#1408')
                    self.__admin_console.select_value_from_dropdown("destinationServer", destination_server)
                    self.__admin_console.select_value_from_dropdown("destinationProxy", proxy)
                    compartment_name = vm_info.compartment_name
                    subnet_id = vm_info.subnet_id
                    datastore = vm_info.datastore
                    shape = vm_info.disk_count
                    for vms in vm_list:
                        self.__select_vm_from_list(vms)
                        self.__admin_console.wait_for_completion()
                        self.__admin_console.select_value_from_dropdown("instanceShape", shape)
                        self.__admin_console.select_value_from_dropdown("compartment", compartment_name)
                        self.__admin_console.select_value_from_dropdown("availabilityDomain", datastore)
                        self.__select_subnet(subnet_id)
                    if power_on:
                        self.__admin_console.enable_toggle(
                            toggle_id="vmFullRestoreOracleCloudInfrastructure_toggle_powerOn")
                    else:
                        self.__admin_console.disable_toggle(
                            toggle_id="vmFullRestoreOracleCloudInfrastructure_toggle_powerOn")
                    if over_write:
                        self.__admin_console.enable_toggle(
                            toggle_id="vmFullRestoreOracleCloudInfrastructure_toggle_overwrite")
                        self.__admin_console.wait_for_completion()
                        self.__admin_console.click_button(self.__admin_console.props['button.delete.yes'])
                    else:
                        self.__admin_console.disable_toggle(
                            toggle_id="vmFullRestoreOracleCloudInfrastructure_toggle_overwrite")
                    self.__admin_console.log.info("Submitting full VM restore")
                    self.__admin_console.wait_for_completion()
                    self.__admin_console.submit_form()
                    return self.__admin_console.get_jobid_from_popup()

                if restore_as != "VMware vCenter":
                    self.__panel_dropdown_obj.select_drop_down_values(
                        values=[restore_as], drop_down_id='restoreVM_isteven-multi-select_#1408')
                    job_id = self.hv_full_vm_restore(hv_restore_options['vm_dict'],
                                                     hv_restore_options['proxy'],
                                                     hv_restore_options['server'],
                                                     hv_restore_options['location'])
                    return job_id

            if different_vcenter:
                self.__panel_dropdown_obj.select_drop_down_values(
                    values=['Select a different vCenter (optional)'],
                    drop_down_id='vmFullRestoreVMWare_isteven-multi-select_#0614')
                self.__admin_console.fill_form_by_id("vcHostName", different_vcenter_info['vcenter_hostname'])
                self.__admin_console.fill_form_by_id("vcLoginName", different_vcenter_info['vcenter_username'])
                self.__admin_console.fill_form_by_id("vcPassword", different_vcenter_info['vcenter_password'])

            if dest_target:
                target = destination_server
                self.__panel_dropdown_obj.select_drop_down_values(
                    values=[target], drop_down_id='vmFullRestoreVMWare_isteven-multi-select_#0614')

            if destination_server:
                self.__panel_dropdown_obj.select_drop_down_values(
                    values=[destination_server], drop_down_id='vmFullRestoreVMWare_isteven-multi-select_#0614')

            self.__panel_dropdown_obj.select_drop_down_values(
                values=[proxy], drop_down_id='vmFullRestoreVMWare_isteven-multi-select_#8259')

            for vms in vm_list:
                self.__driver.find_element_by_xpath("//ul[@class='vm-full-restore-list']"
                                                    "//a[text()='" + vms + "']").click()
                self.__admin_console.wait_for_completion()

                if vm_info:
                    if 'prefix' in vm_info[vms].keys():
                        display_name = vm_info[vms]['prefix'] + vms
                    elif 'suffix' in vm_info[vms].keys():
                        display_name = vms + vm_info[vms]['suffix']
                    self.__driver.find_element_by_xpath("//*[@id='perVmOptions']//label["
                                                        "@for='displayName']/input").clear()
                    self.__driver.find_element_by_xpath("//*[@id='perVmOptions']//label["
                                                        "@for='displayName']"
                                                        "/input").send_keys(display_name)
                    if not dest_target:
                        self.__driver.find_element_by_xpath("//label[@for='destinationHost']/"
                                                            "following-sibling::span/button["
                                                            "@class='btn btn-default browse-btn']"
                                                            ).click()
                        self.__admin_console.wait_for_completion()
                        self.__admin_console.check_error_message()

                        self.__admin_console.select_destination_host(vm_info[vms]['host'])
                        self.__driver.find_element_by_xpath(
                            "//form[@name='selectVmForm']/div/button[2]").click()
                        self.__admin_console.wait_for_completion()
                        self.__admin_console.log.info("Destination host selected")
                        self.__panel_dropdown_obj.select_drop_down_values(
                            values=[vm_info[vms]['datastore']],
                            drop_down_id='vmFullRestoreVMWare_isteven-multi-select_#9782',
                            partial_selection=True)
                        self.__admin_console.log.info("Datastore selected successfully")
                        self.__panel_dropdown_obj.select_drop_down_values(
                            values=[vm_info[vms]['respool']],
                            drop_down_id='vmFullRestoreVMWare_isteven-multi-select_#2149',
                            partial_selection=True)
                        self.__admin_console.log.info("Resource pool selected successfully")

                        if 'network' in vm_info[vms].keys():
                            self.__driver.find_element_by_xpath(
                                "//uib-accordion[@class='restore-network-settings ng-scope' \
                                ]/div/div[1]/div[1]/h4/a").click()
                            xpath = "//cv-grid/div/div[2]/div/div[1]/div[1]/div[2]/div[" \
                                    "@class='ui-grid-canvas']/div"
                            elements = self.__driver.find_elements_by_xpath(xpath)
                            if elements:
                                self.__admin_console.scroll_into_view(xpath)
                                for elem in elements:
                                    elem.find_element_by_xpath(
                                        "./div/div[3]//a[@title='Remove setting']").click()
                            self.__driver.find_element_by_xpath(
                                "//uib-accordion/div/div[1]/div[2]/div/div/ul/li/a").click()
                            self.__admin_console.wait_for_completion()
                            if 'source' in vm_info[vms]['network'].keys():
                                self.__panel_dropdown_obj.select_drop_down_values(
                                    0, [vm_info[vms]['network']['source']])
                                self.__panel_dropdown_obj.select_drop_down_values(
                                    1, [vm_info[vms]['network']['destination']])
                            self.__admin_console.click_button(self.__admin_console.props['OK'])
                        if 'IP' in vm_info[vms].keys():
                            self.__driver.find_element_by_xpath(
                                "//uib-accordion[@class='restore-network-settings ng-scope']/div/"
                                "div[2]/div[1]/h4/a").click()
                            xpath = "//cv-grid/div/div[2]/div/div[1]/div[1]/div[2]/div[@class=" \
                                    "'ui-grid-canvas']/div"
                            self.__admin_console.scroll_into_view(xpath)
                            elements = self.__driver.find_elements_by_xpath(xpath)
                            for elem in elements:
                                elem.find_element_by_xpath(
                                    "./div/div[3]/div/span/a[2]").click()
                            self.__driver.find_element_by_xpath(
                                "//uib-accordion/div/div[2]/div[2]/div/div/ul/li/a").click()
                            self.__admin_console.wait_for_completion()
                            ip_settings = vm_info[vms]['IP']
                            self.__admin_console.fill_form_by_id("sourceIPAddress", ip_settings['sourceIP'])
                            self.__admin_console.fill_form_by_id("sourceSubnetMask", ip_settings['sourceSubnet'])
                            self.__admin_console.fill_form_by_id("sourceDefaultGateway",
                                                                 ip_settings['sourceGateway'])
                            if 'DHCP' in ip_settings.keys():
                                self.__admin_console.toggle_enable("Use DHCP")
                            else:
                                self.__admin_console.fill_form_by_id("destinationIpAddress",
                                                                     ip_settings['destinationIP'])
                                self.__admin_console.fill_form_by_id("destinationSubnetMask",
                                                                     ip_settings['destinationSubnet'])
                                self.__admin_console.fill_form_by_id("destinationDefaultGateway",
                                                                     ip_settings['destinationGateway'])

                                if 'destinationPrefDNS' in ip_settings.keys():
                                    self.__admin_console.fill_form_by_id("destinationPrefDnsServer",
                                                                         ip_settings['destinationPrefDNS'])
                                if 'destinationAltDNS' in ip_settings.keys():
                                    self.__admin_console.fill_form_by_id("destinationAltDnsServer",
                                                                         ip_settings['destinationAltDNS'])
                                if 'destinationPrefWins' in ip_settings.keys():
                                    self.__admin_console.fill_form_by_id("destinationPrefWinsServer",
                                                                         ip_settings['destinationPrefWins'])
                                if 'destinationAltWins' in ip_settings.keys():
                                    self.__admin_console.fill_form_by_id("destinationAltWinsServer",
                                                                         ip_settings['destinationAltWins'])
                        # if live_recovery:
                        #     self.__admin_console.toggle_enable(
                        #         "Restore virtual machine using live recovery (vMotion)")
                        #     self.__panel_dropdown_obj.select_drop_down_values(
                        #         values=[live_recovery_options['redirect_datastore']],
                        #         drop_down_id='vmFullRestoreVMWare_isteven-multi-select_#7353')
                        #     self.__admin_console.fill_form_by_id("delayMigration",
                        #                                          live_recovery_options['delay_migration'])

        else:
            self.__panel_dropdown_obj.select_drop_down_values(
                values=[proxy], drop_down_id='vmFullRestoreVMWare_isteven-multi-select_#8259')

        if live_recovery:
            self.__admin_console.enable_toggle(toggle_id="vmFullRestoreVmware_toggle_liveRecovery")
            if live_recovery_options:
                if [live_recovery_options['redirect_datastore']] is not None:
                    self.__panel_dropdown_obj.select_drop_down_values(
                        values=[live_recovery_options['redirect_datastore']],
                        drop_down_id='vmFullRestoreVMWare_isteven-multi-select_#7353')
                if live_recovery_options['delay_migration'] is not None:
                    self.__admin_console.fill_form_by_id("delayMigration", live_recovery_options['delay_migration'])

        if power_on:
            self.__admin_console.enable_toggle(toggle_id='vmFullRestoreVmware_toggle_powerOn')
        else:
            self.__admin_console.disable_toggle(toggle_id='vmFullRestoreVmware_toggle_powerOn')

        if over_write:
            self.__admin_console.enable_toggle(toggle_id='vmFullRestoreVmware_toggle_overwrite')
            self.__admin_console.click_button(self.__admin_console.props['button.delete.yes'])
        else:
            self.__admin_console.disable_toggle(toggle_id='vmFullRestoreVmware_toggle_overwrite')

        self.__panel_dropdown_obj.select_drop_down_values(
            values=[disk_prov], drop_down_id='vmFullRestoreVMWare_isteven-multi-select_#6106')
        self.__panel_dropdown_obj.select_drop_down_values(
            values=[transport_mode], drop_down_id='vmFullRestoreVMWare_isteven-multi-select_#5144')
        self.__admin_console.submit_form()
        return self.__admin_console.get_jobid_from_popup()

    @PageService()
    def hv_full_vm_restore(
            self,
            vm_list,
            proxy,
            server,
            location,
            name=None,
            path=None,
            power_on=False,
            over_write=False,
            register=False,
            network=None):
        """
        Submits a hyper V full VM restore to a hyperV server

        Args:
            vm_list      (basestring)   --  the name of the VM to restore

            proxy        (basestring)   --  the name of the proxy to be used for restore

            server       (basestring)   --  the hypervisor server to restore to

            location     (basestring)   --  the location where the disks and VM are to be restored

            name         (basestring)   --  the name of the restored VM

            path         (basestring)   --  the path to the restored VM location if not default

            power_on     (bool)  --  if the restored VM has to be powered on

            over_write   (bool)  --  if the restored VM has to overwrite an existing VM

            register     (bool)  --  if the restored VM has to be registered to cluster

            network      (basestring)   --  if a network card has to be attached to the VM

        Returns:
            job_id      (basestring)   --  the restore job ID


        """
        if not self.__admin_console.check_if_entity_exists("xpath",
                                                           "//div[@class='modal fade ng-isolate-scope in']"):
            self.__browse_obj.select_for_restore(vm_list)
            self.__driver.find_element_by_xpath(
                "//*[@id='browseActions']//*[contains(.,'Restore')]").click()
            self.__admin_console.wait_for_completion()
        self.__admin_console.cv_single_select("Virtualization client", server)
        self.__admin_console.cv_single_select("Destination client", proxy)
        if power_on:
            self.__admin_console.checkbox_select("powerOn")
        else:
            self.__admin_console.checkbox_deselect("powerOn")

        if over_write:
            self.__admin_console.checkbox_select("overwrite")
            self.__admin_console.wait_for_completion()
            self.__admin_console.click_button(self.__admin_console.props['button.delete.yes'])
        else:
            self.__admin_console.checkbox_deselect("overwrite")

        if register:
            self.__admin_console.checkbox_select("failover")
        else:
            self.__admin_console.checkbox_deselect("failover")

        for vms in vm_list:
            if len(vm_list) != 1:
                self.__driver.find_element_by_xpath("//ul[@class='vm-full-restore-list']//a["
                                                    "contains(text(),'" + vms + "')]").click()
                self.__admin_console.wait_for_completion()

            if location != "Original folder":
                self.__admin_console.log.info("Performing out of place restore")
                self.__driver.find_element_by_id("outOfPlace").click()
                self.__admin_console.wait_for_completion()
                self.__admin_console.select_value_from_dropdown("restoreLocation", location)
                self.__admin_console.wait_for_completion()
                if location == "Hyper-V default folder":
                    self.__driver.find_element_by_xpath(
                        "//label/input[@name='displayName']").clear()
                    self.__driver.find_element_by_xpath(
                        "//label/input[@name='displayName']") \
                        .send_keys(name + "".join(vms))
                elif location == "Select a folder":
                    self.__driver.find_element_by_xpath(
                        "//label/input[@name='displayName']").clear()
                    self.__driver.find_element_by_xpath(
                        "//label/input[@name='displayName']") \
                        .send_keys(name + "".join(vms))
                    self.__admin_console.fill_form_by_id("destinationFolder", path)

            if network:
                self.__admin_console.select_value_from_dropdown("networkAdapter", network)

        self.__admin_console.log.info("Submitting full VM restore")
        self.__admin_console.wait_for_completion()
        self.__driver.find_element_by_xpath("//div[@class='local-options']/form/div[2]/button["
                                            "contains(text(),'Submit')]").click()

        return self.__admin_console.get_jobid_from_popup()

    @PageService()
    def opc_full_vm_restore(self, vm_list, proxy, server, power_on, restore_prefix,
                            user_account, shape, network_list=None,
                            security_groups=None, ssh_keys=None):
        """
        Full instance restore method for Oracle Public Cloud
        Args:
            vm_list             (list)  --  the list of all VMs to restore

            proxy               (basestring)   --  the proxy to be used for restore

            server              (basestring)   --  the oracle cloud server to restore to

            power_on            (bool)  --  True / False to power on the instance

            restore_prefix      (basestring)   --  The basestringing to be prefixed to the VM name

            user_account        (basestring)   --  The user account where instances are to restored

            shape               (basestring)   --  The shape of the restore instances

            network_list        (list)  --  List of all networks to be associated to instances

            security_groups     (list)  --  List of all security groups to be associated

            ssh_keys            (list)  --  List of all ssh keys to be associated

        Returns:
            job_id  (basestring)   --  the restore job ID

        """
        if not self.__admin_console.check_if_entity_exists("xpath",
                                                           "//div[@class='modal fade ng-isolate-scope in']"):
            self.__browse_obj.select_for_restore(vm_list)
            self.__driver.find_element_by_xpath(
                "//*[@id='browseActions']//*[contains(.,'Restore')]").click()
            self.__admin_console.wait_for_completion()

        self.__admin_console.select_value_from_dropdown("Destination hypervisor", server, search=True)
        self.__admin_console.select_value_from_dropdown("destinationProxy", proxy)

        if power_on:
            self.__admin_console.checkbox_select("powerOn")
        else:
            self.__admin_console.checkbox_deselect("powerOn")

        for vm_name in vm_list:
            restore_vm_name = restore_prefix + vm_name
            if len(vm_list) != 1:
                self.__driver.find_element_by_xpath("//ul[@class='vm-full-restore-list']//a["
                                                    "contains(text(),'" + vm_name + "')]").click()
                self.__admin_console.wait_for_completion()

                # Since there are multiple instances to restore, over writing the shape to Auto
                shape = "Auto"

            self.__driver.find_element_by_xpath("//*[@id='perVmOptions']//label["
                                                "@for='displayName']/input").clear()
            self.__driver.find_element_by_xpath("//*[@id='perVmOptions']//label["
                                                "@for='displayName']"
                                                "/input").send_keys(restore_vm_name)
            self.__admin_console.wait_for_completion()

            self.__driver.find_element_by_xpath(
                "//button[@class='btn btn-default browse-btn']").click()
            self.__admin_console.wait_for_completion()
            self.__admin_console.check_error_message()

            self.__admin_console.select_destination_host(user_account)
            self.__admin_console.submit_form()
            self.__admin_console.log.info("User account selected")

            self.__admin_console.select_value_from_dropdown("instanceType", shape)
            self.__admin_console.log.info("Instance shape selected")

            if network_list:
                self.__admin_console.cvselect_from_dropdown('Network', network_list)
                self.__admin_console.log.info("Selected the networks")

            if security_groups:
                self.__admin_console.cvselect_from_dropdown('Security groups', security_groups)
                self.__admin_console.log.info("Selected the security groups")

            if ssh_keys:
                self.__admin_console.cvselect_from_dropdown('SSH keys', ssh_keys)
                self.__admin_console.log.info("Selected the ssh keys")

            self.__admin_console.log.info("Going to submit restore job")
            self.__admin_console.wait_for_completion()
            self.__driver.find_element_by_xpath("//div[@class='local-options']/form/div[2]/button["
                                                "contains(text(),'Submit')]").click()

            return self.__admin_console.get_jobid_from_popup()

    @PageService()
    def ali_cloud_full_vm_restore(self, instance_list, destination_client, destination_proxy,
                                  power_on, overwrite_instance, in_place=False,
                                  restore_prefix=None, restore_suffix=None,
                                  instance_options=None):
        """
        Performs a full instance restore to Ali Cloud

        Args:
            instance_list       (list):         list of all instances to restore

            destination_client  (basestring):   the destination client where instances are to
                                                be restored

            destination_proxy   (basestring):   the proxy to be used for restore

            power_on            (bool):         if the instances are to be powered on after restore

            overwrite_instance  (bool):         to overwrite existing instances

            restore_prefix      (basestring):   to prepend a prefix to source instance name

            restore_suffix      (basestring):   to append a suffix to a source instance name

            in_place            (bool):         to restore the instances in-place

            instance_options    (dict):         restore options for each individual instance

                    Sample dict:    {'instance1':   {   'availability_zone': '',
                                                        'instance_type: '',
                                                        'network':  '',
                                                        'security_groups':  ['group1', 'group2']
                                                    },
                                    'instance2':   {   'availability_zone': '',
                                                        'instance_type: '',
                                                        'network':  '',
                                                        'security_groups':  ['group1', 'group2']
                                                    }
                                    }

        Raises:
            Exception:
                if there is an error with submitting Ali Cloud Full instance restore

        Returns:
            job_id  (basestring):   the restore job ID

        """
        display_name = None

        if not self.__admin_console.check_if_entity_exists("xpath",
                                                           "//div[@class='modal fade ng-isolate-scope in']"):
            self.__browse_obj.select_for_restore(instance_list)
            self.__driver.find_element_by_xpath(
                "//*[@id='browseActions']//*[contains(.,'Restore')]").click()
            self.__admin_console.wait_for_completion()

        self.__admin_console.select_value_from_dropdown("Destination client", destination_client, search=True)
        if self.__admin_console.check_if_entity_exists("id", "destinationProxy"):
            self.__admin_console.select_value_from_dropdown("destinationProxy", destination_proxy)

        if power_on:
            self.__admin_console.checkbox_select("powerOn")
        else:
            self.__admin_console.checkbox_deselect("powerOn")

        if overwrite_instance:
            self.__admin_console.checkbox_select("overwrite")
            self.__admin_console.wait_for_completion()
            self.__admin_console.click_button(self.__admin_console.props['button.delete.yes'])
        else:
            self.__admin_console.checkbox_deselect("overwrite")

        if len(instance_list) != 1:
            self.__driver.find_element_by_xpath("//ul[@class='vm-full-restore-list']//a["
                                                "contains(text(),'All Instances')]"
                                                ).click()
            self.__admin_console.wait_for_completion()

        if not in_place:
            self.__admin_console.checkbox_deselect("showInPlaceRestore")
            self.__admin_console.wait_for_completion()

            for instance_name in instance_list:
                self.__driver.find_element_by_xpath("//ul[@class='vm-full-restore-list']"
                                                    "//a[text()='" + instance_name + "']").click()
                self.__admin_console.wait_for_completion()

                if restore_prefix:
                    display_name = restore_prefix + instance_name
                elif restore_suffix:
                    display_name = instance_name + restore_suffix

                self.__driver.find_element_by_xpath("//*[@id='perVmOptions']//input["
                                                    "@name='displayName']").clear()
                self.__driver.find_element_by_xpath("//*[@id='perVmOptions']//input["
                                                    "@name='displayName']"
                                                    ).send_keys(display_name)

                options = instance_options[instance_name]
                # Browse for availability zones

                self.__driver.find_element_by_xpath("//*[@id='perVmOptions']/div[3]//button"
                                                    ).click()
                self.__admin_console.wait_for_completion()

                self.__admin_console.select_destination_host(options['availability_zone'])
                self.__admin_console.submit_form()

                self.__admin_console.select_value_from_dropdown("instanceType", options['instance_type'], True)

                self.__driver.find_element_by_xpath("//*[@id='perVmOptions']/div[5]//button"
                                                    ).click()
                self.__admin_console.wait_for_completion()
                self.__admin_console.select_destination_host(options['network'], contains_host_name=True)
                self.__admin_console.submit_form()

                self.__admin_console.cv_single_select("Security groups (browse network to populate)",
                                                      options['security_groups'][0])

        self.__admin_console.log.info("Submitting full instance restore")
        self.__admin_console.wait_for_completion()
        self.__driver.find_element_by_xpath("//div[@class='local-options']/form/div[2]/button["
                                            "contains(text(),'Submit')]").click()
        return self.__admin_console.get_jobid_from_popup()

    @PageService()
    def azure_full_vm_restore(self, vm_list, proxy, server, in_place, vm_info,
                              create_public_ip=False, over_write=False, power_on=False,
                              managed_vm=False, restore_prefix=None):

        """
        Performs a full VM restore for Azure RM
        Args:
            instance_list    (list):     list of all instances to restore

            vm_list          (list):     the name of the VM to restore

            proxy            (str):      the name of the proxy to be used for restore

            power_on         (bool):     if the restored VM has to be powered on

            param over_write (bool):     if the restored VM has to overwrite an
                                         existing VM
            server           (str) :     server name with each restore to be done

            vm_info          (dict):     contains:azure_container,azure_vmsize,
                                        storageAccount,azure_vmnetwork

            managed_vm       (bool):      False

            create_public_ip (bool):      False

            restore_prefix   (str):     Restore prefix

            over_write       (bool):    False

            in_place         (bool):    True if you need in-place restore

        Returns:
            job_id  (basestring):   the restore job ID

        """
        if not self.__admin_console.check_if_entity_exists("xpath",
                                                           "//div[@class='modal fade ng-isolate-scope in']"):
            self.__browse_obj.select_for_restore(vm_list)
            self.__driver.find_element_by_xpath("//span[@id='browseActions']/a").click()
            self.__admin_console.wait_for_completion()
        self.__admin_console.wait_for_completion()
        self.__admin_console.select_value_from_dropdown("destinationProxy", proxy)
        self.__admin_console.wait_for_completion()

        if power_on:
            # self.__admin_console.checkbox_select("powerOn")
            self.__admin_console.enable_toggle(0)
            self.__admin_console.log.info("Selectd powerON option")

        if over_write:
            # self.__admin_console.checkbox_select("overwrite")
            self.__admin_console.enable_toggle(1)
            self.__admin_console.wait_for_completion()
            self.__admin_console.click_button(self.__admin_console.props['button.delete.yes'])
            self.__admin_console.wait_for_completion()

        for vms in vm_list:
            if not in_place:
                # self.__admin_console.checkbox_deselect("showInPlaceRestore")
                self.__admin_console.select_radio('outOfPlace')
                self.__admin_console.log.info("Selected out of place restore option")

                # select destination server
                select = Select(self.__driver.find_element_by_xpath(
                    "//div[@class='global-options']//div[1]//label[1]//select[1]"))
                for option in select.options:
                    if option.text.lower() == server.lower():
                        option.click()
                        break
                self.__admin_console.wait_for_completion()
                self.__driver.find_element_by_id("outOfPlace").click()
                self.__admin_console.log.info("Deselected in place option")

                if len(vm_list) != 1:
                    self.__driver.find_element_by_xpath("//ul[@class='vm-full-restore-list']//a["
                                                        "contains(text(),'" + vms + "')]").click()
                    self.__admin_console.wait_for_completion()

                # Clear and enter new VM display name
                display_name = ""
                if restore_prefix is not None:
                    display_name = restore_prefix + vms
                else:
                    display_name = vms

                self.__driver.find_element_by_xpath("//*[@id='perVmOptions']//label["
                                                    "@for='displayName']/input").clear()
                self.__driver.find_element_by_xpath("//*[@id='perVmOptions']//label["
                                                    "@for='displayName']"
                                                    "/input").send_keys(display_name[:14])
                self.__admin_console.log.info("Entered the name of restored VM")

                if 'resource_group' in vm_info[vms].keys():
                    resource_group = vm_info[vms]['resource_group']
                    self.__admin_console.select_value_from_dropdown("azureContainer", resource_group, False, True)
                    self.__admin_console.log.info("Entered the name of resource_group")
                    self.__admin_console.wait_for_completion()

                if "storage_account" in vm_info[vms].keys():
                    storage_account = vm_info[vms]['storage_account']
                    self.__admin_console.select_value_from_dropdown("storageAccount", storage_account, False, True)
                    self.__admin_console.log.info("Entered the name of storage Account")

                if 'region' in vm_info[vms].keys():
                    region = vm_info[vms]['region']
                    self.__admin_console.select_value_from_dropdown("region", region, False, True)
                    self.__admin_console.log.info("Entered the name of region")

                if 'vm_size' in vm_info[vms].keys():
                    select_azure_vmsize = Select(self.__driver.find_element_by_id("azureVmSize"))
                    azure_vmsize = vm_info[vms]['vm_size']
                    pattern = re.compile(azure_vmsize, re.IGNORECASE)
                    for option in select_azure_vmsize.options:
                        value = option.get_attribute('value')
                        if pattern.search(value):
                            option.click()
                            break
                    self.__admin_console.wait_for_completion()
                    self.__admin_console.log.info("Entered VM size info")

                if 'network_interface' in vm_info[vms].keys():
                    azure_vmnetwork = vm_info[vms]['network_interface']
                    self.__admin_console.cvselect_from_dropdown("Network interface", azure_vmnetwork, False)
                    self.__admin_console.log.info("Entered the name of NetworkInterface")

                if create_public_ip:
                    self.__admin_console.checkbox_select("createPublicIp")
                else:
                    self.__admin_console.checkbox_deselect("createPublicIp")

                if 'security_group' in vm_info[vms].keys():
                    security_group = vm_info[vms]['security_group']
                    self.__admin_console.select_value_from_dropdown("securityGroup", security_group, False, True)
                    self.__admin_console.wait_for_completion()

                if managed_vm:
                    self.__admin_console.checkbox_select("restoreAsManagedVM")
                else:
                    self.__admin_console.checkbox_deselect("restoreAsManagedVM")
            else:
                if not self.__driver.find_element_by_id("inPlace").is_selected():
                    self.__admin_console.checkbox_select("inPlace")
                    self.__admin_console.wait_for_completion()

        self.__admin_console.log.info("Submitting full instance restore")
        self.__admin_console.wait_for_completion()
        self.__driver.execute_script("window.scrollTo(0, document.body.scrollHeight);")
        self.__admin_console.click_button(self.__admin_console.props['action.submit'])
        return self.__admin_console.get_jobid_from_popup()

    @PageService()
    def vcloud_full_vm_restore(self, vm_list, proxy, destination_server, in_place, vm_info, org_vdc,
                               power_on=True, overwrite=False,
                               restore_vapp=True, restore_prefix=None):

        """
        Performs a full VM restore for Azure RM
        Args:
            vm_list                 (list):     the name of the VM to restore

            proxy                   (str):      the name of the proxy to be used for restore

            destination_server      (str) :     server name with each restore to be done

            in_place                (str):      the type of restore to be performed

            vm_info                 (list):     contains:vapp_name ,source network

            org_vdc                 (str) :     VDC name to be used for destination VM

            power_on                (bool):     if the restored VM has to be powered on

            overwrite               (bool):     if the restored VM has to overwrite an
                                                existing VM

           restore_prefix           (str):     Restore prefix

           restore_vapp             (bool):     True

        Returns:
            job_id  (basestring):   the restore job ID

        """
        if not self.__admin_console.check_if_entity_exists("xpath",
                                                           "//div[@class='modal fade ng-isolate-scope in']"):
            self.__browse_obj.select_for_restore(vm_list)
            self.__driver.find_element_by_xpath("//span[@id='browseActions']/a").click()
            self.__admin_console.wait_for_completion()
        if self.__admin_console.check_if_entity_exists("xpath",
                                                       "//button[@id='destinationServer']"):
            self.__admin_console.select_value_from_dropdown("Destination hypervisor", destination_server)
            self.__admin_console.wait_for_completion()
        self.__admin_console.select_value_from_dropdown("destinationProxy", proxy)
        self.__admin_console.wait_for_completion()

        for vms in vm_list:

            if len(vm_list) != 1:
                self.__driver.find_element_by_xpath("//ul[@class='vm-full-restore-list']//a["
                                                    "contains(text(),'" + vms + "')]").click()
                self.__admin_console.wait_for_completion()

            if not in_place:

                self.__admin_console.select_radio(id="outOfPlace")
                self.__admin_console.log.info("Deselectd in place option")
                self.__send_vm_name(restore_prefix + vms)
                self.__admin_console.log.info("Entered the name of restored VM")

                self.__admin_console.select_value_from_dropdown("orgvDC", org_vdc, False, True)
                self.__admin_console.log.info("Entered the name of Organization VDC")

                self.__admin_console.fill_form_by_id("VAppName", vm_info[vms]['vapp_name'])
                self.__admin_console.log.info("Entered the name of storage Account")

                if 'source_network' in vm_info[vms].keys():
                    self.__driver.find_element_by_xpath(
                        "//a[@uib-accordion-transclude='heading']").click()
                    self.__admin_console.wait_for_completion()

                    xpath = "//cv-grid/div/div[3]/div/div[1]/div[1]/div[2]/div[@class='ui-grid-canvas']/div"
                    elements = self.__driver.find_elements_by_xpath(xpath)
                    if elements:
                        self.__admin_console.scroll_into_view(xpath)
                        for elem in elements:
                            elem.find_element_by_xpath(
                                "./div/div[3]//a[@title='Remove setting']").click()
                    self.__driver.find_element_by_xpath(
                        "//a[@data-ng-click='openAddNetworkSettingsDialog()']").click()
                    self.__admin_console.wait_for_completion()
                    self.__admin_console.select_value_from_dropdown("networkSettingsSource",
                                                                    vm_info[vms]['source_network'])
                    self.__admin_console.select_value_from_dropdown("networkSettingsDestination",
                                                                    vm_info[vms]['destination_network'])

                    self.__driver.find_element_by_xpath(
                        "//button[@type='submit'][contains(text(),'OK')]").click()
                    self.__admin_console.wait_for_completion()
                    self.__driver.find_element_by_xpath(
                        "//a[@uib-accordion-transclude='heading']").click()
                    self.__admin_console.wait_for_completion()
            else:
                self.__admin_console.checkbox_select("showInPlaceRestore")
                self.__admin_console.wait_for_completion()
        if power_on:
            self.__admin_console.enable_toggle(toggle_id="vmFullRestoreVcloud_toggle_powerOn")
            self.__admin_console.log.info("Selectd Power on option")
        else:
            self.__admin_console.disable_toggle(toggle_id="vmFullRestoreVcloud_toggle_powerOn")
            self.__admin_console.log.info("Deselectd Power on option")

        if overwrite:
            self.__admin_console.enable_toggle(toggle_id="vmFullRestoreVcloud_toggle_powerOn")
            self.__admin_console.wait_for_completion()
            self.__admin_console.click_button(self.__admin_console.props['button.delete.yes'])
        else:
            self.__admin_console.disable_toggle(toggle_id="vmFullRestoreVcloud_toggle_powerOn")

        if restore_vapp:
            self.__admin_console.enable_toggle(toggle_id="vmFullRestoreVcloud_label_vAppConfiguration")
            self.__admin_console.wait_for_completion()
        self.__admin_console.log.info("Submitting full instance restore")
        self.__driver.find_element_by_xpath("//*[@id='vmFullRestoreVcloud_button_#1007']").click()
        return self.__admin_console.get_jobid_from_popup()

    @PageService()
    def oci_full_vm_restore(
            self,
            vm_list,
            proxy,
            server,
            power_on,
            in_place,
            vm_instance,
            over_write=True,
            restore_prefix=None):
        """
                Performs a full VM restore for OCI VM
                Args:
                    vm_list                 (list):     the name of the VM to restore

                    proxy                   (str):      the name of the proxy to be used for restore

                    server      (str) :     server name with each restore to be done

                    power_on                (bool):     if the restored VM has to be powered on

                    in_place                (str):      the type of restore to be performed

                    vm_instance             (list):     contains:compartment, AD, VCN

                    over_write              (bool):     if the restored VM has to overwrite an
                                                        existing VM

                   restore_prefix           (str):     Restore prefix

                Returns:
                    job_id  (basestring):   the restore job ID

                """
        if not self.__check_entity_exists():
            self.__browse_obj.select_for_restore(vm_list)
            self.__click_restore()
            self.__admin_console.wait_for_completion()
        self.__admin_console.select_value_from_dropdown("Access node client", proxy)
        if power_on:
            self.__admin_console.checkbox_select("vmFullRestoreOracleCloudInfrastructure_toggle_powerOn")
        else:
            self.__admin_console.checkbox_deselect("vmFullRestoreOracleCloudInfrastructure_toggle_powerOn")

        if over_write:
            self.__admin_console.checkbox_select("vmFullRestoreOracleCloudInfrastructure_toggle_overwrite")
            self.__admin_console.wait_for_completion()
            self.__admin_console.click_button(self.__admin_console.props['button.delete.yes'])
        else:
            self.__admin_console.checkbox_deselect("vmFullRestoreOracleCloudInfrastructure_toggle_overwrite")
        if not in_place:
            compartment_name = vm_instance.compartment_name
            subnet_id = vm_instance.subnet_id
            datastore = vm_instance.datastore

            self.__admin_console.checkbox_select("outOfPlace")
            for vm_name in vm_list:
                vm_restore_name = restore_prefix + vm_name
                self.__send_vm_name(vm_restore_name)
                self.__select_vm_options(vm_name)
                self.__admin_console.select_value_from_dropdown("compartment", compartment_name)
                self.__admin_console.select_value_from_dropdown("availabilityDomain", datastore)
                self.__select_subnet(subnet_id)

        else:
            self.__admin_console.checkbox_select("inPlace")

        self.__admin_console.log.info("Submitting full VM restore")
        self.__admin_console.wait_for_completion()
        self.__admin_console.submit_form()
        return self.__admin_console.get_jobid_from_popup()

    @PageService()
    def google_cloud_full_vm_restore(self, instance_list, restore_prefix=None, in_place=False,
                                     proxy=None, power_on=True,
                                     overwrite_instance=False,
                                     instance_options=None):
        """
        Performs a full instance restore to Google Cloud
        Args:
            instance_list       (list):         list of all instances to restore
            restore_prefix   (str):     Restore prefix
            in_place            (bool):         to restore the instances in-place
            proxy   (basestring):   the proxy to be used for restore
            power_on            (bool):         if the instances are to be powered on after restore
            overwrite_instance  (bool):         to overwrite existing instances
            instance_options    (dict):         restore options for each individual instance
                    Sample dict:    {'instance1':   {   'availability_zone': '',
                                                        'instance_type: '',
                                                        'network':  '',
                                                        'security_groups':  ['group1', 'group2']
                                                    },
                                    'instance2':   {   'availability_zone': '',
                                                        'instance_type: '',
                                                        'network':  '',
                                                        'security_groups':  ['group1', 'group2']
                                                    }
                                    }
        Raises:
            Exception:
                if there is an error with submitting Google Cloud Full instance restore
        Returns:
            job_id  (basestring):   the restore job ID
        """
        if not self.__admin_console.check_if_entity_exists("xpath",
                                                           "//div[@class='modal fade ng-isolate-scope in']"):
            self.__browse_obj.select_for_restore(instance_list)
            self.__click_restore()
            self.__admin_console.wait_for_completion()
        if not in_place:
            self.__admin_console.select_radio(id="outOfPlace")
        self.__admin_console.select_value_from_dropdown("Access node", proxy)
        for _vm in instance_list:
            self.__driver.find_element_by_xpath("//*[@id='perVmOptions']//label["
                                                "@for='displayName']/input").clear()
            self.__driver.find_element_by_xpath("//*[@id='perVmOptions']//label["
                                                "@for='displayName']"
                                                "/input").send_keys(restore_prefix + _vm)
            if power_on:
                self.__admin_console.checkbox_select("vmFullRestoreGoogleCloud_toggle_powerOn")
            else:
                self.__admin_console.checkbox_deselect("vmFullRestoreGoogleCloud_toggle_powerOn")
            if overwrite_instance:
                self.__admin_console.checkbox_select("vmFullRestoreGoogleCloud_toggle_overwrite")
                self.__admin_console.wait_for_completion()
                self.__admin_console.click_button(self.__admin_console.props['button.delete.yes'])
            else:
                self.__admin_console.checkbox_deselect("vmFullRestoreGoogleCloud_toggle_overwrite")
        self.__admin_console.log.info("Submitting full instance restore")
        self.__admin_console.wait_for_completion()
        self.__admin_console.submit_form()
        return self.__admin_console.get_jobid_from_popup()
