# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides method for guest files restore.

Classes:

    EndUserGuestFilesRestoreSelectFolder() ---> AdminPage() ---> AdminConsoleBase() ---> Object()


Functions:

    __select_restore_or_download()           -- Selects either restore or download

    submit_this_vm_restore()     -- Submits a VMware guest files restore with the backed up VM as
                                    the destination server

"""

from Web.Common.page_object import WebAction, PageService
from Web.AdminConsole.Components.panel import PanelInfo


class EndUserGuestFilesRestoreSelectFolder:
    """
    This class provides methods to perform guest files restore to same vm
    """

    def __init__(self, admin_console):
        # super(EndUserGuestFilesRestoreSelectFolder, self).__init__(driver)
        self.__admin_console = admin_console
        self.__admin_console._load_properties(self)
        self.__driver = admin_console.driver
        self.__panel_obj = PanelInfo(admin_console)

    @WebAction()
    def __select_restore_or_download(self, action="Restore"):
        """
        Selects restore / download button

        Args:
            action  (basestring):   To Select restore or download button

        """
        self.__driver.execute_script("window.scrollTo(0,0)")
        self.__driver.find_element_by_xpath(
            f"//*[@id='browseActions']//*[contains(.,'{action}')]").click()
        self.__admin_console.wait_for_completion()

    @WebAction()
    def __verify_vm(self, vm_name):
        """
        Verify backup vm is selected by default under myVM

        Args:
        vm_name (base string) -- name of the backup vm

        Raises:
            Exception:
                If the VM name is different from the VM selected under myVM

        """

        if vm_name not in self.__driver.find_element_by_xpath("//button[@id='vm']/div").text:
            raise Exception("The source backup VM was not selected by default.")

    @PageService()
    def enduser_files_restore(
            self,
            files,
            vm_name,
            vm_username,
            vm_password,
            path,
            over_write=True,
            all_files=False):
        """
        Submits a VMware guest files restore with the backed up VM as the destination server

        Args:
            files            (list)         :  the files to be restored

            vm_name          (basestring)   :  the VM to which the files are to be restored

            proxy            (basestring)   :  the proxy to be used for restore

            vm_username      (basestring)   :  the username of the destination VM

            vm_password      (basestring)   :  the password of the destination VM

            path             (basestring)   :  the path in the destination VM where files are
                                                to be restored

            over_write       (bool)         :  if files are to be overwritten during restore

            all_files        (bool)         :  if all the files are to be selected for restore

        Returns:
            job_id           (basestring)   :  the ID of the restore job submitted

        """
        self.__admin_console.select_for_restore(files, all_files)
        self.__admin_console.log.info("Files selected. Submitting restore")
        self.__select_restore_or_download()
        if self.__driver.find_element_by_id("vm"):
            self.__verify_vm(vm_name)

        if self.__admin_console.check_if_entity_exists("name", "vmLoginName"):
            self.__admin_console.log.info("Filling the vm creds")
            self.__admin_console.fill_form_by_name("vmLoginName", vm_username)
            self.__admin_console.fill_form_by_name("vmPassword", vm_password)

        self.__admin_console.log.info("Filling in the restore path")
        self.__admin_console.fill_form_by_id("restorePath", path)

        if over_write:
            self.__admin_console.enable_toggle(toggle_id='fileLevelRestore_toggle_unconditionallyOverwrite')

        self.__admin_console.log.info("Submitting end user Guest files restore to the same VM")
        self.__admin_console.submit_form(False)

        return self.__admin_console.get_jobid_from_popup()
