# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides methods for submitting full vm restore for all hypervisors

Classes:

    EndUserFullVMRestore() --- > AdminPage() ---> AdminConsoleBase() ---> Object()

FullVMRestore --  This class contains methods for submitting full vm restore.

Functions:

    enduser_full_vm_restore       --  Submits a VMware full VM restore as end user
    #end user               -- User having permissions only at the VM level


"""

from Web.AdminConsole.Components.panel import PanelInfo
from Web.Common.page_object import PageService


class EndUserFullVMRestore:
    """
    This class contains methods for submitting full vm restore.
    """

    def __init__(self, admin_console):
        """ Init for FullVMRestore class"""
        self.__admin_console = admin_console
        self.__admin_console._load_properties(self)
        self.__driver = admin_console.driver
        self.__panel_info_obj = PanelInfo(admin_console)

    @PageService()
    def enduser_full_vm_restore(
            self,
            vm_to_restore,
            inplace=False,
            power_on=True,
            over_write=True,
            restore_prefix=""
    ):
        """
        Submits full VM restore as vm end user

        Args:

            vm_to_restore:                   --  VM thats need to be restored

            inplace                  (bool)  --  if the VM needs to be restored in place

            power_on                 (bool)  --  if the restored VM needs to be powered on

            over_write               (bool)  --  if the restored VM needs to be overwritten

            restore_prefix:                  --  prefix word to keep before the restored VM


        Returns:
            job_id      (basestring)   --  the restore job ID

        """
        self.__admin_console.log.info("performing end user Full VM restore")

        if not inplace:
            restore_vm = restore_prefix + vm_to_restore
            self.__admin_console.fill_form_by_name('displayName', restore_vm)

        if power_on:
            self.__admin_console.enable_toggle(toggle_id='vmFullRestoreVmware_toggle_powerOn_endUser')
        else:
            self.__admin_console.disable_toggle(toggle_id='vmFullRestoreVmware_toggle_powerOn_endUser')

        if over_write:
            self.__admin_console.enable_toggle(toggle_id='vmFullRestoreVmware_toggle_overwrite_endUser')
            self.__admin_console.click_button(self.__admin_console.props['button.delete.yes'])
        else:
            self.__admin_console.disable_toggle(toggle_id='vmFullRestoreVmware_toggle_overwrite_endUser')

        self.__admin_console.submit_form()
        return self.__admin_console.get_jobid_from_popup()
