# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""
This module provides functions or operations to do a disk level restore.

Class:

    DiskLevelRestore() ---> AdminPage() ---> AdminConsoleBase() ---> object()

Functions:

    disk_level_restore()        -- Submits a VMware disk level restore for the chosen disk and
                                   attaches it to the specified VM

    aws_attach_disk_restore()       -- Attach volumes to instances restore
"""
from selenium.webdriver.support.ui import Select
from Web.Common.handlers import adminconsole_handler
from Web.AdminConsole.VSAPages.restore_select_vm import RestoreSelectVM


class DiskLevelRestore:
    """
    This class provides method to do a disk level restore
    """

    def __init__(self, admin_console):
        """
        Init method to create objects of classes used in the file.
        """
        self.__admin_console = admin_console
        self.__admin_console._load_properties(self)
        self.__driver = admin_console.driver
        self.res_select_vm_obj = RestoreSelectVM(admin_console)

    @adminconsole_handler()
    def disk_level_restore(
            self,
            virtual_machine,
            disks,
            destination_proxy,
            destination_server,
            esx,
            vm_name,
            datastore,
            disk_prov='Original',
            transport_mode='Auto',
            respool=None,
            overwrite=True,
    ):
        """
        Disk level restore job will be submitted with the given values

        Args:
            virtual_machine         (basestring)   :   the virtual machine whose disks has
                                                        to be restored

            disks                   (list)         :   the list of disks to be restored

            destination_proxy       (basestring)   :   the proxy to be used for restore

            destination_server      (basestring)   :   the hypervisor where the destination
                                                        VM resides

            esx                     (basestring)   :   the esx in which the attach to VM resides

            vm_name                 (basestring)   :   the VM to which the disks are to be added

            datastore               (basestring)   :   the datastore where the disk is to
                                                        be restored

            disk_prov               (basestring)   :   the type of disk provisioning for the
                                                        restored disk

            transport_mode          (basestring)   :   the transport mode to be used for restore

            respool                 (list)         :   the resource pool in which the attach
                                                        VM resides

            overwrite               (bool)         :   if the restored disk has to be over written

        Raises:
            Exception:
                if there is any error while submitting disk level restore

        Returns:
            job_id  (int)   :   the restore job ID

        """
        self.__admin_console.log.info("Attaching disk to virtual_machine %s in server %s",
                                      str(vm_name), str(destination_server))

        if not self.__admin_console.check_if_entity_exists(
                "xpath", "//div[1]/ul/li[2]/span[contains(text(),'" + virtual_machine + "')]"):
            self.res_select_vm_obj.select_vm(virtual_machine)
        self.__admin_console.select_for_restore(disks)

        self.__driver.find_element_by_xpath(
            "//*[@id='browseActions']//*[contains(.,'Restore')]").click()
        self.__admin_console.wait_for_completion()

        self.__admin_console.select_value_from_dropdown("serverId", destination_server)
        self.__admin_console.select_value_from_dropdown("destinationProxy", destination_proxy)

        if overwrite:
            self.__admin_console.checkbox_select("overwrite")

        self.__admin_console.select_value_from_dropdown("diskProvOption", disk_prov)
        self.__admin_console.select_value_from_dropdown("transportMode", transport_mode)
        # disks = [disks[0]]
        # selecting the VM to attach the disk to
        self.__driver.find_element_by_xpath(
            "//*[@id='per-disk-options']/div/span/button[contains(text(),'Browse')]").click()
        self.__admin_console.wait_for_completion()

        self.__admin_console.select_destination_host(esx)

        if respool:
            for resource_pool in respool:
                if self.__admin_console.check_if_entity_exists("xpath", "//span[contains(text(),'" +
                                                                        resource_pool + "')]"):
                    self.__driver.find_element_by_xpath(
                        "//span[contains(text(),'" + resource_pool + "')]").click()
                    self.__admin_console.wait_for_completion()
            self.__admin_console.log.info("selected a resource pool %s", str(respool))

        if self.__admin_console.check_if_entity_exists("xpath", "//span[contains(text(),'" + vm_name + "')]"):
            self.__driver.find_element_by_xpath(
                "//span[contains(text(),'" + vm_name + "')]").click()
            self.__admin_console.log.info("selected vm %s", str(vm_name))
        else:
            raise Exception("There is no VM with the given name to be used to attach disk")

        self.__admin_console.submit_form()

        Select(self.__driver.find_element_by_id(
            "data-store")).select_by_visible_text(datastore)
        self.__admin_console.wait_for_completion()
        self.__admin_console.log.info("selected a datastore %s", str(datastore))

        self.__admin_console.submit_form(False)

        return self.__admin_console.get_jobid_from_popup()

    @adminconsole_handler()
    def aws_attach_disk_restore(self, virtual_machine, disks, attach_vol_to,
                                vir_client, proxy, existing_instance=None, vol_prefix=None,
                                power_on=True, overwrite=False, inst_display_name=None, ami=None,
                                available_zone=None, auto_select_instance_type=True, ins_type=None,
                                network=None, auto_select_security_grp=True, security_group=None):
        """
        Attach volumes to instances restore

        Args:
            virtual_machine     (basestring)   :   name of the VM to be restored

            disks               (list)         :   list of disks to be restore

            attach_vol_to       (basestring)   : New Instance or existing instance

            vir_client          (basestring)   : virtualization client for restore

            proxy               (basestring)   : proxy for restore

            existing_instance   (basestring)   : Existing instance in case of restore to existing
                                                instance

            vol_prefix          (basestring)   : prefix to restore volumes

            power_on            (bool)         : restored vm should be powered on of off

            overwrite           (bool)         : restored vm should be overwritten or not

            inst_display_name   (basestring)   : Display name for restored instance

            ami                 (basestring)   : Ami to be selected for attaching disk

            available_zone      (basestring)   : Availability zone for restoring

            auto_select_instance_type(bool)    : Instance type should be auto selected or not

            ins_type            (basestring)   : If Instance type isnt auto selected,then the instance typ

            network             (basestring)   : Network for restoring

            auto_select_security_grp (bool)    : if auto select securtiy group or not

            security_group      (basestring)   : security group to be selected

        Raises:
            Exception:
                if there is any error while submitting attach disk restore

        Returns:
            job_id  (int)   :   the restore job ID

        """

        self.__admin_console.log.info("Attach volume restore")
        if not self.__admin_console.check_if_entity_exists(
                "xpath", "//div[1]/ul/li[2]/span[contains(text(),'" + virtual_machine + "')]"):
            self.res_select_vm_obj.select_vm(virtual_machine)
        self.__admin_console.select_for_restore(disks)
        self.__driver.find_element_by_xpath(
            "//*[@id='browseActions']//*[contains(.,'Restore')]").click()
        self.__admin_console.wait_for_completion()

        if attach_vol_to == 'Other instance':
            self.__admin_console.cv_single_select("Attach volume to", "Other instance")
            self.__admin_console.click_button('Browse')
            self.__admin_console.select_destination_host(existing_instance)
            self.__admin_console.submit_form()
            self.__admin_console.cv_single_select("Virtualization client", vir_client)
        elif attach_vol_to == 'New instance':
            self.__admin_console.cv_single_select("Attach volume to", "New instance")

            self.__admin_console.cv_single_select("Destination hypervisor", vir_client)

        self.__admin_console.cv_single_select("Proxy client", proxy)

        if attach_vol_to == 'New instance':
            if vol_prefix is not None:
                if self.__admin_console.check_if_entity_exists("xpath", "//uib-accordion[@class='text-left']"):
                    if self.__driver.find_element_by_xpath(
                            "//uib-accordion[@class='text-left']/div/div[1]").get_attribute(
                            'class') == 'accordion panel ng-isolate-scope panel-default':
                        self.__driver.find_element_by_xpath(
                            "//uib-accordion[@class='text-left']/div/div[1]/div[1]/h4/a").click()

        elements = self.__driver.find_elements_by_xpath(
            "//ul[@class='vm-disk-restore-list width-full']/li")
        for disk in disks:
            for elem in elements:
                if elem.find_element_by_xpath("./a").text == disk:
                    elem.find_element_by_xpath("./a").click()
                    new_vol_name = vol_prefix + disk
                    self.__admin_console.fill_form_by_id("displayName", new_vol_name)

        if attach_vol_to == 'New instance':

            if power_on:
                self.__admin_console.checkbox_select("powerOn")
            else:
                self.__admin_console.checkbox_deselect("powerOn")

            if overwrite:
                self.__admin_console.checkbox_select("overwrite")
            else:
                self.__admin_console.checkbox_deselect("overwrite")

            if inst_display_name is not None:
                self.__admin_console.fill_form_by_id("instanceDisplayName", inst_display_name)

            if self.__driver.find_element_by_xpath(
                    "//uib-accordion[@class='text-left']/div/div[2]").get_attribute(
                    'class') == 'accordion panel ng-isolate-scope panel-default':
                self.__driver.find_element_by_xpath(
                    "//uib-accordion[@class='text-left']/div/div[2]/div[1]/h4/a").click()
            self.__admin_console.wait_for_completion()
            if available_zone:
                self.__driver.find_element_by_xpath(
                    "//div/input[@id='availabilityZone']/../span/button["
                    "contains(.,'Browse')]").click()
                self.__admin_console.wait_for_completion()
                self.__admin_console.select_destination_host(available_zone)
                self.__admin_console.submit_form()

            self.__admin_console.cv_single_select("AMI selection", ami)
            if not auto_select_instance_type:
                self.__admin_console.cvselect_from_dropdown("Instance type", ins_type)

            if network is not None:
                self.__driver.find_element_by_xpath(
                    "//div/input[@id='networkSettings']/../span/button[contains("
                    ".,'Browse')]").click()
                self.__admin_console.select_destination_host(network)
                self.__admin_console.submit_form()

                if not auto_select_security_grp:
                    self.__admin_console.checkbox_deselect("securitygroup")
                    self.__admin_console.cvselect_from_dropdown("Security groups", security_group)
        self.__admin_console.submit_form(wait=False)
        return self.__admin_console.get_jobid_from_popup()

    @adminconsole_handler()
    def azure_attach_disk(self, virtual_machine, destination_server, destination_proxy, disks,
                          disk_name, destination_vm, resource_group, storage_account):

        """
        Performs a attach disk  restore for Azure RM
        Args:
            virtual_machine    (str):     machine name for the source disk
            destination_server  (str):     VClient name
            proxy               (str):      the name of the destination proxy to be used
                                            for restore
            disks              (list):     disks to be attached
            disk_name           (str):     name to be given to the disk after attach
            destination_vm      (str):      Disks to be attached to the destination VM

            resource_group      (str) :     resource_group for disk to be attached

            storage_account     (str):     storage_account to be used for attaching disk
            destination_proxy   (str)

        Returns:
            job_id  (basestring):   the restore job ID

        """
        if self.__admin_console.check_if_entity_exists(
                "xpath", "//div[@class = 'ui-grid-canvas']/div/div/div/a[contains(text(),'"
                         + virtual_machine + "')]"):
            self.__admin_console.select_hyperlink(virtual_machine)

        selected = []
        while True:
            elements = self.__driver.find_elements_by_xpath(
                "//div[1]/div[2]/div[2]/div[@class='ui-grid-canvas']/div")
            index = 1
            flag = []
            for elem in elements:
                f_elem = elem.find_element_by_xpath("./div/div[1]").text
                for file in disks:
                    if file in f_elem:
                        flag.append(index)
                        selected.append(f_elem)
                    else:
                        continue
                index = index + 1
            for flg in flag:
                flg = str(flg)
                self.__driver.find_element_by_xpath(
                    "//div[1]/div/div[2]/div[@class='ui-grid-canvas']/div[" +
                    flg +
                    "]/div/div/div/div/div").click()

            break

        self.__admin_console.log.info("Attaching disk to virtual_machine %s in server %s",
                                      str(disk_name), str(destination_server))
        self.__admin_console.log.info("Selecting restore")
        self.__admin_console.driver.find_element_by_xpath("//*[@id='browseActions']/a").click()

        self.__admin_console.select_value_from_dropdown("serverId", destination_server, False, True)
        self.__admin_console.log.info("Entered destination server name")
        self.__admin_console.wait_for_completion()

        self.__admin_console.select_value_from_dropdown("destinationProxy", destination_proxy, False, True)
        self.__admin_console.log.info("Entered destination server name")
        self.__admin_console.wait_for_completion()

        for disk in disks:

            self.__driver.find_element_by_xpath("//ul[@class='vm-disk-restore-list']//a["
                                                "contains(text(),'" + disk + "')]").click()
            self.__admin_console.wait_for_completion()

            self.__driver.find_element_by_xpath("//input[@name='displayName']").clear()
            self.__driver.find_element_by_xpath("//input[@name='displayName']").send_keys \
                (disk_name + disk)

            self.__driver.find_element_by_xpath("//button[contains(text(),'Browse')]").click()
            self.__admin_console.log.info("Browse to select destination VM")
            self.__admin_console.wait_for_completion()

            elements = self.__driver.find_elements_by_xpath("//div[@class='browse-tree']/div")
            for elem in elements:
                elem_text = elem.find_element_by_xpath("./div/span").text
                elem_text = elem_text.strip()
                if elem_text.lower() == resource_group.lower():
                    elem.find_element_by_xpath("./div/button").click()
                    self.__admin_console.wait_for_completion()
                    break

            child_elements = self.__driver.find_elements_by_xpath("//div[@class='children "
                                                                  "ng-scope expanded']/div")
            xpath = "//div[@class='browse-tree']/div//div[@class='children ng-scope expanded']/div"
            for c_elem in child_elements:
                child_elem_text = c_elem.find_element_by_xpath("./div[1]/span").text
                child_elem_text = child_elem_text.strip()
                if child_elem_text.lower() == destination_vm.lower():
                    self.__admin_console.scroll_into_view(xpath)
                    c_elem.find_element_by_xpath("./div[1]/span").click()
                    self.__admin_console.wait_for_completion()
                    break

            self.__admin_console.log.info("Selected Destination VMto attach disk")

            self.__driver.find_element_by_xpath("//button[@type='submit']"
                                                "[contains(text(),'OK')]").click()
            self.__admin_console.wait_for_completion()

            select_id = "data-store"
            self.__admin_console.select_value_from_dropdown(select_id, storage_account, False, True)
            self.__admin_console.log.info("Entered storage account name")

        self.__admin_console.submit_form(wait=False)
        return self.__admin_console.get_jobid_from_popup()
