# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""
This module provides the function or operations that can be performed on the
configure live sync replication group page of the AdminConsole

"""

from Web.Common.page_object import (
    WebAction,
    PageService
)


class ConfigureVSAReplicationGroup:
    """Class for Configure VSA Replication Group Page"""

    def __init__(self, admin_console):
        """ """
        self.__admin_console = admin_console
        self.__admin_console._load_properties(self)
        self.__driver = admin_console.driver

    @WebAction()
    def __get_vms(self):
        """Gets all VMs of VMGroup
            Returns: All VM elements
        """
        return self.__driver.find_elements_by_xpath(
            "//*[@id='selectContentForm']/div[4]/div/div/isteven-multi-select/"
            "span/div/div[2]/div[1]/div/label/span")

    @WebAction()
    def __set_suffix(self):
        """Enters restore suffix"""
        self.__driver.find_element_by_xpath('//*[@id="restoreOptionsOCIForm"]/div'
                                            '/div/div/div/cv-display-name-azure'
                                            '/div[1]/div[2]'
                                            '/input').send_keys("_DeleteMeAC")

    @WebAction()
    def __get_subnet(self):
        """Finds available subnets
        Returns: Available subnets
        """
        return self.__driver.find_elements_by_xpath("//*[@id='subnet']/span/div")[0].text

    @WebAction()
    def __expand_advanced_options(self):
        """Expands Advanced options"""
        self.__driver.find_element_by_xpath(
            '//*[@id="restoreOptionsOCIForm"]/div/div/div/cv-oci-compartments-settings/div[6]'
            '/uib-accordion/div/div/div[1]/span[1]').click()

    @WebAction()
    def __override_options(self):
        """Get override options for the selected vm"""
        self.__driver.find_element_by_xpath('//*[@id="vmSelections"]/span/button').click()

    @WebAction()
    def __select_subnet(self, subnet_name):
        """Select subnet with given subnet ID"""
        _xp = '//*[@id="cloudNetwork"]/span/div/div[2]/div[1]/div/label/span'
        ind = 2
        while self.__admin_console.check_if_entity_exists("xpath", _xp):
            vcn_name = self.__driver.find_element_by_xpath(_xp).get_attribute("innerHTML")
            self.__admin_console.cv_single_select("Virtual cloud network", vcn_name)
            self.__admin_console.wait_for_completion()
            subnet_path = f'//*[@id="subnet"]/span/div/div[2]/div[@title="{subnet_name}"]'
            if self.__admin_console.check_if_entity_exists("xpath", subnet_path):
                self.__admin_console.cv_single_select("Subnet", subnet_name)
                self.__admin_console.wait_for_completion()
                self.__driver.find_element_by_xpath('//*[@id="subnet"]').click()
                break
            self.__admin_console.cv_single_select("Virtual cloud network", "")
            self.__admin_console.wait_for_completion()
            _xp = f'//*[@id="cloudNetwork"]/span/div/div[2]/div[{ind}]/div/label/span'
            ind += 1

    @PageService()
    def configure_vsa_replication_group(self, dict_vals):
        """Configures Live Sync VSA replication group

            Args:
                dict_vals: Dictionary object containing hypervisor, replication_group_name,
                vm_backupgroup, replication_target, proxy, compartment, datastore, vcn and shape

            Sample dict_vals:
            dict_vals = {
            'hypervisor': '',
            'replication_group_name': '',
            'vm_backupgroup': '',
            'replication_target': '',
            'proxy': '',
            'compartment': '',
            'datastore': '',
            'vcn': '',
            'shape': '',
            'vm_name': '',
            'subnet_name": ''
            }

        """
        self.__admin_console.cv_single_select("Hypervisors", dict_vals['hypervisor'])
        self.__admin_console.fill_form_by_id("name", dict_vals['replication_group_name'])
        self.__admin_console.cv_single_select("Select VM group", dict_vals['vm_backupgroup'])
        self.__admin_console.wait_for_completion()
        vms = self.__get_vms()
        for _vm in vms:
            self.__admin_console.cv_single_select("Select VMs", _vm.text)
        self.__admin_console.button_next()
        self.__admin_console.select_hyperlink("Create new")
        self.__admin_console.cv_single_select("Select vendor", 'Oracle Cloud Infrastructure')
        self.__admin_console.fill_form_by_id("replicationTargetName",
                                             dict_vals['replication_target'])
        self.__admin_console.cv_single_select("Destination hypervisor", dict_vals['hypervisor'])
        self.__admin_console.cv_single_select("Access node", dict_vals['proxy'])
        self.__set_suffix()
        self.__admin_console.cv_single_select("Compartment", dict_vals['compartment'])
        self.__admin_console.cv_single_select("Availability domain", dict_vals['datastore'])
        # self.__admin_console.cv_single_select("Virtual cloud network", dict_vals['vcn'])
        self.__select_subnet(dict_vals['subnet_name'])
        self.__admin_console.cv_single_select("Shape", dict_vals['shape'])
        self.__admin_console.cv_single_select("Access node", dict_vals['proxy'])
        self.__admin_console.submit_form()
        self.__admin_console.button_next()
        self.__admin_console.click_button('Yes')
        self.__admin_console.cv_single_select("Virtual machine", dict_vals['vm_name'])
        self.__override_options()
        self.__admin_console.wait_for_completion()
        self.__admin_console.button_next()
        self.__admin_console.click_button("Finish")
