# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations that can be performed on the
DiskStorageDetails page of the AdminConsole

DiskStorageDetails:

    select_backup_location_type()   -- Select the backup location type

    select_media_agent()            -- Select the given media agent

    __click_edit_symbol()           -- Click edits symbol for key management server

    __click_ok_symbol()             -- Click ok symbol for key management server

    edit_key_management_server()    -- Edits the existing key management server

    add_backup_location()           -- To add a new backup location to an existing disk storage

    disk_storage_info()             -- To get the details of disk storage

    delete_backup_location()        -- Deletes the backup location on disk storage

    add_media_agent()               -- Add media agent to backup location on disk storage

    list_backup_locations()         -- Get all the backup location disk storage in the form of a list

    encrypt_storage()               -- To encrypt the storage on the selected disk

    list_associated_plans()         -- Get all the associated plans to the disk storage in the form of a list

"""

from Web.AdminConsole.Components.panel import (PanelInfo, DropDown)
from Web.AdminConsole.Components.table import Table
from Web.Common.page_object import (WebAction, PageService)


class DiskStorageDetails:
    """
    Class for Disk Storage Details page
    """

    def __init__(self, admin_console):
        """
        Initialization method for DiskStorageDetails Class

            Args:
                admin_console (AdminConsole): AdminConsole object
        """
        self.__admin_console = admin_console
        self.__admin_console._load_properties(self)
        self.__driver = self.__admin_console.driver
        self.__props = self.__admin_console.props
        self.__table = Table(self.__admin_console)
        self.__drop_down = DropDown(self.__admin_console)

    @WebAction()
    def select_backup_location_type(self, backup_location_type):
        """
        Select the backup location type

            Args: backup_location_type (str)
                  values: 'networkStorage' or 'localStorage'
        """
        self.__driver.find_element_by_xpath(f"//input[@value='{backup_location_type}']").click()

    @WebAction()
    def select_media_agent(self, media_agent):
        """
        Select the given media agent

            Args:
                media_agent (str)	--	Name of the media_agent to be selected
        """
        self.__driver.find_element_by_xpath(f"//tr[contains(@id,'{media_agent}')]//td").click()

    @WebAction()
    def __click_edit_symbol(self):
        """
            Click edits symbol for key management server
        """
        self.__driver.find_element_by_xpath(
            f"//span[contains(text(),{self.__props['label.keyManagement']})]//parent::li//span[2]//a//i").click()

    @WebAction()
    def __click_ok_symbol(self):
        """
            Click ok symbol for key management server
        """
        self.__driver.find_element_by_xpath(
            f"//span[contains(text(),{self.__props['label.keyManagement']})]//parent::li//a[1]").click()

    @PageService()
    def edit_key_management_server(self, key_management_server):
        """
        Edits the existing key management server

            Args:
                key_management_server (str)	-- New key management server
        """
        self.__click_edit_symbol()
        self.__admin_console.wait_for_completion()
        self.__drop_down.select_drop_down_values(0, [key_management_server])
        self.__click_ok_symbol()
        self.__admin_console.wait_for_completion()
        self.__admin_console.check_error_message()

    @PageService()
    def add_backup_location(self, media_agent, backup_location, saved_credential_name=None,
                            username=None, password=None):
        """
        To add a new backup location to an existing disk storage

        Args:
            media_agent     (str)       -- Media agent to create storage on

            saved_credential_name (str) -- saved credential name created using credential manager

            username        (str)       -- username for the network path

            password        (str)       -- password for the network path

            backup_location (str)       -- local/network path for the backup

        **Note** MediaAgent should be installed prior, for creating a backup location for storage.
                To use saved credentials for network path it should be created prior using credential manager,
        """

        self.__admin_console.select_hyperlink(self.__props['action.add'])
        self.__drop_down.select_drop_down_values(drop_down_id='mediaAgent', values=[media_agent])
        if saved_credential_name:
            self.select_backup_location_type("networkStorage")
            self.__admin_console.wait_for_completion()
            self.__admin_console.toggle_enable(self.__props['label.useSavedCred'])
            self.__drop_down.select_drop_down_values(drop_down_id='credential',
                                               values=[saved_credential_name])
        elif username and password:
            self.__admin_console.toggle_disable(self.__props['label.useSavedCred'])
            self.__admin_console.fill_form_by_id("userName", username)
            self.__admin_console.fill_form_by_id("password", password)

        self.__admin_console.fill_form_by_id("mountPath", backup_location)

        self.__admin_console.click_button(self.__props['action.save'])
        self.__admin_console.check_error_message()

    @PageService()
    def disk_storage_info(self):
        """
        To get the details of disk storage

            Returns:
                info    (dict)  -- details of disk storage
        """
        panel_info = PanelInfo(self.__admin_console, self.__props['label.general'])
        return panel_info.get_details()

    @PageService()
    def delete_backup_location(self, backup_location):
        """
        Deletes the backup location on disk storage

            Args:
                backup_location (str)   --  name of the backup location to delete
        """
        self.__table.access_action_item(backup_location, self.__props['action.delete'])
        self.__admin_console.click_button(self.__props['label.yes'])
        self.__admin_console.check_error_message()

    @PageService()
    def add_media_agent(self, backup_location, media_agent_list):
        """
        Add media agent to backup location on disk storage

            Args:
                backup_location   (str)   --  backup location on which given media agent will be added
                media_agent_list  (list)  --  list of media agents to be added
        """
        self.__table.access_action_item(backup_location, self.__props['title.addMediaAgent'])
        for media_agent in media_agent_list:
            self.select_media_agent(media_agent)
        self.__admin_console.click_button(self.__props['action.save'])
        self.__admin_console.check_error_message()

    @PageService()
    def list_backup_locations(self):
        """
        Get all the backup location disk storage in the form of a list

            Returns:
                    backup_location_list    (list)  --  all backup locations on disk
        """

        return self.__table.get_column_data(self.__props['Name'])

    @PageService()
    def encrypt_storage(self, cipher=None, key_length=None, key_management_server=None):
        """
        To encrypt the storage on the selected disk

            Args:
                cipher      (str)   -- Encryption method to be used

                key_length  (str)   -- Key length for the chosen cipher

                key_management_server   (str)   --  Key management server for the storage pool
        """
        self.__admin_console.access_tab(self.__props['label.scaleOutConfiguration'])
        self.__admin_console.wait_for_completion()

        if cipher and key_length:
            panel_info = PanelInfo(self.__admin_console, self.__props['title.encryption'])
            panel_info.enable_toggle(self.__props['label.encrypt'])
            self.__drop_down.select_drop_down_values(0, [cipher])
            self.__drop_down.select_drop_down_values(1, [key_length])
            self.__admin_console.click_button(self.__props['action.save'])
            self.__admin_console.check_error_message()
        if key_management_server:
            self.edit_key_management_server(key_management_server)

    @PageService()
    def list_associated_plans(self):
        """
        Get all the associated plans to the disk storage in the form of a list

            Returns:
                    plans_list    (list)  --  all associated plans to the disk
        """
        self.__admin_console.access_tab(self.__props['label.associatedPlans'])
        self.__admin_console.wait_for_completion()
        table = Table(self.__admin_console)
        try:
            return table.get_column_data(self.__props['Name'])
        except ValueError:
            return []
