# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations related to disk storage page in AdminConsole
DiskStorage : This class provides methods for disk storage related operations

DiskStorage:

    select_backup_location_type()   --  select the backup location type

    add_disk_storage()      --  adds a new disk storage

    list_disk_storage()     --  returns a list of all disk storage

    access_disk_storage()   --  opens a disk storage

    delete_disk_storage()   --  removes a disk storage

"""

from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.Components.panel import DropDown
from Web.Common.page_object import (WebAction, PageService)


class DiskStorage:
    """
    This class provides the function or operations that can be
    performed on the Disk Storage Page of the Admin Console
    """

    def __init__(self, admin_console):
        """
        Initialization method for DiskStorage Class

            Args:
                admin_console (AdminConsole): AdminConsole object
        """
        self.__admin_console = admin_console
        self.__admin_console._load_properties(self)
        self.__driver = self.__admin_console.driver
        self.__props = self.__admin_console.props
        self.__table = Table(self.__admin_console)
        self.__dropdown = DropDown(self.__admin_console)

    @WebAction()
    def select_backup_location_type(self, backup_location_type):
        """
        Select the backup location type

            Args: backup_location_type (str)
                  values: 'networkStorage' or 'localStorage'
        """
        self.__driver.find_element_by_id(backup_location_type)

    @PageService()
    def add_disk_storage(self, disk_storage_name, media_agent, backup_location,
                         saved_credential_name=None, username=None, password=None,
                         deduplication_db_location=None):
        """
        To add a new disk storage

        Args:
            disk_storage_name (str)     -- Name of the disk storage to be created

            media_agent     (str)       -- Media agent to create storage on

            saved_credential_name (str) -- saved credential name created using credential manager

            username        (str)       -- username for the network path

            password        (str)       -- password for the network path

            backup_location (str)       -- local/network path for the backup

            deduplication_db_location (str) -- local path for the deduplication db

        **Note** MediaAgent should be installed prior, for creating a new backup location for storage.
                To use saved credentials for network path it should be created prior using credential manager,
        """

        self.__admin_console.select_hyperlink(self.__props['action.add'])
        self.__admin_console.fill_form_by_id("storagePoolName", disk_storage_name)
        self.__admin_console.click_by_id('gridEmptyMenuForDiskorCloud')
        self.__dropdown.select_drop_down_values(drop_down_id='mediaAgent', values=[media_agent])
        if saved_credential_name:
            self.select_backup_location_type("networkStorage")
            self.__admin_console.enable_toggle(index=0)
            self.__dropdown.select_drop_down_values(drop_down_id='credential',
                                                    values=[saved_credential_name])
        elif username and password:
            self.select_backup_location_type("networkStorage")
            self.__admin_console.disable_toggle(index=0)
            self.__admin_console.fill_form_by_id("userName", username)
            self.__admin_console.fill_form_by_id("password", password)

        self.__admin_console.fill_form_by_id("mountPath", backup_location)

        if deduplication_db_location:
            self.__admin_console.enable_toggle(
                index=int(bool(saved_credential_name or (username and password))))
            self.__admin_console.fill_form_by_id("partitionPath", deduplication_db_location)
        else:
            self.__admin_console.disable_toggle(
                index=int(bool(saved_credential_name or (username and password))))

        self.__admin_console.click_button(self.__props['action.save'])
        self.__admin_console.click_button(self.__props['action.save'])
        self.__admin_console.check_error_message()

    @PageService()
    def list_disk_storage(self):
        """
        Get the of all the disk storage in the form of a list

            Returns:
               list --  all disk storage
        """
        try:
            return self.__table.get_column_data(self.__props['Name'])
        except ValueError:
            return []

    @PageService()
    def access_disk_storage(self, disk_storage):
        """
        selects the disk storage with the given name

        Args:
            disk_storage    (str)   -- Name of the disk storage to be opened
        """
        self.__table.access_link(disk_storage)

    @PageService()
    def delete_disk_storage(self, disk_storage):
        """
        Deletes the disk storage with the given name

        Args:
            disk_storage (str) - name of the storage to be removed
        """
        self.__table.access_action_item(disk_storage, self.__props['action.delete'])
        self.__admin_console.click_button(self.__props['label.yes'])
        self.__admin_console.check_error_message()
