# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations related to cloud storage page in AdminConsole
CloudStorage : This class provides methods for cloud storage related operations

CloudStorage:

    add_cloud_storage()      --  adds a new cloud storage

    list_cloud_storage()     --  returns a dict of all cloud storage

    select_storage_pool()   --  opens a cloud storage

    action_delete()         --  removes a cloud storage

"""
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.Components.panel import DropDown
from Web.Common.page_object import (WebAction, PageService)


class CloudStorage:
    """
    This class provides the function or operations that can be
    performed on the Cloud Storage Page of the Admin Console
    """

    def __init__(self, admin_console):
        """
        Initialization method for CloudStorage Class

            Args:
                admin_console (AdminConsole): AdminConsole object
        """
        self.__admin_console = admin_console
        self.__admin_console._load_properties(self)
        self.__driver = self.__admin_console.driver
        self.__props = self.__admin_console.props
        self.__table = Table(self.__admin_console)
        self.__dropdown = DropDown(self.__admin_console)

    @WebAction()
    def fill_form_by_name(self, element_name, value):
        """
        Fill the value in a text field with id element id.

        Args:
            element_name (basestring) -- the ID attribute of the element to be filled
            value (basestring)      -- the value to be filled in the element

        Raises:
            Exception:
                If there is no element with given name/id

        """
        element = self.__driver.find_element_by_name(element_name)
        element.clear()
        element.send_keys(value)
        self.__admin_console.wait_for_completion()

    @PageService()
    def add_cloud_storage(self, cloud_storage_name, media_agent, cloud_type, server_host,
                          container, storage_class=None, saved_credential_name=None, username=None,
                          password=None, deduplication_db_location=None):
        """
        To add a new cloud storage

        Args:
            cloud_storage_name (str)    -- Name of the cloud storage to be created

            media_agent     (str)       -- Media agent to create storage on

            cloud_type      (str)       -- type of the cloud storage server

            server_host     (str)       -- cloud server host name

            container       (str)       -- container to be associated with the storage

            storage_class   (str)       --  storage class to be associated with the container

            saved_credential_name (str) -- saved credential name created using credential manager

            username        (str)       -- username for the network path

            password        (str)       -- password for the network path

            deduplication_db_location (str) -- local path for the deduplication db

        **Note** MediaAgent should be installed prior, for creating a new storage,
                To use saved credentials it should be created prior using credential manager.
        """
        self.__admin_console.select_hyperlink(self.__props['action.add'])
        self.__admin_console.click_by_xpath("//*[@data-ng-click='launchOther()']")
        self.fill_form_by_name("libraryName", cloud_storage_name)

        self.__dropdown.select_drop_down_values(drop_down_id='mediaAgent', values=[media_agent])
        self.__admin_console.select_value_from_dropdown("cloudType", cloud_type)
        self.__admin_console.fill_form_by_id("serviceHost", server_host)
        self.__admin_console.fill_form_by_id("bucket", container)
        if storage_class:
            self.__admin_console.select_value_from_dropdown("storageClass", storage_class)

        try:
            self.__dropdown.select_drop_down_values(drop_down_id='credential',
                                                    values=[saved_credential_name])
        except:
            self.__admin_console.click_by_id('selectWithCreateButton_button_#2034_credential')
            self.__admin_console.fill_form_by_id("userName", username)
            self.__admin_console.fill_form_by_id("password", password)
            self.__admin_console.fill_form_by_id("credentialName", saved_credential_name)
            self.__admin_console.click_button(self.__props['action.save'])
        finally:
            self.__dropdown.select_drop_down_values(drop_down_id='credential',
                                                    values=[saved_credential_name])

        if deduplication_db_location:
            self.__admin_console.enable_toggle(index=0)
            self.__admin_console.select_hyperlink(self.__props['action.add'])
            self.__admin_console.fill_form_by_id('ddbPartitionPath', deduplication_db_location)
            self.__dropdown.select_drop_down_values(index=0, values=[media_agent])
            self.__admin_console.click_button(self.__props['action.save'])
        else:
            self.__admin_console.disable_toggle(index=0)
        self.__admin_console.click_button(self.__props['action.save'])
        self.__admin_console.check_error_message()

    @PageService()
    def list_cloud_storage(self):
        """
        Get all the cloud storage in the form of a list

            Returns:
               list --  all cloud storage
        """
        try:
            return self.__table.get_column_data(self.__props['Name'])
        except ValueError:
            return []

    @PageService()
    def select_cloud_storage(self, cloud_storage):
        """
        selects the cloud storage with the given name

        Args:
            cloud_storage    (str)   -- Name of the cloud storage to be opened
        """
        self.__table.access_link(cloud_storage)

    @PageService()
    def action_delete(self, cloud_storage):
        """
        Deletes the cloud storage with the given name

        Args:
            cloud_storage (str) - name of the storage to be removed
        """
        self.__table.access_action_item(cloud_storage, self.__props['action.delete'])
        self.__admin_console.click_button(self.__props['label.yes'])
        self.__admin_console.check_error_message()
