# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations related to getting started page

==============

    get_solutions_listed()  -- Returns the solutions listed

    get_setup_completion_state() -- fetches the setup completion state

    click_complete_setup_for_solution()--Clicks the complete setup in getting started page

    finish_setup_later()--Clicks the finish this later option in getting started or solution page

    click_get_started()--clicks the link, get started or lets finish it

    access_tab()                -- access tabs under guided setup

    navigate_to_services()      -- navigate to services tab under guided setup

    navigate_to_metallic()      -- navigate to metallic tile under services

    link_metallic_account()     -- register to metallic services

    unlink_metallic_account()   -- unregister to metallic services

"""


from Web.AdminConsole.AdminConsolePages.AdminPage import AdminPage
from Web.AdminConsole.Setup.core_setup import Setup
from Web.Common.page_object import (
    WebAction,
    PageService
)


class GettingStarted(AdminPage):
    """
    Class for file servers getting started page in Admin console

    """

    def __init__(self, driver):
        """
        Setup Class initialization

        """
        super(GettingStarted, self).__init__(driver)
        self.setup_obj = Setup(driver)
        self.driver = driver

    @PageService()
    def get_solutions_listed(self):
        """
            To get the solutions getting listed in getting started page

        """
        solutions = []
        if self.check_if_entity_exists('xpath', '//td[@class = "app-name ng-binding"]'):
            tile_elements = self.driver.find_elements_by_xpath(
                '//td[@class = "app-name ng-binding"]')
            if tile_elements:
                for tile_elem in tile_elements:
                    solutions.append(tile_elem.text)
        else:
            self.log.info("No solutions are listed in getting started page")
        return solutions

    @PageService()
    def get_setup_completion_state(self, solution):
        """
            fetches the setup completion state
            Args:
                solution(string):name of the solution
            Returns:True,when setup is completed


        """
        if self.check_if_entity_exists('xpath', '//div[@class="app ng-scope"]'):
            div_elements = self.driver.find_elements_by_xpath('//div[@class="app ng-scope"]')
            for div_element in div_elements:
                listed_solution = div_element.find_element_by_xpath(
                    './/td[@class = "app-name ng-binding"]').text
                if listed_solution == solution:
                    self.log.info("Solution listed, checking the status of completion")
                    status = div_element.find_element_by_xpath(
                        './/div[@class="app-setup-link"]').text
                    if status == self.props['label.setupCompleted']:
                        return True
                    if status == self.props['label.completeSetup']:
                        return False
                    raise Exception(" setup completion status for the solution is unknown")
        else:
            raise Exception("No solution is listed")

    @WebAction()
    def click_complete_setup_for_solution(self, solution):
        """
            Clicks the complete setup in getting started page
            Args:
                solution(string):name of the solution
        """
        solution_found = False
        if self.check_if_entity_exists('xpath', '//div[@class="app ng-scope"]'):
            div_elements = self.driver.find_elements_by_xpath('//div[@class="app ng-scope"]')
            for div_element in div_elements:
                listed_solution = div_element.find_element_by_xpath(
                    './/td[@class = "app-name ng-binding"]').text
                if listed_solution == solution:
                    solution_found = True
                    self.log.info("Found the solution to complete setup for")
                    div_element.click()
                    self.wait_for_completion()
                    break
            if not solution_found:
                raise Exception('Solution could not be found on the getting started page')

    @WebAction()
    def finish_setup_later(self):
        """
        Clicks the finish this later option in getting started or solution page
        Returns: None

        """
        try:
            self.select_hyperlink(self.props['label.finishLater'])
        except Exception as exp:
            try:
                self.log.info("%s", str(exp))
                self.select_hyperlink(self.props['label.finishLaterCTA'])
            except Exception as exp:
                raise Exception(str(exp))

    @WebAction()
    def click_get_started(self):
        """
        clicks the link, get started or lets finish it
        Returns:None

        """
        try:
            if self.check_if_entity_exists('xpath',
                                           '//span[@data-ng-bind="coreSetupSubHeaderCompleted"]'):
                self.log.info("core setup completed")
                return False
            elif self.check_if_entity_exists("xpath",
                                             "//div[@data-ng-if='coreSetupCompleted']"):
                self.log.info("Core setup completed")
                return False
            self.log.info("click on get started if listed")
            self.select_hyperlink(self.props['label.letsGetStarted']+" ")
            return True
        except Exception as exp:
            try:
                self.log.info("%s", str(exp))
                self.select_hyperlink(self.props['label.finishSetupCTA']+" ")
                return True
            except Exception as exp:
                raise Exception(str(exp))

    @WebAction()
    def access_tab(self, tab_header):
        """Method to click on given tab under guided setups"""

        self.driver.find_element_by_xpath(
            f"//a[@class='nav-link ng-binding' and text() ='{tab_header}']").click()
        self.wait_for_completion()

    @PageService()
    def navigate_to_services(self):
        """Navigates to services page"""

        self.navigate_to_getting_started()
        self.access_tab('Services')

    @PageService()
    def navigate_to_metallic(self):
        """Navigates to metallic registration page"""

        self.navigate_to_services()
        self.access_tile('appsCloud_metallicSetup')

    @PageService()
    def link_metallic_account(self, username, password):
        """To register to metallic services

        Args:
             username: metallic company user
             password: metallic company password

        """

        self.fill_form_by_id("userName", username)
        self.fill_form_by_id("userNamePassword", password)
        self.submit_form('linkMetallicAccount')

    @PageService()
    def unlink_metallic_account(self):
        """Unregister to metallic services"""

        self.navigate_to_metallic()
        self.click_button('Unlink')
        self.click_button('Yes')
