# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations related to Commvault getting started page
Setup : This class provides methods for setup related operations

Setup
=====

    select_get_started()        -- To select the lets get started button

    configure_email()           -- To configure email

    add_storage_pool()          -- To add storage pool

    create_server_backup_plan() -- To create a sever backup plan

    core_setup_status()         -- To check if the core setup is completed or not

    **Note** using Admin console default values for filled web elements

"""

from Web.AdminConsole.Helper.AdminConsoleBase import AdminConsoleBase
from Web.Common.page_object import PageService


class Setup(AdminConsoleBase):
    """
    Class for commvault's Setup page

    """

    @PageService()
    def select_get_started(self):
        """
        To select the lets get started button

        """
        self.select_hyperlink('get started')

    @PageService(hide_args=True)
    def configure_email(
            self,
            smtp_server='smtp.commvault.com',
            smtp_port='25',
            sender_email=None,
            sender_name=None,
            authentication=False,
            username=None,
            password=None,
            encryption=None):
        """
        To configure email

        Args:
            smtp_server     (str)   -- SMTP server to send mails

            smtp_port       (str)   -- Port for the smtp server
                                        default: 25

            sender_email    (str)   -- Email address of the sender

            sender_name     (str)   -- name of the sender

            authentication  (bool)  -- set to True if authentication is enabled

            username        (str)   -- Username for authentication

            password        (str)   -- Password for the authentication

            encryption      (str)   -- Encrytion algorithm to be used

        """
        self.fill_form_by_id('smtpServer', smtp_server)
        self.fill_form_by_id('smtpPort', smtp_port)

        # Its automatically populated
        if sender_email:
            self.fill_form_by_id('senderEmail', sender_email)

        self.fill_form_by_id('senderName', sender_name)

        # To select authentication checkbox and to fill the username and password
        if authentication and username and password:
            self.checkbox_select('authentication')
            self.fill_form_by_id('username', username)
            self.fill_form_by_id('password', password)
            self.fill_form_by_id('confirmPassword', password)

        # To set encryption
        if encryption:
            self.select_value_from_dropdown('encryption', encryption)

        self.submit_form()

        # To check for errors
        self.check_error_message()

    @PageService(hide_args=True)
    def add_storage_pool(
            self,
            pool_name=None,
            media_agent=None,
            username=None,
            password=None,
            path=None,
            partition_path=None):
        """
        To add storage pool

        Args:
            pool_name       (str)   -- Name of the storage pool to be created

            media_agent     (str)   -- media agent to be selected, by default it is selected

            username        (str)   -- Username for the network path

            password        (str)   -- Password for the network path

            path            (str)   -- Path to be selected as storage

            partition_path  (str)   -- DDB partition path

        """
        self.fill_form_by_id('storagePoolName', pool_name)

        # By default media agent for the commcell machine is created and selected here
        if media_agent:
            self.select_value_from_dropdown('mediaAgent', media_agent)

        if username and password:
            self.select_radio('Network Path')
            self.fill_form_by_id('userName', username)
            self.fill_form_by_id('password', password)

        self.fill_form_by_id('mountPath', path)
        self.fill_form_by_id('partitionPath', partition_path)

        self.click_button('Save')

        # To check for error messages
        self.check_error_message()

    @PageService()
    def create_server_backup_plan(self):
        """
        To create a sever backup plan

        """
        self.submit_form()

        # To check for error messages
        self.check_error_message()

    @PageService()
    def core_setup_status(self):
        """
        Checks the status of the core setup completion.

        Returns:
            True / False based on the setup completion

        """
        if self.check_if_entity_exists('xpath', '//div[@data-ng-if="coreSetupCompleted"]'):
            return True
        else:
            return False
