# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""
This module provides the function or operations that can be performed on the
individual report page of the AdminConsole
"""
from selenium.common.exceptions import (
    WebDriverException,
    NoSuchElementException
)
from selenium.webdriver import ActionChains

from Web.AdminConsole.Reports.cte import (
    Email,
    ConfigureSchedules
)
from Web.Common.exceptions import CVWebAutomationException
from Web.Common.page_object import (
    WebAction,
    PageService
)


class Report:
    """Class for Report Page"""
    def __init__(self, admin_console):
        """
        Args:
            admin_console: adminconsole base object
        """
        self._admin_console = admin_console
        self.driver = admin_console.driver

    @WebAction()
    def __click_save_as_view(self):
        """clicks "Save as view"""
        save_as_view = self.driver.find_element_by_xpath("//a[contains(text(),'Save as view')]")
        save_as_view.click()

    @WebAction()
    def __click_refresh(self):
        """Clicks refresh"""
        refresh = self.driver.find_element_by_xpath("//a[contains(text(),'Refresh')]")
        refresh.click()

    @WebAction()
    def __click_email(self):
        """Clicks email"""
        email = self.driver.find_element_by_xpath("//a[contains(text(),'Email')]")
        email.click()

    @WebAction()
    def __click_schedule(self):
        """Clicks schedule"""
        schedule = self.driver.find_element_by_xpath("//a[contains(text(),'Schedule')]")
        schedule.click()

    @WebAction()
    def __click_more_actions(self):
        """Clicks 'more actions' """
        # using class instead of text to support different locale
        more_action = self.driver.find_element_by_xpath(
            "//div[contains(@class,'btn-group dropdown')]")
        more_action.click()

    @WebAction()
    def __click(self, element):
        """Clicks the given option"""
        obj = self.driver.find_element_by_xpath(f"//a[contains(text(),'{element}')]")
        obj.click()

    @WebAction()
    def __set_view_name(self, name):
        """Sets the given view name"""
        view = self.driver.find_element_by_xpath("//input[@id='viewName']")
        view.clear()
        view.send_keys(name)

    @WebAction()
    def __set_url(self, url):
        """Sets the given URL"""
        view = self.driver.find_element_by_xpath("//input[@id='urlName']")
        view.clear()
        view.send_keys(url)

    @WebAction()
    def __set_as_default(self):
        """Selects the 'Set as Default' Checkbox"""
        view = self.driver.find_element_by_xpath("//*[@for='chkSetAsDefault']")
        view.click()

    @WebAction()
    def __click_save(self):
        """Clicks Save button"""
        button = self.driver.find_element_by_xpath("//button[contains(text(),'Save')]")
        button.click()

    @WebAction()
    def __fetch_current_view(self):
        """Fetches current view"""
        view = self.driver.find_element_by_xpath("//a[@id='views-dropdown']")
        return view.text

    @WebAction()
    def __fetch_all_views(self):
        """Fetches all views"""
        objects = self.driver.find_elements_by_xpath("//li[contains(@class,'report-view')]")
        objects.pop()
        return [each.text.strip() for each in objects]

    @WebAction()
    def __click_view_drop_down(self):
        """Clicks the view dropdown"""
        view = self.driver.find_element_by_xpath("//a[@id='views-dropdown']")
        view.click()

    @WebAction()
    def __hover_over_the_view(self, view_name):
        """Hovers over the given view"""
        view = self.driver.find_element_by_xpath(f"//ul//span[contains(text(),'{view_name}')]")
        action_chain = ActionChains(self.driver)
        action = action_chain.move_to_element(view)
        action.perform()

    @WebAction()
    def __click_delete_icon(self, view_name):
        """Click delete icon on the given view name"""
        trash_icon = self.driver.find_element_by_xpath(
            f"//span[contains(text(),'{view_name}')]/ancestor::li//"
            f"*[contains(@class,'glyphicon-trash')]")
        trash_icon.click()

    @WebAction()
    def __click_delete(self):
        """clicks delete button"""
        button = self.driver.find_element_by_xpath("//button[contains(text(),'Delete')]")
        button.click()

    @WebAction()
    def __check_for_permission(self):
        """Checks for permission errors"""
        notification = self.driver.find_elements_by_xpath("//*[@id='notificationContainer']//p")
        if notification:
            raise CVWebAutomationException(f"{notification[0].text}")

    @WebAction()
    def __click_yes_on_delete_confirmation(self):
        """Click yes on delete confirmation"""
        button = self.driver.find_element_by_xpath(
            "//button[.='Yes' and @data-ng-disabled='!confirmBox']")
        button.click()

    @PageService()
    def save_as_pdf(self):
        """Save as PDF"""
        self.__click_more_actions()
        self._admin_console.select_hyperlink('Save as')
        self.__click("PDF")

    @PageService()
    def save_as_html(self):
        """Save as HTML"""
        self.__click_more_actions()
        self._admin_console.access_page_action_menu_by_class("SAVEAS")
        self.__click("HTML")

    @PageService()
    def save_as_csv(self):
        """Save as CSV"""
        self.__click_more_actions()
        self._admin_console.select_hyperlink('Save as')
        self.__click("CSV")

    @PageService()
    def access_security(self):
        """Access security of the report"""
        self.__click_more_actions()
        self.__click("Security")
        self._admin_console.wait_for_completion()

    @PageService()
    def edit_report(self):
        """Edit Report"""
        self.__click_more_actions()
        try:
            self.__click("Edit")
        except WebDriverException:
            self.__click_more_actions()
            raise CVWebAutomationException("User has no privileges")
        self.driver.switch_to_window(self.driver.window_handles[1])
        self.__check_for_permission()

    @PageService()
    def delete_report(self):
        """Delete Report"""
        self.__click_more_actions()
        try:
            self.__click("Delete")
        except WebDriverException:
            self.__click_more_actions()
            raise CVWebAutomationException("User has no privileges")
        self.__click_yes_on_delete_confirmation()
        notification = self._admin_console.get_notification()
        if "successfully" not in notification:
            self._admin_console.wait_for_completion()
            raise CVWebAutomationException(notification)
        self._admin_console.wait_for_completion()

    @PageService()
    def save_as_view(self, name, url=None, set_as_default=True):
        """Saves as view

        Args:
            name (str): Name of the view

            url (str):  URL string

                defaults: None.(keeps the url at the time of on clicking 'save as view' intact)

            set_as_default (bool):  Sets as default

                defaults: True

        """
        self.__click_save_as_view()
        self._admin_console.wait_for_completion()
        self.__set_view_name(name)
        if url:
            self.__set_url(url)
        if set_as_default:
            self.__set_as_default()
        self.__click_save()
        self._admin_console.wait_for_completion()

    @PageService()
    def schedule(self):
        """Schedules the report"""
        self.__click_more_actions()
        self.__click_schedule()
        self._admin_console.wait_for_completion()
        return ConfigureSchedules(self._admin_console)

    @PageService()
    def email(self):
        """Emails the report"""
        self.__click_more_actions()
        self.__click_email()
        self._admin_console.wait_for_completion()
        return Email(self._admin_console)

    @PageService()
    def refresh(self):
        """Refreshes the report"""
        self.__click_refresh()
        self._admin_console.wait_for_completion()

    @PageService()
    def get_current_view(self):
        """Gets the name of the view which is set as default"""
        return self.__fetch_current_view()

    @PageService()
    def get_all_views(self):
        """Returns a list of views available for the report"""
        try:
            self.__click_view_drop_down()
            views = self.__fetch_all_views()
            self.__click_view_drop_down()
        except NoSuchElementException:
            return []
        return views

    @PageService()
    def delete_view(self, name):
        """Deletes the given view

        Args:
            name (str): Name of the view to be deleted

        """
        self.__click_view_drop_down()
        self.__hover_over_the_view(name)
        self.__click_delete_icon(name)
        self.__click_delete()
