# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""
This module provides the function or operations that can be performed on the
reports listing page of the AdminConsole
"""

import time

from selenium.common.exceptions import (
    WebDriverException,
    NoSuchElementException
)
from selenium.webdriver import ActionChains

from Web.Common.exceptions import CVWebAutomationException
from Web.Common.page_object import (
    WebAction,
    PageService
)
from Reports.storeutils import StoreUtils
from Web.WebConsole.Store import storeapp
from Web.AdminConsole.Adapter.WebConsoleAdapter import WebConsoleAdapter

_STORE_CONFIG = StoreUtils.get_store_config()


class _MockBrowser:

    def __init__(self, driver):
        self.driver = driver


class ManageReport:
    """Class for Reports Page"""

    def __init__(self, admin_console):
        """
        Args:
            admin_console: adminconsole base object
        """
        self._admin_console = admin_console
        self._driver = admin_console.driver

    @WebAction()
    def __click_page_action_item(self, item):
        """Clicks on the given item under the page level action menu"""
        item = self._driver.find_element_by_xpath(f"//a[.='{item}']")
        item.click()

    @WebAction()
    def __set_search(self, string):
        """Sets the given search string"""
        search = self._driver.find_element_by_xpath("//input[@title='Search']")
        search.clear()
        search.send_keys(string)
        time.sleep(2)

    @WebAction()
    def __hover_on_report(self, report_name):
        """Hovers on the given report"""
        elements = self._driver.find_elements_by_xpath(f"//*[@title='{report_name}']")
        for each_element in elements:
            if each_element.is_displayed():
                hover = ActionChains(self._driver).move_to_element(each_element)
                hover.perform()
                break

    @WebAction()
    def __click_actions_of_a_report(self, report_name):
        """Clicks the actions of a given report"""
        actions = self._driver.find_element_by_xpath(
            f"//*[@title='{report_name}']/ancestor::div[@class='report-details-wrapper']"
            f"//*[contains(@class,'k-icon')]")
        actions.click()

    @WebAction()
    def __click_actions_menu_item(self, report_name, menu_item):
        """Click actions menu option"""
        item = self._driver.find_element_by_xpath(
            f"//*[*/*[@title='{report_name}']]//*[@title='{menu_item}']"
        )
        item.click()

    @WebAction()
    def __actions_menu(self):
        """Click Actions menu"""
        menu = self._driver.find_element_by_xpath("//ul[@role='menu']/../a/span")
        menu.click()

    @WebAction()
    def __click_report(self, report_name):
        """Clicks Report"""
        xp = f"//*[@title='{report_name}' and @href]"
        links = [link for link in self._driver.find_elements_by_xpath(xp)
                 if link.is_displayed()]
        if not links:
            raise NoSuchElementException(f"XPath {xp} not found")
        links[0].click()

    @WebAction()
    def __is_report_displayed(self, report_name):
        """Checks whether the report is displayed """
        report = self._driver.find_elements_by_xpath(f"//*[@title='{report_name}']")
        if not report:
            return False
        return True if report[0].is_displayed() else False

    @WebAction()
    def __read_tags(self):
        """Reads the Tags from reports landing page"""
        return [
            tag.text.strip() for tag in
            self._driver.find_elements_by_xpath("//*[contains(@class,'group-tag-name-content')]")
            if len(tag.text.strip()) > 0
        ]

    @WebAction()
    def __click_on_edit_tag(self, tag_name):
        """clicks on the tags name"""
        tag_xpath = f"//*[@class='group-tag-name']//span[text() = '{tag_name}']"
        tag_element = self._driver.find_element_by_xpath(tag_xpath)
        edit_tag_xpath = f"//span[text() = '{tag_name}']/..//span[@title='Edit tag']"
        edit_tag_element = self._driver.find_element_by_xpath(edit_tag_xpath)
        self._admin_console.mouseover_and_click(
            mouse_move_over=tag_element, mouse_click=edit_tag_element
        )

    @WebAction()
    def __set_tag_name(self, new_name):
        """ passes tag name to the edited tag field"""
        self._admin_console.fill_form_by_id('tagNameField', new_name)

    @WebAction()
    def __view_menu(self):
        """Clicks the view dropdown"""
        view = self._driver.find_element_by_xpath(
            "//div[contains(@class,'page-action-item')][1]//span[contains(@class,'dropdownArrow')]"
        )
        view.click()

    @WebAction()
    def __click_update_report_icon(self):
        """Click update button"""
        link = self._driver.find_element_by_xpath(
            "//*[contains(@uib-tooltip,'Needs update')]/a"
        )
        link.click()

    @WebAction()
    def __click_goto_store_for_update(self):
        """Click goto store"""
        btn = self._driver.find_element_by_xpath(
            "//button[.='Go to store to update']"
        )
        btn.click()
        self._driver.switch_to.window(
            self._driver.window_handles[-1]
        )

    @WebAction()
    def __get_reports_available_for_update(self):
        """Get the list of reports available for update"""
        reports = self._driver.find_elements_by_xpath(
            """//*[@data-ng-repeat="reportUpdate in updates | orderBy: 'reportName'"]"""
        )
        return [report.text.strip() for report in reports]

    @PageService()
    def view_schedules(self):
        """View Schedules"""
        self.__actions_menu()
        self.__click_page_action_item("View schedules")
        self._admin_console.wait_for_completion()

    @PageService()
    def view_alerts(self):
        """View alerts"""
        self.__actions_menu()
        self.__click_page_action_item("View alerts")
        self._admin_console.wait_for_completion()

    @PageService()
    def is_report_exists(self, report_name):
        """ Returns true if report exists else false

        Args:
            report_name: Name of the report

        Returns(bool):  True/False

        """
        return self.__is_report_displayed(report_name)

    @PageService()
    def add_report(self):
        """Adds report"""
        self.__actions_menu()
        try:
            self.__click_page_action_item('Add report')
        except WebDriverException:
            self.__actions_menu()
            raise CVWebAutomationException("Add report is not available")

        self._driver.switch_to_window(self._driver.window_handles[1])
        self._admin_console.wait_for_completion()

    @PageService()
    def import_template(self):
        """Imports template"""
        self.__actions_menu()
        self.__click_page_action_item("Import template")

    @PageService()
    def connect_to_store(self):
        """Connects to store"""
        self.__actions_menu()
        self.__click_page_action_item("Connect to store")
        self._admin_console.wait_for_completion()

    @PageService()
    def access_report(self, report_name):
        """Search reports page

        Args:
            report_name(str): Name of the report

        """
        self.__set_search(report_name)
        self.__click_report(report_name)
        self._admin_console.wait_for_completion()

    @PageService()
    def tag_report(self, report_name):
        """Tags the given report

        Args:
            report_name(str): Name of the report

        """
        self.__set_search(report_name)
        self.__hover_on_report(report_name)
        self.__click_actions_of_a_report(report_name)
        self.__click_actions_menu_item(report_name, "Tag")
        self._admin_console.wait_for_completion()

    @PageService()
    def edit_report(self, report_name):
        """Edits the given report

        Args:__click_actions
            report_name(str): Name of the report

        """
        self.__hover_on_report(report_name)
        self.__click_actions_of_a_report(report_name)
        try:
            self.__click_actions_menu_item(report_name, "Edit")
        except WebDriverException:
            self.__click_actions_of_a_report(report_name)
            raise CVWebAutomationException("User has no privileges")
        self._admin_console.wait_for_completion()

    @PageService()
    def report_permissions(self, report_name):
        """Access the security panel for the given report

        Args:
            report_name(str): Name of the report

        """
        self.__hover_on_report(report_name)
        self.__click_actions_of_a_report(report_name)
        self.__click_actions_menu_item(report_name, "Security")
        self._admin_console.wait_for_completion()

    @PageService()
    def delete_report(self, report_name):
        """Deletes the given report

        Args:
            report_name(str): Name of the report

        """
        self.__hover_on_report(report_name)
        self.__click_actions_of_a_report(report_name)
        try:
            self.__click_actions_menu_item(report_name, "Delete")
        except WebDriverException:
            self.__click_actions_of_a_report(report_name)
            raise CVWebAutomationException("User has no privileges")
        self._admin_console.wait_for_completion()

    @PageService()
    def get_tags(self):
        """Returns the visible Tags in reports landing page"""
        return self.__read_tags()

    @PageService()
    def edit_tag(self, current_name, new_name):
        """Edits Tag name

        Args:
            current_name: current name for the tag

            new_name: new name which replaces current name

        """
        self.__click_on_edit_tag(current_name)
        self.__set_tag_name(new_name)
        self._admin_console.click_button('Save')
        self._admin_console.wait_for_completion()

    @PageService()
    def get_reports_available_for_update(self):
        """Get the reports which has an update to install from store"""
        self.__click_update_report_icon()
        reports = self.__get_reports_available_for_update()
        self.__click_update_report_icon()
        return reports

    @PageService(hide_args=True)
    def goto_store_for_update(self,
                              username=_STORE_CONFIG.PREMIUM_USERNAME,
                              password=_STORE_CONFIG.PREMIUM_USERNAME):
        """Goto to store to update reports which have updates"""
        browser = _MockBrowser(self._driver)
        self.__click_update_report_icon()
        self.__click_goto_store_for_update()
        wc = WebConsoleAdapter(self._admin_console, browser)
        storeapp.StoreLogin(wc).login(username, password)
        storeapp.StoreApp(wc).wait_till_load_complete()


class Tag:
    """Class to deal with the Tags panel"""

    def __init__(self, admin_console):
        """
        Args:
            admin_console: adminconsole base object
        """
        self._admin_console = admin_console
        self._driver = admin_console.driver

    @WebAction(delay=1)
    def _enter_tag_name(self, tag_name):
        """enter tag name"""
        inp_tag = self._driver.find_element_by_xpath(
            "//*[@class='select2-search-field']/input"
        )
        inp_tag.clear()
        inp_tag.send_keys(tag_name + '\n')

    @WebAction()
    def _get_tags(self):
        """reads the associated tag names"""
        tags = []
        tags_obj = self._driver.find_elements_by_xpath("//div[@class='users-list']//label/span")
        for e_tag in tags_obj:
            tags.append(e_tag.text)
        return tags

    @WebAction()
    def _click_delete_tag(self, tag_name):
        """click on delete icon on tag name"""
        self._driver.find_element_by_xpath(f"//span[@title='Remove {tag_name}']").click()

    @PageService()
    def apply_to_all(self):
        """Enable appy to all"""
        self._admin_console.checkbox_select('publishToAll')

    @PageService()
    def save(self):
        """Saves the Tag panel"""
        self._admin_console.click_button('Save')
        self._admin_console.wait_for_completion()

    @PageService()
    def create_tag(self, tag_name, apply_to_all=False):
        """Create a Tag

        Args:
            tag_name (str): name of the Tag

            apply_to_all (bool): True/False

        """
        self._enter_tag_name(tag_name)
        if apply_to_all:
            self._admin_console.checkbox_select('publishToAll')
        else:
            self._admin_console.checkbox_deselect('publishToAll')
        self.save()

    @PageService()
    def get_associated_tags(self):
        """Returns the assoicated Tags"""
        return self._get_tags()

    @PageService()
    def delete_tag(self, tag_name):
        """
        Delete the tag
        Args:
            tag_name: Name of the Tag

        """
        self._click_delete_tag(tag_name)
        self._admin_console.wait_for_completion()
