# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""
File for all CTE operations like export, schedule, alert and Security operations
any common functionality across Reports APP can be added here.

Email:

    _select_format     --     Select email attachment export_type

    _set_recipient     --     Set recipient email id

    _choose_format     --     Choose export_type

    email_now          --     emails the report

ConfigureSchedules:

    _select_format     --     Select schedule email attachment export_type

    _select_daily      --     select daily schedule option

    _select_weekly     --     select weekly schedule option

    _select_monthly    --     select monthly schedule option

    set_schedule_name  --     Set schedule name

    select_frequency   --     select schedule frequency

    create_schedule    --     Create schedule on report

"""
from datetime import time
from time import sleep

from AutomationUtils import config

from Web.AdminConsole.Components.table import Table
from Web.Common.page_object import (
    WebAction,
    PageService
)

_CONFIG = config.get_config()


class Email:
    """
    EmailNow has the interfaces to work on Email now panel and do email operations on admin console
    Reports

    email_now(file_format, recipient_mail_id) -- emails the report with given export_type
                                                 to recipient mail given in input
    """
    class Format:
        """
        export_type is a constant on different type of file formats in Email now panel on reports
        """
        PDF = "PDF"
        HTML = "HTML"
        CSV = "CSV"

    def __init__(self, admin_console):
        """
        Args:
            admin_console: adminconsole base object
        """
        self._admin_console = admin_console
        self._driver = admin_console.driver

    @WebAction()
    def _switch_frame(self):
        """
        Switch to email frame
        """
        frame = self._driver.find_element_by_class_name("modal-iframe")
        self._driver.switch_to_frame(frame)

    @WebAction()
    def _select_format(self, attachment_type):
        """
        Select pdf/html/csv export_type in email frame
        Args:
            attachment_type: select attachment type from export_type class
        """
        self._driver.find_element_by_xpath("//*[contains(@class, 'checkboxLayer')]"
                                           f"//div[@title='{attachment_type}']").click()

    @WebAction()
    def _set_recipient(self, recipient_mail_id):
        """
        Set recipient name
        Args:
            recipient_mail_id       (String)   --    recipient email id

        Returns:

        """
        self._driver.find_element_by_name("emailRecipients").clear()
        self._driver.find_element_by_name("emailRecipients").send_keys(recipient_mail_id)

    @WebAction()
    def _click_format(self):
        """
        Click on export_type before selecting file type in email frame
        """
        self._driver.find_element_by_xpath("//*[@id='outputFormatSelect']").click()

    @PageService()
    def _choose_format(self, file_format):
        """
        Choose file export_type
        Args:
            file_format           (String)   --   Select formats available from export_type class
        """
        self._click_format()
        if file_format == self.Format.PDF:
            self._select_format(self.Format.PDF)
        elif file_format == self.Format.HTML:
            self._select_format(self.Format.HTML)
        elif file_format == self.Format.CSV:
            self._select_format(self.Format.CSV)
        else:
            raise ValueError("Invalid attachment type is passed!")

    @PageService()
    def email_now(self, file_format, recipient_mail_id):
        """
        Emails the report
        Args:
            file_format           (String)   --   Select formats available from export_type class
            recipient_mail_id:    (String)   --    recipient email id
        """
        self._choose_format(file_format)
        self._set_recipient(recipient_mail_id)
        self._admin_console.submit_form()


class ConfigureSchedules:
    """
    ConfigureSchedules has the interfaces to schedule reports and
    to operate on Reports schedule panels

    create_schedule(schedule_name, email_recipient, file_format) -- creates a schedule
                    with email notification with given frequency type of default time.
    """
    class Frequency:
        """
        Frequency is a constant on different frequency types available on Reports schedule panel
        """
        DAILY = 'Daily'
        WEEKLY = "Weekly"
        Monthly = "Monthly"

    class Format:
        """
        export_type is a constant on different type of export_type available on schedule panel of reports
        """
        PDF = "PDF"
        HTML = "HTML"
        CSV = "CSV"
        EXECUTIVE_SUMMARY = "Executive Summary"  # executive summary for WW activity report.
        INLINE = "INLINE"

    def __init__(self, admin_console):
        """
        Args:
            admin_console: adminconsole base object
        """
        self._admin_console = admin_console
        self._driver = admin_console.driver


    @WebAction()
    def _access_schedule_frame(self):
        """
        Switch to schedule frame
        """
        frame = self._driver.find_element_by_class_name("modal-iframe")
        self._driver.switch_to_frame(frame)

    @WebAction()
    def _select_format(self, attachment_type):
        """
        Select pdf/html/csv export_type in schedule frame
        Args:
            attachment_type: select attachment type from export_type class
        """
        self._driver.find_element_by_xpath("//*[contains(@class, 'checkboxLayer')]"
                                           f"//div[@title='{attachment_type}']").click()

    @WebAction()
    def _user_to_notify(self, user_or_user_group_name):
        """
        select user to notify for the user or user group
        Args:
            user_or_user_group_name: Enter string user or group name
        """
        self._admin_console.fill_form_by_id('field',user_or_user_group_name)
        self._admin_console.wait_for_completion()
        self._admin_console.driver.find_element_by_xpath(
            f"//*[@class='suggestedText vw-text']/..//*[@title='{user_or_user_group_name}']").click()

    @WebAction()
    def _select_inline(self):
        """
        Select inline as email body
        """
        self._admin_console.select_radio("Inline (as email body)")

    @WebAction()
    def _select_daily(self):
        """
        Select daily schedule
        """
        self._admin_console.select_radio("Daily")

    @WebAction()
    def _select_weekly(self):
        """
        Select weekly
        """
        self._admin_console.select_radio("Weekly")

    def _select_monthly(self):
        """
        Select monthly
        """
        self._admin_console.select_radio("Monthly")

    @WebAction()
    def _click_on_save(self):
        """
        Click on save button
        """
        self.save()

    @WebAction()
    def set_schedule_name(self, schedule_name):
        """
        sets name for the schedule
           Args:
           schedule_name               (String)   --    name for schedule
        """
        self._driver.find_element_by_id("name").clear()
        self._driver.find_element_by_id("name").send_keys(schedule_name)

    @WebAction()
    def _click_format(self):
        """
        Click on export_type before selecting file type in schedule frame
        """
        self._driver.find_element_by_xpath("//*[@id='outputFormatSelect']").click()

    @PageService()
    def select_frequency(self, frequency):
        """
        selects the frequency pattern for schedules
        :param frequency: frequency to select, use self.frequency for available types
        """
        if frequency == self.Frequency.DAILY:
            self._select_daily()
        elif frequency == self.Frequency.WEEKLY:
            self._select_weekly()
        elif frequency == self.Frequency.Monthly:
            self._select_monthly()
        else:
            raise Exception("Undefined Frequency type [%s] " % frequency)

    @PageService()
    def save(self):
        """
        saves the schedule
        """
        self._admin_console.click_button("Save")

    @PageService()
    def cancel(self):
        """
        closes the schedule panel
        """
        self._admin_console.click_button("Cancel")

    @WebAction()
    def set_recipient(self, recipient):
        """
        sets recipient mail id in schedule
        Args:
            recipient: mail id of recipient, use comma seperation for multiple id
        """
        self._driver.find_element_by_name("emailRecipients").clear()
        self._driver.find_element_by_name("emailRecipients").send_keys(recipient)

    @PageService()
    def select_format(self, file_type='PDF'):
        """
        selects the export_type given in input
        :param file_type: file export_type to select in schedule
        """
        self._click_format()
        if file_type == self.Format.PDF:
            self._select_format(self.Format.PDF)
        elif file_type == self.Format.HTML:
            self._select_format(self.Format.HTML)
        elif file_type == self.Format.CSV:
            self._select_format(self.Format.CSV)
        elif file_type == self.Format.INLINE:
            self._select_inline()
        else:
            raise Exception('Given export_type %s not available in schedule panel' % file_type)

    @PageService()
    def create_schedule(self, schedule_name, email_recipient, file_format=Format.PDF):
        """
        Method to create schedule with basic options with email
        notification and with given export_type and frequency
        Args:
            schedule_name: name of schedule to be created
            email_recipient: email id for notfication, comma seperated to multiple id
            file_format: export_type of file type, use self.export_type for available formats
        """
        self._admin_console.log.info(
            "Creating schedule:%s ; email recipient:%s; export_type:%s;",
            schedule_name, email_recipient, file_format
        )
        self.set_schedule_name(schedule_name)
        self.set_recipient(email_recipient)
        self.select_format(file_format)
        self.save()
        sleep(4)

    @PageService()
    def enable_end_user_security(self):
        """ select the end user security"""
        xpath = '//*[@toggle-name="endUserSecurity"]'
        self._driver.find_element_by_xpath(xpath).click()

    @PageService()
    def select_user_to_notify(self, user_or_group):
        """
        select the user to notify and send user or user group
        Args:
            user_or_group: string user or group name
        """
        self._user_to_notify(user_or_group)


class CommonFeatures:
    """Commonly used options such as alerts and schedules options"""
    def __init__(self, admin_console):
        """
        Args:
            admin_console: adminconsole base object
        """
        self._admin_console = admin_console
        self._driver = admin_console.driver
        self.table = Table(self._admin_console)

    @WebAction()
    def _click_enable(self):
        """Click on enable"""
        self._driver.find_element_by_id('enable').click()

    @WebAction()
    def _click_disable(self):
        """Click on disable"""
        self._driver.find_element_by_id('disable').click()

    @WebAction()
    def _click_more_menu(self):
        """Click more menu"""
        xpath = "//span[@class='k-link k-menu-link']/span['More actions']"
        self._driver.find_element_by_xpath(xpath).click()

    @WebAction()
    def _click_run_now(self):
        """Click run now"""
        self._driver.find_element_by_xpath(
            "//*[@class='k-animation-container']//a[text()='Run now']").click()

    @WebAction()
    def _click_delete(self):
        """Click delete"""
        sleep(2)
        self._driver.find_element_by_xpath(
            "//*[@class='k-animation-container']//a[@id='delete']").click()

    @WebAction()
    def accept_delete_warning(self):
        """accept delete warning"""
        self._driver.find_element_by_xpath("//*[@ng-click='yes()']").click()

    @PageService()
    def delete_entity(self, names):
        """
        Delete particular row entity
        Args:
            names              (String)     --        names of the entity to delete
        """
        self.table.select_rows(names)
        self._click_more_menu()
        sleep(1)
        self._click_delete()
        self.accept_delete_warning()

    @PageService()
    def enable_entity(self, names):
        """
        Enable entity
        Args:
            names             (list)     --        name of the entity to delete
        """
        self.table.select_rows(names)
        self._click_enable()

    @PageService()
    def disable_entity(self, names):
        """
        Disable entity
        Args:
            names             (String)    --        name of the entity to delete
        """
        self.table.select_rows(names)
        self._click_disable()

    @PageService()
    def run_entity(self, names):
        """
        Run Entity
        Args:
            names            (String)     --        name of the entity to run
        """
        self.table.select_rows(names)
        self._click_more_menu()
        sleep(1)
        self._click_run_now()
