# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations that can be used to run
testcases of OneDrive for Business module.

To begin, create an instance of OneDrive for test case.

To initialize the instance, pass the testcase object to the OneDrive class.

Call the required definition using the instance object.

This file consists of only one class OneDrive.
"""


from selenium.common.exceptions import (ElementNotInteractableException,
                                        NoSuchElementException,
                                        ElementClickInterceptedException)

from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.Office365Pages.office365_apps import Office365Apps
from Web.Common.exceptions import CVWebAutomationException
from Web.Common.page_object import PageService, WebAction
from . import constants


class OneDrive(Office365Apps):
    """ Class for OneDrive object """

    def __init__(self, tcinputs, admin_console):
        super(OneDrive, self).__init__(tcinputs, admin_console)
        self._navigator = self._admin_console.navigator
        self.__table = Table(self._admin_console)

    @WebAction(delay=2)
    def _view_user_details(self, user_name):
        """
        Go to user details page for given user

        Args:
            user_name (str):    Email address of user to be accessed

        """
        self._admin_console.access_tab(self.app_type.ACCOUNT_TAB.value)
        xp = (f"//*[@id='cv-k-grid-td-URL']/span[normalize-space()='{user_name}']"
              f"/ancestor::tr/td[contains(@id,'cv-k-grid-td-DISPLAY_NAME')]/a")
        self._driver.find_element_by_xpath(xp).click()
        self._admin_console.wait_for_completion()

    @WebAction(delay=2)
    def _click_selectall_browse(self):
        """Select all items during browse"""
        self._driver.find_element_by_xpath(
            "//th[@id='cv-k-grid-th-:checkbox']"
        ).click()

    @WebAction(delay=2)
    def _click_refresh_cache(self):
        """Clicks on the Refresh Cache option"""
        self._driver.find_element_by_xpath(
            f"//a[normalize-space()='{self._admin_console.props['label.forceCacheRefresh']}']"
        ).click()
        self._admin_console.wait_for_completion()
        self._modal_dialog.click_cancel()
        self._admin_console.wait_for_completion()

    @WebAction(delay=2)
    def _click_add_group(self, all_users=False):
        """Clicks on Add User Group button of Office365 OneDrive Client"""
        self._admin_console.access_tab(self._admin_console.props['label.content'])
        self._driver.find_element_by_xpath("//a[@id='ID_ADD_AD_GROUPS']").click()
        try:
            if all_users:
                self._driver.find_element_by_xpath("//a[@id='ALL_USERS']").click()
            else:
                self._driver.find_element_by_xpath("//a[@id='ADD_GROUP']").click()
        except ElementClickInterceptedException:
            if all_users:
                self._driver.find_element_by_xpath(
                    f"//span[text()='{self._admin_console.props['subtext.add.allOneDriveUsers']}']"
                ).click()
            else:
                self._driver.find_element_by_xpath(
                    f"//span[text()='{self._admin_console.props['desc.text.addOneDriveADGroup']}']"
                ).click()
        self._admin_console.wait_for_completion()

    @WebAction(delay=2)
    def _select_restore_time(self, restore_time):
        """
        Clicks on the given restore time

        Args:
            restore_time (str): Start time of backup job for
                                which items have to be restored
                                Format: Nov 5, 2020 5:35:58 PM

        """
        split_time = restore_time.split(' ')
        split_time[3] = split_time[3].split(':')
        split_time[3] = split_time[3][0] + ':' + split_time[3][1] + ' '
        restore_time = split_time[3] + split_time[4]
        self._driver.find_element_by_xpath(
            f"//div[contains(@class,'cv-ProductNav_Link ng-scope')]//span[text()='{restore_time}']"
        ).click()

    @WebAction(delay=2)
    def open_discover_cache_info_modal(self):
        """Opens the Discover cache info modal from Add User panel"""
        try:
            self._driver.find_element_by_xpath(
                f"//a[normalize-space()='{self._admin_console.props['info.usersFromCache']}']"
            ).click()
        except ElementClickInterceptedException:
            element = self._driver.find_element_by_xpath(
                f"//a[normalize-space()='{self._admin_console.props['info.usersFromCache']}']"
            )
            self._admin_console.driver.execute_script(
                "arguments[0].scrollIntoView();", element)
            self._admin_console.driver.execute_script(
                "arguments[0].click();", element)

    @WebAction(delay=2)
    def _enter_browse_destination_path(self, path):
        """Clicks the browse button on the app page

                path (basestring)   --  The destination path to which files/users are to be restored
        """
        input_xp = (f"//div[@class='modal-content']//label[text()='"
                    f"{self._admin_console.props['label.restoreoptions.machinePath']}"
                    f"']/following-sibling::div//input")
        self._driver.find_element_by_xpath(input_xp).send_keys(path)

    @PageService()
    def get_onedrive_discovery_count(self):
        """Get the count of users in tenant that was added by discovery"""
        self._open_add_user_panel()
        self._admin_console.wait_for_completion()
        try:
            self.open_discover_cache_info_modal()
            self._admin_console.wait_for_completion()
            cache_info_details = self.get_details_from_discover_cache_info()
        except (ElementNotInteractableException, NoSuchElementException):
            self.wait_while_discovery_in_progress()
            self.open_discover_cache_info_modal()
            self._admin_console.wait_for_completion()
            cache_info_details = self.get_details_from_discover_cache_info()
        user_count = int(cache_info_details[self._admin_console.props['count.lastCacheUsersCount']])
        group_count = int(cache_info_details[self._admin_console.props['count.lastCacheGroupsCount']])
        cache_update_time = cache_info_details[self._admin_console.props['label.cacheUpdateTime']]
        self._modal_dialog.click_cancel()
        self._admin_console.wait_for_completion()
        return user_count, group_count, cache_update_time

    @PageService()
    def refresh_cache(self):
        """Refreshes the OneDrive discovery cache"""
        self._admin_console.access_tab(self.app_type.ACCOUNT_TAB.value)
        self._open_add_user_panel()
        self._admin_console.wait_for_completion()
        try:
            self.open_discover_cache_info_modal()
        except (ElementNotInteractableException, NoSuchElementException):
            self.wait_while_discovery_in_progress()
            self.open_discover_cache_info_modal()
        self._admin_console.wait_for_completion()
        self._click_refresh_cache()
        self._modal_dialog.click_cancel()
        self._admin_console.wait_for_completion()

    @PageService()
    def add_group(self, groups=None, plan=None):
        """
        Adds user groups to the Office 365 App
        Args:
            groups (list):  Groups to be added to the app
            plan (str):     Plan to be associated to each group

        """
        self._click_add_group()
        if plan:
            o365_plan = plan
        else:
            o365_plan = self.tcinputs['Office365Plan']
        try:
            self._dropdown.select_drop_down_values(
                values=[o365_plan],
                drop_down_id=self.app_type.O365_PLAN_DROPDOWN_ID.value)
        except (ElementNotInteractableException, NoSuchElementException):
            self.wait_while_discovery_in_progress()
            self._dropdown.select_drop_down_values(
                values=[o365_plan],
                drop_down_id=self.app_type.O365_PLAN_DROPDOWN_ID.value)
        if groups:
            for group in groups:
                search_element = self._driver.find_element_by_id('searchInput')
                if search_element.is_displayed():
                    self._admin_console.fill_form_by_id(element_id='searchInput', value=group)
                self.__table.select_rows([group])
            self._LOG.info(f'Added groups: {groups}')
        else:
            for group in self.groups:
                search_element = self._driver.find_element_by_id('searchInput')
                if search_element.is_displayed():
                    self._admin_console.fill_form_by_id(element_id='searchInput', value=group)
                self.__table.select_rows([group])
            self._LOG.info(f'Added groups: {self.groups}')
        self._admin_console.submit_form()

    @PageService()
    def _enter_details_in_restore_panel(self, destination, file_server=None, dest_path=None, user_name=None):
        """

        Args:
            destination --  Specifies the destination
                            Acceptable values: to_disk, in_place, out_of_place
            dest_path   --  Path to which the files have to be restored
            user_name   --  User to which files have to be restored
        """
        if destination == constants.RestoreType.TO_DISK:
            self._dropdown.select_drop_down_values(
                values=['File location'],
                drop_down_id='restoreDestinationSummaryDropdown',
                partial_selection=True
            )
            self._dropdown.select_drop_down_values(
                values=[file_server],
                drop_down_id='office365AccessNodesDropdown'
            )
            self._enter_browse_destination_path(dest_path)
        elif destination == constants.RestoreType.IN_PLACE:
            self._dropdown.select_drop_down_values(
                values=['OneDrive for Business'],
                drop_down_id='restoreDestinationSummaryDropdown',
                partial_selection=True
            )
            self._driver.find_element_by_xpath("//input[@value='INPLACE']").click()
        elif destination == constants.RestoreType.OOP:
            self._dropdown.select_drop_down_values(
                values=['OneDrive for Business'],
                drop_down_id='restoreDestinationSummaryDropdown',
                partial_selection=True
            )
            self._driver.find_element_by_xpath("//input[@value='OUTOFPLACE']").click()
            self._driver.find_element_by_xpath(
                "//input[@id='userAccountInput']").send_keys(user_name)

    @PageService()
    def run_restore(self, destination, file_server=None, dest_path=None, user_name=None, user=True):
        """
        Runs the restore by selecting all users associated to the app

        Args:
            destination (str):  Specifies the destination
                                Acceptable values: to_disk, in_place, out_of_place
            file_server (str):  Name of file server in case of restore to disk
            dest_path (str):    Path to which the files have to be restored
            user_name (str):    User to which files have to be restored in case of OOP
            user (bool):        Whether to restore user or files

        Returns:
            job_details (dict): Details of the restore job

        """
        for user in self.users:
            self._select_user(user)
        self._click_restore(account=True)
        self._admin_console.wait_for_completion()
        self._enter_details_in_restore_panel(
            destination=destination,
            file_server=file_server,
            dest_path=dest_path,
            user_name=user_name)
        self._modal_dialog.click_submit()
        job_details = self._job_details()
        self._LOG.info('job details: %s', job_details)
        return job_details

    @PageService()
    def point_in_time_restore(self, user_name, time, file_server, dest_path):
        """
        Perform Point-in-time Restore for given user

        Args:
            user_name (str):    User to be restored
            time (str):         Time for which restore has to be performed
                                Format: Nov 5, 2020 5:35:58 PM
            file_server (str):  Server to which data has to be restored
            dest_path (str):    Path in the file server

        Returns:
            job_details (dict): Details of the restore job

        """
        self._view_user_details(user_name)
        self._select_restore_time(time)
        self._admin_console.select_hyperlink('Restore')
        self._admin_console.wait_for_completion()
        self._click_selectall_browse()
        self._click_restore()
        self._admin_console.wait_for_completion()
        self._enter_details_in_restore_panel(
            destination=constants.RestoreType.TO_DISK,
            file_server=file_server,
            dest_path=dest_path)
        self._modal_dialog.click_submit()
        return self._job_details()

    @PageService()
    def verify_cache_is_fetched(self):
        """Verify that existing cache is fetched"""
        self._click_add_group()
        if self._admin_console.check_if_entity_exists(
                'id', 'office365OneDriveAddAssociation_isteven-multi-select_#5444'):
            self._LOG.info('VERIFIED: Existing cache is fetched')
            self._modal_dialog.click_cancel()
        else:
            raise CVWebAutomationException('Existing cache is not fetched when Auto-discovery is running')

    @PageService()
    def remove_from_content(self, user=None, is_group=False):
        """
        Excludes the given user from backup
        Args:
            user (str):         User which has to be deleted
            is_group (bool):    Whether user/group is to be deleted
        """
        if user:
            self._select_user(user, is_group=is_group)
        else:
            self._select_user(self.users[0], is_group=is_group)
        self._click_remove_content(user=user, is_group=is_group)
        self._admin_console.wait_for_completion()
        self._modal_dialog.click_submit()
        self._admin_console.wait_for_completion()
        self._LOG.info(f'User removed from content: {user}')

    @PageService()
    def get_jobs_count(self):
        """Get the count of jobs in the job history of client"""
        return int(self.__table.get_total_rows_count())
