# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides the function or operations that can be used to run
testcases of Office365 module.

To begin, create an instance of Office365Apps for test case.

To initialize the instance, pass the testcase object to the Office365Apps class.

Call the required definition using the instance object.

This file consists of only one class Office365Apps.
"""

import datetime
from enum import Enum
import time
import collections
from selenium.common.exceptions import (ElementNotInteractableException,
                                        NoSuchElementException,
                                        ElementClickInterceptedException)

from AutomationUtils import logger
from Web.AdminConsole.Components.table import Table
from Web.Common.page_object import PageService, WebAction
from Web.Common.exceptions import CVWebAutomationException
from Web.AdminConsole.Components.dialog import ModalDialog
from Web.AdminConsole.Components.panel import DropDown, PanelInfo
from Web.AdminConsole.Components.browse import Browse
from Web.AdminConsole.AdminConsolePages.Jobs import Jobs
from Web.AdminConsole.AdminConsolePages.Plans import Plans
from . import constants


class Office365Apps:
    """Class for all Office 365 Apps page"""

    class AppType(Enum):
        """App types for Office 365"""
        exchange_online = 'ADD_EXCHANGEONLINE_APP'
        share_point_online = 'ADD_SHAREPOINT_ONLINE_APP'
        one_drive_for_business = 'ADD_ONEDRIVE_FOR_BUSINESS_APP'

    def __init__(self, tcinputs, admin_console):
        """Initializes the Office365Apps class instance

                Args:
                    tc (Object)  --  Testcase object

        """
        self.tcinputs = tcinputs
        self._admin_console = admin_console
        self._driver = admin_console.driver
        self._LOG = logger.get_log()
        self.modern_authentication = False
        self.newly_created_app_name = None
        self._modal_dialog = ModalDialog(self._admin_console)
        self.app_stats_dict = {}
        # Required Components
        self.__table = Table(self._admin_console)
        self._dropdown = DropDown(self._admin_console)
        self._jobs = Jobs(self._admin_console)
        self._plans = Plans(self._admin_console)
        self._browse = Browse(self._admin_console)

        # Call Localization method
        self._admin_console._load_properties(self)
        if 'Users' in self.tcinputs:
            self.users = self.tcinputs['Users'].split(",")
        if 'Groups' in self.tcinputs:
            self.groups = self.tcinputs['Groups']

        if self.tcinputs['office_app_type'] == Office365Apps.AppType.exchange_online:
            self.app_type = constants.ExchangeOnline
        elif self.tcinputs['office_app_type'] == Office365Apps.AppType.one_drive_for_business:
            self.app_type = constants.OneDrive
        elif self.tcinputs['office_app_type'] == Office365Apps.AppType.share_point_online:
            self.app_type = constants.SharePointOnline

    @WebAction()
    def _create_app(self, app_type):
        """Creates app with given app type

                Args:
                    app_type  (basestring)  -- Type of app to create

        """

        add_apps_xpath = "//a[@id='ADD_APPS']"
        add_apps_new_id = "addOffice365App"
        exchange_online_xpath = f"//div[text()='{self._admin_console.props['label.nav.exchangeOnline']}']"
        onedrive_v2_xpath = f"//div[text()='{self._admin_console.props['label.nav.oneDriveForBusiness']}']"
        sharepoint_online_v2_xpath = f"//div[text()='{self._admin_console.props['label.nav.sharepointOnline']}']"

        if self._admin_console.check_if_entity_exists('xpath', add_apps_xpath):
            self._driver.find_element_by_xpath(add_apps_xpath).click()
            self._driver.find_element_by_xpath("//a[@id='" + app_type.value + "']").click()

        elif self._admin_console.check_if_entity_exists('id', add_apps_new_id):
            self._driver.find_element_by_id(add_apps_new_id).click()
            self._admin_console.wait_for_completion()
            if app_type == Office365Apps.AppType.exchange_online:
                self._driver.find_element_by_xpath(exchange_online_xpath).click()
            elif app_type == Office365Apps.AppType.one_drive_for_business:
                self._driver.find_element_by_xpath(onedrive_v2_xpath).click()
            elif app_type == Office365Apps.AppType.share_point_online:
                self._driver.find_element_by_xpath(sharepoint_online_v2_xpath).click()

        elif self._admin_console.check_if_entity_exists('link', self._admin_console.props['label.add.office365']):
            self._driver.find_element_by_link_text(self._admin_console.props['label.add.office365']).click()
            self._admin_console.wait_for_completion()
            if app_type == Office365Apps.AppType.exchange_online:
                self._driver.find_element_by_xpath(exchange_online_xpath).click()
            elif app_type == Office365Apps.AppType.one_drive_for_business:
                self._driver.find_element_by_xpath(onedrive_v2_xpath).click()
            elif app_type == Office365Apps.AppType.share_point_online:
                self._driver.find_element_by_xpath(sharepoint_online_v2_xpath).click()

        else:
            if app_type == Office365Apps.AppType.exchange_online:
                self._driver.find_element_by_link_text(self._admin_console.props['link.addExchangeOnline']).click()
            elif app_type == Office365Apps.AppType.share_point_online:
                self._driver.find_element_by_link_text(self._admin_console.props['link.addSharePointOnline']).click()
            else:
                self._driver.find_element_by_link_text(self._admin_console.props['link.addOneDriveforBusiness']).click()

        self._admin_console.wait_for_completion()

    @WebAction(delay=2)
    def _enter_shared_path_for_multinode(self, shared_jrd=None):
        """Enters the shared Job Results directory for multiple access nodes"""
        if shared_jrd:
            self._driver.find_element_by_id('jobResultsDirectory').clear()
            self._driver.find_element_by_id('jobResultsDirectory').send_keys(shared_jrd)
        else:
            self._driver.find_element_by_id('jobResultDirectory').send_keys(self.tcinputs["UNCPath"])

    @WebAction(delay=2)
    def _add_account_for_shared_path(self, user_account=None, password=None):
        """Enters the user account and password to access the shared path"""
        app_type = self.tcinputs['office_app_type']
        if user_account:
            if app_type == Office365Apps.AppType.share_point_online:
                self._driver.find_element_by_id('localAccessAccountName').clear()
                self._driver.find_element_by_id('localAccessAccountName').send_keys(user_account)
            else:
                self._driver.find_element_by_id('username').clear()
                self._driver.find_element_by_id('username').send_keys(user_account)

            self._driver.find_element_by_id('password').send_keys(password)
            self._driver.find_element_by_id('confirmPassword').send_keys(password)
        else:
            self._driver.find_element_by_id('addLocalAccount').click()
            self._admin_console.wait_for_completion()
            self._driver.find_element_by_id('localSystemAccountUser').send_keys(self.tcinputs["UserAccount"])
            self._driver.find_element_by_id('password').send_keys(self.tcinputs["UserAccPwd"])
            self._driver.find_element_by_id('confirmPassword').send_keys(self.tcinputs["UserAccPwd"])
        self._modal_dialog.click_submit()
        self._admin_console.wait_for_completion()

    @WebAction(delay=2)
    def _edit_global_admin(self, new_global_admin, new_admin_pwd):
        """
        Modifies the global admin to the new value given as argument
        Args:
            new_global_admin:   New value of global admin to be set
            new_admin_pwd:      Password for global admin account
        """
        self.tcinputs['GlobalAdmin'] = new_global_admin
        self.tcinputs['Password'] = new_admin_pwd

        self._driver.find_element_by_xpath(
            f"//span[text()='{self._admin_console.props['label.globalAdministrator']}'"
            f"]//ancestor::li//a[text()='Edit']"
        ).click()
        self._admin_console.wait_for_completion()
        self._driver.find_element_by_id('globalAdministrator').clear()
        self._driver.find_element_by_id('globalAdministrator').send_keys(new_global_admin)
        self._driver.find_element_by_id('password').send_keys(new_admin_pwd)
        self._driver.find_element_by_id('confirmPassword').send_keys(new_admin_pwd)
        self._modal_dialog.click_submit()
        self._admin_console.wait_for_completion()

    @WebAction(delay=2)
    def _edit_server_plan(self, server_plan):
        """
        Modifies the server plan to the new value given as argument
        Args:
            server_plan: New value of server plan to be set
        """
        self.tcinputs['ServerPlan'] = server_plan
        panel = PanelInfo(self._admin_console,
                          title=self._admin_console.props['label.infrastructurePane'])
        self._driver.find_element_by_xpath(
            f"//span[text()='{self._admin_console.props['label.o365.serverPlan']}'"
            f"]//ancestor::li//a[contains(text(), 'Edit')]"
        ).click()
        self._admin_console.wait_for_completion()
        self._dropdown.select_drop_down_values(values=[server_plan],
                                               drop_down_id='planSummaryDropdown')
        panel.save_dropdown_selection(self._admin_console.props['label.o365.serverPlan'])
        self._modal_dialog.click_submit()
        self._admin_console.wait_for_completion()
        self._modal_dialog.click_submit()
        self._admin_console.wait_for_completion()

    @WebAction(delay=2)
    def _edit_stream_count(self, new_stream_count):
        """
        Modifies the count of max streams to the new value given as argument
        Args:
            new_stream_count:   New value of 'Max streams' to be set
        """
        self.tcinputs['MaxStreams'] = new_stream_count
        panel = PanelInfo(self._admin_console,
                          title=self._admin_console.props['label.infrastructurePane'])
        self._driver.find_element_by_xpath(
            f"//span[text()='Max streams']//ancestor::li//a[contains(text(), 'Edit')]"
        ).click()
        self._admin_console.wait_for_completion()
        self._driver.find_element_by_id("maxStream").clear()
        self._driver.find_element_by_id("maxStream").send_keys(new_stream_count)
        panel.save_dropdown_selection('Max streams')

    @WebAction(delay=2)
    def _edit_index_server(self, new_index_server):
        """
        Modifies the index server to the new value given as argument
        Args:
            new_index_server:   New value of index server to be set
        """
        self.tcinputs['IndexServer'] = new_index_server
        self._driver.find_element_by_xpath(
            f"//span[text()='Index server']//ancestor::li//a[contains(text(), 'Edit')]"
        ).click()
        self._admin_console.wait_for_completion()
        self._dropdown.select_drop_down_values(
            values=[new_index_server], drop_down_id='updateIndexServer_isteven-multi-select_#2168')
        self._modal_dialog.click_submit()
        self._modal_dialog.click_submit()
        self._admin_console.wait_for_completion()

    @WebAction(delay=2)
    def _edit_access_node(self,
                          new_shared_path,
                          new_user_account,
                          new_password,
                          new_access_node=None):
        """
        Modifies the shared path and access node values
        Args:
            new_access_node:    New value(s) to be set for access node
            new_shared_path:    New value to be set for shared path
            new_user_account:   Local system account to access shared path
            new_password:       Password for local system account
        """
        self.tcinputs['UNCPath'] = new_shared_path
        self.tcinputs['UserAccount'] = new_user_account
        self.tcinputs['UserAccPwd'] = new_password

        self._driver.find_element_by_xpath(
            f"//span[text()='Access nodes']//ancestor::li//a[contains(text(), 'Edit')]"
        ).click()
        self._admin_console.wait_for_completion()
        if new_access_node:
            self.tcinputs['AccessNode'] = new_access_node
            if isinstance(new_access_node, list):
                self._dropdown.select_drop_down_values(values=new_access_node,
                                                       drop_down_id='accessNodes')
            else:
                self._dropdown.select_drop_down_values(values=[new_access_node],
                                                       drop_down_id='accessNodes')
        self._enter_shared_path_for_multinode(new_shared_path)
        self._add_account_for_shared_path(new_user_account, new_password)
        self._admin_console.wait_for_completion()
        close_xpath = "//div[contains(@class, 'modal-content')]//button[contains(@class, 'btn btn-default')]"
        if self._admin_console.check_if_entity_exists('xpath', close_xpath):
            self._modal_dialog.click_cancel()
        self._admin_console.wait_for_completion()

    @WebAction(delay=2)
    def _click_create_azure_ad_app(self):
        """Clicks create azure ad app button"""
        create_azure_ad_app_xpath = "//button[@id='createOffice365App_button_#6055']"
        self._driver.find_element_by_xpath(create_azure_ad_app_xpath).click()
        self._check_for_errors()

    @WebAction()
    def _check_for_errors(self):
        """Checks for any errors while creating app"""
        error_xpaths = [
            "//div[@class='help-block']",
            "//div[@ng-bind='addOffice365.appCreationErrorResponse']"
        ]

        for xpath in error_xpaths:
            if self._admin_console.check_if_entity_exists('xpath', xpath):
                if self._driver.find_element_by_xpath(xpath).text:
                    raise CVWebAutomationException(
                        'Error while creating the app: %s' %
                        self._driver.find_element_by_xpath(xpath).text
                    )

    @WebAction(delay=3)
    def _click_authorize_now(self):
        """Clicks on authorize now button"""
        self._admin_console.select_hyperlink(self._admin_console.props['action.authorize.app'])

    @WebAction(delay=3)
    def _switch_to_window(self, window):
        """Switch to specified window

                Args:
                    window (WebElement)  -- Window element to switch to
        """
        self._driver.switch_to_window(window)

    @WebAction(delay=2)
    def _enter_email(self, email):
        """Enter email in email type input

                Args:
                    email (basestring)  --  Microsoft Global Admin email
        """
        if self._admin_console.check_if_entity_exists('id', 'i0116'):
            self._admin_console.fill_form_by_id('i0116', email)
            self._click_submit()

    @WebAction(delay=2)
    def _enter_password(self, password):
        """Enter password into password type input

                Args:
                    password (basestring)  --  Global Admin password
        """
        if self._admin_console.check_if_entity_exists('id', 'i0118'):
            self._admin_console.fill_form_by_id('i0118', password)
            self._click_submit()

    @WebAction(delay=3)
    def _click_submit(self):
        """Click submit type button"""
        # This xpath is used to click submit button on Microsoft login pop up window
        ms_submit_xpath = "//input[@type='submit']"
        self._admin_console.scroll_into_view(ms_submit_xpath)
        self._driver.find_element_by_xpath(ms_submit_xpath).click()

    @WebAction(delay=2)
    def _get_new_app_name(self):
        """Fetches the newly created Azure app name from MS permission dialog"""
        return self._driver.find_element_by_xpath("//div[@class='row app-name']").text

    @WebAction()
    def _authorize_permissions(self, global_admin, password):
        """Clicks authorize now and enables permissions for the app

            Args:
                global_admin (basestring)  --  Microsoft Global admin email id
                password (basestring)  --  Global admin password
        """
        self._click_authorize_now()

        admin_console_window = self._driver.window_handles[0]
        azure_window = self._driver.window_handles[1]

        self._switch_to_window(azure_window)

        self._enter_email(global_admin)

        self._enter_password(password)

        self.newly_created_app_name = self._get_new_app_name()

        # Final Accept button
        self._click_submit()

        self._switch_to_window(admin_console_window)
        self._admin_console.wait_for_completion()

        # Clicking multiple times is required sometimes
        authorize_now_xpath = "//button[@id='createOffice365App_button_#1691']"
        attempts = 5
        while self._admin_console.check_if_entity_exists('xpath', authorize_now_xpath) or \
                self._admin_console.check_if_entity_exists('link', self._admin_console.props['action.authorize.app']):
            if attempts == 0:
                raise CVWebAutomationException("Failed to grant permissions.")

            self._LOG.info('Accept permissions button did not work. Trying again..')
            self._click_authorize_now()

            admin_console_window = self._driver.window_handles[0]
            azure_window = self._driver.window_handles[1]

            self._switch_to_window(azure_window)
            self._click_submit()

            self._switch_to_window(admin_console_window)
            self._admin_console.wait_for_completion()

            attempts -= 1

    @WebAction(delay=2)
    def _click_show_details(self):
        """Clicks the Show details link for Azure App"""
        if self._admin_console.check_if_entity_exists("link", self._admin_console.props['label.showDetails']):
            self._admin_console.select_hyperlink(self._admin_console.props['label.showDetails'])

    @WebAction(delay=2)
    def _verify_modern_authentication(self):
        """Verifies if app is created using modern authentication enabled."""
        self._click_show_details()
        try:
            modern_xpath = "//div[@data-ng-if='addOffice365.office365Attributes.exchangeAttributes.onePassProp.isModernAuthEnabled']"
            text = self._driver.find_element_by_xpath(modern_xpath).text
            if "Using modern authentication" in text:
                self.modern_authentication = True
        except:
            self.modern_authentication = False

    @WebAction(delay=2)
    def _verify_sharepoint_modern_authentication(self):
        """Verifies if SharePoint online app is created using modern authentication enabled."""
        try:
            sp_modern_xpath = "//div[contains(@data-ng-if, " \
                              "'addOffice365.office365Attributes.sharepointAttributes.sharepointBackupSet.spOffice365BackupSetProp.isModernAuthEnabled')]"
            text = self._driver.find_element_by_xpath(sp_modern_xpath).text
            if "Using modern authentication" in text:
                self.modern_authentication = True
                self._LOG.info('Modern authentication is enabled on the tenant and app is created with mod auth')
        except:
            self._LOG.info('Modern authentication is not enabled on the tenant and app is created with basic auth')
            self.modern_authentication = False

    @WebAction(delay=2)
    def _verify_sharepoint_service_account(self):
        """Verify if one sharepoint service account is created"""
        try:
            sp_service_acc_xpath = "//span[@ng-bind='addOffice365.office365Attributes.generalAttributes.sharepointOnlineServiceAccount']"
            text = self._driver.find_element_by_xpath(sp_service_acc_xpath).text
            if "CVSPBackupAccount" in text:
                self._LOG.info('Sharepoint Service account created from web server:%s', text)
        except:
            raise CVWebAutomationException("Service account not created for SharePoint Online from web server")

    @WebAction(delay=2)
    def _verify_app_principal(self):
        """Verify if one sharepoint app principal is authorized"""
        try:
            app_principal_xpath = "//div[@data-ng-if='addOffice365.isAzureAppPrincipleCreated']"
            text = self._driver.find_element_by_xpath(app_principal_xpath).text
            if "App principal created" in text:
                self._LOG.info('For SharePoint Online:%s', text)
        except:
            raise CVWebAutomationException("App principal not created for sharepoint online")

    @WebAction(delay=2)
    def _create_app_principal(self):
        """Creates app principal required for SharePoint V2 Pseudo Client Creation"""
        sharepoint_app_id = self._driver.find_element_by_xpath \
            ("//code[@data-ng-bind='o365SPCreateAppPrincipleCtrl.azureAppDetails.azureAppId']").text
        sharepoint_tenant_id = self._driver.find_element_by_xpath \
            ("//code[@data-ng-bind='o365SPCreateAppPrincipleCtrl.azureAppDetails.azureDirectoryId']").text
        sharepoint_request_xml = self._driver.find_element_by_xpath \
            ("//code[@data-ng-bind='o365SPCreateAppPrincipleCtrl.appPrincipleXML']").text
        self._driver.find_element_by_xpath(
            ".//a[contains(@href, '_layouts/15/appinv.aspx')]").click()

        admin_console_window = self._driver.window_handles[0]
        app_principal_window = self._driver.window_handles[1]

        if not self._admin_console.check_if_entity_exists("class", "ms-input"):
            self._switch_to_window(app_principal_window)
            self._admin_console.wait_for_completion()

        self._driver.find_element_by_xpath("//input[contains(@id, 'TxtAppId')]").send_keys(sharepoint_app_id)
        self._driver.find_element_by_xpath("//input[ @value = 'Lookup']").click()
        self._admin_console.wait_for_completion()
        self._driver.find_element_by_xpath("//input[@title = 'App Domain']").send_keys(sharepoint_tenant_id)
        self._driver.find_element_by_xpath("//textarea[@title = 'Permission Request XML']").send_keys(
            sharepoint_request_xml)
        self._driver.find_element_by_xpath("//input[ @value = 'Create']").click()
        self._admin_console.wait_for_completion()
        self._driver.find_element_by_xpath("//input[ @value = 'Trust It']").click()
        self._admin_console.wait_for_completion()
        self._switch_to_window(admin_console_window)
        self._driver.find_element_by_xpath(
            "//button[@data-ng-click = 'o365SPCreateAppPrincipleCtrl.confirm()']").click()
        self._driver.find_element_by_xpath("//button[@ng-click = 'yes()']").click()
        self._admin_console.wait_for_completion()

    @WebAction(delay=2)
    def _click_show_details_for_client_readiness(self):
        """Clicks on show details for client readiness check"""
        self._driver.find_element_by_xpath(
            f"//a[contains(text(), '{self._admin_console.props['label.showDetails']}')]"
        ).click()

    @WebAction(delay=2)
    def _get_client_readiness_value(self):
        """Gets the value of Client Readiness check"""
        elements = self._driver.find_elements_by_xpath(
            "//td[@data-ng-bind-html='item.status']")
        readiness = list()
        for row in elements:
            readiness.append(row.text)
        return readiness

    @WebAction(delay=2)
    def _open_add_user_panel(self):
        """Opens the panel to add users to the client"""
        try:
            self._driver.find_element_by_id(self.app_type.ADD_BUTTON_ID.value).click()
            self._driver.find_element_by_id(self.app_type.ADD_USER_BUTTON_ID.value).click()
        except (ElementClickInterceptedException, NoSuchElementException):
            if self.tcinputs['office_app_type'] == Office365Apps.AppType.exchange_online:
                self._driver.find_element_by_xpath(
                    f"//span[text()='{self._admin_console.props['label.addMailbox']}']").click()

    @WebAction(delay=2)
    def _select_all_users(self):
        """Selects all the users associated with the app"""
        self._driver.find_element_by_xpath("// label[@class ='k-checkbox-label k-no-text']").click()

    @WebAction(delay=2)
    def _select_user(self, user_name, is_group=False):
        """
        Selects the user specified
        Args:
            user_name (str):    Item to be selected
            is_group (bool):    Whether or not the item is a group

        """
        if is_group:
            self._admin_console.access_tab(self.app_type.CONTENT_TAB.value)
            xp = (f"//*[@id='cv-k-grid-td-DISPLAY_NAME']/span[normalize-space()='{user_name}']"
                  f"/ancestor::tr/td[contains(@id,'checkbox')]")
        else:
            self._admin_console.access_tab(self.app_type.ACCOUNT_TAB.value)
            xp = (f"//*[@id='cv-k-grid-td-URL']/span[normalize-space()='{user_name}']"
                  f"/ancestor::tr/td[contains(@id,'checkbox')]")
        self._driver.find_element_by_xpath(xp).click()

    @WebAction(delay=2)
    def _click_backup(self):
        """Clicks the backup button on the app page"""
        self.__table.access_toolbar_menu(menu_id=self.app_type.BACKUP_MENU_ID.value)

    @WebAction(delay=2)
    def _click_add_autodiscover(self):
        """Clicks the Add button on Content tab"""
        self.__table.access_toolbar_menu(menu_id=self.app_type.ADD_MENU_ID.value)

    @WebAction(delay=2)
    def _click_more_actions(self):
        """Clicks the more actions link on the app page"""
        more_actions_xpath = "//li[@id='batch-action-menu_moreItem']"
        self._driver.find_element_by_xpath(more_actions_xpath).click()

    @WebAction(delay=2)
    def _click_restore(self, account=True):
        """Clicks the browse button on the app page

                account (boolean)  --   If true - whole mailbox/user is restored.
                                        If false messages/files are restored
        """
        try:
            self._driver.find_element_by_id(self.app_type.CLICK_RESTORE_ID.value).click()
            if account:
                self._driver.find_element_by_xpath(self.app_type.ACCOUNT_RESTORE_XPATH.value).click()
            else:
                self._driver.find_element_by_xpath(self.app_type.BROWSE_RESTORE_XPATH.value).click()
                if self.tcinputs['office_app_type'] == Office365Apps.AppType.exchange_online:
                    self._restore_messages()
        except Exception:
            self._driver.find_element_by_xpath("//span[normalize-space()='Restore']").click()

    @WebAction(delay=2)
    def _click_selectall_browse(self):
        """Clicks the Select all checkbox"""
        self._browse.select_for_restore(all_files=True)

    @WebAction(delay=2)
    def _restore_messages(self):
        """Restores messages from browse window"""
        self._click_selectall_browse()
        self._admin_console.select_hyperlink("Restore")
        self._driver.find_element_by_xpath("//a[@class='nowrap']//span[text()='Items in current page']").click()

    @WebAction(delay=2)
    def _fetch_all_apps(self):
        """Fetches the list of apps from configuration page"""
        self._admin_console.refresh_page()
        panel = PanelInfo(self._admin_console, title='Exchange connection settings')
        panel.access_tab(tab_text=self._admin_console.props['label.azureApps'])
        details = self.__table.get_column_data(self._admin_console.props['label.addAzureApp'])
        return details

    @WebAction(delay=2)
    def _verify_app(self):
        """Verifies whether the newly created app is present in the list on configuration page"""
        apps_list = self._fetch_all_apps()
        self._LOG.info('Apps list:%s', apps_list)
        if self.newly_created_app_name not in apps_list:
            raise CVWebAutomationException("New app was not found in the list on configuration page")

    @WebAction(delay=2)
    def _get_job_id(self):
        """Fetches the jobID from the toast"""
        return self._admin_console.get_jobid_from_popup()

    @WebAction(delay=2)
    def _verify_connection_text(self):
        """Verifies the text on the modal after verifying connection"""
        modal_content_xpath = "//div[@class='form-group ng-scope']"
        modal_text = self._driver.find_element_by_xpath(modal_content_xpath).text
        if not modal_text == 'The status of Azure apps has been updated.':
            raise CVWebAutomationException('Verify connection did not succeed.')

    @WebAction(delay=2)
    def _add_plan_and_verify(self, change=False):
        """Changes exchange plan and verifies the same

                Args:
                    change (boolean)  --  True if it's a change plan operation
        """

        if change:
            value = self.tcinputs['NewPlan']
            self._create_inline_o365_plan()
        else:
            value = self.tcinputs['Office365Plan']
        self._dropdown.select_drop_down_values(values=[value],
                                               drop_down_id='exchangePlan_isteven-multi-select_#2')
        self._modal_dialog.click_submit()
        self._admin_console.wait_for_completion()
        notification = self._admin_console.get_notification()
        if "Successfully updated" not in notification:
            CVWebAutomationException('Change exchange plan was not successful')

    @WebAction(delay=2)
    def _job_details(self):
        """Waits for job completion and gets the job details"""
        job_id = self._get_job_id()
        job_details = self._jobs.job_completion(job_id=job_id)
        job_details['Job Id'] = job_id
        self._LOG.info('job details: %s', job_details)
        if job_details[self._admin_console.props['Status']] not in [
                "Committed", "Completed", "Completed w/ one or more errors"]:
            raise CVWebAutomationException('Job did not complete successfully')
        return job_details

    @WebAction(delay=2)
    def _get_app_stat(self, stat_type):
        """Gets the stats from the app page

                Args:
                    stat_type (basestring)  --  Type of stat we want to fetch from App details page
                        Valid options:
                            Mails -- To fetch the number of mails displayed on App page
                            Mailboxes -- To fetch the number of mailboxes displayed on App page
                            Indexed mails -- To fetch the number of Indexed mails displayed on App page

        """
        if stat_type == 'Mails':
            stat_type_temp = 'Number of emails'
        elif stat_type == 'Mailboxes':
            stat_type_temp = 'number of mailboxes'
        elif stat_type == 'Indexed mails':
            stat_type_temp = 'Index collection'
        stat_xpath = f"//a[contains(@data-uib-tooltip, '{stat_type_temp}')]/span[@ng-bind='entity.value']"
        return self._driver.find_element_by_xpath(stat_xpath).text

    @WebAction(delay=2)
    def _click_view_jobs(self):
        """Clicks the view jobs link from the app page"""
        view_jobs_xpath = f"//span[text()='{self._admin_console.props['action.jobs']}']"
        self._driver.find_element_by_xpath(view_jobs_xpath).click()
        self._admin_console.wait_for_completion()

    @WebAction(delay=2)
    def _click_view_exports(self):
        """Clicks the view jobs link from the app page"""
        view_exports_xpath = "//span[text()='View exports']"
        self._driver.find_element_by_xpath(view_exports_xpath).click()
        self._admin_console.wait_for_completion()

    @WebAction(delay=2)
    def _create_inline_o365_plan(self):
        """Create Office365 plan from edit association modal"""

        self._driver.find_element_by_id('createNewPlanInline').click()
        self._admin_console.fill_form_by_id('planName', self.tcinputs['NewPlan'])
        self._driver.find_element_by_id(self.app_type.CREATE_OFFICE365_PLAN_BUTTON_ID.value).click()
        self._admin_console.wait_for_completion()

    @WebAction(delay=2)
    def _click_exclude_plan(self):
        """Clicks the exclude plan button on action menu"""
        exclude_plan_id = self.app_type.USERS_DELETE_ID.value
        self._driver.find_element_by_id(exclude_plan_id).click()

    @WebAction(delay=2)
    def _click_change_plan_individual(self, user=None, is_group=False):
        """Clicks the change plan button for individual user"""
        if not is_group:
            manage_id = self.app_type.USERS_MANAGE_ID.value
            change_plan_id = self.app_type.USER_CHANGE_OFFICE365_PLAN_ID.value
        else:
            manage_id = self.app_type.GROUP_MANAGE_ID.value
            change_plan_id = self.app_type.GROUP_CHANGE_OFFICE365_PLAN_ID.value
        if not user:
            user = self.users[0]
        self.__table.hover_click_actions_sub_menu(entity_name=user,
                                                  mouse_move_over_id=manage_id,
                                                  mouse_click_id=change_plan_id)

    @WebAction(delay=2)
    def _click_exclude_user(self, user=None, is_group=False):
        """Clicks the exclude user button under action context menu"""
        if not is_group:
            manage_id = self.app_type.USERS_MANAGE_ID.value
            disable_id = self.app_type.USERS_DISABLE_ID.value
        else:
            manage_id = self.app_type.GROUP_MANAGE_ID.value
            disable_id = self.app_type.GROUP_DISABLE_ID.value
        if not user:
            user = self.users[0]
        self.__table.hover_click_actions_sub_menu(entity_name=user,
                                                  mouse_move_over_id=manage_id,
                                                  mouse_click_id=disable_id)

    @WebAction(delay=2)
    def _click_include_user(self, user=None, is_group=False):
        """Clicks the exclude user button under action context menu"""
        if not is_group:
            manage_id = self.app_type.USERS_MANAGE_ID.value
            enable_id = self.app_type.USERS_ENABLE_ID.value
        else:
            manage_id = self.app_type.GROUP_MANAGE_ID.value
            enable_id = self.app_type.GROUP_ENABLE_ID.value
        if not user:
            user = self.users[0]
        self.__table.hover_click_actions_sub_menu(entity_name=user,
                                                  mouse_move_over_id=manage_id,
                                                  mouse_click_id=enable_id)

    @WebAction(delay=2)
    def _click_remove_content(self, user=None, is_group=False):
        """Clicks the exclude user button under action context menu"""
        if not is_group:
            manage_id = self.app_type.USERS_MANAGE_ID.value
            delete_id = self.app_type.USERS_DELETE_ID.value
        else:
            manage_id = self.app_type.GROUP_MANAGE_ID.value
            delete_id = self.app_type.GROUP_DELETE_ID.value
        if not user:
            user = self.users[0]
        self.__table.hover_click_actions_sub_menu(entity_name=user,
                                                  mouse_move_over_id=manage_id,
                                                  mouse_click_id=delete_id)

    @WebAction(delay=2)
    def _click_batch_change_plan(self):
        """Clicks the change plan button from Manage content"""
        change_plan_xpath = "//li[@id='batch-action-menu_moreContextMenu_ADD_PLAN']//a[@id='ADD_PLAN']"
        self._driver.find_element_by_xpath(change_plan_xpath).click()

    @WebAction(delay=2)
    def _click_manage(self):
        """Clicks on the manage button under more actions"""
        manage_xpath = "//li[@id='batch-action-menu_moreContextMenu_MANAGE']//a[@id='MANAGE']"
        self._driver.find_element_by_xpath(manage_xpath).click()

    @WebAction(delay=2)
    def _select_content(self, content_type):
        """Selects the auto discover content.
                Args:
                    content_type -- Type of auto discovery to select
                        Valid options:
                            ALL_USERS
                            ALL_PUBLIC_FOLDERS
                            ALL_OFFICE365_GROUPS
        """
        select_content_xpath = f"//li[@data-cv-menu-item-id='{content_type}']"
        self._driver.find_element_by_xpath(select_content_xpath).click()

    @PageService()
    def get_all_office365_apps(self):
        """
        List of all App Names
        """
        return self.__table.get_column_data(self._admin_console.props['label.name'])

    @PageService()
    def access_office365_app(self, app_name):
        """Accesses the Office365 app from the Office365 landing page

                Args:
                    app_name (basestring)  --  Name of the Office365 app to access

        """
        self.__table.access_link(app_name)

    @PageService()
    def check_if_app_exists(self, app_name):
        """Checks if the given app already exists

                Args:
                    app_name (basestring)  --  Name of the Office365 app to check
        """
        return self.__table.is_entity_present_in_column(
            self._admin_console.props['label.name'], app_name)

    @PageService()
    def create_office365_app(self, time_out=600, poll_interval=10):
        """
        Creates O365 App

        Args:

            time_out (int): Time out for app creation

            poll_interval (int): Regular interval for app creating check
        """

        # General Required details
        app_type = self.tcinputs['office_app_type']
        name = self.tcinputs['Name']

        if name in set(self.get_all_office365_apps()):
            raise CVWebAutomationException("App name already exists")

        self._create_app(app_type)

        if (app_type == Office365Apps.AppType.exchange_online
                or app_type == Office365Apps.AppType.one_drive_for_business
                or app_type == Office365Apps.AppType.share_point_online):
            # Required details for Exchange Online and OneDrive for Business
            index_server = None
            access_node = None
            client_group = None
            global_admin = None
            password = None
            application_id = None
            application_key_value = None
            azure_directory_id = None
            server_plan = self.tcinputs['ServerPlan']
            if 'IndexServer' in self.tcinputs:
                index_server = self.tcinputs['IndexServer']
            if 'AccessNode' in self.tcinputs:
                access_node = self.tcinputs['AccessNode']
            elif 'ClientGroup' in self.tcinputs:
                client_group = self.tcinputs['ClientGroup']
            if 'GlobalAdmin' in self.tcinputs:
                global_admin = self.tcinputs['GlobalAdmin']
                password = self.tcinputs['Password']
            elif 'application_id' in self.tcinputs:
                application_id = self.tcinputs['application_id']
                application_key_value = self.tcinputs['application_key_value']
                azure_directory_id = self.tcinputs['azure_directory_id']

            self._admin_console.fill_form_by_id('appName', name)

            self._dropdown.select_drop_down_values(values=[server_plan],
                                                   drop_down_id='planSummaryDropdown')

            # Check if infrastructure settings are inherited from plan or not
            if self._admin_console.check_if_entity_exists('xpath',
                                                          "//button[contains(@id, 'indexServers')]"
                                                          ):
                self._dropdown.select_drop_down_values(values=[index_server],
                                                       drop_down_id='createOffice365App_isteven-multi-select_#2568')
                if access_node:
                    if isinstance(access_node, list):
                        self._dropdown.select_drop_down_values(
                            values=access_node, drop_down_id='createOffice365App_isteven-multi-select_#5438')
                        self._enter_shared_path_for_multinode()
                        self._add_account_for_shared_path()
                    else:
                        self._dropdown.select_drop_down_values(
                            values=[access_node], drop_down_id='createOffice365App_isteven-multi-select_#5438')
                elif client_group:
                    self._dropdown.select_drop_down_values(
                        values=[client_group], drop_down_id='createOffice365App_isteven-multi-select_#5438')
                    self._enter_shared_path_for_multinode()
                    self._add_account_for_shared_path()
            if 'Region' in self.tcinputs:
                self._dropdown.select_drop_down_values(values=[self.tcinputs['Region']],
                                                       drop_down_id='createOffice365App_isteven-multi-select_#1167')
            if global_admin:
                self._admin_console.fill_form_by_id('globalUserName', global_admin)
                self._admin_console.fill_form_by_id('globalPassword', password)

                self._click_create_azure_ad_app()

                attempts = time_out // poll_interval
                while True:
                    if attempts == 0:
                        raise CVWebAutomationException('App creation exceeded stipulated time.'
                                                       'Test case terminated.')
                    self._LOG.info("App creation is in progress..")

                    self._admin_console.wait_for_completion()
                    self._check_for_errors()

                    # Check authorize app available

                    if self._admin_console.check_if_entity_exists("link",
                                                                  self._admin_console.props['action.authorize.app']):
                        break

                    time.sleep(poll_interval)
                    attempts -= 1
                self._verify_modern_authentication()
                self._authorize_permissions(global_admin, password)

            elif application_id:
                self._driver.find_element_by_xpath(
                    "//input[@type='radio' and @value='MANUALLY']"
                ).click()
                self._admin_console.fill_form_by_id('applicationId', application_id)
                self._admin_console.fill_form_by_id('secretAccessKey', application_key_value)
                self._admin_console.fill_form_by_id('tenantName', azure_directory_id)
            #Required changes for SharePoint Online V2
            if app_type == Office365Apps.AppType.share_point_online:
                self._create_app_principal()
                self._verify_sharepoint_service_account()
                self._verify_sharepoint_modern_authentication()
                time.sleep(5)
                self._verify_app_principal()
            self._admin_console.submit_form()

            self._admin_console.wait_for_completion()
            time.sleep(5)  # Wait for Discovery process to launch in proxy server

    @PageService()
    def delete_office365_app(self, app_name):
        """Deletes the office365 app
                Args:
                    app_name (str)  --  Name of the office365 app to delete

        """
        self.__table.access_action_item(app_name, self._admin_console.props['action.releaseLicense'])
        self._modal_dialog.click_submit()
        self.__table.access_action_item(app_name, self._admin_console.props['action.delete'])
        self._modal_dialog.type_text_and_delete(text_val='erase and reuse media',
                                                checkbox_id='deleteClientConfirmationChx')
        self._admin_console.wait_for_completion()

    @PageService()
    def get_azure_app_details(self):
        """Get the azure app details from Configuration Tab"""
        self._admin_console.select_configuration_tab()
        details = self.__table.get_table_data()
        return details

    @PageService()
    def get_details_from_discover_cache_info(self):
        """Get the details from Discover cache info component"""
        details = self._modal_dialog.get_details()
        self._modal_dialog.click_cancel()
        self._admin_console.wait_for_completion()
        return details

    @PageService()
    def wait_while_discovery_in_progress(self, time_out=600, poll_interval=60):
        """Waits for cache to get populated
        Args:
            time_out (int): Time out
            poll_interval (int): Regular interval for check
        """
        attempts = time_out // poll_interval
        if self._admin_console.check_if_entity_exists(
                'link', self._admin_console.props['action.refresh']):
            while attempts != 0:
                if self._admin_console.check_if_entity_exists(
                        'link', self._admin_console.props['action.refresh']):
                    self._LOG.info('Please wait. Discovery in progress...')
                    time.sleep(poll_interval)
                    self._admin_console.select_hyperlink(self._admin_console.props['action.refresh'])
                    self._admin_console.wait_for_completion()
                else:
                    break
                attempts -= 1
        if attempts == 0:
            raise CVWebAutomationException('Discovery exceeded stipulated time.'
                                           'Test case terminated.')

    @PageService()
    def add_user(self, users=None, plan=None):
        """
        Adds users to the office 365 app
        Args:
            users (list):   Users to be added to the app
            plan (str):     Office 365 Plan to be associated to the user
        """
        self._open_add_user_panel()
        self._admin_console.wait_for_completion()
        if plan:
            o365_plan = plan
        else:
            o365_plan = self.tcinputs['Office365Plan']
        try:
            self._dropdown.select_drop_down_values(
                values=[o365_plan],
                drop_down_id=self.app_type.O365_PLAN_DROPDOWN_ID.value)
        except (ElementNotInteractableException, NoSuchElementException):
            self.wait_while_discovery_in_progress()
            self._dropdown.select_drop_down_values(
                values=[o365_plan],
                drop_down_id=self.app_type.O365_PLAN_DROPDOWN_ID.value)
        if users:
            for user in users:
                search_element = self._driver.find_element_by_id('searchInput')
                if search_element.is_displayed():
                    self._admin_console.fill_form_by_id(element_id='searchInput', value=user)
                self.__table.select_rows([user.split("@")[0]])
            self._LOG.info(f'Users added: {users}')
        else:
            for user in self.users:
                search_element = self._driver.find_element_by_id('searchInput')
                if search_element.is_displayed():
                    self._admin_console.fill_form_by_id(element_id='searchInput', value=user)
                self.__table.select_rows([user.split("@")[0]])
            self._LOG.info(f'Users added: {self.users}')
        self._admin_console.submit_form()

    @PageService()
    def verify_plan_association(self, users=None, plan=None, is_group=False):
        """
        Verifies addition of users and check if plan is associated correctly
        Args:
            users (list):   Users to be verified
            plan (string):  Office 365 plan to which user should be associated
            is_group (bool): Should be set to true if verifying plan association for group
        """
        if is_group:
            self._admin_console.access_tab(self.app_type.CONTENT_TAB.value)
            column_name = 'Name'
            item_list = self.groups
        else:
            self._admin_console.access_tab(self.app_type.ACCOUNT_TAB.value)
            column_name = 'Email address'
            item_list = self.users
        if users:
            item_list = users
        if plan is None:
            plan = self.tcinputs['Office365Plan']
        self.__table.set_pagination(pagination_value=500)
        email_data = self.__table.get_column_data(column_name)
        try:
            plan_data = self.__table.get_column_data('Plan')
        except ValueError:
            plan_data = self.__table.get_column_data('plan')
        ui_association = dict(zip(email_data, plan_data))
        association = {user: plan for user in item_list}
        for key, value in association.items():
            if ui_association[key] != value:
                raise CVWebAutomationException(f"Office 365 Plan has not been "
                                               f"associated to each user/group correctly "
                                               f"--> {key} is associated to {ui_association[key]} "
                                               f"when it should be associated to {value}")
        self._LOG.info("Users/Groups and Office 365 plans have been associated correctly")

    @PageService()
    def verify_group_members(self, members):
        """
        Verifies that all the group members are present in the client
        Args:
            members (list): List of members belonging to the group
        """
        self._admin_console.access_tab(self.app_type.ACCOUNT_TAB.value)
        self.__table.set_pagination(pagination_value=500)
        user_list = self.__table.get_column_data(column_name='Email address')
        for member in members:
            if member not in user_list:
                raise CVWebAutomationException('Members of group not added to client')

    @PageService()
    def run_backup(self):
        """Runs backup by selecting all the associated users to the app"""
        if self.tcinputs['office_app_type'] == Office365Apps.AppType.exchange_online:
            self._select_all_users()
        elif self.tcinputs['office_app_type'] == Office365Apps.AppType.one_drive_for_business:
            for user in self.users:
                self._select_user(user)
        self._click_backup()
        self._modal_dialog.click_submit()
        job_details = self._job_details()
        self._LOG.info('job details: %s', job_details)
        if 'Number of files transferred' in job_details:
            self.backedup_mails = job_details['Number of files transferred']
        return job_details

    @PageService()
    def initiate_backup(self, users=None):
        """
        Initiates backup for the given users

        Args:
            users (list):   List of users to be backed up

        Returns:
            job_id (str): Job Id of the initiated backup job

        """
        if not users:
            if self.tcinputs['office_app_type'] == Office365Apps.AppType.exchange_online:
                self._select_all_users()
            elif self.tcinputs['office_app_type'] == Office365Apps.AppType.one_drive_for_business:
                for user in self.users:
                    self._select_user(user)
        else:
            for user in users:
                self._select_user(user)
        self._click_backup()
        self._modal_dialog.click_submit()
        job_id = self._get_job_id()
        return job_id

    @PageService()
    def add_azure_app_and_verify(self):
        """Adds a new azure app from configuration page and verifies it"""

        self._admin_console.select_configuration_tab()
        self._admin_console.select_hyperlink(f"{self._admin_console.props['dialog.azureapp.title']} ")
        self._admin_console.wait_for_completion()
        self._modal_dialog.click_submit()
        self._admin_console.wait_for_completion()
        self._authorize_permissions(self.tcinputs['GlobalAdmin'], self.tcinputs['Password'])
        self._modal_dialog.click_cancel()
        self._verify_app()

    @PageService()
    def add_service_account(self):
        """Adds a new Service account from configuration page and verifies it"""
        self._admin_console.select_configuration_tab()
        self._admin_console.select_hyperlink(self._admin_console.props['label.addServiceAccount'] + " ")
        self._admin_console.wait_for_completion()
        self._modal_dialog.click_submit()
        self._admin_console.wait_for_completion()

    @PageService()
    def verify_app_config_values(self, infra_pool=False):
        """Verifies the configuration values on app configuration page against the input values provided"""
        self._admin_console.select_configuration_tab()
        general_panel = PanelInfo(self._admin_console, title=self._admin_console.props['label.generalPane'])
        details = general_panel.get_details()
        if 'GlobalAdmin' in self.tcinputs:
            global_admin = details['Global Administrator'].split('\n')[0]
            if not global_admin == self.tcinputs['GlobalAdmin']:
                raise CVWebAutomationException("Global admin value is incorrect on configuration page")
        if not details['Use modern authentication'] == 'ON':
            raise CVWebAutomationException("Modern authentication toggle on configuration page is disabled.")

        infra_panel = PanelInfo(self._admin_console, title=self._admin_console.props['label.infrastructurePane'])
        infra_details = infra_panel.get_details()
        if self.tcinputs['ServerPlan'] not in infra_details['Server plan']:
            raise CVWebAutomationException("Server plan value is incorrect on configuration page")
        if not infra_pool:
            if 'AccessNode' in self.tcinputs:
                if isinstance(self.tcinputs['AccessNode'], list):
                    access_node_details = infra_details['Access nodes'].split('\n')
                    proxies = [node.strip() for node in access_node_details[1].split(',')]
                    user_account = access_node_details[2]
                    shared_jrd = access_node_details[3]
                    if shared_jrd.find("JobResults") == -1:
                        shared_jrd = shared_jrd + "\\JobResults"
                    if len(proxies) != len(self.tcinputs['AccessNode']):
                        raise CVWebAutomationException("Access node values on configuration page "
                                                       "do not match with entered values")
                    else:
                        for node in self.tcinputs['AccessNode']:
                            if node not in proxies:
                                raise CVWebAutomationException("Access node value is incorrect on configuration page")
                    if user_account != self.tcinputs['UserAccount']:
                        raise CVWebAutomationException("Account to access shared path is"
                                                       " incorrect on configuration page")
                    if shared_jrd != self.tcinputs["UNCPath"] + "\\JobResults":
                        raise CVWebAutomationException("Shared Job Results Directory is "
                                                       "incorrect on configuration page")
                elif self.tcinputs['AccessNode'] not in infra_details['Access nodes']:
                    raise CVWebAutomationException("Access node value is incorrect on configuration page")
            elif 'ClientGroup' in self.tcinputs:
                access_node_details = infra_details['Access nodes'].split('\n')
                client_group = access_node_details[1]
                user_account = access_node_details[2]
                shared_jrd = access_node_details[3]
                if client_group != self.tcinputs['ClientGroup']:
                    raise CVWebAutomationException('Client Group value is incorrect on configuration page')
                if user_account != self.tcinputs['UserAccount']:
                    raise CVWebAutomationException("Account to access shared path is incorrect on configuration page")
                if shared_jrd != self.tcinputs["UNCPath"] + "\\JobResults":
                    raise CVWebAutomationException("Shared Job Results Directory is incorrect on configuration page")

            if self.tcinputs['IndexServer'] not in infra_details['Index server']:
                raise CVWebAutomationException("Index server value is incorrect on configuration page")
            try:
                if not infra_details['Max streams'].split('\n')[0] == self.tcinputs['MaxStreams']:
                    raise CVWebAutomationException("Max streams value is incorrect on configuration page")
            except KeyError:
                if not infra_details['Max streams'].split('\n')[0] == self.app_type.MAX_STREAMS_COUNT.value:
                    raise CVWebAutomationException("Max streams value is incorrect on configuration page")
        else:
            if not infra_details['Max streams'].split('\n')[0] == self.app_type.INFRA_POOL_MAX_STREAMS.value:
                raise CVWebAutomationException("Max streams value is incorrect on configuration page")
            if self.app_type.INFRA_POOL_INDEX_SERVER.value not in infra_details['Index server']:
                raise CVWebAutomationException("Index server value is incorrect on configuration page")
            access_node_details = infra_details['Access nodes'].split('\n')
            client_group = access_node_details[1]
            user_account = access_node_details[2]
            shared_jrd = access_node_details[3]
            if client_group != self.app_type.INFRA_POOL_CLIENT_GROUP.value:
                raise CVWebAutomationException('Client Group value is incorrect on configuration page')
            if user_account != self.app_type.INFRA_POOL_SHARED_ACCOUNT.value:
                raise CVWebAutomationException("Account to access shared path is incorrect on configuration page")
            if shared_jrd != self.app_type.INFRA_POOL_SHARED_DIR.value:
                raise CVWebAutomationException("Shared Job Results Directory is incorrect on configuration page")
        client_readiness = infra_details['Client readiness']
        if 'Unknown' in client_readiness or 'Not available' in client_readiness:
            self._click_show_details_for_client_readiness()
            self._admin_console.wait_for_completion()
            client_readiness = self._get_client_readiness_value()
            self._driver.find_element_by_link_text(self.tcinputs['Name']).click()
            self._admin_console.wait_for_completion()
        if isinstance(client_readiness, list):
            for value in client_readiness:
                if 'Ready' not in value:
                    raise CVWebAutomationException('Client is not ready.')
        else:
            if 'Ready' not in client_readiness:
                raise CVWebAutomationException('Client is not ready.')

        if 'GlobalAdmin' in self.tcinputs:
            status = self.get_azure_app_details()['Status']
            for app_status in status:
                if app_status != 'Authorized':
                    raise CVWebAutomationException('Azure App is not authorized')

    @PageService()
    def modify_app_config_values(self,
                                 new_global_admin=None,
                                 new_admin_pwd=None,
                                 new_server_plan=None,
                                 new_stream_count=None,
                                 new_index_server=None,
                                 new_access_node=None,
                                 new_shared_path=None,
                                 new_user_account=None,
                                 new_password=None):
        """
        Modifies the values set in the configuration page to the new values given as arguments
        Args:
            new_global_admin:   New value to be set for global admin
            new_admin_pwd:      Password for new value of global admin
            new_server_plan:    New value to be set for server plan
            new_stream_count:   New vale to be set for 'Max streams'
            new_index_server:   New value to be set for Index Server
            new_access_node:    New value(s) to be set for access node
            new_shared_path:    New value to be set for shared path
            new_user_account:   Local system account to access shared path
            new_password:       Password for local system account
        """
        self._admin_console.select_configuration_tab()
        if new_global_admin:
            self._edit_global_admin(new_global_admin, new_admin_pwd)
        if new_server_plan:
            self._edit_server_plan(new_server_plan)
        if new_stream_count:
            self._edit_stream_count(new_stream_count)
        if new_index_server:
            self._edit_index_server(new_index_server)
        if new_access_node:
            self._edit_access_node(new_shared_path, new_user_account, new_password, new_access_node)
        elif new_shared_path:
            self._edit_access_node(new_shared_path, new_user_account, new_password)

    @PageService()
    def is_app_associated_with_plan(self):
        """
        Verifies if client is associated with plan
        Returns: True if client is listed in the associated entities page of plans
                 else False
        """
        self._admin_console.navigator.navigate_to_plan()
        self._plans.select_plan(self.tcinputs['ServerPlan'])
        self._admin_console.access_tab('Associated entities')
        if not self.check_if_app_exists(self.tcinputs['Name']):
            raise CVWebAutomationException('The client has not been associated with Server Plan')

    @PageService()
    def run_restore(self, mailbox=True):
        """Runs the restore by selecting all users associated to the app

                Args:
                    mailbox  (Boolean)  --  Whether to restore mailbox or messages
        """
        self._select_all_users()
        self._click_restore(mailbox)
        self._admin_console.wait_for_completion()
        self._admin_console.submit_form(wait=False)
        job_details = self._job_details()
        self._LOG.info('job details: %s', job_details)
        self.restored_mails = job_details['Successful messages']

    @PageService()
    def verify_backedup_mails(self):
        """Verifies whether backed up mails number is correct"""

        if not self.backedup_mails > self.restored_mails:
            raise CVWebAutomationException("Number of mails in the mailbox and "
                                           "number of backed up messages on command center are not matching")

    @PageService()
    def verify_connection(self):
        """Verifies the Azure app connection from configuration tab"""
        self._admin_console.select_configuration_tab()
        self._admin_console.select_hyperlink("Verify connection ")
        self._verify_connection_text()
        self._modal_dialog.click_submit()

    @PageService()
    def verify_added_users(self, users=None):
        """Verifies the users added to the app as against the input file"""
        user_list = self.__table.get_column_data(column_name='Email address')
        if users:
            for user in users:
                if user not in user_list:
                    raise CVWebAutomationException("User list on the app page does not"
                                                   " match the input file user list")
            self._LOG.info(f'Added users verified: {users}')
        else:
            if not collections.Counter(user_list) == collections.Counter(self.users):
                raise CVWebAutomationException("User list on the app page does "
                                               "not match the input file user list")
            self._LOG.info(f'Added users verified: {self.users}')

    @PageService()
    def verify_added_groups(self, groups=None):
        """Verifies the groups added to the app"""
        self._admin_console.access_tab(self.app_type.CONTENT_TAB.value)
        group_list = self.__table.get_column_data(column_name='Name')
        if groups:
            for group in groups:
                if group not in group_list:
                    raise CVWebAutomationException("Group list on the app page does "
                                                   "not match the input file group list")
            self._LOG.info(f'Added groups verified: {groups}')
        else:
            if not collections.Counter(group_list) == collections.Counter(self.groups):
                raise CVWebAutomationException("Group list on the app page does "
                                               "not match the input file group list")
            self._LOG.info(f'Added groups verified: {self.groups}')

    @PageService()
    def change_office365_plan(self, user, plan, is_group=False):
        """
        Changes the Office 365 Plan for the given user
        Args:
            user (str):         User for which plan has to be changed
            plan (str):         The value of the new plan
            is_group (bool):    Whether plan has to changed for user or group
        """
        self._select_user(user, is_group=is_group)
        self._click_change_plan_individual(user=user, is_group=is_group)
        self._admin_console.wait_for_completion()
        self._dropdown.select_drop_down_values(
            values=[plan],
            drop_down_id=self.app_type.O365_PLAN_DROPDOWN_ID.value)
        self._modal_dialog.click_submit()
        self._admin_console.wait_for_completion()
        self._LOG.info(f'Office 365 Plan changed to value: {plan}')

    @PageService()
    def change_plan(self):
        """Changes the exchange plan for office365 user"""

        # Changing plan for all users using more actions link

        self._select_all_users()
        self._click_more_actions()
        self._click_manage()
        time.sleep(2)
        self._click_batch_change_plan()
        self._admin_console.wait_for_completion()
        self._add_plan_and_verify(change=True)

        # Changing plan for an individual user

        self._click_change_plan_individual()
        self._add_plan_and_verify()

    @PageService()
    def run_ci_job(self):
        """Runs the content indexing job and verifies job completes successfully"""
        self._select_all_users()
        self._click_more_actions()
        self._click_ci_path()
        self._modal_dialog.click_submit()
        job_details = self._job_details()
        self.indexed_mails = job_details['Successful messages']

    @WebAction(delay=2)
    def _click_ci_path(self):
        """Clicks the content indexing job link"""
        ci_xpath = "//li[@id='batch-action-menu_moreContextMenu_RUN_CONTENT_INDEXING_GRID']//a[@id='RUN_CONTENT_INDEXING_GRID']"
        self._driver.find_element_by_xpath(ci_xpath).click()

    @PageService()
    def verify_app_stats(self):
        """Gets the stats from app page

                Raises:
                    Exception if stats do not match that of input file
        """
        self.app_stats_dict['Mails'] = self._get_app_stat(stat_type='Mails')
        self.app_stats_dict['Mailboxes'] = self._get_app_stat(stat_type='Mailboxes')
        self.app_stats_dict['Indexed Mails'] = self._get_app_stat(stat_type='Indexed mails')

        if not str(len(self.users)) == self.app_stats_dict['Mailboxes']:
            raise CVWebAutomationException('Mailboxes app stat is not matching with input file users')
        if not self.app_stats_dict['Mails'] == self.backedup_mails:
            raise CVWebAutomationException('Mails app stat is not matching with backed up files from job details')
        if not self.app_stats_dict['Indexed Mails'] == self.indexed_mails:
            raise CVWebAutomationException('Indexed Mails app stat is not matching with Indexed files from job details')

    @PageService()
    def remove_from_content(self):
        """Removes the content from app

                Raises:
                    Exception if user removal is unsuccessful
        """
        # Get the list of existing users
        user_list = self.__table.get_column_data(column_name=self._admin_console.props['column.email'])
        # Now remove one user from the app

        self._click_remove_content()
        self._modal_dialog.click_submit()
        self._admin_console.wait_for_completion()
        # Verify that user was removed
        user_list_new = self.__table.get_column_data(column_name=self._admin_console.props['column.email'])
        diff_list = list(set(user_list) - set(user_list_new))
        if diff_list and diff_list[0] == self.users[0]:
            self._LOG.info(f'user {diff_list[0]} was removed from the app')
            self.users.remove(diff_list[0])
        else:
            raise CVWebAutomationException('There was an error in removing user')

    @PageService()
    def exclude_user(self, user=None, is_group=False):
        """
        Excludes the given user from backup

        Args:
            user (str):         User which has to be disabled
            is_group (bool):    Whether user/group is to be disabled

        """
        if self.tcinputs['office_app_type'] == Office365Apps.AppType.one_drive_for_business:
            if user:
                self._select_user(user, is_group=is_group)
            else:
                self._select_user(self.users[0], is_group=is_group)
        self._click_exclude_user(user=user, is_group=is_group)
        self._admin_console.wait_for_completion()
        self._modal_dialog.click_submit()
        self._admin_console.wait_for_completion()
        self._LOG.info(f'User excluded from backup: {user}')

    @PageService()
    def include_in_backup(self, user=None, is_group=False):
        """
        Includes the given user to backup

        Args:
            user (str):         User which has to be enabled
            is_group (bool):    Whether user/group is to be enabled

        """
        if user:
            self._select_user(user, is_group=is_group)
        else:
            self._select_user(self.users[0], is_group=is_group)
        self._click_include_user(user=user, is_group=is_group)
        self._admin_console.wait_for_completion()
        self._modal_dialog.click_submit()
        self._admin_console.wait_for_completion()
        self._LOG.info(f'User included to backup: {user}')

    @PageService()
    def view_jobs(self):
        """Verifies the view jobs page from the app"""
        self._click_view_jobs()

    @PageService()
    def open_active_jobs_tab(self):
        """Opens the active jobs page for the client"""
        self._admin_console.access_tab(self._admin_console.props['label.activeJobs'])

    @PageService()
    def select_content(self, content_type='All Users'):
        """Selects the auto discover content

                Args:
                    content_type (str)  --  Type of auto discovery content
                        Valid Options:
                            All Users
                            All Public Folders
                            All O365 group mailboxes
                        Default:
                            All Users

        """
        content_type_dict = {"All Users": "ALL_USERS",
                             "All Public Folders": "ALL_PUBLIC_FOLDERS",
                             "All O365 group mailboxes": "ALL_OFFICE365_GROUPS"
                             }

        self._admin_console.access_tab(self._admin_console.props['label.content'])
        self._click_add_autodiscover()
        self._select_content(content_type=content_type_dict[content_type])
        self._admin_console.wait_for_completion()
        self._add_plan_and_verify()

    @PageService()
    def deselect_content(self, content_type='All Users'):
        """Disables the auto discover content

                Args:
                    content_type (str)  --  Type of auto discovery content
                        Valid Options:
                            All Users
                            All Public Folders
                            All O365 group mailboxes
                        Default:
                            All Users

        """
        self._admin_console.access_tab(self._admin_console.props['label.content'])
        self.__table.access_action_item(content_type, self._admin_console.props['label.manage'])
        self._click_exclude_plan()
        self._modal_dialog.click_submit()
        self._admin_console.wait_for_completion()

    @PageService()
    def verify_content_association(self, content_list):
        """Verifies whether all user association is properly added in command center

                Args:
                    content_list (List)  --  List of users or AD groups

        """
        self._admin_console.access_tab(self._admin_console.props['label.mailboxes'])
        self._admin_console.wait_for_completion()
        users_from_local_db = content_list
        local_db_count = len(users_from_local_db)
        cc_user_count = self.__table.get_total_rows_count()
        if cc_user_count == 0:
            cc_user_count = len(self.__table.get_column_data(self._admin_console.props['column.email']))
        self._LOG.info("Total rows count: %s" % cc_user_count)
        self._LOG.info("User from local db count : %s" % local_db_count)
        if not local_db_count == int(cc_user_count):
            self._LOG.info("Users from local db: %s" % users_from_local_db)
            raise CVWebAutomationException("All User association had mismatch of number of Users")

    @PageService()
    def verify_content_deassociation(self):
        """Verifies whether the content is deassociated"""
        self._admin_console.access_tab(self._admin_console.props['label.mailboxes'])
        try:
            cc_user_count = str(self.__table.get_total_rows_count())
            if cc_user_count != '0':
                raise CVWebAutomationException(
                    "Auto discovery de-association has some issues. Users/groups were not removed.")
        except NoSuchElementException:
            self._LOG.info(
                "NoSuchElementException caught. In this case it means content has been deassociated.Positive")

    @PageService()
    def verify_cache_update_time(self, ui_time, db_time):
        """
        Verifies the cache update time from db against time displayed in the UI
        Args:
            ui_time:    Time displayed in UI - format: Nov 3, 3;01 PM
            db_time:    Time stored in discovery cache - format: epoch time
        """
        epoch_time = int(time.mktime(datetime.datetime.strptime(
            ui_time, "%b %d, %I:%M %p").replace(
            year=datetime.datetime.now().year).timetuple()))
        if abs(db_time - epoch_time) > 900:
            raise CVWebAutomationException('Cache is not showing latest update time')
        else:
            self._LOG.info('Latest cache update time verified')

    @PageService()
    def delete_plan(self, plan_name):
        """Deletes the plan"""
        self._plans.delete_plan(plan_name)

    @WebAction(delay=2)
    def _get_user_status(self, user, is_group=False):
        """
        Gets the status of the user

        Args:
            user (str):         User whose status we need to get
            is_group (bool):    Whether user/group

        Returns:
            status (str):       Status of the user
                                Valid values - Active, Disabled, Deleted

        """
        columns = self.__table.get_visible_column_names()
        if 'Status' not in columns:
            self.__table.display_hidden_column('Status')
        if not is_group:
            self._admin_console.access_tab(self.app_type.ACCOUNT_TAB.value)
            self.__table.apply_filter_over_column('Email address', user)
            status = self.__table.get_table_data().get('Status')[0]
            self.__table.clear_column_filter('Email address')
        else:
            self._admin_console.access_tab(self.app_type.CONTENT_TAB.value)
            self.__table.apply_filter_over_column('Name', user)
            status = self.__table.get_table_data().get('Status')[0]
            self.__table.clear_column_filter('Name')
        return status

    @PageService()
    def verify_user_status(self, status, user, is_group=False):
        """
        Verify the status of the user in the Command Center

        Args:
            status (str):   Status of the user
                            Valid values - Active, Disabled, Deleted
            user:           User whose status has to be checked
            is_group:       Whether user/group

        """
        if status == constants.StatusTypes.DELETED.value:
            if is_group:
                self._admin_console.access_tab(self.app_type.CONTENT_TAB.value)
                self.__table.apply_filter_over_column('Name', user)
                if self.__table.get_total_rows_count() != 0:
                    raise CVWebAutomationException(f'Group {user} has not been deleted')
                ui_status = constants.StatusTypes.DELETED.value
                self.__table.clear_column_filter('Name')
            else:
                self._admin_console.access_tab(self.app_type.ACCOUNT_TAB.value)
                self.__table.apply_filter_over_column_selection('Status', status)
                ui_status = self._get_user_status(user=user, is_group=is_group)
                self.__table.clear_column_filter('Status')
        else:
            ui_status = self._get_user_status(user=user, is_group=is_group)
        if ui_status != status:
            raise CVWebAutomationException('User/Group Status Verification Failed')
        self._LOG.info(f'Status of {user} Verified: {status}')

    @PageService()
    def verify_no_users_configured(self, is_group=False):
        """
        Verify that no users are configured

        Args:
            is_group (bool):    Whether to check User/Content Tab

        """
        if is_group:
            self._admin_console.access_tab(self.app_type.CONTENT_TAB.value)
        else:
            self._admin_console.access_tab(self.app_type.ACCOUNT_TAB.value)
        if self.__table.get_total_rows_count() != 0:
            raise CVWebAutomationException('Users/Groups are configured for backup')
