# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""
This module provides the function or operations that can be performed on the
Object storage Accounts listing page page


    add_ibm_client()        --  adds IBM Object Storage

    add_ail_client()        -- adds Ali Object Storage
"""
from abc import abstractmethod

from Web.Common.page_object import (
    PageService,
    WebAction
)
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.Components.panel import ModalPanel
from Web.AdminConsole.adminconsole import AdminConsole


class ObjectStorage:
    """Class for Object storage Accounts page"""

    def __init__(self, admin_console):
        """
        Args:
            admin_console (AdminConsole): adminconsole base object
        """
        self.__admin_console = admin_console
        self.__table = Table(self.__admin_console)
        self.__admin_console._load_properties(self)

    @PageService()
    def access_account(self, name):
        """
        Access account details page
        Args:
            name (str): Object storage name
        """
        self.__table.view_by_title('All')
        self.__table.access_link(name)

    @PageService()
    def add_azure_gen2_client(
            self,
            name,
            proxy_client,
            backup_plan,
            account_name,
            account_pwd,
            url='dfs.core.windows.net'
    ):
        """
        Creates Azure gen2 client
        Args:
            name: client name
            proxy_client: proxy client
            backup_plan: backup plan
            account_name: cloud storage account
            account_pwd: storage account password
            url: host url
        """
        self.__table.access_toolbar_menu('ADD_CLOUD_STORAGE')
        _CreateAzureGen2(self.__admin_console).create_client(
            name,
            proxy_client,
            backup_plan,
            account_name,
            account_pwd,
            url
        )
        self.__admin_console.check_error_message()

    @PageService()
    def add_ali_client(
            self,
            name,
            proxy_client,
            backup_plan,
            access_key,
            secret_key,
            url
    ):
        """Creates Ali Client
            Args:
                name (str)      -- name of client to be created
                proxy_client (str)  --name of proxy client
                backup_plan(str)    --name of backup plan
                access_key (str)    -- access key of Ali Cloud account
                secret_key (str)    -- secret key of Ali Cloud account
                url (str)           -- Region to which the account is pointed

            Returns:
                object - Object storage if client is created successfully
        """
        self.__table.access_toolbar_menu('ADD_CLOUD_STORAGE')
        _CreateAliClient(self.__admin_console).create_client(
            name,
            proxy_client,
            backup_plan,
            access_key,
            secret_key,
            url
        )
        self.__admin_console.check_error_message()

    @PageService()
    def add_ibm_client(
            self,
            name,
            proxy_client,
            backup_plan,
            access_key,
            secret_key,
            url
    ):
        """Creates IBM Client
            Args:
                name (str)      -- name of client to be created
                proxy_client (str)  --name of proxy client
                backup_plan(str)    --name of backup plan
                access_key (str)    -- access key of IBM Cloud account
                secret_key (str)    -- secret key of IBM Cloud account
                url (str)           -- Region to which the account is pointed

            Returns:
                object - Object storage if client is created successfully
        """
        self.__table.access_toolbar_menu('ADD_CLOUD_STORAGE')
        _CreateIbmClient(self.__admin_console).create_client(
            name,
            proxy_client,
            backup_plan,
            access_key,
            secret_key,
            url
        )
        self.__admin_console.check_error_message()

    @PageService()
    def get_sla(self, client_name):
        """
        Gets sla value for client
        Args:
            client_name (str): Object storage name
        Returns:
            (str): SLA value
        """
        self.__table.search_for(client_name)
        sla_column = self.__table.get_column_data(self.__admin_console.props['label.nav.sla'])
        return sla_column[0]

    @PageService()
    def view_by_vendor(self, name):
        """
        Filter table view by vendor
        Args:
            name (str): vendor name
        """
        self.__table.view_by_title(name)

    @PageService()
    def get_clients(self):
        """
        gets all visbile client names
        """
        return self.__table.get_column_data(self.__admin_console.props['label.name'])


class _NewClient(ModalPanel):
    """class for object storage client creation"""

    @property
    @abstractmethod
    def vendor_name(self):
        """Override this method and implement it as a variable whose
        value is set to the host url as per the vendor"""
        raise NotImplementedError

    @WebAction()
    def _select_vendor(self):
        """Selects vendor"""

        self._admin_console.access_sub_menu(self.vendor_name)

    @WebAction()
    def _set_client_name(self, name):
        """
        set Client name
        Args:
            name (str): Client name
        """
        self._admin_console.fill_form_by_id('cloudStorageName', name)

    @WebAction()
    def _set_access_name(self, name):
        """
        set Cloud storage account name
        Args:
            name (str): account name
        """
        self._admin_console.fill_form_by_id('cappsInstanceUserName', name)

    @WebAction()
    def _set_access_pwd(self, key):
        """
        set Cloud storage account key or secret key
        Args:
            key (str): account key
        """
        self._admin_console.fill_form_by_id('cappsInstancePassword', key)

    @WebAction()
    def _set_user_name(self, name):
        """
        set Cloud storage name
        Args:
            name (str): Client name
        """
        self._admin_console.fill_form_by_id('cappsInstanceUserName', name)

    @WebAction()
    def _set_host_url(self, url):
        """
        set Cloud storage host url
        Args:
            url (str): Cloud storage host url
        """
        self._admin_console.fill_form_by_id('hostURL', url)

    @WebAction()
    def _set_access_node(self, name):
        """
        set Cloud storage access node
        Args:
            name (str): Plan name
        """
        self._dropdown.select_drop_down_values(drop_down_id='addCloudStorageContent_isteven-multi-select_#2484',
                                               values=[name])

    @WebAction()
    def _set_plan(self, name):
        """
        set Cloud storage name
        Args:
            name (str): Plan name
        """
        self._dropdown.select_drop_down_values(drop_down_id='planSummaryDropdown', values=[name])


class _CreateAzureGen2(_NewClient):
    """
    used to create Azure Gen2 client
    """

    @property
    def vendor_name(self):
        """For vendor selection
        """
        return self._admin_console.props['label.clientType.azureDataLake']

    @PageService()
    def create_client(
            self, name, proxy_client,
            backup_plan,
            account_name,
            account_pwd,
            url='dfs.core.windows.net'
    ):
        """
        Creates Azure gen2 client
        Args:
            name            (str): client name
            proxy_client    (str): proxy client
            backup_plan     (str): backup plan
            account_name    (str): cloud storage account
            account_pwd     (str): storage account password
            url             (str): host url
        """

        self._select_vendor()
        self._set_client_name(name)
        if url != 'dfs.core.windows.net':
            self._set_host_url(url)
        self._set_user_name(account_name)
        self._set_access_pwd(account_pwd)
        self._set_access_node(proxy_client)
        self._set_plan(backup_plan)
        self.submit()


class _CreateAliClient(_NewClient):
    """
    Class used to create Ali client
    """

    @property
    def vendor_name(self):
        """Returns:

            object-  host url of the vendor
        """
        return self._admin_console.props['label.clientType.alibabaOSS']

    @PageService()
    def create_client(
            self, name, proxy_client,
            backup_plan,
            access_key,
            secret_key,
            url
    ):
        """
        Creates Ali Client
        Args:
            name (str)      -- name of client to be created
            proxy_client (str)  --name of proxy client
            backup_plan(str)    --name of backup plan
            access_key (str)    -- access key of Ali Cloud account
            secret_key (str)    -- secret key of Ali Cloud account
            url (str)           -- Region to which the account is pointed

        Returns:
            object - Object storage if client is created successfully

        """
        self._select_vendor()
        self._set_client_name(name)
        self._set_host_url(url)
        self._set_user_name(access_key)
        self._set_access_pwd(secret_key)
        self._set_access_node(proxy_client)
        self._set_plan(backup_plan)
        self.submit()


class _CreateIbmClient(_NewClient):
    """
        Class used to create IBM client
    """

    @property
    def vendor_name(self):
        """
        Returns:
            object-  host url of the vendor
        """
        return self._admin_console.props['label.clientType.ibmCOS']

    @PageService()
    def create_client(
            self, name, proxy_client,
            backup_plan,
            access_key,
            secret_key,
            url
    ):
        """
               Creates IBM Client
               Args:
                   name (str)      -- name of client to be created
                   proxy_client (str)  --name of proxy client
                   backup_plan(str)    --name of backup plan
                   access_key (str)    -- access key of IBM Cloud account
                   secret_key (str)    -- secret key of IBM Cloud account
                   url (str)           -- Region to which the account is pointed

               Returns:
                   object - Object storage if client is created successfully

        """
        self._select_vendor()
        self._set_client_name(name)
        self._set_host_url(url)
        self._set_user_name(access_key)
        self._set_access_pwd(secret_key)
        self._set_access_node(proxy_client)
        self._set_plan(backup_plan)
        self.submit()
