# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""
This module provides the function or operations that can be performed on the
Object storage Account details page
"""
from Web.Common.page_object import (
    PageService,
    WebAction
)
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.Components.panel import PanelInfo
from Web.AdminConsole.Components.panel import ModalPanel, Backup
from Web.AdminConsole.Components.browse import Browse
from Web.AdminConsole.Components.dialog import ModalDialog
from Web.AdminConsole.adminconsole import AdminConsole

class ObjectDetails:
    """Class for Object storage Accounts details page"""

    def __init__(self, admin_console):
        """
        Args:
            admin_console (AdminConsole): adminconsole base object
        """
        self.__admin_console = admin_console
        self.driver = admin_console.driver
        self.__table = Table(self.__admin_console)
        self.__panel = PanelInfo(admin_console)
        self.__browse = Browse(self.__admin_console)

    @WebAction()
    def __click_add_content_group(self):
        """Add content group"""
        self.__admin_console.access_menu(self.__admin_console.props['pageHeader.addContentGroup'])

    @PageService
    def get_details(self):
        """returns details of client"""
        return self.__panel.get_details()

    @PageService()
    def get_content_groups(self):
        """
        gets all visbile client names
        """
        return self.__table.get_column_data(self.__admin_console.props['label.name'])

    @PageService()
    def create_content_group(self, name, plan, content_path):
        """
        Create content group
        Args:
            name (str): content group name
            plan (str): plan
            content_path (str): content

        """
        self.__click_add_content_group()
        _AddContentGroup(self.__admin_console).add(name, plan, content_path)
        self.__admin_console.check_error_message()

    @PageService()
    def submit_backup(self, group_name, backup_type=Backup.BackupType.INCR):
        """
        submits backup job
        Args:
            group_name    (str)             : content group name
            backup_type (Backup.BackupType) : backup type

        Returns
            (str) -- Backup job id

        """
        self.__table.access_context_action_item(
            group_name,
            self.__admin_console.props['action.backup']
        )
        return Backup(self.__admin_console).submit_backup(backup_type)

    @PageService()
    def access_restore(self, group_name):
        """
        Args:
            group_name    (str): content group name
        """
        self.__table.access_context_action_item(
            group_name,
            self.__admin_console.props['action.restore']
        )
        self.__admin_console.wait_for_completion()

    @PageService()
    def submit_restore(self, file_folders):
        """
        Args:
            group_name    (str): content group name
            file_folders (list): list of files and folders to be restored

        Returns
          (str):  Restore options object

        """
        self.__browse.clear_all_selection()
        self.__browse.select_for_restore(file_folders)
        self.__browse.submit_for_restore()
        return RestoreOptions(self.__admin_console)

    @PageService()
    def delete_account(self):
        """Delete account"""
        self.__admin_console.access_menu_from_dropdown(
            self.__admin_console.props['action.delete']
        )
        _dialog = ModalDialog(self.__admin_console)
        _dialog.type_text_and_delete('DELETE')
        self.__admin_console.wait_for_completion()


class RestoreOptions(ModalPanel):

    @WebAction()
    def __click_overwrite_newer(self):
        """clicks overwrite only newer files radio button"""
        self._admin_console.select_radio(
            self._admin_console.props['label.overwriteIfFileInBackupIsNewer']
        )

    @WebAction()
    def __set_disk_path(self, path):
        """
        Sets disk path
        Args:
            path (str): path to restore
        """
        self._driver.find_element_by_xpath(
        '//input[contains(@data-ng-model, "destinationPathRTD")]'
        ).send_keys(path)

    def in_place(self, overwrite=True):
        """
        submits inplace restore
        Args:
            overwrite (bool): Overwrite files unconditionally
        Returns
            (str): Restore Job id

        """
        if not overwrite:
            self.__click_overwrite_newer()
        self.submit(wait_for_load=False)
        _jobid = self._admin_console.get_jobid_from_popup()
        self._admin_console.wait_for_completion()
        return _jobid

    def out_of_place(self, destination_client, dest_path, overwrite=True):
        """
        submits inplace restore
        Args:
            destination_client (str): client name
            dest_path (str)         : desitnation path
            overwrite (bool)        : Overwrite files unconditionally

        Returns
            (str): Restore Job id

        """
        self.access_tab(self._admin_console.props['label.OOPRestore'])
        self._dropdown.select_drop_down_values(0, [destination_client])
        if not overwrite:
            self.__click_overwrite_newer()
        self._admin_console.fill_form_by_id('destinationPathOOP', dest_path),
        self.submit(wait_for_load=False)
        _jobid = self._admin_console.get_jobid_from_popup()
        self._admin_console.wait_for_completion()
        return _jobid

    def to_disk(self, destination_client, dest_path, overwrite=True):
        """
        submits restore to disk
        Args:
            destination_client (str): client name
            dest_path (str)         : desitnation path
            overwrite (bool)        : Overwrite files unconditionally

        Returns
            (str): Restore Job id

        """
        self.access_tab(self._admin_console.props['label.rtdRestore'])
        self._dropdown.select_drop_down_values(1, [destination_client])
        if not overwrite:
            self.__click_overwrite_newer()
        self.__set_disk_path(dest_path)
        self.submit(wait_for_load=False)
        _jobid = self._admin_console.get_jobid_from_popup()
        self._admin_console.wait_for_completion()
        return _jobid


class _AddContentGroup(ModalPanel):

    @WebAction()
    def __click_plus(self):
        """clicks plus near custom content"""
        self._driver.find_element_by_xpath("//i[contains(@class,'glyphicon-plus')]").click()

    @WebAction()
    def __set_custom_path(self, path):
        """
        sets custom path
        Args:
            path (str): content path
        """
        elem = self._driver.find_element_by_xpath(
            "//input[contains(@class, 'content-custom-path')]"
        )
        elem.send_keys(path)

    def add(self, name, plan, content_path):
        self._admin_console.fill_form_by_id('subclientName', name)
        self._dropdown.select_drop_down_values(0, [plan])
        self.__set_custom_path(content_path)
        self.__click_plus()
        self.submit()
