#!/usr/bin/env python

"""
This module provides all the methods that can be done of the NAS_File_Servers page.


Classes:

    NAS_File_Servers() ---> AdminPage() ---> LoginPage --->
    AdminConsoleBase() ---> object()


NAS_File_Servers  --  This class contains all the methods for action in NAS_File_Servers page and
                  is inherited by other classes to perform NAS Client related actions

    Functions:

    select_nas_client()        		--  Opens the server with the given name
    action_jobs()                	--  Opens the job details page for the chosen NAS File Server
    action_add_software()			--	Allows user to select different iDAs under NAS Client
    action_release_license()        --  Allows user to Releases Licenses consumed by NAS Client
    action_send_logs()           	--  Sends logs of a server with the specified name
    action_backup()					--	Runs Backup of specified subClient of NAS File Server
    action_restore()				--	Runs Restore from defaultbackupSet of NAS File Server
    delete_client()                 --  Deletes NAS Client.
    add_nas_client()                --  Add a new NAS Client.

"""
from enum import Enum
from Web.AdminConsole.Components.panel import Backup
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.Components.panel import PanelInfo, DropDown, ModalPanel
from Web.Common.page_object import PageService, WebAction
from Web.AdminConsole.FileServerPages.network_share import NFS, CIFS
from selenium.webdriver.common.keys import Keys

class Vendor(Enum):
    """
    List of all Vendors.
    """
    DELL_EMC_ISILON = "Dell EMC Isilon"


class NASFileServers:
    """
    This class contains all the methods for action in NAS_File_Servers page
    """
    def __init__(self, admin_console):
        self.admin_console = admin_console
        self.driver = admin_console.driver
        self.admin_console._load_properties(self)
        self.__table = Table(self.admin_console)
        self.__drop_down = DropDown(self.admin_console)
        self.__panel = PanelInfo(self.admin_console)
        self.__modal_panel = ModalPanel(self.admin_console)

    @staticmethod
    def _create_array(admin_console, vendor, array_details):
        """
        Creates and returns an instance of the array vendor.

        Args:

            admin_console   (obj)   --  Instance of admin_console object

            vendor          (enum)  --  Vendor name, values can be found under class Vendor.

            array_details   (dict)  :   The dictionary of array details.
            Dictionary contains the keys array_name, control_host, username and password.

                array_name      (basestring)    :   Name of the array.

                control_host    (basestring)    :   Control host name if applicable to the array.

                username        (basestring)    :   Username for the array, defined in config.json.

                password        (basestring)    :   Password for the array, defined in config.json.

        """
        if vendor == Vendor.DELL_EMC_ISILON:
            return DellEMCIsilonArray(admin_console, array_details)
        return
    
    @WebAction()
    def __get_access_nodes(self, accessnode_ma):
        """Returns MA count and opt"""
        ma_cnt = 0
        opt = 0
        element = self.driver.find_element_by_id('detectMa')
        for option in element.find_elements_by_tag_name('option'):
            ma_cnt = ma_cnt + 1
            cmptxt = option.text
            cmptxt = cmptxt[17: len(cmptxt) - 14]
            if not accessnode_ma and str(cmptxt) == str(accessnode_ma):
                opt = ma_cnt
        return ma_cnt, opt

    @WebAction()
    def __get_text(self):
        """Gets text"""
        return self.driver.find_element_by_tag_name("html").text

    @WebAction()
    def __click_backup(self):
        """Clicks backup"""
        self.driver.find_element_by_link_text("Backup").click()

    @WebAction()
    def __click_subclient(self, subclient_name):
        """Clicks subclient"""
        self.driver.find_element_by_xpath("//label[text()='" + subclient_name + "']/../..//input").click()

    @WebAction()
    def __click_submit(self):
        """Clicks submit"""
        self.driver.find_element_by_xpath("//button[@type='submit']").click()

    @WebAction
    def __click_restore(self):
        """Clicks restore"""
        self.driver.find_element_by_xpath("*//a[.='Restore']/..").click()

    @PageService()
    def select_nas_client(self, server_name):
        """
        Opens the NAS File Server with the given name

        Args:
            server_name  (basestring):  the name of the server to be opened

        """
        self.__table.access_link(server_name)

    @PageService()
    def add_nas_client(self, name, host_name, plan, vendor=None, **kwargs):
        """
        Adds a new NAS File Server with the Chosen iDAs and Access Nodes.

        Args:
            name        (basestring)    :   The  name of the NAS/Network Share client to be created.

            host_name   (basestring)    :   The host name of the NAS/Network Share client to be created.

            plan        (basestring)    :   The name of the plan that needs to be associated to the client.

            vendor      (Vendor(Enum))  :   The name of the vendor, supports following values.
                -   DELL_EMC_ISILON

            kwargs  (dict)              --  Optional arguments.

            Available kwargs Options:

                    array_details   (dict)  :   The dictionary of array details.
                     Dictionary contains the keys array_name, control_host, username and password.

                        array_name      (basestring)    :   Name of the array.

                        control_host    (basestring)    :   Control host name if applicable to the array.

                        username        (basestring)    :   Username for the array, defined in CoreUtils\config.json.

                        password        (basestring)    :   Password for the array, defined in CoreUtils\config.json.

                    cifs            (dict)  :   The dictionary of CIFS Agent details.
                    Dictionary contains the keys access_nodes, impersonate_user and content.

                        access_nodes        (list)  :   List of access node names, access node names are strings.

                        impersonate_user    (dict)  :   The dictionary of impersonation account details.
                        Dictionary contains the keys username and password.

                            username    (basestring)    :   Username of the account, defined in config.json.

                            password    (basestring)    :   Password of the account, defined in config.json.

                        content             (list)  :   List of content paths, content paths are strings.

        """

        self.admin_console.click_by_id("addFileServer")
        self.admin_console.access_sub_menu("NAS")

        self.admin_console.fill_form_by_id("clientName", name)
        self.admin_console.fill_form_by_id("hostName", len(name)*Keys.DELETE)
        self.admin_console.fill_form_by_id("hostName", host_name)

        # SELECT PLAN
        self.__drop_down.select_drop_down_values(values=[plan], drop_down_id='planSummaryDropdown')

        # EXPAND ACCORDION Network share configuration
        self.__modal_panel._expand_accordion("Network share configuration")

        # SELECT VENDOR IF PROVIDED
        if vendor:
            self.__modal_panel._expand_accordion("Add")
            self.__drop_down.select_drop_down_values(values=[vendor.value], drop_down_id='nas-vendor-dropdown')
            if kwargs.get('array_details', None):
                array = self._create_array(self.admin_console, vendor, kwargs.get('array_details', None))
                array.fill_array_details(kwargs['array_details'])
                self.admin_console.submit_form()

        if kwargs.get('cifs', None):
            cifs = CIFS(self.admin_console, kwargs.get('cifs', None))
            cifs.add_access_nodes()
            cifs.edit_impersonate_user()
            cifs.edit_content()

        if kwargs.get('nfs', None):
            nfs = NFS(self.admin_console, kwargs.get('nfs', None))
            nfs.add_access_nodes()
            nfs.edit_content()

        self.admin_console.submit_form()
        
    @PageService()
    def action_jobs(self, server_name):
        """
        Opens the job details page for the chosen server

        Args:
            server_name  (basestring):  the name of the server to be opened

        """
        self.__table.access_action_item(server_name, "Jobs")

    @PageService()
    def action_send_logs(self, server_name):
        """
        Send logs of a server with the specified name

        Args:
            server_name  (basestring):  the name of the server to be opened

        """
        self.__table.access_action_item(server_name, "Send logs")

    @PageService()
    def action_release_license(self, server_name):
        """"
        Release CIFS and NFS Licenses

        Args:
            server_name  (basestring):  the name of the server to be opened

        """
        self.__table.access_action_item(server_name, "Release license")
        if "Server File System - Linux File System" in self.__get_text():
            self.admin_console.heckbox_select("Server File System - Linux File System")
        if "Server File System - Windows File System" in self.__get_text():
            self.admin_console.checkbox_select("Server File System - Windows File System")
        if "NDMP - NDMP" in self.__get_text():
            self.admin_console.checkbox_select("NDMP - NDMP")
        self.admin_console.click_button("OK")
        #self.log.info("Release license successful on Client:" + server_name)

    @PageService()
    def action_backup(self, server_name, idataagent, subclient_name, backup_type):
        """"
        Running backup of subClients under defaultbackupset

        Args:
            server_name  (basestring):  the name of the server to be opened

            idataagent  (basestring):   the name of the iDA (NDMP or CIFS or NFS)

            subclient_name (basestring):    name of the subclient to be backed up
            backup_type (BackupType):   the backup type to be run, among the type in Backup.BackupType enum

        Returns (int) : the backup job ID
        """

        self.__table.search_for(server_name)
        self.__click_backup()
        self.admin_console.wait_for_completion()
        if idataagent == 'NDMP':
            #self.log.info("Finding for SubClient under NDMP Agent")
            self.__click_subclient(subclient_name)
            self.admin_console.wait_for_completion()
        elif idataagent == 'CIFS':
            #self.log.info("Finding for subClient under CIFS Agent")
            self.__click_subclient(subclient_name)
            self.admin_console.wait_for_completion()
        elif idataagent == 'NFS':
            #self.log.info("Finding for subClient under NFS Agent")
            self.__click_subclient(subclient_name)
            self.admin_console.wait_for_completion()
        else:
            raise Exception("Invalid iDA name passed")
        self.__click_submit()
        self.admin_console.wait_for_completion()
        backup = Backup(self)
        return backup.submit_backup(backup_type)

    @PageService()
    def action_restore(self, server_name, idataagent, subclient_name):
        """"
            Running Restore of Data backed up by SubClient under defaultbackupSet of NAS Client

            Args:
                server_name  (basestring):  the name of the server to be opened

                idataagent  (basestring):   the name of the iDA (NDMP or CIFS or NFS)

                subclient_name (basestring):    name of the subclient to be backed up

        """

        #self.log.info("Checking Action Item.....")
        self.__table.search_for(server_name)
        self.__click_restore()
        self.admin_console.wait_for_completion()
        #self.log.info("Clicked on Restore")
        if "Backup content" in self.__get_text():
            #self.log.info("Looks like there is only one backed up subclient")
            self.admin_console.wait_for_completion()
        elif "Please select a subclient or instance to restore" in self.__get_text():
            #self.log.info("Multiple subClients present")
            if idataagent == 'NDMP':
                #self.log.info("SubClient  under NDMP Agent")
                self.__click_subclient(subclient_name)
                self.admin_console.wait_for_completion()
            elif idataagent == 'Windows File System':
                #self.log.info("Finding for subClient under CIFS Agent")
                self.__click_subclient(subclient_name)
                self.admin_console.wait_for_completion()
            elif idataagent == 'Linux File System':
                self.__click_subclient(subclient_name)
                self.admin_console.wait_for_completion()
            else:
                #self.log.info("Invalid iDA passed")
                raise Exception("Invalid Agent Passed")
            self.admin_console.click_button("Select")
        else:
            #self.log.info("Looks like there is nothing to Restore")
            return

    @PageService()
    def delete_client(self, server_name):
        """
            Deletes Deconfigured NAS NDMP Client

            Args:
                server_name  (basestring):  the name of the server to be opened

        """

        self.__table.access_action_item(server_name, "Delete")
        #self.log.info("Proceeding with Client Deletion")
        self.admin_console.click_button("Yes")


class __Array:
    """
    This class contains all the methods for action on the Add Array Page.
    """

    @PageService()
    def fill_array_details(self, array_details):
        """
        Fill in all the array details.

        Args:
            array_details   (dict)  :   Dictionary of array details, refer to add_nas_client() docstring.

        """
        self.admin_console.fill_form_by_id("arrayNameInput", array_details['array_name'])
        self.admin_console.fill_form_by_id("arrayUserName", array_details['username'])
        self.admin_console.fill_form_by_id("arrayPassword", array_details['password'])


class DellEMCIsilonArray(__Array):
    """
    Class for Dell EMC Isilon Array Vendor.
    """
    def __init__(self, admin_console, array_details):
        self.admin_console = admin_console
        self.array_details = array_details

    @PageService()
    def fill_array_details(self, array_details):
        """
        Fill in all the array details.

        Args:
            array_details   (dict)  :   Dictionary of array details, refer to add_nas_client() docstring.

        """
        super().fill_array_details(self.array_details)
        self.get_control_host_element().send_keys(self.array_details['control_host'])

    @WebAction()
    def get_control_host_element(self):
        """
        Returns the control host element.
        """
        return self.admin_console.driver.find_elements_by_name("arrayNameInput")[1]


class NutanixFilesArray(__Array):

    def __init__(self, admin_console, array_details):
        self.admin_console = admin_console
        self.array_details = array_details


class NetApp(__Array):
    """
    Class for NetApp Array Vendor.
    """
    def __init__(self, array_details, driver):
        raise NotImplementedError

