# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""
This module provides the function or operations that can be performed on
on the Laptops page in admin console

Class:

    Laptops()

Functions:

_init_()                        :     initialize the class object

__set_host_name()               :     fill the hostname field with the specified name

__set_user_name()               :     fill the username field with the specified user name

__set_password()                :     fill the password field with the value

__set_confirm_password()        :     fill the confirm password field with the value

__select_os_option()            :     select the radio button based on given input

__get_text_from_dialog()        :     Gets the notification text

__client_configuration_status   : Configuartion status of the client

get_jobid_from_screen()         :     Gets the job id from screen

get_client_configuration_status() : to get the client configuration status

check_if_error_text()           :     Read the error text from popup

add_windows_laptop()            :     Method to add new windows laptop

add_mac_laptop()                :     Method to add new mac laptop

deactivate_laptop()             :     Method to deactivate the laptop

action_update_software()        :     To invoke update software operation in the laptop page

action_check_readiness()        :     To run check readiness operation on the given client in Laptop page

action_send_logs()              :     To perform send logs operation on the given client in Laptop page

action_retire_client()          :     Retire the given client

activate_laptop_byuser()        :     activate the laptop by enduser

activate_laptop_byplan()        :     activate the laptop by given plan name

subclient_backup_now()          :    To perform backup operation

cancel_the_form()               :    To perform cancel operation

laptop_summary_info()           :    get the summary info from admin console laptop page

restore_from_job()              :    Method for job based restore

restore_from_actions()          :    Method to Restore the given client data from actions

restore_from_details_page()     :    Method to Restore the given client data from details page

"""

import re
import time
from AutomationUtils import logger
from Web.Common.page_object import (WebAction, PageService)
from Web.AdminConsole.Components.dialog import ModalDialog
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.Components.panel import PanelInfo, Backup
from Web.AdminConsole.Components.panel import DropDown
from Web.AdminConsole.Components.browse import Browse, ContentBrowse
from Web.AdminConsole.FileServerPages.file_servers import RestorePanel
from Web.AdminConsole.AdminConsolePages.Jobs import Jobs
from Web.Common.exceptions import CVWebAutomationException


class Laptops:
    """
    Class for Laptops page
    """

    def __init__(self, admin_page):
        """
        Method to initiate the Laptops class

        Args:
            admin_page   (Object) :   Admin Page Class object
        """
        self.__admin_console = admin_page
        self.__driver = admin_page.driver
        self.__admin_console._load_properties(self)
        self.__dialog = ModalDialog(admin_page)
        self.__table = Table(admin_page)
        self.__browse = Browse(admin_page)
        self.__content_browse = ContentBrowse(admin_page)
        self.__panel = PanelInfo(admin_page)
        self.__backup = Backup(admin_page)
        self.__navigator = admin_page.navigator
        self.__drop_down = DropDown(admin_page)
        self.__jobs = Jobs(admin_page)
        self.__restore_panel = RestorePanel(admin_page)

        self.log = logger.get_log()

    @WebAction()
    def __set_host_name(self, host_name):
        """
        Method to set the hostname

        Args:
            host_name (string): Machine Host name

        """
        self.__admin_console.fill_form_by_id("hostName", host_name)

    @WebAction()
    def __set_user_name(self, user_name):
        """
        Method to set username

        Args:
            user_name (string): Machine User name

        """
        self.__admin_console.fill_form_by_id("fakeUserName", user_name)

    @WebAction()
    def __set_password(self, password):
        """
        Method to set password

        Args:
            password (string): Machine password

        """
        self.__admin_console.fill_form_by_id("fakePassword", password)

    @WebAction()
    def __set_confirm_password(self, confirm_password):
        """
        Method to set confirm password

        Args:

            confirm_password (string): confirm password

        """
        self.__admin_console.fill_form_by_id("confirmPassword", confirm_password)

    @WebAction()
    def __select_os_option(self, value):
        """
        Method to select the radio button based on "Value"

        Args:

            value (string): value of the element

        """

        self.__driver.find_element_by_xpath(
            "//input[@type='radio' and @value='" + value + "']").click()

    @WebAction()
    def __get_text_from_dialog(self):
        """
        Gets the notification text

        Returns:

            notification_text (basestring): the notification string

        """
        xpath = "//div[@class='modal-content']/div/div/div/p"
        if self.__admin_console.check_if_entity_exists("xpath", xpath):
            return self.__driver.find_element_by_xpath(xpath).text
        xpath = "//div[@class='modal-content']/div/div/div/div"
        return self.__driver.find_element_by_xpath(xpath).text

    @WebAction()
    def __client_configuration_status(self):
        """
        Get configuration status of the client"
       """

        xpath = f"//td[@id='cv-k-grid-td-deviceSummary.isClientDeconfigured']"\
        f"//span[contains(@class,'k-icon')and @cv-toggle='tooltip']"

        elem = self.__driver.find_element_by_xpath(xpath)

        return elem.get_attribute('cv-toggle-content')

    @PageService()
    def get_jobid_from_screen(self):
        """
        Gets the job id from screen

        Returns:

            job_id (int):  the job id for the submitted request

        """
        job_text = self.__get_text_from_dialog()
        if not job_text:
            raise Exception("No notification is popped up to extract job id")
        job_id = re.findall(r'\d+', job_text)[0]
        self.log.info("Job %s has started", str(job_id))
        return job_id

    @PageService()
    def get_client_configuration_status(self, client_name):
        """
        Gets the clint configuration status

        Returns:

        """
        self.__navigator.navigate_to_devices()
        self.__admin_console.wait_for_completion()
        self.__table.apply_filter_over_column("Name", client_name)
        client_status = self.__client_configuration_status()
        return client_status

    @PageService()
    def add_windows_laptop(self,
                           host_name,
                           user_name,
                           password=None,
                           confirm_password=None):
        """
        Method to add new windows laptop

        Args:
            host_name (string): hostname of the laptop

            user_name (string): user_name of the laptop

            password (string): password of the laptop

            confirm_password (string): password of the laptop

        Returns:
            None

        Raises:
            Exception:
                if failed to add laptop

        """
        self.__admin_console.select_hyperlink('Add laptop')
        self.__select_os_option('WINDOWS')
        self.__set_host_name(host_name)
        self.__set_user_name(user_name)
        self.__set_password(password)
        self.__set_confirm_password(confirm_password)

        self.__admin_console.submit_form()
        job_id = self.get_jobid_from_screen()
        self.__admin_console.submit_form()
        self.__admin_console.check_error_message()
        return job_id

    @PageService()
    def add_mac_laptop(self,
                       host_name,
                       user_name,
                       password=None,
                       confirm_password=None):
        """
        Method to add new mac laptop

        Args:
            host_name (string): hostname of the laptop

            user_name (string): user_name of the laptop

            password (string): password of the laptop

            confirm_password (string): password of the laptop

        Returns:
            None

        Raises:
            Exception:
                if failed to add laptop

        """
        self.__admin_console.select_hyperlink('Add laptop')
        self.__select_os_option('UNIX_LINUX')

        self.__set_host_name(host_name)
        self.__set_user_name(user_name)
        self.__set_password(password)
        self.__set_confirm_password(confirm_password)

        self.__admin_console.submit_form()
        job_id = self.get_jobid_from_screen()
        self.__admin_console.submit_form()
        self.__admin_console.check_error_message()
        return job_id


    @PageService()
    def deactivate_laptop(self, client_name):
        """
        Method to deactivate the laptop

        Args:

            client_name (string): client_name of the laptop

        Returns:
            None

        Raises:
            Exception:
                -- if fails to deactivate laptop
        """
        self.__table.access_action_item(client_name, 'Deactivate')
        self.__admin_console.click_button('Yes')
        self.__admin_console.check_error_message()

    @PageService()
    def action_update_software(self, client_name=None, reboot=False):
        """
        To invoke update software method in the laptop page

        Args:
            client_name     (str) -- client to update software on

            reboot         (bool)    -- set to True if reboot required
                                            default: False

        Returns:
            None

        Raises:
            Exception
                --if given input is invalid

                --if there is no update software option for the client

                --if Job failed

        """

        self.__table.access_action_item(client_name, 'Update software')
        if reboot:
            self.__admin_console.toggle_enable('forceReboot')

        self.__admin_console.click_button('Yes')
        job_id = self.get_jobid_from_screen()
        self.__admin_console.click_button("OK")
        self.__admin_console.check_error_message()
        return job_id


    @PageService()
    def action_check_readiness(self, client_name=None):
        """To run check readiness operation on the given client in Laptop page

        Args:
            client_name     (str) -- client on which check readiness to be performed

            Returns:
               None

            Raises:
                Exception

                    if given input is invalid

                    if there is an error message after submitting the request

        """

        self.__table.access_action_item(client_name, 'Check readiness')
        self.__admin_console.check_error_message()


    @PageService()
    def action_retire_client(self, client_name):
        """Retire the given client

        Args:

        client_name     (str) -- client which need to be retired

        Returns:
            None

        Raises:
            Exception
                if client_name is invalid

                if there is an error message after submitting the requestt
        """
        self.__table.access_action_item(client_name, 'Retire')
        self.__admin_console.fill_form_by_id('typedConfirmationRetire', 'Retire')
        self.__admin_console.click_button('Retire')
        job_id = self.__admin_console.get_jobid_from_popup(wait_time=10)
        self.__admin_console.check_error_message()
        return job_id

    @PageService()
    def activate_laptop_byuser(self, client_name):
        """activate the laptop by enduser

        Args:

        client_name     (str) -- client which need to be activated

        Returns:
            None

        Raises:
            Exception
                if client_name is invalid

                if there is an error message after submitting the requestt
        """

        self.__table.access_action_item(client_name, 'Activate')
        self.__admin_console.select_radio('activateByUser')
        self.__admin_console.submit_form()

    @PageService()
    def activate_laptop_byplan(self, client_name, plan_name):
        """activate the laptop by given plan name

        Args:

        client_name     (str) -- client which need to be activated

        Returns:
            None

        Raises:
            Exception
                if client_name is invalid

                if there is an error message after submitting the requestt
        """
        self.__table.access_action_item(client_name, 'Activate')
        self.__admin_console.select_radio('activateByPlan')
        if plan_name:
            self.__drop_down.select_drop_down_values(0, [plan_name])
        self.__admin_console.submit_form()
        details = self.__panel.get_details()
        self.log.info(details)

    @PageService()
    def subclient_backup_now(self, client_name, backup_type):
        """To perform backup operation

        Args:

        client_name     (str) -- client which need to be retired

        Backup_type    (str)  -- Type the of backup to be performed on client
        (INCR , FULL, SYNTHETIC FULL)

        Returns:
            None

        Raises:
            Exception
                if client_name is invalid

                if there is an error message after submitting the request
        """
        self.__table.access_link(client_name)
        self.__admin_console.wait_for_completion(600)
        i = 1
        while i<=5:
            if self.__admin_console.check_if_entity_exists("link", 'Back up now'):
                break
            time.sleep(60)
            i=i+1
        self.__admin_console.refresh_page()
        self.__admin_console.select_hyperlink('Back up now')
        job_id = self.__backup.submit_backup(backup_type)
        self.log.info("backup job {0} started to for the client {1}".format(job_id, client_name))
        return job_id


    @PageService()
    def cancel_the_form(self):
        """To perform cancel operation

        Raises:
            Exception

                if unable to click on cancel button
        """

        self.__admin_console.cancel_form()

    @PageService()
    def laptop_summary_info(self, client_name):
        """get the summary info from admin console laptop page"""
        self.__navigator.navigate_to_devices()
        self.__admin_console.wait_for_completion()
        self.__table.apply_filter_over_column("Name", client_name)
        summary_info_dict = self.__table.get_table_data()
        summary_info_dict.pop('Configured')
        summary_info_dict.pop('Actions')
        client_status = self.__client_configuration_status()
        summary_info_dict['Status'] = [client_status]
        return summary_info_dict

    @PageService()
    def restore_from_job(self, jobid, source_data_path=None, dest_path=None):
        """
        Method for job based restore

        Args:

             Jobid (str)      : job id to restore the data

             source_data_path(str)   : Sourece data of teh client to be restored

             dest_path(str)         : Destination place to restore the data

        Returns :
              job id  (int) : Job id of the restore job

        Raises:
            Exception :
             -- if fails to run the restore operation

        """
        backup_jobid = str(jobid)
        self.__navigator.navigate_to_jobs()
        if self.__jobs.if_job_exists(backup_jobid):
            self.__jobs.access_job_by_id(backup_jobid)
        else:
            self.__jobs.access_job_history()
            if not self.__jobs.if_job_exists(backup_jobid):
                raise CVWebAutomationException("jobs is not present in Active jobs or job history")
        self.__jobs.initiate_job_based_subclient_restore(backup_jobid)
        self.__browse.select_path_for_restore(source_data_path)
        self.__browse.submit_for_restore()
        if dest_path:
            self.__admin_console.checkbox_deselect('inplace')
            self.__restore_panel._click_browse()
            self.__admin_console.wait_for_completion()
            self.__content_browse.select_path(dest_path)
            self.__content_browse.save_path()
        self.__admin_console.submit_form()
        restore_job_id = self.__admin_console.get_jobid_from_popup()
        return restore_job_id

    @PageService()
    def restore_from_actions(self, client_name, source_data_path=None, dest_path=None):
        """
        Method to Restore the given client data from actions

        Args:

             client_name (str)      : Name of the client

             source_data_path(str)   : Sourece data of teh client to be restored

             dest_path(str)         : Destination place to restore the data

        Returns :
              job id  (int) : Job id of the restore job

        Raises:
            Exception :
             -- if fails to run the restore operation

        """
        self.__navigator.navigate_to_devices()
        self.__table.access_action_item(client_name, 'Restore')
        self.__admin_console.wait_for_completion()
        self.__browse.select_path_for_restore(source_data_path)
        self.__browse.submit_for_restore()
        if dest_path:
            self.__admin_console.checkbox_deselect('inplace')
            self.__restore_panel._click_browse()
            self.__admin_console.wait_for_completion()
            self.__content_browse.select_path(dest_path)
            self.__content_browse.save_path()
        self.__admin_console.submit_form()
        restore_job_id = self.__admin_console.get_jobid_from_popup()
        return restore_job_id

    @PageService()
    def restore_from_details_page(self, client_name, source_data_path=None, dest_path=None):
        """
        Method to Restore the given client data from details page

        Args:

             client_name (str)      : Name of the client

             source_data_path(str)   : Sourece data of teh client to be restored

             dest_path(str)         : Destination place to restore the data

        Returns :
              job id  (int) : Job id of the restore job

        Raises:
            Exception :
             -- if fails to run the restore operation

        """
        self.__navigator.navigate_to_devices()
        self.__table.access_link(client_name)
        self.__admin_console.wait_for_completion()
        self.__admin_console.select_hyperlink('Restore')
        self.__admin_console.wait_for_completion()
        self.__browse.select_path_for_restore(source_data_path)
        self.__browse.submit_for_restore()
        if dest_path:
            self.__admin_console.checkbox_deselect('inplace')
            self.__restore_panel._click_browse()
            self.__admin_console.wait_for_completion()
            self.__content_browse.select_path(dest_path)
            self.__content_browse.save_path()
        self.__admin_console.submit_form()
        restore_job_id = self.__admin_console.get_jobid_from_popup()
        return restore_job_id

