# -*- coding: utf-8 -*-
# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""
This module provides the function or operations that can be performed on the
LaptopDetails Page on the AdminConsole

Class:

    LaptopDetails ()

Functions:

extract_laptop_info()                -- Extracts and returns contained information from the laptop

laptop_info()                        -- Returns all information about the laptop


"""

from Web.Common.page_object import (WebAction, PageService)
from Server.JobManager.jobmanager_helper import JobManager
from AutomationUtils import logger
from AutomationUtils.idautils import CommonUtils
from Web.AdminConsole.Components.table import Table
from Web.AdminConsole.Components.table import CVTable
from Web.AdminConsole.Components.panel import PanelInfo
from Web.AdminConsole.Components.panel import DropDown
from Web.AdminConsole.Components.panel import ModalPanel


class LaptopDetails:
    """
    This class provides the operations that can be performed on LaptopDetails Page of
    admin console
    """
    def __init__(self, admin_page, commcell):
        """
        Method to initiate Laptop class

        Args:
            admin_page   (Object) :   Admin Page Class object
        """
        self.__admin_console = admin_page
        self.__commcell = commcell
        self.__table = Table(self.__admin_console)
        self.__cv_table = CVTable(self.__admin_console)
        self.__driver = admin_page.driver
        self.__drop_down = DropDown(self.__admin_console)
        self.__model_panel = ModalPanel(self.__admin_console)
        self.__job_manager = JobManager(commcell=self.__commcell)
        self.__ida_utils = CommonUtils(self.__commcell)
        self.__navigator = admin_page.navigator
        self.last_online_time = None
        self.last_backupsize = None
        self.log = logger.get_log()


    @WebAction()
    def __extract_laptop_info(self):
        """
        Extracts all the information about the laptop

        Args:
            None

        Returns:
            laptop_info (dict) : all info about laptop

        Raises:
            Exception:
                -- if fails to return laptop info
        """
        laptop_info = {}
        elements = self.__admin_console.driver.find_elements_by_xpath(
            "//div[@class='page-details group']")

        for elem in elements:
            key = elem.find_element_by_xpath("./span").text
            if key == 'Summary':
                summary_values = PanelInfo(self.__admin_console, key).get_details()

                laptop_info.update({key: summary_values})

            elif key == 'Security':
                values_list = []
                div_elements = elem.find_elements_by_xpath("./div[@class='tile-accordion ng-scope']")
                for div_elem in div_elements:
                    owners_rows = div_elem.find_elements_by_xpath(
                        "//div[@id='tileContent_Security']/div/cv-tabset-component/div/div/ul/li")
                    for each_owner in owners_rows:
                        val = each_owner.find_element_by_xpath("./span").text
                        values_list.append(val)
                laptop_info.update({key: values_list})

            elif key == 'Schedules':
                schedule_values = PanelInfo(self.__admin_console, key).get_details()
                laptop_info.update({key: schedule_values})

            elif key == 'Content':
                values_list = elem.find_element_by_xpath(
                    "./div[@class='tile-accordion ng-scope']").text.split(',')
                laptop_info.update({key: values_list})

            elif key == 'Plan':
                values_list = []
                values_list = elem.find_element_by_xpath(
                    "./div[@class='tile-accordion ng-scope']").text
                laptop_info.update({key: values_list})

        return laptop_info

    @PageService()
    def laptop_info(self, client_name):
        """
        collates and returns all the information about the Laptop

        Returns:
            displayed_val(dict) : displayed values of the laptop

        Raises:
            Exception:
                -- if fails to return displayed_val
        """
        self.__navigator.navigate_to_devices()
        self.__table.access_link(client_name)
        self.__admin_console.wait_for_completion()
        displayed_val = self.__extract_laptop_info()
        self.__admin_console.log.info(displayed_val)
        return displayed_val


