# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""
This module provides all the methods to perform actions on k8s clusters.

Clusters  --  This class contains methods to perform actions on k8s clusters like
              opening a cluster, restore, view jobs, Retire etc.

    Functions:

    add_cluster()        --  add clusters

    open_cluster()       --  Opens cluster with the given name

    action_retire()      --  Retire the k8s cluster

    action_view_jobs()        --    Displays all the jobs of the given cluster

    action_restore()     -- Restores the given cluster

    action_change_company()    -- Change company

    run_validate_backup()   --  Runs backup validation job

    add_k8s_cluster(self)   -- add kubernetes clusters

    is_cluster_exists()     --  check if cluster exists

    delete_cluster_name()    --  delete clusters

    access_cluster_tab()    --  access cluster tab

    access_application_group_tab()    --  access application groups tab

    access_application_tab()    -- access cluster tab

K8sServer:

    set_hostname()    -- set api server endpoint

    set_servername()    --  set server name

    set_authentication()    --  set authentication type

    set_service_account()    -- set service account

    set_service_token()    -- set service token

    set_access_nodes()    -- select access nodes

    save()    --  click save button on add cluster screen

    click_add_application_group()    -- on successfully added cluster screen, click
                                        add application group

    click_skip_for_now()    -- on successfully added cluster screen, click skip for now
"""
from Web.AdminConsole.Components.panel import DropDown, ModalPanel
from Web.AdminConsole.Components.table import Table
from Web.Common.page_object import PageService, WebAction


class K8SClusters:
    """
     This class contains methods to perform actions on k8s clusters like restore,
     view jobs, Retire...
    """

    def __init__(self, admin_console):
        self._admin_console = admin_console
        self.__table = Table(admin_console)

    @PageService()
    def add_cluster(self):
        """
        Selects Add Cluster
        """
        # local message only have "Add clusters"
        self._admin_console.access_menu("Add cluster")

    @PageService()
    def open_cluster(self, cluster_name):
        """
        Opens clusters with the given name

        Args:
            cluster_name     (basestring):   name of the cluster to open

        """
        self.__table.access_link(cluster_name)

    @PageService()
    def action_view_jobs(self, cluster_name):
        """
        Displays all the jobs of the given cluster

        Args:
            cluster_name  (basestring):  the name of the cluster whose jobs to open

        """
        self.__table.access_action_item(
            cluster_name, self._admin_console.props['action.jobs'])

    @PageService()
    def action_restore(self, cluster_name):
        """
        Restores the given cluster

        Args:
            cluster_name  (basestring):  the name of the cluster to restore

        """
        self.__table.access_action_item(
            cluster_name, self._admin_console.props['label.restore'])

    @PageService()
    def action_change_company(self, cluster_name):
        """
        Restores the given cluster

        Args:
            cluster_name  (basestring):  the name of the cluster to restore

        """
        self.__table.access_action_item(
            cluster_name, self._admin_console.props['action.commonAction.changeCompany'])

    @PageService()
    def add_k8s_cluster(self):
        """Add kubernetes cluster"""
        self.add_cluster()
        return K8sServer(self._admin_console)

    @PageService()
    def is_cluster_exists(self, server_name):
        """Check if cluster exists"""
        return self.__table.is_entity_present_in_column(
            self._admin_console.props['label.name'], server_name)

    @PageService()
    def delete_cluster_name(self, servername):
        """Delete clusters
        Args:
            servername     (String)       --     cluster name
        """
        self.__table.access_action_item(
            servername, self._admin_console.props['action.retire'])
        self._admin_console.fill_form_by_id(
            "deleteTypedConfirmation",
            self._admin_console.props['action.retire'])
        self._admin_console.click_button(
            self._admin_console.props['action.retire'])
        self._admin_console.wait_for_completion()
        self._admin_console.refresh_page()
        if self.is_cluster_exists(servername):
            self.__table.access_action_item(
                servername, self._admin_console.props['action.delete'])
            self._admin_console.fill_form_by_id(
                "typedConfirmationRetire",
                self._admin_console.props['action.delete'])
            self._admin_console.click_button(
                self._admin_console.props['action.delete'])
            self._admin_console.wait_for_completion()

    @PageService()
    def access_cluster_tab(self):
        """
        Access cluster tab
        """
        self._admin_console.access_tab(
            self._admin_console.props['label.nav.clusters'])

    @PageService()
    def access_application_group_tab(self):
        """
        Access application group tab
        """
        self._admin_console.access_tab(
            self._admin_console.props['label.nav.applicationGroups'])

    @PageService()
    def access_applications_tab(self):
        """
        Access applications tab
        """
        self._admin_console.access_tab(
            self._admin_console.props['label.nav.applications'])


class K8sServer(ModalPanel):
    """Add k8s cluster"""

    def __init__(self, admin_console):
        super().__init__(admin_console)
        self.__dropdown = DropDown(admin_console)

    @PageService()
    def set_hostname(self, api_server_endpoint):
        """
        Set api server endpoint
        Args:
            api_server_endpoint   (String)      -- Set API server endpoint (hostname)
        """
        self._admin_console.fill_form_by_id('hostname', api_server_endpoint)

    @PageService()
    def set_servername(self, server_name):
        """
        Set server name
        Args:
            server_name        (String)      -- Set k8s cluster server name
        """
        self._admin_console.fill_form_by_id('serverName', server_name)

    @PageService()
    def select_authentication(self, authentication_type):
        """Select Authentication type
        Args:
            authenticaiton_type      (String)       -- Set Authentication type
        """
        self.__dropdown.select_drop_down_values(
            drop_down_id='authenticationType',
            values=[authentication_type])

    @PageService()
    def set_service_account(self, serviceaccount):
        """Set service account
        Args:
            serviceaccount             (String)         --     service account
        """
        self._admin_console.fill_form_by_id('serviceName', serviceaccount)

    @PageService()
    def set_service_token(self, servicetoken):
        """Set service token
        Args:
            servicetoken            (String)         --     service token
        """
        self._admin_console.fill_form_by_id('serviceToken', servicetoken)

    @PageService()
    def select_access_nodes(self, access_node):
        """Select access node
        Args:
            access_node      (String)       -- Set access nodes
        """
        defaultvalue = (self.__dropdown.get_values_of_drop_down(
            "selectProxy_isteven-multi-select_#6148"))[0]
        self.__dropdown.select_drop_down_values(
            drop_down_id='selectProxy_isteven-multi-select_#6148',
            values=[access_node])
        if defaultvalue.lower() != access_node.lower():
            self.__dropdown.deselect_drop_down_values(1, [defaultvalue])

    @PageService()
    def save(self):
        """Click save"""
        self._admin_console.click_button(
            self._admin_console.props['action.save'])
        if not self._admin_console.get_error_message():
            self.click_add_application_group()

    @PageService()
    def click_add_application_group(self):
        """on successfully added cluster screen, Click Add application group"""
        self._admin_console.click_button('Add Application group')

    @PageService()
    def click_skip_for_now(self):
        """Click Skip for now"""
        self._admin_console.driver.find_element_by_link_text(
            self._admin_console.props['action.skip']).click()
